/* 
 *   Creation Date: <2001/05/05 23:33:49 samuel>
 *   Time-stamp: <2001/05/24 02:46:03 samuel>
 *   
 *	<hfsp_fs.c>
 *	
 *	HFS+ file system interface (and ROM lookup support)
 *   
 *   Copyright (C) 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

#include "libhfsp.h"
#include "volume.h"
#include "record.h"
#include "unicode.h"
#include "blockiter.h"
#include "fs.h"

#define MAC_OS_ROM_CREATOR	0x63687270	/* 'chrp' */
#define MAC_OS_ROM_TYPE		0x74627869	/* 'tbxi' */
#define MAC_OS_ROM_NAME		"Mac OS ROM"

#define FINDER_TYPE		0x464E4452	/* 'FNDR' */
#define FINDER_CREATOR		0x4D414353	/* 'MACS' */
#define SYSTEM_TYPE		0x7A737973	/* 'zsys' */
#define SYSTEM_CREATOR		0x4D414353	/* 'MACS' */


typedef struct {
	record		rec;
	char		*path;

	off_t		pos;
} hfsp_file_t;


/************************************************************************/
/*	File System Interface						*/
/************************************************************************/

static void		close_fs( fs_ops_t *fs );
static file_desc_t 	open_path( fs_ops_t *fs, char *path );
static file_desc_t	search_rom( fs_ops_t *fs );
static file_desc_t 	search_file( fs_ops_t *fs, char *name );

static void		file_close( file_desc_t fd);
static int		file_read( file_desc_t fd, void *buf, size_t count );
static int		file_lseek( file_desc_t fd, off_t offset, int whence );
static char		*get_path( file_desc_t fd, char *buf, int len );

static fs_ops_t fs_ops = {
	close_fs:	close_fs,
	open_path:	open_path,
	search_rom:	search_rom,
	search_file:	search_file,

	get_path:	get_path,
	close:		file_close,
	read:		file_read,
	lseek:		file_lseek
};


/************************************************************************/
/*	Search implementation						*/
/************************************************************************/

typedef int (*match_proc_t)( record *r, record *parent, void *match_data, hfsp_file_t *pt );

static int
search_files( record *par, int recursive, match_proc_t proc, void *match_data, hfsp_file_t *pt )
{
	hfsp_file_t t;
	record r;
	int ret = 1;

	t.path = NULL;

	record_init_parent( &r, par );
	do{
		if( r.record.type == HFSP_FOLDER || r.record.type == HFSP_FILE )
			ret = (*proc)( &r, par, match_data, &t );

		if( ret && r.record.type == HFSP_FOLDER && recursive )
			ret = search_files( &r, 1, proc, match_data, &t );

	} while( ret && !record_next(&r) );

	if( !ret && pt ) {
		char name[256];
		char *s2 = t.path ? t.path : "";
		unicode_uni2asc( name, &r.key.name, sizeof(name));

		pt->rec = t.rec;
		pt->path = malloc( strlen(name) + strlen(s2) + 2 );
		strcpy( pt->path, name );
		if( strlen(s2) ) {
			strcat( pt->path, "/" );
			strcat( pt->path, s2 );
		}
	}

	if( t.path )
		free( t.path );

	return ret;
}

static int
root_search_files( fs_ops_t *fs, int recursive, match_proc_t proc, void *match_data, hfsp_file_t *pt )
{
	volume *vol = (volume*)fs->fs_data;
	record r;

	record_init_root( &r, &vol->catalog );
	return search_files( &r, recursive, proc, match_data, pt );
}


static int 
match_file( record *r, record *parent, void *match_data, hfsp_file_t *pt )
{
	char name[255];
	int ret;
	
	if( r->record.type != HFSP_FILE )
		return 1;

	(void) unicode_uni2asc(name, &r->key.name, sizeof(name));
	if( !(ret = strcasecmp( (char*)match_data, name)) && pt )
		pt->rec = *r;

	return ret;
}

static int
match_rom( record *r, record *par, void *match_data, hfsp_file_t *pt )
{
	hfsp_cat_file *file = &r->record.u.file;
	FInfo *fi = &file->user_info;
	int ret = 1;
	
	if( r->record.type == HFSP_FILE && fi->fdCreator == MAC_OS_ROM_CREATOR && fi->fdType == MAC_OS_ROM_TYPE ) {
		ret = (search_files( par, 0, match_file, "System", NULL )) 
			|| (search_files( par, 0, match_file, "Finder", NULL ));
		if( !ret && pt )
			pt->rec = *r;
	}
	return ret;
}

static int
match_path( record *r, record *par, void *match_data, hfsp_file_t *pt )
{
	char name[256], *s, *next, *org;
	int ret=1;

 	next = org = strdup( (char*)match_data );
	while( (s=strsep( &next, ":/" )) && !strlen(s) )
		;
	if( !s ) {
		free( org );
		return 1;
	}

	(void) unicode_uni2asc(name, &r->key.name, sizeof(name));

	if( !strcasecmp( s, name ) ) {
		if( r->record.type == HFSP_FILE && !next ) {
			if( pt )
				pt->rec = *r;
			ret = 0;
		} else /* must be a directory */
			ret = search_files( r, 0, match_path, next, pt );
	}
	free( org );
	return ret;
}


/************************************************************************/
/*	File System Operations						*/
/************************************************************************/

fs_ops_t *
fs_hfsp_open( unsigned long long offs, char *path )
{
	fs_ops_t *fs;
	volume *vol = malloc(sizeof(volume));

	if( volume_open( vol, path, offs ) ) {
		free( vol );
		return NULL;
	}

	fs = malloc( sizeof(fs_ops_t) );
	*fs = fs_ops;
	fs->fs_data = vol;
	fs->dev_name = strdup( path );
	
	return fs;
}

static void
close_fs( fs_ops_t *fs )
{
	volume *vol = (volume*)fs->fs_data;
	volume_close( vol );
	free( vol );
	free( fs->dev_name );
	free( fs );
}

static file_desc_t
_create_fops( hfsp_file_t *t ) 
{
	hfsp_file_t *r = malloc( sizeof(hfsp_file_t) );

	*r = *t;
	r->pos = 0;
	return (file_desc_t)r;
}

static file_desc_t
open_path( fs_ops_t *fs, char *path )
{
	hfsp_file_t t;

	if( !root_search_files( fs, 0, match_path, path, &t ))
		return _create_fops( &t );
	return NULL;
}

static file_desc_t
search_rom( fs_ops_t *fs )
{
	hfsp_file_t t;

	if( !root_search_files( fs, 1, match_rom, NULL, &t ))
		return _create_fops( &t );
	return NULL;
}

static file_desc_t
search_file( fs_ops_t *fs, char *name )
{
	hfsp_file_t t;

	if( !root_search_files( fs, 1, match_file, name, &t ))
		return _create_fops( &t );
	return NULL;
}


/************************************************************************/
/*	File Operations							*/
/************************************************************************/

static char *
get_path( file_desc_t fd, char *buf, int len )
{
	hfsp_file_t *t = (hfsp_file_t*)fd;
	if( !t->path )
		return NULL;

	strncpy( buf, t->path, len );
	buf[len-1] = 0;
	return buf;
}

void
file_close( file_desc_t fd )
{
	hfsp_file_t *t = (hfsp_file_t*)fd;
	if( t->path )
		free( t->path );
	free( t );
}

int
file_lseek( file_desc_t fd, off_t offs, int whence )
{
	hfsp_file_t *t = (hfsp_file_t*)fd;
	hfsp_cat_file *file = &t->rec.record.u.file;
	int total = file->data_fork.total_size;

	switch( whence ){
	case SEEK_CUR:
		t->pos += offs;
		break;
	case SEEK_END:
		t->pos = total + offs;
		break;
	default:
	case SEEK_SET:
		t->pos = offs;
		break;
	}

	if( t->pos < 0 )
		t->pos = 0;

	if( t->pos > total )
		t->pos = total;

	return t->pos;
}

int
file_read( file_desc_t fd, void *buf, size_t count )
{
	hfsp_file_t 	*t = (hfsp_file_t*)fd;
	volume 		*vol = t->rec.tree->vol;
	UInt32 		blksize = vol->blksize;
	hfsp_cat_file 	*file = &t->rec.record.u.file;
	blockiter 	iter;

	char 		buf2[blksize];
	int		act_count, curpos=0;

	blockiter_init( &iter, vol, &file->data_fork, HFSP_EXTENT_DATA, file->id );
	while( curpos + blksize < t->pos ) {
		if( blockiter_next( &iter ) )
			return -1;
		curpos += blksize;
	}
	act_count = 0;

	while( act_count < count ){
		UInt32 block = blockiter_curr(&iter);
		int max = blksize, add = 0, size;

		if( volume_readinbuf( vol, buf2, block ) )
			break;

		if( curpos < t->pos ){
			add += t->pos - curpos;
			max -= t->pos - curpos;
		}
		size = (count-act_count > max)? max : count-act_count;
		memcpy( buf + act_count, &buf2[add], size );
		
		curpos += blksize;
		act_count += size;

		if( blockiter_next( &iter ) )
			break;
	}

	t->pos += act_count;
	return (act_count > 0)? act_count : -1;
}


