/*
**	new_videocard.c 11/13/97 Mikael Forselius <mikaelf@comenius.se>
**	Inspiration and code from Brian Gaeke <brg@dgate.org>.
**
**	This file is subject to the terms and conditions of the GNU General Public
**	License.  See the file COPYING in the main directory of this archive
**	for more details.
**
**	Does the enabling and disabling of video card interrupts.
**	Compatible with internal (slot 0) and external (NuBus) video cards.
**
**	Main entry points are:
**
**	turn_on_interrupts(void)
**		Enable interrupts for all video cards found
** 
**	turn_off_interrupts(void)
**		Disable interrupts for all video cards found
** 
*/

#include <stdio.h>
#include <string.h>
#include <Slots.h>
#include <ROMDefs.h>
#include <Video.h>
#include <Errors.h>
#include <Traps.h>
#include <Devices.h>
#include <StringCompare.h>
#include <Aliases.h>
#include "penguin_prototypes.h"
#include "penguin_utils.h"
#include "bootstrap_prototypes.h"
#include "videocard_prototypes.h"

typedef int		(*card_callback)(SInt8, SInt8, SInt8, short);

static int		slot_scan(card_callback card_fn);

static int		slot_int_get(SInt8 spSlot, SInt8 spID, SInt8 spExtDev, short refNum);
static int		slot_int_set(SInt8 spSlot, SInt8 spID, SInt8 spExtDev, short refNum, SInt8 intOn);

static int		card_int_on(SInt8 spSlot, SInt8 spID, SInt8 spExtDev, short refNum);
static int		card_int_off(SInt8 spSlot, SInt8 spID, SInt8 spExtDev, short refNum);
static int		card_int_show(SInt8 spSlot, SInt8 spID, SInt8 spExtDev, short refNum);

static short	is_buggy_iici(short ref_num);
static short	get_driver_info(short ref_num, Str255 d_name, short *d_version);
static short	patch_iici_driver(short ref_num);

/*
 * turn_on_interrupts
 *
 * Turn on interrupts for all slot video drivers
 *
 */
void
turn_on_interrupts(void)
{
	int		result;

	if (have_trap(_SlotManager)) {
		result = slot_scan(card_int_on);
		if (result != 0) {
			cprintf("Warning: %d error(s) while turning ON slot interrupts...booting anyway!\n", result);
		}
	}
}

/*
 * turn_off_interrupts
 *
 * Turn off interrupts for all slot video drivers
 *
 */
void
turn_off_interrupts(void)
{
	int		result;
	
	if (have_trap(_SlotManager)) {
		result = slot_scan(card_int_off);
		if (result != 0) {
			cprintf("Warning: %d error(s) while turning OFF slot interrupts...booting anyway!\n", result);
		}
	}
}

/*
 * slot_scan
 *
 * Scan through slot drivers, calling card_fn for each
 * video driver found. Handles slot 0x0 and 0x9 to 0xE
 *
 * card_fn : function to call for each slot driver found
 *
 * Only looks for:
 *	category	: catDisplay
 *  type		: typeVideo
 *	software	: drSwApple
 *
 */
int
slot_scan(card_callback card_fn)
{
	SpBlock		spBlk;
	OSErr		err;
	int			result;

	result = 0;

	/* Initialize parameter block
	 * Set search mask to include spCategory, spCType and spDrvrSW
	 * This is exactly what the ROM code does at system startup
	 */
	memset(&spBlk, 0, sizeof(spBlk));
	spBlk.spSlot = 0;
	spBlk.spID = 0;
	spBlk.spExtDev = 0;

	/* Loop through all slot drivers
	 * Calling card_fn() for each driver found
	 */
	err = noErr;
	while(err == noErr) {
		err = SNextSRsrc(&spBlk);
		if (err == noErr) {
			if (spBlk.spRefNum == 0) continue; /* no driver here, so don't call it */
			if ((*card_fn)(spBlk.spSlot, spBlk.spID, spBlk.spExtDev, spBlk.spRefNum) != 0)
				++result;
		} else if (err != smNoMoresRsrcs) {
			cprintf(
				"Error %d processing slot 0x%02X sRsrcID 0x%02x\n",
				(int)err, (int)spBlk.spSlot, (int)((UInt8)spBlk.spID));
			++result;
		}
	}
	
	return result;
}

/*
 * slot_int_get
 *
 * Print interrupt setting for driver in slot spSlot
 *
 * refNum : driver reference number
 *
 * Print and return error if driver does not handle call
 *
 */
static int
slot_int_get(SInt8 spSlot, SInt8 spID, SInt8 spExtDev, short refNum)
{
	CntrlParam		pb;
	VDFlagRecord	flagRec;
	OSErr			err;

	err = noErr;

	/* Check for buggy IIci .Display_Video_Apple_RBV1 version 0 driver.
	 * If this is the buggy driver beeing used, we can not make the interrupt
	 * calls without generating a system error. Explain and fail.
	 */
	if (is_buggy_iici(refNum)) {
		if (!patch_iici_driver(refNum)) {
			cprintf("*** IIci buggy version 0 internal video driver\n");
			cprintf("*** Not possible to disable video interrupts\n");
			cprintf("*** Hot-fix (patch) failed\n");
			cprintf("*** Upgrade to System >= 7.5 for new drivers\n");
			return -1;
		}
	}

	/* Initialize parameter block */
	memset(&pb, 0, sizeof(pb));
	pb.ioCRefNum = refNum;
	pb.csCode = cscGetInterrupt;

	/* Inform about what we're going to do */
	cprintf("slot_int_get: slot 0x%02X, drvr_refnum %d, spID 0x%02X, spExtDev 0x%02X\n",
		(int)spSlot, (int)refNum, (int)((UInt8)spID), (int)((UInt8)spExtDev));

	/* Get driver interrupt setting, 99 were for debugging
	 * purposes, driver will not read the value, only set it
	 */
	flagRec.csMode = 99;
	*((unsigned long *)(&pb.csParam[0])) = (unsigned long)&flagRec;

	/* Call driver GetInterrupt routine
	 * Return error if driver does not handle call
	 */
	err = PBStatusSync((ParmBlkPtr)&pb);
	if (err != noErr)
		cprintf("Error %d slot_int_get for slot 0x%02X\n", (int)err, (int)spSlot);

	return err;
}

/*
 * slot_int_set
 *
 * Set interrupt flag for driver
 *
 * refNum : driver reference number
 * 
 * intOn = 0 : turn interrupts off
 * intOn = 1 : turn interrupts on
 *
 * Print and return error if driver does not handle call
 *
 */
static int
slot_int_set(SInt8 spSlot, SInt8 spID, SInt8 spExtDev, short refNum, SInt8 intOn)
{
//#pragma unused(spID, spExtDev)

	CntrlParam		pb;
	VDFlagRecord	flagRec;
	OSErr			err;

	err = noErr;

	/* Check for buggy IIci .Display_Video_Apple_RBV1 version 0 driver.
	 * If this is the buggy driver beeing used, we can not make the interrupt
	 * calls without generating a system error. Explain and fail.
	 */
	if (is_buggy_iici(refNum)) {
		if (!patch_iici_driver(refNum)) {
			cprintf("*** IIci buggy version 0 internal video driver\n");
			cprintf("*** Not possible to disable video interrupts\n");
			cprintf("*** Hot-fix (patch) failed\n");
			cprintf("*** Upgrade to System >= 7.5 for new drivers\n");
			return -1;
		}
	}

	/* Initialize parameter block */
	memset(&pb, 0, sizeof(pb));
	pb.ioCRefNum = refNum;
	pb.csCode = cscSetInterrupt;

	/* Inform about what we're going to do */
	cprintf("slot_int_set: slot 0x%02X, drvr_refnum %d, spID 0x%02X, spExtDev 0x%02X, ON %d\n",
		(int)spSlot, (int)refNum, (int)((UInt8)spID), (int)((UInt8)spExtDev), (int)intOn);

	/* Set interrupt flag
	 * csMode = 1 : interrupts off (blocked)
	 * csMode = 0 : interrupts on (unblocked)
	 */
	flagRec.csMode = (intOn ? 0 : 1);
	*((unsigned long *)(&pb.csParam[0])) = (unsigned long)&flagRec;

	/* Call driver SetInterrupt routine
	 * Return error if driver does not handle call
	 */
	err = PBControlSync((ParmBlkPtr)&pb);
	if (err != noErr)
		cprintf("Error %d slot_int_set for slot 0x%02X\n", (int)err, (int)spSlot);

	return err;
}

/*
 * card_int_on
 *
 * Turn interrupts ON for driver
 *
 * refNum : driver reference number
 * 
 * Return result from slot_int_set() routine
 *
 */
static int
card_int_on(SInt8 spSlot, SInt8 spID, SInt8 spExtDev, short refNum)
{
	return slot_int_set(spSlot, spID, spExtDev, refNum, 1);
}

/*
 * card_int_off
 *
 * Turn interrupts OFF for driver
 *
 * refNum : driver reference number
 * 
 * Return result from slot_int_set() routine
 *
 */
static int
card_int_off(SInt8 spSlot, SInt8 spID, SInt8 spExtDev, short refNum)
{
	return slot_int_set(spSlot, spID, spExtDev, refNum, 0);
}

/*
 * card_int_off
 *
 * Print driver interrupt status
 *
 * refNum : driver reference number
 * 
 * Return result from slot_int_get() routine
 *
 */
static int
card_int_show(SInt8 spSlot, SInt8 spID, SInt8 spExtDev, short refNum)
{
	return slot_int_get(spSlot, spID, spExtDev, refNum);
}

/*
 * is_buggy_iici
 *
 * Return 1 if IIci buggy driver, otherwise 0
 *
 * refNum : driver reference number
 * 
 */
static short
is_buggy_iici(short ref_num)
{
	Str255			drvr_name;
	short			drvr_vers;
	short			result;

	result = 0;

	if (get_driver_info(ref_num, drvr_name, &drvr_vers)) {
		if (EqualString(drvr_name, "\p.Display_Video_Apple_RBV1", 1, 1)) {
			if (drvr_vers == 0) {
				result = 1;
			}
		}
	} else {
		cprintf("Could not get slot driver information\n");
	}

	return result;
}

/*
 * get_driver_info
 *
 * Get driver name and version
 *
 * refNum : driver reference number
 * 
 */
static short
get_driver_info(short ref_num, Str255 d_name, short *d_version)
{
	AuxDCEHandle	auxDCEH;
	char			*drvr_ptr, *drvr_name_ptr;
	short			drvr_name_len;
	OSErr			result;

	result = 0;

	auxDCEH = (AuxDCEHandle)GetDCtlEntry(ref_num);
	if (auxDCEH != nil) {
		if((**auxDCEH).dCtlFlags & dRAMBasedMask){
			/* RAM-based driver. */
			drvr_ptr = *(char **)(**auxDCEH).dCtlDriver;
		} else {
			/* ROM-based driver. */
			drvr_ptr = (char *) (**auxDCEH).dCtlDriver;
		}
		if (drvr_ptr != nil) {
			/* Get driver name and version, version is first word-aligned word
			 * following the driver name - this is true for the video drivers */
			drvr_name_ptr = drvr_ptr+18;
			drvr_name_len = (short)(*(unsigned char *)drvr_name_ptr) + 1;
			memcpy(d_name, drvr_name_ptr, drvr_name_len);
			*d_version = *(short *)(drvr_name_ptr + ((drvr_name_len + 1) & -2));
			result = 1;
		}
	}

	return result;
}

/*
 * patch_iici_driver
 *
 * Patch buggy version 0 IIci video driver
 *
 * refNum : driver reference number
 * 
 */

#pragma options align=mac68k

typedef struct {
	short flags;
	short blanks[3];
	short open;
	short prime;
	short control;
	short status;
	short close;
	Str255 name;
} VideoDriver;

#pragma options align=reset

static short
patch_iici_driver(short ref_num)
{
	AuxDCEHandle	auxDCEH;
	long			bytes;
	VideoDriver		*driver;
	short			*w;
	Handle			handle;

	auxDCEH = (AuxDCE **) GetDCtlEntry(ref_num);
	
	// Move ROM-based driver into RAM.
	if(!((**auxDCEH).dCtlFlags & dRAMBasedMask)){
		
		driver = (VideoDriver *)(**auxDCEH).dCtlDriver;

		// Sometimes the word preceding the driver in ROM seems to be the
		// driver size, but not always, e.g. the built-in driver on the Mac IIsi.
		bytes = *((short *)driver-1);

		// Driver size unknown, guessing (generously) at twice the highest offset.
		bytes = driver->open;
		if (bytes < driver->prime)
			bytes = driver->prime;
		if (bytes < driver->control)
			bytes = driver->control;
		if (bytes < driver->status)
			bytes = driver->status;
		if (bytes < driver->close)
			bytes = driver->close;
		bytes *= 2;

		// We know the Mac IIci driver size to be 1896
		// when ROM version is 124 rev. 1, but who knows for later ROMs?
		//bytes = 1896;
		handle = NewHandleSys(bytes);
		if (handle == NULL)
			return 0;	// Insufficient room on System heap.
		HLockHi(handle);
		BlockMove((Ptr)driver, *handle, bytes);
		FlushInstructionCache();
		FlushDataCache();
		(**auxDCEH).dCtlDriver = (Ptr)handle;
		(**auxDCEH).dCtlFlags |= dRAMBasedMask;
	}
	
	// Patch RAM-based driver.
	handle = (Handle)(**auxDCEH).dCtlDriver;
	w = *(short **)handle;
	if( (w[0x51e/2] != 0x818) || (w[0x590/2] != 0x1810) || (w[0x2c/2] != 0) )
	{
		return 0;
	}
	w[0x51e/2] = 0x4858;
	w[0x590/2] = 0x1a12;
	w[0x2c/2] += 100;									// Change version number.
	FlushInstructionCache();
	FlushDataCache();
	if( (w[0x51e/2] !=0x4858) || (w[0x590/2]!=0x1a12) )
	{
		return 0;
	}

	HNoPurge(handle);
	return 1;
}
