package org.apache.turbine.torque;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and 
 *    "Apache Turbine" must not be used to endorse or promote products 
 *    derived from this software without prior written permission. For 
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Turbine", nor may "Apache" appear in their name, without 
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

import java.sql.*;
import java.io.*;
import java.util.*;

import org.w3c.dom.*;

import org.apache.xerces.dom.*;
import org.apache.xml.serialize.*;

/**
 * This class generates an XML schema of an existing database from
 * JDBC metadata.
 *
 *  @author <a href="mailto:jvanzyl@periapt.com">Jason van Zyl</a>
 *  @version $Id: TorqueJDBCTransformTask.java,v 1.3 2001/03/19 22:30:00 dlr Exp $
 */
public class TorqueJDBCTransformTask
{
    /** Torque properties. */
    protected Properties props;

    /** Name of XML database schema produced. */
    protected String xmlSchema;

    /** JDBC URL. */
    protected String dbUrl;

    /** JDBC driver. */
    protected String dbDriver;

    /** JDBC user name. */
    protected String dbUser;

    /** JDBC password. */
    protected String dbPassword;

    /** DOM document produced. */
    protected DocumentImpl doc;

    /** Database Node to start things off. */
    protected Node database;

    /** Hashtable of columns that have primary keys. */
    protected Hashtable primaryKeys;

    /** Hashtable to track what table a column belongs to. */
    protected Hashtable columnTableMap;

    /** Map of java.sql.Types: integer -> string representation. */
    protected Properties sqlTypes;

    XMLSerializer xmlSerializer;

    /**
     * Default constructor.
     */
    public TorqueJDBCTransformTask()
    {
        //props = Utils.loadProps("config/torque.props");
        //sqlTypes = Utils.loadProps("config/torque.types");
        
        // This is just to get this thing to compile.
        // This has to be turned into an Ant task proper.
        props = new Properties();
        sqlTypes = new Properties();
        
        xmlSchema = props.getProperty("jdbcXMLSchema");
        dbUrl = props.getProperty("dbUrl");
        dbDriver = props.getProperty("dbDriver");
        dbUser = props.getProperty("dbUser");
        dbPassword = props.getProperty("dbPassword");

        System.err.println("Torque - JDBCToXMLSchema starting\n");
        System.err.println("Your DB settings are:");
        System.err.println("driver : "+dbDriver);
        System.err.println("URL : "+dbUrl);
        System.err.println("user : "+dbUser);
        System.err.println("password : "+dbPassword);

        doc = new DocumentImpl();
        doc.appendChild(doc.createComment(" Autogenerated by JDBCToXMLSchema! "));

        try
        {
            generateXML();
            xmlSerializer = new XMLSerializer(
                new PrintWriter(
                new FileOutputStream(xmlSchema)),
                new OutputFormat(Method.XML,null,true));
            xmlSerializer.serialize(doc);
        }
        catch (Exception e)
        {
            System.err.println(e);
            e.printStackTrace();
        }

        System.err.println("\nTorque - JDBCToXMLSchema finished");
    }

    /**
     * Generates an XML database schema from JDBC metadata.
     *
     * @exception Exception, a generic exception.
     */
    public void generateXML()
        throws Exception
    {
        // Load the Interbase Driver.
        Class.forName(dbDriver);
        System.err.println("DB driver sucessfuly instantiated");

        // Attemtp to connect to a database.
        Connection con = DriverManager.getConnection(dbUrl,
                                                     dbUser,
                                                     dbPassword);
        System.err.println("DB connection established");

        // Get the database Metadata.
        DatabaseMetaData dbMetaData = con.getMetaData();

        // The database map.
        Vector tableList = getTableNames(dbMetaData);

        database = doc.createElement("database");

        // Build a database-wide column -> table map.
        columnTableMap = new Hashtable();

        for (int i = 0; i < tableList.size(); i++)
        {
            String curTable = (String) tableList.elementAt(i);
            Vector columns = getColumns(dbMetaData, curTable);

            for (int j = 0; j < columns.size(); j++)
            {
                Vector v = (Vector) columns.elementAt(j);
                String name = (String) v.elementAt(0);

                columnTableMap.put(name, curTable);
            }
        }

        for (int i = 0; i < tableList.size(); i++)
        {
            // Add Table.
            String curTable = (String) tableList.elementAt(i);
            // dbMap.addTable(curTable);

            Element table = doc.createElement("table");
            table.setAttribute("name", curTable);

            // Add Columns.
            // TableMap tblMap = dbMap.getTable(curTable);

            List columns = getColumns(dbMetaData, curTable);
            List primKeys = getPrimaryKeys(dbMetaData, curTable);
            List forgnKeys = getForeignKeys(dbMetaData, curTable);

            // Set the primary keys.
            primaryKeys = new Hashtable();

            for (int k = 0; k < primKeys.size(); k++)
            {
                String curPrimaryKey = (String) primKeys.get(k);
                primaryKeys.put(curPrimaryKey, curPrimaryKey);
            }

            // Foreign keys for this table.
            for (int l = 0; l < forgnKeys.size(); l++)
            {
                String curForeignKey = (String) forgnKeys.get(l);
                String foreignKeyTable =
                        (String) columnTableMap.get(curForeignKey);
                System.out.println(curForeignKey + " => " +
                        foreignKeyTable);
            }

            for (int j = 0; j < columns.size(); j++)
            {
                Vector v = (Vector) columns.get(j);
                String name = (String) v.elementAt(0);
                int type = ((Integer) v.elementAt(1)).intValue();
                int size = ((Integer) v.elementAt(2)).intValue();

                // From DatabaseMetaData.java
                //
                // Indicates column might not allow NULL values.  Huh?
                // Might? Boy, that's a definitive answer.
                /* int columnNoNulls = 0; */

                // Indicates column definitely allows NULL values.
                /* int columnNullable = 1; */

                // Indicates NULLABILITY of column is unknown.
                /* int columnNullableUnknown = 2; */

                Integer nullType = (Integer) v.elementAt(3);

                Element column = doc.createElement("column");
                column.setAttribute("name", name);
                column.setAttribute("type",
                        (String) sqlTypes.get(
                        new Integer(type).toString()));

                if (size > 0 &&
                    (type == Types.CHAR ||
                     type == Types.VARCHAR ||
                     type == Types.LONGVARCHAR))
                {
                    column.setAttribute("size",
                            new Integer(size).toString());
                }

                if (nullType.intValue() == 0)
                {
                    column.setAttribute("null", "false");
                }

                if (primaryKeys.containsKey(name))
                {
                    column.setAttribute("primaryKey", "true");
                }

                table.appendChild(column);
            }
            database.appendChild(table);
        }
        doc.appendChild(database);
    }

    /**
     * Get all the table names in the current database that are not
     * system tables.
     *
     * @param dbMeta JDBC database metadata.
     * @return A Vector with all the tables in a database.
     * @exception SQLException.
     */
    public Vector getTableNames(DatabaseMetaData dbMeta)
        throws SQLException
    {
        ResultSet tableNames = dbMeta.getTables("",null, "%",null);
        Vector tables = new Vector();
        while (tableNames.next())
        {
            String name = tableNames.getString(3);
            String type = tableNames.getString(4);
            if (type.equals("TABLE"))
            {
                tables.addElement(name);
            }
        }
        return tables;
    }

    /**
     * Retrieves all the column names and types for a given table from
     * JDBC metadata.  It returns a vector of vectors.  Each element
     * of the returned vector is a vector with:
     *
     * element 0 => a String object for the column name.
     * element 1 => an Integer object for the column type.
     * element 2 => size of the column.
     * element 3 => null type.
     *
     * @param dbMeta JDBC metadata.
     * @param tableName Table from which to retrieve column
     * information.
     * @return A Vector with the list of columns in tableName.
     */
    public Vector getColumns(DatabaseMetaData dbMeta,
                             String tableName)
        throws SQLException
    {
        ResultSet columnSet = dbMeta.getColumns("",null, tableName, null);
        Vector columns = new Vector();
        while (columnSet.next())
        {
            String name = columnSet.getString(4);
            Integer sqlType = new Integer(columnSet.getString(5));
            Integer size = new Integer(columnSet.getInt(7));
            Integer nullType = new Integer(columnSet.getInt(11));

            Vector v = new Vector();
            v.addElement (name);
            v.addElement (sqlType);
            v.addElement (size);
            v.addElement (nullType);
            columns.addElement (v);
        }
        return columns;
    }

    /**
     * Retrieves a list of the columns composing the primary key for a given
     * table.
     *
     * @param dbMeta JDBC metadata.
     * @param tableName Table from which to retrieve PK information.
     * @return A list of the primary key parts for <code>tableName</code>.
     */
    public List getPrimaryKeys(DatabaseMetaData dbMeta, String tableName)
        throws SQLException
    {
        ResultSet parts = dbMeta.getPrimaryKeys("", null, tableName);
        List pk = new Vector();
        while (parts.next())
        {
            pk.add(parts.getString(4));
        }
        return pk;
    }

    /**
     * Retrieves a list of foreign key columns for a given table.
     *
     * @param dbMeta JDBC metadata.
     * @param tableName Table from which to retrieve FK information.
     * @return A list of foreign keys in <code>tableName</code>.
     */
    public List getForeignKeys(DatabaseMetaData dbMeta, String tableName)
        throws SQLException
    {
        ResultSet foreignKeys = dbMeta.getImportedKeys("", null, tableName);
        List keys = new Vector();
        while (foreignKeys.next())
        {
            keys.add(foreignKeys.getString(8));
        }
        return keys;
    }
}
