/* ------------------------------------------------------------------------
 * $Id: AttributeListImpl.hh,v 1.1 2001/08/16 11:32:21 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * This C++ SAX implementation (Simple API for XML) is derived from
 * David Megginson's original Java SAX that can be found on 
 * http://www.megginson.com/SAX/.
 * ------------------------------------------------------------------------
 * File created 2001-08-13 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------ 
 */

#ifndef _SAXAttributeListImpl_hh_
#define _SAXAttributeListImpl_hh_

// -- System Includes
#include <vector>

// -- Local Includes
#include "Celsius/SAX/AttributeList.hh"

namespace SAX {

    // -- Class Declarations

    /**
     * Default implementation for AttributeList. 
     **/
    class AttributeListImpl : public SAX::AttributeList {
    public:
    
	/**
	 * Constructor.
	 **/
	AttributeListImpl() { }
    
	/**
	 * Destructor.
	 **/
	virtual ~AttributeListImpl() { }
    
	/**
	 * Return the number of attributes in this list.
	 *
	 * @return number of attributes in the list.
	 **/
	virtual int getLength() const;
    
	/**
	 * Return the name of an attribute in this list (by position).
	 *
	 * @param i index of the attribute in the list (starting at 0). 
	 * @return name of the indexed attribute.
	 **/
	virtual const std::string &getName(int i) const;
    
	/**
	 * Return the type of an attribute in the list (by position).
	 * The attribute type is one of the strings "CDATA", "ID",
	 * "IDREF", "IDREFS", "NMTOKEN", "NMTOKENS", "ENTITY",
	 * "ENTITIES", or "NOTATION" (always in upper case).
	 *
	 * @param i index of the attribute in the list (starting at 0).
	 * @return attribute type as a string.
	 **/
	virtual const std::string &getType(int i) const;
    
	/**
	 * Return the type of an attribute in the list (by name). The
	 * attribute type is one of the strings "CDATA", "ID",
	 * "IDREF", "IDREFS", "NMTOKEN", "NMTOKENS", "ENTITY",
	 * "ENTITIES", or "NOTATION" (always in upper case).
	 *
	 * @param name name of the attribute.
	 * @return attribute type as a string.
	 **/
	virtual const std::string &getType(const std::string &name) const;
    
	/**
	 * Return the value of an attribute in the list (by
	 * position). If the attribute value is a list of tokens
	 * (IDREFS, ENTITIES, or NMTOKENS), the tokens will be
	 * concatenated into a single string separated by whitespace.
	 *
	 * @param i index of the attribute in the list (starting at 0).
	 * @return attribute value as a string.
	 **/
	virtual const std::string &getValue(int i) const;
    
	/**
	 * Return the value of an attribute in the list (by name). If
	 * the attribute value is a list of tokens (IDREFS, ENTITIES,
	 * or NMTOKENS), the tokens will be concatenated into a single
	 * string separated by whitespace.
	 *
	 * @param name name of the attribute.
	 * @return attribute value as a string.
	 **/
	virtual const std::string &getValue(const std::string &name) const;
    
	/**
	 * Add an attribute to an attribute list. This method is
	 * provided for SAX parser writers, to allow them to build up
	 * an attribute list incrementally before delivering it to the
	 * application.
	 *
	 * @param name attribute name.
	 * @param type attribute type.
	 * @param value attribute value.
	 **/
	void addAttribute(const std::string &name, const std::string &type,
			  const std::string &value);
    
	/**
	 * Set the attribute list, discarding previous contents.
	 *
	 * @param atts attribute list to copy.
	 **/
	void setAttributeList(const AttributeList &atts);
    
	/**
	 * Remove an attribute from the list. 
	 *
	 * @param attribute name.
	 **/
	void removeAttribute(const std::string &name);
	
	/**
	 * Clear the attribute list. 
	 **/
	void clear();
    
    private:
    
	/**
	 * Is the given index in the range of the attribute list?
	 *
	 * @param i index to check.
	 * @return true if the index is in range, false otherwise.
	 **/
	bool inRange(int i) const {
	    return i >= 0 && i < getLength();
	}
    
	/**
	 * Attribute struct.
	 **/
	struct attrib_t {
	    std::string name;
	    std::string type;
	    std::string value;
	    bool operator == (const std::string &n) const { return n == name; }
	};
    
	std::vector<attrib_t> _data;
    };

};

#endif /* AttributeListImpl.hh */
