/* ------------------------------------------------------------------------
 * $Id: Socket.hh,v 1.5 2001/06/29 15:00:38 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-09-23 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Socket_hh_
#define _Socket_hh_

// -- System Includes
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <utility>

// -- Class Declarations

/**
 * Socket address abstract base class.
 **/
class Address {
public:
    
    int getType() const { return _type; }
    virtual const sockaddr *getAddr() const = 0;
    virtual int getAddrLen() const = 0;

protected:

    Address() : _type(AF_INET) { }
    Address(int type) : _type(type) { }
    virtual ~Address() { }
    
private:
    int _type;
};

class InetAddress : public Address {
public:
    
    InetAddress(int port);
    InetAddress(const sockaddr_in &sin);
    InetAddress(const char *host, int port);
    
    const sockaddr *getAddr() const { return (sockaddr *) &_sin; }
    int getAddrLen() const { return sizeof(_sin); }

private:
    sockaddr_in _sin;
};

/**
 * Simple socket abstraction class. Wraps BSD Sockets in a cleaner,
 * easy-to-use interface.
 **/
class Socket {

public:
    
    /**
     * Default constructor.
     * 
     * @param protocol the protocol type.
     **/
    Socket(int protocol = AF_INET);

    /**
     * Constructor.
     *
     * @param fd an opened file descriptor to use.
     * @param protocol the protocol type.
     **/
    Socket(int fd, int protocol);

    /**
     * Copy constructor.
     **/
    Socket(const Socket &s);
    
    /**
     * Destructor.
     **/
    virtual ~Socket();

    /**
     * Connect the socket to a given address.
     * 
     * @param addr the address of the host to connect to.
     **/ 
    void connect(const Address &addr);
    
    /**
     * Send data over the socket.
     *
     * @param data the data buffer to send.
     * @param len the number of bytes to send.
     * @return the number of bytes actually sent.
     **/
    int send(const void *data, int len);
    
    /**
     * Receive data from the socket.
     *
     * @param data the data buffer to store the received info in.
     * @param len the length of the data buffer.
     * @return the number of bytes actually received.
     **/
    int receive(void *data, int len);

    /**
     * Shutdown the socket.
     **/
    void shutdown(bool norecv = true, bool nosend = true);

    /**
     * Close the socket.
     **/
    void close();
    
    /**
     * Start listening for connections on a socket.
     *
     * @param int backlog the maximum number of queued connections.
     **/
    void listen(int backlog);

    /**
     * Accept an incoming connection to the socket.
     * 
     * @return a pair of the new socket and the client address.
     **/
    std::pair<Socket *, Address *> accept(); 

    /**
     * Bind the socket to an address.
     *
     * @param addr the address to bind the socket to. 
     **/
    void bind(const Address &addr);

    /**
     * Set socket to be nonblocking. 
     **/ 
    void setNonblocking();

    /**
     * Retrieve socket protocol type.
     *
     * @return the protocol type.
     **/ 
    int getProtocol() const { return _protocol; }

    /**
     * Is the remote peer located on the same host as this socket?
     * 
     * @return true if the peer and socket are colocated.
     **/
    bool isSameHost() const;
    
protected:

    // Socket file descriptor
    int _socket;

    // Protocol type
    int _protocol;
};

#endif /* socket.hh */

