/* ------------------------------------------------------------------------
 * $Id: Vector3D.hh,v 1.7 2001/07/30 15:18:55 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 1999-06-02 by Niklas Elmqvist.
 *
 * Copyright (c) 1999-2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2001 Steve Houston <steve@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Vector3D_hh_
#define _Vector3D_hh_

// -- System Includes
#include <iostream>

// -- Celsius Includes
#include "Matrix3D.hh"

// -- Class Declarations

/**
 * Three-dimensional vector class.
 **/
class Vector3D {

public:
    
    /**
     * Default constructor.
     **/
    Vector3D(float x = 0., float y = 0., float z = 0., float w = 1.);

    /**
     * Constructor.
     **/
    Vector3D(const float c[3]);

    /**
     * Copy constructor.
     **/
    Vector3D(const Vector3D &v);

    /**
     * Destructor.
     **/
    ~Vector3D() { }

    /**
     * Access operator (indexing).
     **/
    float &operator () (int n) { return _coord[n]; }

    /**
     * Access operator (indexing). Read-only.
     **/
    float operator () (int n) const { return _coord[n]; }
    
    float x() const { return _coord[0]; }
    float y() const { return _coord[1]; }
    float z() const { return _coord[2]; }
    float w() const { return _coord[3]; }

    float &x() { return _coord[0]; }
    float &y() { return _coord[1]; }
    float &z() { return _coord[2]; }
    float &w() { return _coord[3]; }
    
    /**
     * Vector addition assignment.
     **/
    Vector3D &operator += (const Vector3D &v);
    
    /**
     * Vector subtraction assignment.
     **/
    Vector3D &operator -= (const Vector3D &v);

    /**
     * Vector addition.
     **/    
    Vector3D operator + (const Vector3D &v) const;

    /**
     * Vector subtraction.
     **/
    Vector3D operator - (const Vector3D &v) const;

    /**
     * Vector division (element-wise).
     **/
    Vector3D operator / (const Vector3D &v) const;

    /**
     * Vector multiplication (element-wise).
     **/
    Vector3D operator * (const Vector3D &v) const;
    
    /**
     * Matrix multiplication 
     */
    Vector3D operator * (const Matrix3D &m) const;

    /**
     * Scalar addition assignment.
     **/
    Vector3D &operator += (float f);

    /**
     * Scalar addition assignment.
     **/
    Vector3D &operator -= (float f);

    /**
     * Scalar subtraction assignment.
     **/
    Vector3D &operator /= (float f);

    /**
     * Scalar multiplication assignment.
     **/
    Vector3D &operator *= (float f);

    /**
     * Scalar addition.
     **/
    Vector3D operator + (const float f) const;

    /**
     * Scalar subtraction.
     **/
    Vector3D operator - (const float f) const;

    /**
     * Scalar division.
     **/
    Vector3D operator / (const float f) const;

    /**
     * Scalar multiplication.
     **/
    Vector3D operator * (const float f) const;

    /**
     * Unary negation.
     **/
    Vector3D operator - () const;

    /**
     * Assignment operator.
     **/
    Vector3D &operator = (const Vector3D &v);
    
    /**
     * Compute magnitude of vector.
     **/
    float magnitude();

    /**
     * Normalize vector (make the vector unit-length).
     **/
    void normalize();

    /**
     * Clear the vector (set it to [x y z w] = [0 0 0 1]).
     **/
    void clear();
    
    /**
     * Zero the vector (same as clear, set it to [0 0 0 1]).
     **/
    void zero();

    /**
     * Vector output operator.
     **/
    friend std::ostream & operator << (std::ostream &f, const Vector3D &v);

    /**
     * Compute the (vector) dot product.
     * 
     * @param a first vector.
     * @param b second vector.
     * @return the dot product of the two vectors.
     **/
    static float dot(const Vector3D &a, const Vector3D &b);

    /**
     * Compute the (vector) cross product.
     **/
    static Vector3D cross(const Vector3D &a, const Vector3D &b);
    
private:
    
    /// Vector coordinates array
    float _coord[4];
};

/**
 * Vector equality operator.
 * 
 * @param a first vector.
 * @param b second vector.
 * @return true if the vectors are equal.
 **/
bool operator == (const Vector3D &a, const Vector3D &b);

/**
 * Scalar vector multiplication.
 *
 * @param f scalar.
 * @param v vector.
 * @return resulting vector.
 **/
inline Vector3D operator * (float f, Vector3D v) {
    return v * f;
}

#endif /* Vector3D.hh */

