/* ------------------------------------------------------------------------
 * CORBA.hh
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-06-16 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _CORBA_HH_
#define _CORBA_HH_

// -- System Includes
#include <string>
#include <iostream>
#include <assert.h>

//using namespace std;

// -- omniORB Includes
#include <omniORB3/CORBA.h>

// -- Celsius Includes
#include "Celsius/Exception.hh"

// -- Class Declarations

template <typename Servant>
inline Servant activate(Servant servant)
{
    // Retrieve the default POA (usually Root POA, override servant
    // function to change this), activate the object (ignoring the
    // object identifier), and then remove the original reference to
    // the servant so that the POA is the sole holder of it.

    PortableServer::POA_var poa = servant->_default_POA();
    PortableServer::ObjectId_var oid = poa->activate_object(servant);
    servant->_remove_ref();
    return servant;
}

inline void deactivate(PortableServer::Servant servant) 
{
    // Retrieve the default POA (usually Root POA, override the
    // servant to change this), retrieve the object identifier, and
    // deactivate the servant (thus causing it to be deleted).

    PortableServer::POA_var poa = servant->_default_POA();
    const PortableServer::ObjectId_var oid = poa->servant_to_id(servant);
    poa->deactivate_object(oid);
}

template <class T>
typename T::_ptr_type
resolve_init(CORBA::ORB_ptr orb, const char *id)
{
    // Resolve the initial reference, check that it is not nil (i.e.
    // that it does exist), narrow it to the desired type, and make
    // sure that it worked (i.e. that the narrowing was successful).
    // [This code from the Henning and Vinooski CORBA C++ book.]
    
    // Resolve identifier
    CORBA::Object_var obj;
    try {
	obj = orb->resolve_initial_references(id);
    }
    catch (const CORBA::ORB::InvalidName &e) {
	throw;
    }
    catch (const CORBA::Exception &e) {
	throw Exception(std::string("Cannot get initial reference for ") + id);
    }

    // Make sure it is NOT nil!
    assert(CORBA::is_nil(obj) == false);

    // Narrow reference to desired type (if possible)
    typename T::_var_type ref;
    try {
	ref = T::_narrow(obj);
    } 
    catch (const CORBA::Exception &e) {
	throw Exception(std::string("Cannot narrow reference for ") + id);
    }
    if (CORBA::is_nil(ref) == true) {
	throw Exception(std::string("Incorrect type of reference for ") + id);
    }
    return ref._retn();
}

template<class T>
typename T::_ptr_type
resolve_name(CosNaming::NamingContext_ptr nc, const CosNaming::Name &name)
{
    // Resolve the name using the given naming context, narrow the
    // reference return it, checking for nils and errors along the
    // way.
    
    CORBA::Object_var obj;
    try {
	obj = nc->resolve(name);
    }
    catch (const CosNaming::NamingContext::NotFound &e) {
	throw;
    }
    catch (const CORBA::Exception &e) {
	throw Exception("Cannot resolve binding.");
    }

    // Make sure this is not a nil binding
    if (CORBA::is_nil(obj) == true) {
	throw Exception("Nil binding in Naming Service.");
    }

    // Narrow reference to desired type (if possible)
    typename T::_var_type ref;
    try {
	ref = T::_narrow(obj);
    } 
    catch (const CORBA::Exception &e) {
	throw Exception("Cannot narrow reference.");
    }
    if (CORBA::is_nil(ref) == true) {
	throw Exception("Incorrect type of reference.");
    }
    return ref._retn();
}

template<class T>
typename T::_ptr_type
resolve_name(CosNaming::NamingContext_ptr nc, const char *name)
{
    // Build a naming identifier and resolve it using the CORBA Naming
    // Service.

    CosNaming::Name n;
    n.length(1);
    n[0].id = name;
    n[0].kind = CORBA::string_dup("Object");
    return resolve_name<T>(nc, n);
}

inline void
bind_name(CORBA::ORB_ptr orb, CORBA::Object_ptr obj, const char *name) 
{
    // Resolve the naming service root context, build the Naming
    // Service identifier, and bind the object to the name in the
    // Naming Service.
    
    CosNaming::NamingContext_var context =
	resolve_init<CosNaming::NamingContext>(orb, "NameService");
    CosNaming::Name n;
    n.length(1);
    n[0].id = name;
    n[0].kind = CORBA::string_dup("Object");
    context->rebind(n, obj);
}

inline std::ostream &operator << (std::ostream &os,
				  const CORBA::Exception &exception)
{
    CORBA::Any any;
    any <<= exception;
    CORBA::TypeCode_var tc = any.type();
    const char *p = tc->name();
    if (*p != '\0') os << p;
    else os << tc->id();
    return os;
}

inline std::ostream &operator << (std::ostream &os,
				  CORBA::Exception *exception) {
    return os << *exception;
}

#endif /* CORBA.hh */
