;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;David M. Russinoff
;;;Advanced Micro Devices, Inc.
;;;February, 1998
;;;***************************************************************

(in-package "ACL2")

(include-book "fp")

;natp
(defthm natp-compound-recognizer
  (equal (natp x)
         (and (integerp x)
              (>= x 0)))
  :rule-classes :compound-recognizer)

(in-theory (disable natp))

; The fpaf3a proof of far-exp-low-lemma-1 in far.lisp requires the
; following to be a :rewrite rule, not just a :type-prescription rule.
; Let's make most or all of our :type-prescription rules into :rewrite
; rules as well.
(defthm natp+
    (implies (and (natp x) (natp y))
	     (natp (+ x y))))

(defthm natp*
    (implies (and (natp x) (natp y))
	     (natp (* x y))))


(defun natp-induct (k)
  (if (zp k)
      t
    (natp-induct (1- k))))

;abs

(defthm abs-type 
  (>= (abs x) 0)
  :rule-classes (:TYPE-PRESCRIPTION))

;kill some of these 4 abs lemmas (they are from divsqrt and don't seem to be
;needed in support or exported in lib)

(defthm abs-+
  (implies (and (rationalp x)
                (rationalp y)
                (rationalp z))
           (<= (abs (- x y))
               (+ (abs (- x z))
                  (abs (- y z)))))
  :rule-classes ())

(defthm abs->=
  (implies (and (rationalp x)
                (rationalp y))
           (>= (abs (- y x)) (- (abs x) (abs y))))
  :rule-classes ())

;kill?
(local 
 (defthm abs+
   (implies (and (rationalp x)
                 (rationalp y))
            (<= (abs (+ x y))
                (+ (abs x) (abs y))))
   :rule-classes ()
   :hints (("goal" :in-theory (enable abs)))))

(defthm abs-
  (implies (and (rationalp x)
                (rationalp y))
           (<= (abs (- x y))
               (+ (abs x) (abs y))))
  :rule-classes ()
  :hints (("goal" :in-theory (enable abs))))


;fl

(defthm fl-def-linear
  (implies (case-split (rationalp x))
           (and (<= (fl x) x)
                (< x (1+ (fl x)))))
  :rule-classes :linear)

(in-theory (disable a13)) ;the same rule as fl-def-linear!

;bad? free var.
(defthm fl-monotone-linear
  (implies (and (<= x y)
                (rationalp x)
                (rationalp y))
           (<= (fl x) (fl y)))
  :rule-classes :linear)

(defthm n<=fl-linear
    (implies (and (<= n x)
		  (rationalp x)
		  (integerp n))
	     (<= n (fl x)))
  :rule-classes :linear)


;may need to disable?
(defthm fl+int-rewrite
    (implies (and (integerp n)
		  (rationalp x))
	     (equal (fl (+ x n)) (+ (fl x) n))))

(local
 (defthm fl/int-1
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (<= (fl (/ (fl x) n))
		 (fl (/ x n))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance fl-def-linear))))))

(local
 (defthm fl/int-2
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (fl (/ (fl x) n))
		 (fl (/ x n))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance fl-def-linear)
			(:instance n<=fl-linear (n (* n (fl (/ x n)))))
			(:instance n<=fl-linear (n (fl (/ x n))) (x (/ (fl x) n)))
			(:instance fl-def-linear (x (/ x n)))
			(:instance fl-def-linear (x (/ (fl x) n))))))))

(defthm fl/int-rewrite
  (implies (and (integerp n)
                (> n 0)
                (rationalp x))
           (equal (fl (/ (fl x) n))
                  (fl (/ x n))))
  :hints (("Goal" :use ((:instance fl/int-1)
			(:instance fl/int-2)))))


;needed?
(defthm fl-integer-type
  (integerp (fl x))
  :rule-classes ( :type-prescription))

;this rule is no better than fl-integer-type and might be worse:
(in-theory (disable (:type-prescription fl)))

(defthm fl-int
    (implies (integerp x)
	     (equal (fl x) x)))

(encapsulate
 ()
 (local (include-book "fl2"))
 (DEFTHM FL-INT-2
   (equal (EQUAL (FL X) X)
          (INTEGERP X))))

(defthm fl-unique
    (implies (and (rationalp x)
		  (integerp n)
		  (<= n x)
		  (< x (1+ n)))
	     (equal (fl x) n))
  :rule-classes ())




;cg


(defun cg (x)
;  (declare (xargs :guard (real/rationalp x)))
  (- (fl (- x))))

(defthm cg-def-linear
  (implies (rationalp x)
           (and (>= (cg x) x)
                (> (1+ x) (cg x))))
  :rule-classes :linear)

(defthm cg-monotone-linear
  (implies (and (rationalp x)
                (rationalp y)
                (<= x y))
           (<= (cg x) (cg y)))
  :rule-classes :linear)

(defthm n>=cg-linear
  (implies (and (>= n x)
                (rationalp x)
                (integerp n))
           (>= n (cg x)))
  :rule-classes :linear)

(defthm cg+int-rewrite
  (implies (and (integerp n)
                (rationalp x))
           (equal (cg (+ x n)) (+ (cg x) n))))

(local
 (defthm cg/int-1
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (cg (/ (cg x) n))
		 (cg (/ x n))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance cg-def-linear)
			(:instance cg-monotone-linear (x (/ x n)) (y (/ (cg x) n))))))))

(local
 (defthm cg/int-2
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (<= (cg (/ (cg x) n))
		 (cg (/ x n))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance n>=cg-linear (n (* n (cg (/ x n)))))
			(:instance n>=cg-linear (n (cg (/ x n))) (x (/ (cg x) n)))
			(:instance cg-def-linear (x (/ x n))))))))

(defthm cg/int-rewrite
  (implies (and (integerp n)
                (> n 0)
                (rationalp x))
           (equal (cg (/ (cg x) n))
                  (cg (/ x n))))
  :hints (("Goal" :use ((:instance cg/int-1)
			(:instance cg/int-2)))))

(defthm int-cg-rules
  (implies (rationalp x)
           (integerp (cg x)))
  :rule-classes (:rewrite :type-prescription))

(defthm cg-int
    (implies (integerp x)
	     (equal (cg x) x)))

(defthm cg-int-2
    (implies (rationalp x)
	     (iff (equal (cg x) x)
		  (integerp x))))

(defthm cg-unique
    (implies (and (rationalp x)
		  (integerp n)
		  (>= n x)
		  (> (1+ x) n))
	     (equal (cg x) n))
  :rule-classes ())



(defthm fl-cg
    (implies (and (rationalp x)
		  (not (integerp x)))
	     (equal (cg x) (1+ (fl x))))
  :rule-classes ())





(defthm delta1-1
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp d)
		  (>= y 0)
		  (>= x (+ y y))
		  (>= d 0))
	     (<= (* y d) (* (- x y) d)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance *-weakly-monotonic (x d) (y+ (- x y)))))))


(defthm delta1-a
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp d)
		  (>= y 0)
		  (>= x (+ y y))
		  (>= d 0))
	     (>= (- x (* y (+ 1 d)))
		 (* (- x y) (- 1 d))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance delta1-1)))))

(defthm delta1-b
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp d)
		  (>= y 0)
		  (>= x (+ y y))
		  (>= d 0))
	     (<= (- x (* y (- 1 d)))
		 (* (- x y) (+ 1 d))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance delta1-1)))))

(defthm delta2
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp d)
		  (>= (* x d) 0))
	     (>= (+ x (* y (- 1 d)))
		 (* (+ x y) (- 1 d))))
  :rule-classes ())


(encapsulate
 ()
 (local (include-book "expt0")) ;can't include expt or expt2 (circular)

 (defthm expt-2-positive-rational-type
   (and (rationalp (expt 2 i))
        (< 0 (expt 2 i)))
   :rule-classes ((:type-prescription :typed-term (expt 2 i))))

 (defthm expt-2-positive-integer-type
   (implies (<= 0 i)
            (and (integerp (expt 2 i))
                 (< 0 (expt 2 i))))
   :rule-classes (:type-prescription))

 (in-theory (disable a14)) ;the rules above are better than this one for (expt 2 i)

;the rewrite rule counterpart to expt-2-positive-integer-type
 (defthm expt-2-integerp
   (implies (<= 0 i)
            (integerp (expt 2 i))))

 (defthm expt-2-type-linear
   (implies (<= 0 i)
            (<= 1 (expt 2 i)))
   :rule-classes ((:linear :trigger-terms ((expt 2 i)))))

 (defthm expt-split
   (implies (and (integerp i1)
                 (integerp i2)
                 (case-split (acl2-numberp r)) ;(integerp r)
                 (case-split (not (equal r 0)))
                 )
            (equal (expt r (+ i1 i2))
                   (* (expt r i1)
                      (expt r i2))))
   :hints (("Goal" :in-theory (enable expt)))
   )
 (in-theory (disable expt-split))

 (theory-invariant
  (not (and (member-equal '(:rewrite expt-split)
                          theory)
            (member-equal '(:definition a15)
                          theory)))
  :key :expt-split-invariant)

 )

(defthm expt-monotone-1
    (implies (and (integerp n)
		  (integerp k)
		  (>= k 0))
	     (<= (expt 2 n) (expt 2 (+ n k))))
  :rule-classes ())

(defthm expt-monotone
    (implies (and (integerp n)
		  (integerp m)
		  (<= n m))
	     (<= (expt 2 n) (expt 2 m)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance expt-monotone-1 (k (- m n)))))))

(defthm expt-strong-monotone-1
    (implies (and (integerp n)
		  (integerp k)
		  (> k 0))
	     (< (expt 2 n) (expt 2 (+ n k))))
  :rule-classes ())

(defthm expt-strong-monotone-2
    (implies (and (integerp n)
		  (integerp m)
		  (< n m))
	     (< (expt 2 n) (expt 2 m)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance expt-strong-monotone-1 (k (- m n)))))))

(defthm expt-strong-monotone
    (implies (and (integerp n)
		  (integerp m))
	     (iff (< n m) (< (expt 2 n) (expt 2 m))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance expt-strong-monotone-2)
			(:instance expt-strong-monotone-2 (n m) (m n))))))

(defthm expt-weak-monotone
    (implies (and (integerp n)
		  (integerp m))
	     (iff (<= n m)
		  (<= (expt 2 n) (expt 2 m))))
  :rule-classes ()
  :hints (("Goal" :use (expt-strong-monotone
			(:instance expt-strong-monotone (m n) (n m))))))

(defthm expt-
    (implies (and (integerp a)
		  (integerp b))
	     (= (/ (expt 2 a) (expt 2 b)) (expt 2 (- a b))))
  :rule-classes ())





(defthm expo+
    (implies (and (integerp n)
		  (integerp m))
	     (= (expt 2 (+ m n))
		(* (expt 2 m) (expt 2 n))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance my-exponents-add (r 2) (i m) (j n))))))  



(defthm exp+1-1
    (implies (and (integerp m)
		  (integerp n)
		  (<= n m))
	     (<= (+ (expt 2 m) (expt 2 n))
		 (expt 2 (1+ m))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance expt-monotone)
			(:instance expo+ (n 1))))))

(defthm exp+1
    (implies (and (integerp m)
		  (integerp n)
		  (<= n m))
	     (> (* (- 1 (expt 2 m)) (- 1 (expt 2 n)))
		(- 1 (expt 2 (1+ m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable)
		  :use ((:instance exp+1-1)
                        ))))

(defthm exp+2-1
    (implies (and (integerp n)
		  (integerp m)
		  (<= n m)
		  (<= m 0))
	     (<= (* (expt 2 m) (expt 2 n))
		 (expt 2 m)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance expt-monotone (n (+ m n)))
			(:instance expo+)))))

(defthm exp+2-2
    (implies (and (integerp n)
		  (integerp m)
		  (<= n m)
		  (<= m 0))
	     (<= (+ (expt 2 m) (expt 2 n) (* (expt 2 m) (expt 2 n)))
		 (* 3 (expt 2 m))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable)
		  :use ((:instance expt-monotone)
			(:instance exp+2-1)))))

(defthm exp+2-3
    (implies (and (integerp n)
		  (integerp m)
		  (<= n m)
		  (<= m 0))
	     (< (+ (expt 2 m) (expt 2 n) (* (expt 2 m) (expt 2 n)))
		 (* 4 (expt 2 m))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable)
		  :use (
			(:instance exp+2-2)
			(:instance *-strongly-monotonic (x (expt 2 m)) (y 3) (y+ 4))))))

(defthm exp+2
    (implies (and (integerp n)
		  (integerp m)
		  (<= n m)
		  (<= m 0))
	     (< (* (1+ (expt 2 m)) (1+ (expt 2 n)))
		(1+ (expt 2 (+ m 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance exp+2-3)
			(:instance expo+ (n 2))))))


(defthm exp-invert-1
    (implies (and (integerp n)
		  (<= n -1))
	     (<= (* (expt 2 n) (expt 2 (1+ n)))
		 (expt 2 n)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance expt-monotone (n (1+ n)) (m 0))
			(:instance *-weakly-monotonic (x (expt 2 n)) (y (expt 2 (1+ n))) (y+ 1))))))

(defthm exp-invert-2
    (implies (and (integerp n)
		  (<= n -1))
	     (>= (* (- 1 (expt 2 n))
		    (1+ (expt 2 (1+ n))))
		 1))
  :rule-classes ()
  :hints (("Goal" :use ((:instance expo+ (m 1))
			(:instance exp-invert-1)))))

(defthm cancel-x
    (implies (and (rationalp x)
		  (rationalp y)
		  (> x 0)
		  (<= 1 (* x y)))
	     (<= (/ x) y))
  :rule-classes ())

(defthm exp-invert
    (implies (and (integerp n)
		  (<= n -1))
	     (<= (/ (- 1 (expt 2 n)))
		 (1+ (expt 2 (1+ n)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance cancel-x (x (- 1 (expt 2 n))) (y (1+ (expt 2 (1+ n)))))
			(:instance exp-invert-1)
			(:instance expt-monotone (m -1))))))

(defthm *-doubly-monotonic
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp a)
		  (rationalp b)
		  (<= 0 x)
		  (<= 0 y)
		  (<= 0 a)
		  (<= 0 b)
		  (<= x y)
		  (<= a b))
	     (<= (* x a) (* y b)))
  :rule-classes ())

(local
 (defthm sq-sq-1
    (implies (and (rationalp a)
		  (rationalp b)
		  (rationalp p)
		  (integerp n)
		  (>= p 0)
		  (<= (* (- 1 (expt 2 n)) p) (* a a))
		  (<= (* a a) p)
		  (<= (* b b) (* (expt 2 (- (* 2 n) 2)) p)))
	     (<= (* (* a b) (* a b))
		 (* (expt 2 (- (* 2 n) 2)) (* p p))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance *-doubly-monotonic 
				   (x (* a a)) (a (* b b)) 
				   (y p) (b (* (expt 2 (- (* 2 n) 2)) p))))))))

(defthm sqrt<=
    (implies (and (rationalp x)
		  (rationalp a)
		  (>= a 0)
		  (<= (* x x) (* a a)))
	     (<= x a))
  :rule-classes ()
  :hints (("Goal" :use ((:instance *-strongly-monotonic (y a) (y+ x))
			(:instance *-strongly-monotonic (x a) (y a) (y+ x))))))
(local
 (defthm sq-sq-2
    (implies (and (rationalp a)
		  (rationalp b)
		  (rationalp p)
		  (integerp n)
		  (>= p 0)
		  (<= (* (- 1 (expt 2 n)) p) (* a a))
		  (<= (* a a) p)
		  (<= (* b b) (* (expt 2 (- (* 2 n) 2)) p)))
	     (<= (* (* (expt 2 (1- n)) p) (* (expt 2 (1- n)) p))
		 (* (expt 2 (- (* 2 n) 2)) (* p p))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance expo+ (m (1- n)) (n (1- n))))))))


(local
 (defthm sq-sq-3
    (implies (and (rationalp a)
		  (rationalp b)
		  (rationalp p)
		  (integerp n)
		  (>= p 0)
		  (<= (* (- 1 (expt 2 n)) p) (* a a))
		  (<= (* a a) p)
		  (<= (* b b) (* (expt 2 (- (* 2 n) 2)) p)))
	     (>= (* (expt 2 (1- n)) p) 0))
  :rule-classes ()))

(local
 (defthm sq-sq-4
    (implies (and (rationalp a)
		  (rationalp b)
		  (rationalp p)
		  (integerp n)
		  (>= p 0)
		  (<= (* (- 1 (expt 2 n)) p) (* a a))
		  (<= (* a a) p)
		  (<= (* b b) (* (expt 2 (- (* 2 n) 2)) p)))
	     (<= (* a b)
		 (* (expt 2 (1- n)) p)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance sq-sq-1)
			(:instance sq-sq-3)
			(:instance sqrt<= (x (* a b)) (a (* (expt 2 (1- n)) p)))
			(:instance sq-sq-2))))))

(local
 (defthm sq-sq-5
    (implies (and (rationalp x)
		  (rationalp p)
		  (integerp n)
		  (<= x (* (expt 2 (1- n)) p)))
	     (<= (* 2 x) (* (expt 2 n) p)))
  :rule-classes ()
  :hints (("Goal" ;; :in-theory (disable *-weakly-monotonic)  ;; RBK: f-c and lin
		  :use ((:instance expo+ (m (1- n)) (n 1))
			(:instance *-weakly-monotonic (x 2) (y x) (y+ (* (expt 2 (1- n)) p))))))))


(local
 (defthm sq-sq-6
    (implies (and (rationalp a)
		  (rationalp b)
		  (rationalp p)
		  (integerp n)
		  (>= p 0)
		  (<= (* (- 1 (expt 2 n)) p) (* a a))
		  (<= (* a a) p)
		  (<= (* b b) (* (expt 2 (- (* 2 n) 2)) p)))
	     (<= (* 2 a b)
		 (* (expt 2 n) p)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance sq-sq-4)
			(:instance sq-sq-5 (x (* a b))))))))

(local
 (defthm sq-sq-7
    (implies (and (rationalp a)
		  (rationalp b))
	     (>= (* (- a b) (- a b))
		 (- (* a a) (* 2 a b))))
  :rule-classes ()))

(local
 (defthm sq-sq-8
    (implies (and (rationalp a)
		  (rationalp b)
		  (rationalp p)
		  (integerp n)
		  (>= p 0)
		  (<= (* (- 1 (expt 2 n)) p) (* a a))
		  (<= (* a a) p)
		  (<= (* b b) (* (expt 2 (- (* 2 n) 2)) p)))
	     (>= (* (- a b) (- a b))
		 (- (* (- 1 (expt 2 n)) p)
		    (* (expt 2 n) p))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance sq-sq-6)
			(:instance sq-sq-7))))))

(local
 (defthm sq-sq-9
    (implies (and (rationalp p)
		  (integerp n))
	     (= (- (* (- 1 (expt 2 n)) p)
		   (* (expt 2 n) p))
		(* (- 1 (expt 2 (1+ n))) p)))
  :rule-classes ()))

;where is this used?
(defthm sq-sq
    (implies (and (rationalp a)
		  (rationalp b)
		  (rationalp p)
		  (integerp n)
		  (<= (* (- 1 (expt 2 n)) p) (* a a))
		  (<= (* a a) p)
		  (<= (* b b) (* (expt 2 (- (* 2 n) 2)) p)))
	     (>= (* (- a b) (- a b))
		 (* (- 1 (expt 2 (1+ n))) p)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance sq-sq-8)
			(:instance sq-sq-9)))))

;stuff from logdefs

(defun INDUCT-NAT (x)
  (if (and (integerp x)
	   (> x 0))
      (induct-nat (1- x))
    ()))

(local
(defthm x-or-x/2-1
    (implies (integerp n)
	     (integerp (1+ n)))
  :rule-classes ()
:hints (("Goal" :induct (induct-nat x)))))

(local
(defthm x-or-x/2-2
    (implies (rationalp x)
	     (equal (+ 1 -1/2 x) (+ 1/2 x)))
  :rule-classes ()))

(local
(defthm x-or-x/2-3
    (implies (and (rationalp x)
		  (integerp (+ -1/2 x)))
	     (integerp (+ 1/2 x)))
  :hints (("Goal" :use ((:instance x-or-x/2-1 (n (+ -1/2 x)))
			(:instance x-or-x/2-2))))))

(local
(defthm x-or-x/2-4
    (implies (and (integerp x) (>= x 0))
	     (or (integerp (/ x 2)) (integerp (/ (1+ x) 2))))
  :rule-classes ()
  :hints (("Goal" :induct (induct-nat x)))))

(local
(defthm x-or-x/2-5
    (IMPLIES (integerp x)
	     (integerp (- x)))
  :rule-classes ()))

(local
(defthm x-or-x/2-6
    (IMPLIES (integerp x)
	     (= (- (* -1/2 x)) (* 1/2 x)))
  :rule-classes ()))

(local
(defthm x-or-x/2-7
    (implies (and (integerp x)
		  (integerp (* -1/2 x)))
	     (integerp (* 1/2 x)))
  :hints (("Goal" :in-theory (disable a2)
		  :use ((:instance x-or-x/2-5 (x (* -1/2 x)))
			(:instance x-or-x/2-6))))))

(local
(defthm x-or-x/2-8
    (implies (and (integerp x)
		  (integerp y))
	     (integerp (+ x y)))
  :rule-classes ()))

(local
(defthm x-or-x/2-9
    (implies (integerp x)
	     (= (+ x (+ 1/2 (* -1/2 x)))
		(+ 1/2 (* 1/2 x))))
  :rule-classes ()))

(local
(defthm x-or-x/2-10
    (implies (and (integerp x)
		  (integerp (+ 1/2 (* -1/2 x))))
	     (integerp (+ 1/2 (* 1/2 x))))
  :hints (("Goal" :in-theory (disable a2)
		  :use ((:instance x-or-x/2-8 (y (+ 1/2 (* -1/2 x))))
			(:instance x-or-x/2-9))))))

(local
(defthm x-or-x/2-11
    (implies (and (integerp x) (<= x 0))
	     (or (integerp (/ x 2)) (integerp (/ (1+ x) 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance x-or-x/2-4 (x (- x)))
			(:instance x-or-x/2-5 (x (* -1/2 x))))))))

(defthm X-OR-X/2
    (implies (integerp x) 
	     (or (integerp (/ x 2)) (integerp (/ (1+ x) 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance x-or-x/2-4)
			(:instance x-or-x/2-11)))))


;from divsqrt:

(defun fl-half (x)
;  (declare (xargs :guard (real/rationalp x)))
  (1- (fl (/ (1+ x) 2))))


(defthm fl-half-lemma
    (implies (and (integerp x)
		  (not (integerp (/ x 2))))
	     (= x (1+ (* 2 (fl-half x)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable fl-int)
           :use ((:instance x-or-x/2)
			(:instance fl-int (x (/ (1+ x) 2)))))))

(in-theory (disable fl-half))

(defmacro local-defthm (&rest body)
  (list 'local (cons 'defthm body)))
