#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <usb.h>
#include "pcscdefines.h"
#include "usbserial_linux.h"

#ifndef FALSE
#define FALSE           0
#endif

#ifndef TRUE
#define TRUE            1
#endif

#define MAX_READERS     16
#define ACR38U_VID      0x072f
#define ACR38U_PID      0x9000
#define ACSCM_VID       0x072f
#define ACSCM_PID       0x9006

// USB device
typedef struct _USB_DEVICE {
    usb_dev_handle *hDevice;
    char dirname[PATH_MAX + 1];
    char filename[PATH_MAX + 1];
} USB_DEVICE, *PUSB_DEVICE;

// Array of pointer to USB device
static PUSB_DEVICE g_pUsbDevice[MAX_READERS];

// Initialize usbserial
void __attribute__ ((constructor)) InitUSB(void)
{
    // Initialize array of pointer to USB device
    memset(g_pUsbDevice, 0, sizeof(g_pUsbDevice));
    
    // Initialize USB library
    usb_init();
}

// Finalize usbserial
void __attribute__ ((destructor)) FiniUSB(void)
{
    int i;
    
    for (i = 0; i < MAX_READERS; i++)
    {
        if (g_pUsbDevice[i] != NULL)
        {
            // Release interface
            usb_release_interface(g_pUsbDevice[i]->hDevice, 0);

            // Close device
            usb_close(g_pUsbDevice[i]->hDevice);
            
            // Free device
            free(g_pUsbDevice[i]);
            g_pUsbDevice[i] = NULL;
        }
    }
}

RESPONSECODE OpenUSB( DWORD lun, DWORD channel )
{
    RESPONSECODE ret = STATUS_UNSUCCESSFUL;
    int found;
    int opened;
    
    struct usb_bus *bus;
    struct usb_bus *busses;
    struct usb_device *dev;
    usb_dev_handle *hDevice;
    int err;
    
    // GET_ACR_STAT
    BYTE cmd[] = { 0x01, 0x01, 0x00, 0x00 };
    BYTE rsp[128];
    int bytes;
    int i;

#ifdef PCSC_DEBUG
    int j;
#endif

#ifdef PCSC_DEBUG
    printf("OpenUSB: LUN %d\n", lun);
#endif

    // Check parameters
    if (lun >= MAX_READERS)
        return STATUS_UNSUCCESSFUL;

    // If device is already opened
    if (g_pUsbDevice[lun] != NULL)
        return STATUS_UNSUCCESSFUL;

    usb_find_busses();
    usb_find_devices();

    // Find ACR38U device
    found = FALSE;
    busses = usb_get_busses();
    for (bus = busses; bus; bus = bus->next)
    {
        for (dev = bus->devices; dev; dev = dev->next)
        {
            if (((dev->descriptor.idVendor == ACR38U_VID) && (dev->descriptor.idProduct == ACR38U_PID)) ||
                ((dev->descriptor.idVendor == ACSCM_VID) && (dev->descriptor.idProduct == ACSCM_PID)))
            {
#ifdef PCSC_DEBUG
                printf("OpenUSB: %s/%s\n", bus->dirname, dev->filename);
#endif
                opened = FALSE;
                for (i = 0; i < MAX_READERS; i++)
                {
                    if (g_pUsbDevice[i] != NULL)
                    {
                        if ((strcmp(g_pUsbDevice[i]->dirname, bus->dirname) == 0) &&
                            (strcmp(g_pUsbDevice[i]->filename, dev->filename) == 0))
                        {
                            opened = TRUE;
                            break;
                        }
                    }
                }

#ifdef PCSC_DEBUG
                printf("OpenUSB: opened: %d\n", opened);
#endif
                // If device is not opened
                if (!opened)
                {
                    // Open device
                    hDevice = usb_open(dev);
#ifdef PCSC_DEBUG
                    printf("OpenUSB: usb_open: 0x%X\n", hDevice);
#endif
                    if (hDevice != NULL)
                    {
                        err = usb_claim_interface(hDevice, 0);
#ifdef PCSC_DEBUG
                        printf("OpenUSB: usb_claim_interface: %d\n", err);
#endif
                        // Claim interface
                        if (err >= 0)
                        {
#ifdef PCSC_DEBUG
                            printf("OpenUSB: Testing LUN %d...\n", lun);
#endif
                            // Try 10 times to test command/response from reader
                            for (i = 0; i < 10; i++)
                            {
                                bytes = usb_bulk_write(hDevice, 0x02, (char *) cmd, sizeof(cmd), 2 * 1000);
#ifdef PCSC_DEBUG
                                printf("OpenUSB: bytes: %d, cmd: ", bytes);
                                for (j = 0; j < bytes; j++)
                                    printf("%02X ", cmd[j]);
                                printf("\n");
#endif
                                bytes = usb_bulk_read(hDevice, 0x82, (char *) rsp, sizeof(rsp), 2 * 1000);
#ifdef PCSC_DEBUG
                                printf("OpenUSB: bytes: %d, rsp: ", bytes);
                                for (j = 0; j < bytes; j++)
                                    printf("%02X ", rsp[j]);
                                printf("\n");
#endif
                                if (bytes > 0)
                                {
                                    found = TRUE;
                                    break;
                                }
                            }
#ifdef PCSC_DEBUG
                            printf("OpenUSB: found: %d\n", found);
#endif
                            if (found)
                            {
                                // Allocate device
                                g_pUsbDevice[lun] = (PUSB_DEVICE) malloc(sizeof(USB_DEVICE));
                                if (g_pUsbDevice[lun] != NULL)
                                {
#ifdef PCSC_DEBUG
                                    printf("OpenUSB: connected: %s/%s\n", bus->dirname, dev->filename);
#endif
                                    // Store device information
                                    g_pUsbDevice[lun]->hDevice = hDevice;
                                    strcpy(g_pUsbDevice[lun]->dirname, bus->dirname);
                                    strcpy(g_pUsbDevice[lun]->filename, dev->filename);

                                    ret = STATUS_SUCCESS;
                                }
                                else
                                {
                                    // Release interface if device cannot be allocated
                                    usb_release_interface(hDevice, 0);
                                    found = FALSE;
                                }
                            }
                            else
                            {
                                // Release interface if test is failed
                                usb_release_interface(hDevice, 0);
                            }
                        }
                        
                        if (!found)
                        {
                            // Close device if test is failed
                            usb_close(hDevice);
                        }
                    }

                    if (found)
                        break;
                }
            }
        }
        
        if (found)
            break;
    }

#ifdef PCSC_DEBUG
    printf("OpenUSB: return 0x%X\n", ret);
#endif
    return ret;
}

RESPONSECODE WriteUSB( DWORD lun, DWORD length, unsigned char *buffer )
{
    RESPONSECODE ret = STATUS_SUCCESS;
    int bytes;
    
    // Check parameters
    if ((buffer == NULL) || (length == 0))
        return STATUS_UNSUCCESSFUL;

    // Check LUN
    if (lun >= MAX_READERS)
        return STATUS_UNSUCCESSFUL;
        
    // Check open
    if (g_pUsbDevice[lun] == NULL)
        return STATUS_UNSUCCESSFUL;
        
    bytes = usb_bulk_write(g_pUsbDevice[lun]->hDevice, 0x02, (char *) buffer, length, 50 * 1000);
    if (bytes != length)
    {
#ifdef PCSC_DEBUG
        printf("WriteUSB: %d, %d\n", bytes, length);
#endif
        ret = STATUS_UNSUCCESSFUL;
    }

    return ret;
}

RESPONSECODE ReadUSB( DWORD lun, DWORD *length, unsigned char *buffer )
{
    RESPONSECODE ret = STATUS_SUCCESS;
    int bytes;
    
    // Check parameters
    if ((buffer == NULL) || (length == NULL))
        return STATUS_UNSUCCESSFUL;
        
    // Check length
    if (*length == 0)
        return STATUS_UNSUCCESSFUL;

    // Check LUN
    if (lun >= MAX_READERS)
        return STATUS_UNSUCCESSFUL;

    // Check open
    if (g_pUsbDevice[lun] == NULL)
        return STATUS_UNSUCCESSFUL;
        
    bytes = usb_bulk_read(g_pUsbDevice[lun]->hDevice, 0x82, (char *) buffer, *length, 50 * 1000);
    if (bytes < 0)
    {
#ifdef PCSC_DEBUG
        printf("ReadUSB: %d\n", bytes);
#endif
        ret = STATUS_UNSUCCESSFUL;
        *length = 0;
    }
    else
        *length = bytes;

    return ret;
}

RESPONSECODE CloseUSB( DWORD lun )
{
#ifdef PCSC_DEBUG
    printf("CloseUSB: LUN %d\n", lun);
#endif
    // Check parameters
    if (lun >= MAX_READERS)
        return STATUS_UNSUCCESSFUL;
        
    if (g_pUsbDevice[lun] != NULL)
    {
        // Release interface
        usb_release_interface(g_pUsbDevice[lun]->hDevice, 0);

        // Close device
        usb_close(g_pUsbDevice[lun]->hDevice);

        // Free device
        free(g_pUsbDevice[lun]);
        g_pUsbDevice[lun] = NULL;
    }
    
    return STATUS_SUCCESS;
}
