#
# Module:     CVSAccess.pm
# Date:       03/05/04
# Version:    1.3
# Edited by:  teon@teon.org
# Description:
#
#  This is a plugin, which allows you to control access to CVS
#  repositories available over pserver.
#
# License:
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License Version 2 as
#  published by the Free Software Foundation.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
#  USA.
#
# This file is part of adduser-ng program.
#
# Copyright (C) 2003-2004 Bartosz Oler <liar@bzimage.us>
# Copyright (C) 2003-2004 Robert Olejnik <teon@teon.org>
#

package AddUser::plugins::CVSAccess;

use strict;
use AddUser::StdLib;

my $_PLUGIN = 'CVSAccess';
my $_APIVERSION = 0x1;


# new
# {{{

sub new {
	my ($c, %args) = @_;
	my $class = ref($c) || $c;

	$args{opts} = {
		'passwd_file' => '',
		'user_login' => '',
		'local_user' => '',
		'password' => '',
		'crypt' => 'yes',
		'readers_file' => '',
		'writers_file' => ''};

	$args{APIVERSION} = $_APIVERSION;
        $args{NAME} = $_PLUGIN;

	bless \%args, $class;
}

# }}}




# configure
# =========
#
# {{{

sub configure {
	my $self     = shift;
        my $conf     = $self->{plugins_options}{GroupConfig};
        my $ui       = $self->{plugins_options}{UI};
        my $verbose  = $self->{plugins_options}{verbose};
        my $keywords = $self->{plugins_options}{keywords};
	my $docs     = &get_documentation($self->{plugins_options}{documentation_dir}, $_PLUGIN);

	$self->{plugins_options}{docs} = $docs->{options}{option};
	$ui->display("$_PLUGIN", $docs->{description}."\n") if $verbose;

	while (wrapper_getlong($self, $_PLUGIN, 'passwd_file') =~ m/^\s*$/) {};
	while (wrapper_getlong($self, $_PLUGIN, 'user_login') =~ m/^\s*$/) {};
	while (wrapper_getlong($self, $_PLUGIN, 'local_user') =~ m/^\s*$/) {};

	wrapper_getsingle($self, $_PLUGIN, 'crypt', ('yes', 'no'));
	wrapper_getpassword($self, $_PLUGIN, 'password');
	wrapper_getlong($self, $_PLUGIN, 'readers_file');
	wrapper_getlong($self, $_PLUGIN, 'writers_file');

	open(FH, '<', $self->{opts}{passwd_file}) || return $ERRNO{'ERROR'};
	if (scalar(grep {/^$self->{opts}{user_login}$/} <FH>)) {
		close FH;
		return $ERRNO{'DONE'};
	}
	close FH;

	if ($self->{opts}{readers_file}) {
		open(FH, '<', $self->{opts}{readers_file}) || return $ERRNO{'ERROR'};
		if (scalar(grep {/^$self->{opts}{user_login}$/} <FH>)) {
			close FH;
			return $ERRNO{'DONE'};
		}
		close FH;
	}

	if ($self->{opts}{writers_file}) {
		open(FH, '<', $self->{opts}{writers_file}) || return $ERRNO{'ERROR'};
		if (scalar(grep {/^$self->{opts}{user_login}$/} <FH>)) {
			close FH;
			return $ERRNO{'DONE'};
		}
		close FH;
	}

	return $ERRNO{'OK'};
}

# }}}




# execute
# =======
#
# {{{

sub execute {
	my $self    = shift;
        my $conf    = $self->{plugins_options}{GroupConfig};
        my $ui      = $self->{plugins_options}{UI};
        my $verbose = $self->{plugins_options}{verbose};
	my $line;

	umask 0077;

	my $pass = $self->{opts}{password};
	if ($self->{opts}{crypt} eq 'yes') {
		$pass = crypt($self->{opts}{password}, ('.', '/', 0..9, 'A'..'Z', 'a'..'z')[rand 64, rand 64]);
	}

	open(FH, '>>', $self->{opts}{passwd_file}) || return $ERRNO{'ERROR'};
	print FH $self->{opts}{user_login} . ':' . $pass . ':' .
	    $self->{opts}{local_user} . "\n";
	close FH;

	if ($self->{opts}{readers_file}) {
		open(FH, '>>', $self->{opts}{readers_file}) || return $ERRNO{'ERROR'};
		print FH $self->{opts}{user_login}, "\n";
		close FH;
	}

	if ($self->{opts}{writers_file}) {
		open(FH, '>>', $self->{opts}{writers_file}) || return $ERRNO{'ERROR'};
		print FH $self->{opts}{user_login}, "\n";
		close FH;
	}

	return $ERRNO{'OK'};
}

# }}}


# rollback
# ========
#
# {{{

sub rollback {
	my $self    = shift;
        my $conf    = $self->{plugins_options}{GroupConfig};
        my $ui      = $self->{plugins_options}{UI};
        my $verbose = $self->{plugins_options}{verbose};

	$ui->display("Rollback...", "Doing a rollback.");

        my $tmp_p = create_file("/tmp", "adduser.");
        my $tmp_r = create_file("/tmp", "adduser.");
        my $tmp_w = create_file("/tmp", "adduser.");

	#
	# Rollback passwd file modifications.
	# 
	open(IN, '<', $self->{opts}{passwd_file}) || return $ERRNO{'ERROR'};
	open(OUT, '>', $tmp_p) || return $ERRNO{'ERROR'};
	while (<IN>) {
		print OUT $_ unless /^$self->{opts}{user_login}:/;
	}
	close OUT;
	close IN;
	au_move($tmp_p, $self->{opts}{passwd_file});

	#
	# Rollback readers file modifications.
	# 
	if ($self->{opts}{readers_file}) {
		open(IN, '<', $self->{opts}{readers_file}) || return $ERRNO{'ERROR'};
		open(OUT, '>', $tmp_r) || return $ERRNO{'ERROR'};
		while (<IN>) {
			print OUT $_ unless /^$self->{opts}{user_login}$/;
		}
		close OUT;
		close IN;
		au_move($tmp_r, $self->{opts}{readers_file});
	}

	#
	# Rollback writers file modifications.
	#
	if ($self->{opts}{writers_file}) {
		open(IN, '<', $self->{opts}{writers_file}) || return $ERRNO{'ERROR'};
		open(OUT, '>', $tmp_w) || return $ERRNO{'ERROR'};
		while (<IN>) {
			print OUT $_ unless /^$self->{opts}{user_login}$/;
		}
		close OUT;
		close IN;
		au_move($tmp_w, $self->{opts}{writers_file});
	}

	#
	# We don't care about result of the rollback.
	#
	return $ERRNO{'OK'};
}

# }}}

1;
