/*--- tree.h -------------------------------------------------------------------
Copyright (C) 2004, 2005, 2006 Sylvain Fourmanoit <syfou@users.sourceforge.net>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.   
------------------------------------------------------------------------------*/
/* This implements a minimalistic tree API with error checking.
*/

/*----------------------------------------------------------------------------*/
#ifndef HAVE_TREE_H
#define HAVE_TREE_H

#ifndef HAVE_CONFIG_H
#error Autogenerated config.h should be used.
#endif

/*----------------------------------------------------------------------------*/
#include "config.h"			/* Autoconf header */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>			/* C includes: malloc, free, realloc */
#endif

#include "types.h"			/* Various typedef */
#include "error.h"			/* Error wrapper */
#include "vector.h"			/* Vector API */

/*----------------------------------------------------------------------------*/
typedef void * (*tree_free_content_func) (void*);

typedef struct s_tree {
  struct s_tree * parent;		/* Reference to parent */
  vector        * children;             /* Ref. to vector of children  */
  void          * content;		/* Ref. to dynamic, unspecified data */
  tree_free_content_func                /* The content freeing function */
     tree_free_content;
} tree;

typedef int (*tree_callback_func) (tree*,uint,uint,void*);

typedef enum e_tree_iter_type {
  TREE_TOP_DOWN,
  TREE_BOTTOM_UP,
  TREE_DESTROY} tree_iter_type;

/*----------------------------------------------------------------------------*/
/* Create a tree. 
   A default wrapping of free() is put in place as the tree_free_content
   function reference. Users needing different desallocation should 
   overwrite it. Return the tree address, or NULL in case of failure.
*/
tree * tree_init(void);

/* Push a reference to an untyped element into a tree. 
   Return 1 if successful, 0 otherwise.
*/ 
int tree_push(tree*, void*);

/* Pop the last reference out of a tree (including all its children)
   Return 1 if successful, 0 otherwise.
*/
int tree_pop(tree*);

/* Iterate through a tree. The iteration process goes:
   - from the top node to the bottom leaves if  last argument 
     has `TREE_TOP_DOWN' value
   - from the bottom leaves to the top if it has `TREE_BOTTOM_UP' 
     value
   - like `THREE_BOTTOM_UP', but stopping as soon as something goes wrong 
     on the way up with the `TREE_DESTROY' value, and iterating through 
     children in reverse order.  
   Return 1 if all calls to tree_callback_func
   were successfull, 0 if not.
*/ 
int tree_iterate(tree*,tree_callback_func,void*,tree_iter_type);

/* Destruct a tree.
   Return NULL if successful, original tree otherwise, minus what 
   could have been removed: the most is made to preserve a consistent tree 
   (bottom-up iteration stopping as soon as something goes wrong), 
   by `nullifiying' dead branches...
*/ 
tree * tree_free(tree*);

/* This function is a wrapper to make possible recursive memory free 
   in leaves: it is the default vector_free_item function for the children 
   vector: it is nothing more than a wrapper around the tree_free() function.
   Note: We export it here because it can be usefull to create more complex
   structures such as vectors of trees.
*/
void * tree_vector_free_item(void *);

/*----------------------------------------------------------------------------*/
#endif
