/*
 *	aegis - project change supervisor
 *	Copyright (C) 2001 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate deldate2edits
 */

#include <change/branch.h>
#include <change/file.h>
#include <error.h>
#include <project.h>
#include <trace.h>


static time_t change_completion_timestamp _((change_ty *));

static time_t
change_completion_timestamp(cp)
	change_ty	*cp;
{
	cstate		cstate_data;
	cstate_history	chp;
	cstate_history_list chlp;

	cstate_data = change_cstate_get(cp);
	assert(cstate_data->state == cstate_state_completed);
	chlp = cstate_data->history;
	assert(chlp);
	assert(chlp->length);
	chp = chlp->list[chlp->length - 1];
	assert(chp);
	return chp->when;
}


/*
 * NAME
 *	change_branch_delta_to_edit
 *
 * SYNOPSIS
 *	string_ty *change_branch_delta_to_edit(project_ty *pp, long delta_number,
 *		string_ty *file_name);
 *
 * DESCRIPTION
 *	The change_branch_delta_to_edit function is used to map a delta number
 *	into a specific edit number for a project source file.  This requires
 *	roll-forward of the edits to the named file, until the relevant
 *	delta is reached.
 *
 * ARGUMENTS
 *	cp		- branch the file is in
 *	delta_number	- delta number wanted
 *	file_name	- name of file
 *
 * RETURNS
 *	string_ty *;	string containing edit number,
 *			NULL if file does not exist at this delta.
 *
 * CAVEAT
 *	It is the caller's responsibility to free the string returned
 *	when not futher required.
 */

fstate_src
change_branch_delta_date_to_edit(pp, delta_date, file_name)
	project_ty	*pp;
	time_t		delta_date;
	string_ty	*file_name;
{
	change_ty	*cp;
	cstate		cstate_data;
	cstate_branch	bp;
	long		j;
	int		first_time_seen;
	fstate_src	result;

	trace(("change_branch_delta_to_edit(pp = %08lX, delta_date = %ld)\n{\n",
		(long)pp, (long)delta_date));
	cp = project_change_get(pp);
	result = 0;
	cstate_data = change_cstate_get(cp);
	bp = cstate_data->branch;
	assert(bp);
	first_time_seen = 1;
	if (!bp->history)
		bp->history = cstate_branch_history_list_type.alloc();
	for (j = 0; j < bp->history->length; ++j)
	{
		cstate_branch_history hp;
		change_ty	*cp2;
		time_t		when;
		fstate_src	src_data;

		hp = bp->history->list[j];
		cp2 = change_alloc(pp, hp->change_number);
		change_bind_existing(cp2);
		when = change_completion_timestamp(cp2);
		src_data = change_file_find(cp2, file_name);
		if (src_data)
		{
			if
			(
				when > delta_date
			&&
				first_time_seen
			&&
				src_data->action != file_action_create
			&&
				pp->parent
			)
			{
				assert(src_data->edit);
				assert(src_data->edit->revision);
				if (result)
					fstate_src_type.free(result);
				result = fstate_src_copy(src_data);
			}
			else if (when <= delta_date)
			{
				if (src_data->action == file_action_remove)
				{
					if (result)
						fstate_src_type.free(result);
					result = 0;
				}
				else
				{
					/*
					 * there should always be an edit number,
					 * because it is checked for when the
					 * file is read in.
					 */
					assert(src_data->edit);
					assert(src_data->edit->revision);
					if (result)
						fstate_src_type.free(result);
					result = fstate_src_copy(src_data);
				}
			}
			first_time_seen = 0;
		}
		change_free(cp2);
		if (when > delta_date)
			break;
	}

	/*
	 * If we have not seen this file at all in this branch, try the
	 * parent branch.
	 */
	if (first_time_seen && pp->parent)
	{
		assert(!result);
		result =
			change_branch_delta_date_to_edit
			(
				pp->parent,
				delta_date,
				file_name
			);
	}

	/*
	 * report results
	 */
	trace(("return %s;\n",
		result ? result->edit->revision->str_text : "NULL"));
	trace(("}\n"));
	return result;
}
