/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <qstring.h>
#include <qstringlist.h>
#include <qvaluelist.h>
#include <qcache.h>
#include <qdict.h>
#include <ataddressbook.h>
#include <addressclass.h>
#include <vfolder.h>
#include <vobject.h>
#include <stdio.h>
#include <kconfig.h>
#include <clientvfs.h>
#include <contactobj.h>
#include <groupobj.h>
//#include <miscfunctions.h>
#include <qfile.h>
#include <qtextstream.h>
#include <esplash.h>

#define IDSTRING "AddressBook: "

//#define DEBUG_ADDRESSBOOK

extern KConfig *GlobalConfig;
extern ESplash *s;
AddressBook *AddressBook::item=0;


AddressBook *AddressBook::thisInstance()
{
	if(!item)
		item=new AddressBook;
	return item;
}


AddressBook::AddressBook()
{
	// this is the default history size. should be customizable
	GlobalConfig->setGroup("Address book");
	if(GlobalConfig->hasKey("Cache size"))
		historySize=GlobalConfig->readEntry("Cache size").toInt();
	else
		historySize=100;
	entries.setAutoDelete(true);
	reverse.setAutoDelete(true);
	aliasesDict.setAutoDelete(true);
// TWEAK! These are not my Group Objects to delete...
	go.setAutoDelete(false);
	// setting up the tics stuff for the nifty progress bar
	int ticks=0;
	VFolder *folder;
	QDict<VFolder> dict=ClientVFS::thisInstance()->folderDict();
	QDictIterator<VFolder> it(dict);
	while((folder=it.current()))
	{
		if(folder->type()=="contacts" || folder->type()=="groups")
		{
#ifdef DEBUG_ADDRESSBOOK
			printf(IDSTRING"ticks: %d\n", folder->objectList().count());
#endif
			ticks+=folder->objectList().count()+1;
		}
		++it;
	}
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"The number of ticks is: %d\n", ticks);
#endif
	s->setTicks(ticks,PROGRESS_ROLODEX);
	s->setActive(PROGRESS_TOOLS,false);		// deactivate tools
	s->setActive(PROGRESS_ROLODEX,true);		// activate rolodex
	// groups before contacts to avoid skipping a group due identical name with
	// an ordinary contact
	scanGroups();
	scanContacts();
	loadHistory();
}

void AddressBook::scanContacts()
{
	VFolder *folder;
	QDict<VFolder> dict=ClientVFS::thisInstance()->folderDict();
	QDictIterator<VFolder> it(dict);
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"scanning %d folders for contacts\n", it.count());
#endif
	while((folder=it.current()))
	{
		if(folder->type()=="contacts")
		{
#ifdef DEBUG_ADDRESSBOOK
			printf(IDSTRING"scanContacts: contacts folder found: \"%s\"\n",
				(const char *)folder->name());
#endif
			QList<VObject> entries=folder->objectList();
#ifdef DEBUG_ADDRESSBOOK
			printf(IDSTRING "scanContacts: adding %d contacts\n", entries.count());
#endif
			s->tick(QString("Scanning: [") + folder->name() + "] contacts folder...");
			for(int i=0;i<entries.count();i++)
			{
			  if( folder->isSimpleObject(entries.at(i)->name()) )
			  {
  				ContactObject *contact=(ContactObject *)entries.at(i);
  				addContact(contact);
  				s->tick();
			  }
			}
		}
		++it;
	}
}

void AddressBook::scanGroups()
{
	VFolder *folder;
	QDict<VFolder> dict=ClientVFS::thisInstance()->folderDict();
	QDictIterator<VFolder> it(dict);
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"scanning %d folders for groups\n",
		it.count());
#endif
	while((folder=it.current()))
	{
		if(folder->type()=="groups")
		{
#ifdef DEBUG_ADDRESSBOOK
			printf(IDSTRING"scanGroups: groups folder found: \"%s\"\n",
				(const char *)folder->name());
#endif
			s->tick(QString("Scanning: [") + folder->name() + "] groups folder...");
			QList<VObject> entries=folder->objectList();
			for(int i=0;i<entries.count();i++)
			{
			  if( folder->isSimpleObject(entries.at(i)->name()) )
			  {
  				GroupObject *group=(GroupObject *)entries.at(i);
  				addGroup(group);
  				s->tick();
			  }
			}
		}
		++it;
	}
}

void AddressBook::removeHistoryItem(QString str)
{
	entries.remove(str);
	aliasesDict.remove(str);
	reverse.remove(str);
	historyAddresses.remove(str);
}

void AddressBook::loadHistory()
{
	GlobalConfig->setGroup("Directory");
	QString mdir=GlobalConfig->readEntry("AetheraDir");
	QFile f(mdir+"/internal/addrcache");
	if(f.open(IO_ReadOnly))
	{
		QTextStream txt(&f);
		int count=0;
		while(!txt.atEnd() && count<historySize)
		{
			QString addr=txt.readLine();
			historyAddresses.append(addr);
			if(entries[addr]==0)
				entries.insert(addr,new QString(addr));
			if(aliasesDict[addr]==0)
				aliasesDict.insert(addr,new QStringList(addr));
			if(reverse[addr]==0)
				reverse.insert(addr,new QStringList(addr));
		}
	}
}

void AddressBook::saveHistory()
{
	GlobalConfig->setGroup("Directory");
	QString mdir=GlobalConfig->readEntry("AetheraDir");
	QFile f(mdir+"/internal/addrcache");
	if(f.open(IO_WriteOnly))
	{
		QTextStream txt(&f);
		for(int i=0;i<historyAddresses.count();i++)
			txt << historyAddresses[i] << "\n";
	}
}

QString AddressBook::makeProper(ContactObject *contact, QString address)
{
	if(isProper(address))
		return address;
	bool proper=false;
	QString name=contact->n()[0]+" "+contact->n()[1];
	int l,r,at;
	l =address.find("<");
	r =address.find(">");
	at=address.find("@");
	if(l!=-1 && r!=-1 && at!=-1 && l<at && at<r) proper=true;
	if(l==0)
	{
		proper=false;
		address.remove(0,1);
		address.remove(r-1,1);
	}
	if(!proper)
		address=QString("\""+name+"\" <") + address + QString(">");
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"Proper address: %s\n", (const char*)address);
#endif
	return address;
}

bool AddressBook::isProper(QString &addr)
{
	bool _proper=true;
	// check for the first quote
	int count=addr.find("\"");
	if(count==-1) return false;
	// and for the second
	count=addr.find("\"",count+1);
	if(count==-1) return false;
	int count2=count;
	// check for "<"
	count=addr.find("<",count);
	if(count==-1)
	{
		// try with []
		count=addr.find("[",count2);
		if(count==-1) return false;
		addr.replace(count,1,"<");
		count=addr.find("]",count);
		if(count==-1) return false;
		addr.replace(count,1,">");
		return true;
	}
	// check for ">"
	count=addr.find(">",count);
	if(count==-1) return false;
	return true;
}

void AddressBook::addContact(ContactObject *contact)
{
	//first look in the entries to see if that already existed
	if(entries[contact->url()]!=0)
	{
		// remove all the occurences from the dicts & lists
		removeContact(contact->url());
	}
	if(contact->emails().count()==0)
	{
		printf(IDSTRING"No e-mails for [%s]. Not adding into addressbook.\n",
			(const char *)contact->fn());
		return;
	}
	// building the list of emails
	QStringList emails;
	emails.append(makeProper(contact, contact->emails()[contact->defaultEmail()].simplifyWhiteSpace()));
	for(int i=0;i<contact->emails().count();i++)
	{
		if(i!=contact->defaultEmail())
			emails.append(makeProper(contact, contact->emails()[i].simplifyWhiteSpace()));
	}
	// done building the list of e-mails
	QStringList aliases; // for reverse
	QStringList temp;
	/*
	 * adding:
	 *		the FN as defined by the contact label
	 *		the default e-mail
	 *		the proper address
	 *		the full name
	 *		the first name
	 *		the surname
	 *		the nicknames
	 */
	aliases.append(contact->fn());
	QCString __temp=(const char *)emails[0];
	AddressClass *ac=new AddressClass(__temp);
	aliases.append((const char *)ac->email);
	aliases.append(emails[0]);
	aliases.append(contact->n()[0]+" "+contact->n()[1]);
	aliases.append(contact->n()[0].simplifyWhiteSpace());
	aliases.append(contact->n()[1].simplifyWhiteSpace());
	QStringList nicks=QStringList::split(',',contact->nick());
	for(int i=0;i<nicks.count();i++)
	{
		aliases.append(nicks[i].simplifyWhiteSpace());
	}
	/*
	 * add the contact to the entries dict
	 */
	entries.insert(contact->url(), new QString(emails.join(",")));
	/*
	 * add the link to the reverse dict (for easy removing purposes)
	 */
	reverse.insert(contact->url(), new QStringList(aliases));
	for(int i=0;i<aliases.count();i++)
	{
		contacts.append(aliases[i]);
		/*
		 * If there is already the entry, just add the url
		 * otherwise, just create the entry
		 */
		if(aliasesDict[aliases[i]])
		{
			aliasesDict[aliases[i]]->append(contact->url());
		}
		else
		{
			aliasesDict.insert(aliases[i],new QStringList(contact->url()));
		}
	}
}

void AddressBook::removeContact(QString url)
{
	QStringList *keys;
	keys=reverse[url];
	if(!keys) return;
	// remove
	for(int i=0;i<keys->count();i++)
	{
		// removes all the aliases for the contacts list & dict
		if(aliasesDict[(*keys)[i]])
		{
			aliasesDict[(*keys)[i]]->remove(url);
			if(aliasesDict[(*keys)[i]]->count()==0)
				aliasesDict.remove((*keys)[i]);
		}
		contacts.remove((*keys)[i]);
	}
	reverse.remove(url);
}

void AddressBook::addGroup(GroupObject *group)
{
	//first look in the entries to see if that already existed
	if(entries[group->url()]!=0)
	{
		// remove all the occurences from the dicts & lists
		removeGroup(group->url());
	}
	/*
	 * The group's aliases (various classifications pending)
	 *	for now, just the name...
	 */
	QStringList aliases;
	aliases.append(group->groupName());
	/*
	 * add the contact to the entries dict
	 */
	entries.insert(group->url(), new QString(group->members().join(",")));
	/*
	 * add the link to the reverse dict (for easy removing purposes)
	 */
	reverse.insert(group->url(), new QStringList(aliases));
	for(int i=0;i<aliases.count();i++)
	{
		groups.append(aliases[i]);
		go.insert(aliases[i],group);
		if(aliasesDict[aliases[i]])
		{
			aliasesDict[aliases[i]]->append(group->url());
		}
		else
		{
			aliasesDict.insert(aliases[i],new QStringList(group->url()));
		}
	}
}

void AddressBook::removeGroup(QString url)
{
	QStringList *keys;
	keys=reverse[url];
	// remove
	for(int i=0;i<keys->count();i++)
	{
		// removes all the aliases for the contacts list & dict
		if(aliasesDict[(*keys)[i]])
		{
			aliasesDict[(*keys)[i]]->remove(url);
			if(aliasesDict[(*keys)[i]]->count()==0)
				aliasesDict.remove((*keys)[i]);
		}
		groups.remove((*keys)[i]);
		go.remove((*keys)[i]);
	}
	reverse.remove(url);
}

GroupObject *AddressBook::getGroupByUrl(QString url)
{
	if(reverse[url])
		return go[(*(reverse[url]))[0]];
	return 0;
}

GroupObject *AddressBook::getGroupByName(QString name)
{
	return go[name];
}



void AddressBook::addHistoryAddress(QString address)
{
	if(historyAddresses.findIndex(address)==-1)
	{
		printf(IDSTRING"Element not found\n");
		historyAddresses.prepend(address);
		if(historyAddresses.count()>historySize)
		{
			QString oldAddress=historyAddresses[historyAddresses.count()-1];
			entries.remove(oldAddress);
			reverse.remove(oldAddress);
			aliasesDict.remove(oldAddress);
			entries.insert(address,new QString(address));
			reverse.insert(address,new QStringList(address));
			aliasesDict.insert(address,new QStringList(address));
			historyAddresses.remove(historyAddresses.at(historyAddresses.count()-1));
		}
	}
	else
	{
		printf(IDSTRING"Element found. Move on top.\n");
		historyAddresses.remove(address);
		historyAddresses.prepend(address);
	}
	saveHistory();
}


void AddressBook::addUnique(QStringList &list, QString entry)
{
	QString tmp=entry.stripWhiteSpace().simplifyWhiteSpace();
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"Add unique <%d> : [%s]\n",list.count(), (const char *)tmp);
#endif
	if(!tmp.isEmpty() && list.findIndex(tmp)==-1)
	{
		list.append(tmp);
		list.sort();
	}
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"AddUnique - done\n");
#endif
}

QStringList AddressBook::completeNamesContacts(QString key)
{
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"completeNamesContacts (%s)\n", (const char *)key);
#endif
	QStringList result;
	for(int i=0;i<contacts.count();i++)
	{
		if(key.lower()==contacts[i].left(key.length()).lower() &&
					!contacts[i].simplifyWhiteSpace().isEmpty())
		{
#ifdef DEBUG_ADDRESSBOOK
			printf(IDSTRING"\t added: |%s|\n", (const char *)contacts[i]);
#endif
			result.append(contacts[i]);
		}
	}
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"completeNamesContacts. %d\n", result.count());
#endif
	return result;
}

QStringList AddressBook::completeNamesGroups(QString key)
{
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"completeNamesGroups (%s)\n", (const char *)key);
#endif
	QStringList result;
	for(int i=0;i<groups.count();i++)
	{
		if(key.lower()==groups[i].left(key.length()).lower() &&
					!groups[i].simplifyWhiteSpace().isEmpty())
		{
#ifdef DEBUG_ADDRESSBOOK
			printf(IDSTRING"\t added: |%s|\n", (const char *)contacts[i]);
#endif
			result.append(groups[i]);
		}
	}
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"completeNamesGroups. %d of %d\n", result.count(), groups.count());
#endif
	return result;
}

QStringList AddressBook::completeNamesHistory(QString key)
{
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"completeNamesHistory (%s)\n", (const char *)key);
#endif
	QStringList result;
	for(int i=0;i<historyAddresses.count();i++)
	{
		if(key.lower()==historyAddresses[i].left(key.length()).lower() &&
					!historyAddresses[i].simplifyWhiteSpace().isEmpty())
		{
#ifdef DEBUG_ADDRESSBOOK
			printf(IDSTRING"\t added: |%s|\n", (const char *)contacts[i]);
#endif
			result.append(historyAddresses[i]);
		}
	}
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"completeNamesHistory. %d\n", result.count());
#endif
	return result;
}

QStringList &AddressBook::getContacts()
{
	return contacts;
}

QStringList &AddressBook::getGroups()
{
	return groups;
}

QStringList &AddressBook::getHistory()
{
	return historyAddresses;
}

/*
 * Gets the *names* and **NOT** the e-mail addresses
 */
QStringList AddressBook::completeNames(QString key, int mode)
{
	QStringList result;
	if(mode & Contacts)
		result+=completeNamesContacts(key);
	if(mode & Groups)
		result+=completeNamesGroups(key);
	if(mode & History)
		result+=completeNamesHistory(key);
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"completeNames : %d\n", result.count());
#endif
	return result;
}

/*
 *	Gets the e-mails addresses based on "completeNames"
 */
QStringList AddressBook::complete(QString key)
{
	QStringList result;
	QStringList names=completeNames(key);
	for(int i=0;i<names.count();i++)
	{
		QStringList *temp=aliasesDict[names[i]];
		if(temp)
			for(int i=0;i<temp->count();i++)
			{
				QString *name=entries[(*temp)[i]];
				if(name) addUnique(result, *name);
			}
	}
#ifdef DEBUG_ADDRESSBOOK
	printf(IDSTRING"complete : %d\n", result.count());
#endif
	return result;
}


/*
 *	Gets the e-mail addresses given a key. if it's a group then splits them
 * Otherwise, it just returns a list of e-mails.
 *	WARNING: The key MAY have multiple significances!
 *		It will return ALL the entries corresponding to "John"...
 */
QStringList AddressBook::getEntry(QString key)
{
	QStringList result, res;
	QStringList *aliasesKey=aliasesDict[key.simplifyWhiteSpace()];
	if(!aliasesKey || key.simplifyWhiteSpace().isEmpty())
	{
#ifdef DEBUG_ADDRESSBOOK
		printf(IDSTRING"Non-existent key: <%s>\n", (const char *)key);
#endif
		return result;
	}
	for(int i=0;i<aliasesKey->count();i++)
	{
		QString *tmp=entries[(*aliasesKey)[i]];
		if(!tmp)
		{
#ifdef DEBUG_ADDRESSBOOK
			printf(IDSTRING"Non-existent key: <%s>\n", (const char *)key);
#endif
			return result;
		}
#ifdef DEBUG_ADDRESSBOOK
		printf(IDSTRING"Existent key: <%s>\n", (const char *)key);
#endif
		if(contacts.findIndex(key)!=-1)
			result=*tmp;
		else
				result=QStringList::split(',',*tmp);
		for(int i=0;i<result.count();i++)
		{
			QString str=result[i].simplifyWhiteSpace();
			if(res.findIndex(str)==-1)
				res+=str;
		}
#ifdef DEBUG_ADDRESSBOOK
		printf(IDSTRING"            : %d elements:\n",res.count());
#endif
	}
	return res;
	
}

/*
 *	returns a full length list for the contact as well.
 */
QStringList AddressBook::getRealEntry(QString key)
{
	QStringList result, res;
	QStringList *aliasesKey=aliasesDict[key.simplifyWhiteSpace()];
	if(!aliasesKey || key.simplifyWhiteSpace().isEmpty())
	{
#ifdef DEBUG_ADDRESSBOOK
		printf(IDSTRING"Non-existent key: <%s>\n", (const char *)key);
#endif
		return result;
	}
	for(int i=0;i<aliasesKey->count();i++)
	{
		QString *tmp=entries[(*aliasesKey)[i]];
		if(!tmp)
		{
#ifdef DEBUG_ADDRESSBOOK
			printf(IDSTRING"Non-existent key: <%s>\n", (const char *)key);
#endif
			return result;
		}
#ifdef DEBUG_ADDRESSBOOK
		printf(IDSTRING"Existent key: <%s> <==> |%s|\n", (const char *)key, (const char *)*tmp);
#endif
		if(contacts.findIndex(key)!=-1)
			if(key.find("@")!=-1)
				result=QStringList::split(',',*tmp);//.grep(key,false);
			else
				result=QStringList::split(',',*tmp)[0];
		else
			result=QStringList::split(',',*tmp);
		for(int j=0;j<result.count();j++)
		{
			QString str=result[j].simplifyWhiteSpace();
			if(res.findIndex(str)==-1)
				res+=str;
		}
#ifdef DEBUG_ADDRESSBOOK
		printf(IDSTRING"            : %d elements:\n",res.count());
#endif
	}
	return res;
}

/*
 *	Get Default Name
 */
QString AddressBook::getDefaultName(QString contact)
{
	for(int i=0;i<contacts.count();i++)
	{
		QStringList myURLs, iterURLs;
		myURLs=getEntry(contact);
		iterURLs=getEntry(contacts[i]);
		for(int j=0;j<myURLs.count();j++)
			if(iterURLs.findIndex(myURLs[j])!=-1)
				return contacts[i];
	}
	return QString::null;
}

/*
 *	A wrapper over getEntry
 */
QStringList AddressBook::getEntries(QStringList keys)
{
	QStringList result;
	for(int i=0;i<keys.count();i++)
	{
#ifdef DEBUG_ADDRESSBOOK
		printf(IDSTRING"getEntries for |%s|\n", (const char *)keys[i]);
#endif
		QStringList r=getRealEntry(keys[i]);
#ifdef DEBUG_ADDRESSBOOK
		printf(IDSTRING"\thas %d occurences.\n", r.count());
#endif
		for(int j=0;j<r.count();j++)
			addUnique(result, r[j]);
	}
	return result;
}

bool AddressBook::isGroup(QString item)
{
	return groups.findIndex(item)!=-1;
}

bool AddressBook::isContact(QString item)
{
	return contacts.findIndex(item)!=-1;
}




