/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <syncfileiohandler.h>
#include <datacollection.h>
#include <qstring.h>
#include <magellan.h>
#include <qdatastream.h>
#include <qfile.h>
#include <qdir.h>
#include <qlist.h>

QString SyncFileIOHandler::err;

//#define DEBUG_SYNC

bool SyncFileIOHandler::save(DataCollection *ref)
{
	if(!ref)
	{
		err="SyncFileIOHandler: Cannot save collections pointed by null references";
		return false;
	}
	
	if(ref->type==DataCollection::FileStorage)
	{
		QFile tfile(ref->storageDevice);
		if(tfile.open(IO_WriteOnly))
		{
			QDataStream stream(&tfile);
			ObjectReference *tobj;
#ifdef DEBUG_SYNC
			printf(" SAVING %d ELEMENTS\n", ref->objectList.count());
#endif
			for(int i=0;i<ref->objectList.count();i++)
			{
				tobj=ref->objectList.at(i);
				stream<<tobj->id<<tobj->rawData;
			}
		}
		else
		{
			err="SyncFileIOHandler: Could not open collection "+ref->storageDevice+" for writing, check file permissions";
			return false;
		}
	}
	else
	{
		// save all the file objects inside the collection
		ObjectReference *tobj;
#ifdef DEBUG_SYNC
		printf("syncfileio: save: start\n");
#endif
		for(int i=0;i<ref->objectList.count();i++)
		{
			tobj=ref->objectList.at(i);
			QFile tfile(ref->storageDevice+"/"+tobj->filename);
			if(tfile.open(IO_WriteOnly))
			{
				tfile.writeBlock((const char *)tobj->rawData, tobj->rawData.size());
				tfile.close();
			}
			else
			{
				printf("Warning: could not save %s, check file permissions\n", (const char *)(ref->storageDevice+"/"+tobj->filename));
			}
		}
#ifdef DEBUG_SYNC
		printf("syncfileio: save: stop\n");
#endif
	}
	
	return true;
}

bool SyncFileIOHandler::save(ObjectReference *ref)
{
	if(!ref)
	{
		err="SyncFileIOHandler: Cannot save objects pointed by null references";
		return false;
	}
	
	if(ref->collection->type==DataCollection::FileStorage)
	{
//		printf("<<<< FileStorage <%s>\n",
//			(const char *)ref->collection->storageDevice);
		QFile tfile(ref->collection->storageDevice);
#ifdef DEBUG_SYNC
		printf("<><><><><><><><> (%s) vs (%s): %d\n", (const char *)ref->id,
			(const char *)ref->collection->lastID(),
				ref->isNewObject);
#endif
		if(ref->isNewObject)
		{
			if(tfile.open(IO_ReadWrite | IO_Append))
			{
#ifdef DEBUG_SYNC
				printf("<><><><><><><><> SyncFileIOHandler: APPEND\n");
#endif
				QDataStream stream(&tfile);
				stream<<ref->id<<ref->rawData;
				tfile.close();
				ref->isNewObject=false;
			}
			else
			{
				err="SyncFileIOHandler: Could not open collection "+ref->collection->storageDevice+" for writing, check file permissions";
				return false;
			}
		}
		else
		{
			if(tfile.open(IO_WriteOnly))
			{
#ifdef DEBUG_SYNC
				printf("<><><><><><><><> SyncFileIOHandler: Write all\n");
#endif
				QDataStream stream(&tfile);
				ObjectReference *tobj;
				for(int i=0;i<ref->collection->objectList.count();i++)
				{
					tobj=ref->collection->objectList.at(i);
					stream<<tobj->id<<tobj->rawData;
				}
				tfile.close();
			}
			else
			{
				err="SyncFileIOHandler: Could not open collection "+ref->collection->storageDevice+" for writing, check file permissions";
				return false;
			}
		}
	}
	else
	{
		QFile tfile(ref->collection->storageDevice+"/"+ref->filename);
		if(tfile.open(IO_WriteOnly))
		{
			tfile.writeBlock((const char *)ref->rawData, ref->rawData.size());
			tfile.close();
		}
		else
		{
			printf("Warning: could not save %s, check file permissions\n", (const char *)(ref->collection->storageDevice+"/"+ref->filename));
		}
	}
	
	return true;
}

bool SyncFileIOHandler::remove(DataCollection *ref)
{
	if(ref->type==DataCollection::FileStorage)
	{
		// remove the block device file
		return QDir(ref->storageDevice.left(ref->storageDevice.find('/'))).remove(ref->storageDevice, true);
	}
	else
	{
		// delete all the files in the folder
		QDir dir(ref->storageDevice);
		for(int i=0;i<dir.count();i++)
			dir.remove(dir[i]);
		
		// delete the folder
		dir.cdUp();
		return dir.rmdir(ref->storageDevice, true);
	}
}

bool SyncFileIOHandler::remove(ObjectReference *ref)
{
	if(ref->collection->type==DataCollection::FolderStorage)
	{
		// remove the file
		return QDir(ref->collection->storageDevice).remove(ref->filename, true);
	}
	
	// we can't remove a random chunk inside a block device; try "save" instead
	return false;
}

QString SyncFileIOHandler::lastError()
{
	return err;
}
