// ---------------------------------------------------------------------------
// - PrintTable.hpp                                                          -
// - standard object library - printable table class definition              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_PRINTTABLE_HPP
#define  AFNIX_PRINTTABLE_HPP

#ifndef  AFNIX_BUFFER_HPP
#include "Buffer.hpp"
#endif

namespace afnix {

  /// The PrintTable class is a special class that can be used to format
  /// a table for output. At construction the number of columns is given.
  /// The default constructor create a print table of one column. Any
  /// literal object can be added to the table and are formated according
  /// to a format descriptor. Once the table is created, it can be written
  /// to an output stream or a buffer.
  /// @author amaury darsch

  class PrintTable : public virtual Object {
  private:
    /// the table size
    long d_size;
    /// the number of columns
    long d_cols;
    /// the number of rows
    long d_rows;
    /// the table body
    String** p_data;
    /// the table head
    String* p_head;
    /// the column format size
    long*   p_csiz;
    /// the column filling character
    t_quad* p_fill;
    /// the column direction
    bool*   p_cdir;
    /// the column maximum width
    long*   p_cwth;

  public:
    /// create a default print table
    PrintTable (void);

    /// create a print table with a number of columns
    /// @param cols the number of columns
    PrintTable (const long cols);

    /// create a print table with a number of columns and rows
    /// @param cols the number of columns
    PrintTable (const long cols, const long rows);

    /// destroy this print table
    ~PrintTable (void);

    /// @return the class name
    String repr (void) const;

    /// @return true if the head is defined
    bool ishead (void) const;

    /// @return the number of rows
    long getrows (void) const;

    /// @return the number of columns
    long getcols (void) const;

    /// set the column size
    /// @param col  the column to set
    /// @param size the column size to set
    void setsize (const long col, const long size);

    /// @return the column size
    long getsize (const long col) const;

    /// set the column filling character
    /// @param col  the column to set
    /// @param fill the character used for filling
    void setfill (const long col, const t_quad fill);

    /// @return the column fill character
    t_quad getfill (const long col) const;

    /// set the column direction flag
    /// @param col  the column to set
    /// @param cdir the column direction 
    void setcdir (const long col, const bool cdir);

    /// @return the column direction flag
    bool getcdir (const long col) const;

    /// add a row to the table and return the row index
    long add (void);
    
    /// set a head value by column
    /// @param col the table column
    /// @param val the table value
    void sethead (const long col, const String& val);

    /// get a head value by column
    /// @param col the table column
    String gethead (const long col) const;

    /// set a value by row and column
    /// @param row the table row
    /// @param col the table column
    /// @param val the table value
    void set (const long row, const long col, const String& val);

    /// set a value by row and column
    /// @param row the table row
    /// @param col the table column
    /// @param val the table value
    void set (const long row, const long col, const long val);

    /// set a literal by row and column
    /// @param row the table row
    /// @param col the table column
    /// @param obj the literal object to set
    void set (const long row, const long col, Literal* obj);

    /// get a value by row and column
    /// @param row the table row
    /// @param col the table column
    String get (const long row, const long col) const;

    /// @return a formatted row suitable for dumping
    String dump (const long row) const;

    /// dump the table into a buffer
    void dump (Buffer& buffer) const;

    /// dump the table for an output stream
    void dump (Output& os) const;

    /// @return a formatted head suitable for printing
    String fmhead (void) const;

    /// @return a formatted row suitable for printing
    String format (const long row) const;

    /// format the table into a buffer
    void format (Buffer& buffer) const;

    /// format the table for an output stream
    void format (Output& os) const;

    /// resize this table
    /// @param size the new table size
    void resize (const long size);

    /// merge a print table into this one
    /// @param ptbl the table to merge
    void merge (const PrintTable& ptbl);

  private:
    // make the copy constructor private
    PrintTable (const PrintTable&);
    // make the assignment operator private
    PrintTable& operator = (const PrintTable&);

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
