/**********************************************************************
 ** Ability - contains methods and attributes for an ability which
 **           could be either a spell or a skill
 **    
 ** Last reviewed: v.51
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef ABILITY_C
#define ABILITY_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "ability.h"
#include "objtype.h"
#include "errlog.h"
#include "utils.h"
#include "newfuncts.h"
#include "builder.h"
#include "skillflags.h"
#include "spellflags.h"
#include "player.h"

char *depend_name[] = {"ReqIntel", "ReqStrength", "ReqDexterity", 
		       "ReqConstitution", "ReqHealth", "ReqMaxHealth",
		       "ReqEndurance", "ReqMagic", "ReqWisdom", "ReqCharisma",
		       "MustHaveItem", NULL};

/***********************************************************************
 ** Ability (constructor) - loads ability name and initializes attributes
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Ability::Ability()
{
   drain = 0;
   depend_list = NULL;
}


/***********************************************************************
 ** ~Ability (destructor) - cleans up for destruction
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Ability::~Ability(void)
{
	dependency *tmp_depend;

	while (depend_list != NULL)
	{
		tmp_depend = depend_list;
		depend_list = depend_list->next_depend;
		delete tmp_depend;
	}
	depend_list = NULL;
}

/***********************************************************************
 ** get_acting_str - gets the string that is seen when the ability is
 **                  performed
 **
 ** Parameters: the_type - where does this string go to, either
 **                        Room, Actor, or Target
 **
 ** Returns: pointer to the string for success, NULL for failure
 **
 ***********************************************************************/

char *Ability::get_acting_str(abil_str_type the_type)
{
   switch(the_type)
   {
      case Room:
         return acting_room.str_show();
         break;   

      case Actor:
         return acting_actor.str_show();
         break;   

      case Target:
         return acting_target.str_show();
         break;   

      default:

         return NULL;
         break;
   }
}


/***********************************************************************
 ** set_acting_str - sets the string that is seen when the ability is
 **                  perfomed
 **
 ** Parameters: the_type - where does this string go to that we are
 **                        setting, either Room, Actor, or Target
 **             the_string - the string to set it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_acting_str(abil_str_type the_type, char *the_string)
{
   if (the_string == NULL)
      return -1;

   switch(the_type)
   {
      case Room:
         acting_room = the_string;
         break;   

      case Actor:
         acting_actor = the_string;
         break;   

      case Target:
         acting_target = the_string;
         break;   

      default:

         return -1;
         break;
   }
   return 1;
}


/***********************************************************************
 ** get_success_str - gets the string that is seen when the ability is
 **                   successfully performed
 **
 ** Parameters: the_type - where does this string go to, either
 **                        Room, Actor, or Target
 **
 ** Returns: pointer to the string for success, NULL for failure
 **
 ***********************************************************************/

char *Ability::get_success_str(abil_str_type the_type)
{
   switch(the_type)
   {
      case Room:
         return success_room.str_show();
         break;   

      case Actor:
         return success_actor.str_show();
         break;   

      case Target:
         return success_target.str_show();
         break;   

      default:

         return NULL;
         break;
   }
}


/***********************************************************************
 ** set_success_str - sets the string that is seen when the ability is
 **                   a success
 **
 ** Parameters: the_type - where does this string go to that we are
 **                        setting, either Room, Actor, or Target
 **             the_string - the string to set it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_success_str(abil_str_type the_type, char *the_string)
{
   if (the_string == NULL)
      return -1;

   switch(the_type)
   {
      case Room:
         success_room = the_string;
         break;   

      case Actor:
         success_actor = the_string;
         break;   

      case Target:
         success_target = the_string;
         break;   

      default:

         return -1;
         break;
   }
   return 1;
}


/***********************************************************************
 ** get_failure_str - gets the string that is seen when the ability 
 **                   is attempted and fails
 **
 ** Parameters: the_type - where does this string go to, either
 **                        Room, Actor, or Target
 **
 ** Returns: pointer to the string for success, NULL for failure
 **
 ***********************************************************************/

char *Ability::get_failure_str(abil_str_type the_type)
{
   switch(the_type)
   {
      case Room:
         return failure_room.str_show();
         break;   

      case Actor:
         return failure_actor.str_show();
         break;   

      case Target:
         return failure_target.str_show();
         break;   

      default:

         return NULL;
         break;
   }
}


/***********************************************************************
 ** set_failure_str - sets the string that is seen when the ability is
 **                   attempted and fails
 **
 ** Parameters: the_type - where does this string go to that we are
 **                        setting, either Room, Actor, or Target
 **             the_string - the string to set it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_failure_str(abil_str_type the_type, char *the_string)
{
   if (the_string == NULL)
      return -1;

   switch(the_type)
   {
      case Room:
         failure_room = the_string;
         break;   

      case Actor:
         failure_actor = the_string;
         break;   

      case Target:
         failure_target = the_string;
         break;   

      default:

         return -1;
         break;
   }
   return 1;
}


/***********************************************************************
 ** get_special_name - gets the special name string
 **
 ** Returns: pointer to the string for success, NULL for failure
 **
 ***********************************************************************/

char *Ability::get_special_name()
{
   return special_name.str_show();
}


/***********************************************************************
 ** set_special_name - sets the name of the special to run on use
 **
 ** Parameters: the_string - the string we set it to
 **
 ***********************************************************************/

void Ability::set_special_name(char *the_string)
{
   special_name = the_string;
}


/***********************************************************************
 ** get_succ_trig - gets the success trigger name
 **
 ** Returns: pointer to the string for success, NULL for failure
 **
 ***********************************************************************/

char *Ability::get_succ_trig()
{
   return success_trigger.str_show();
}


/***********************************************************************
 ** set_succ_trig - sets the success trigger name
 **
 ** Parameters: the_string - the string we set it to
 **
 ***********************************************************************/

void Ability::set_succ_trig(char *the_string)
{
   success_trigger = the_string;
}


/***********************************************************************
 ** get_fail_trig - gets the failure trigger name
 **
 ** Returns: pointer to the string for success, NULL for failure
 **
 ***********************************************************************/

char *Ability::get_fail_trig()
{
   return failed_trigger.str_show();
}


/***********************************************************************
 ** set_fail_trig - sets the failure trigger name
 **
 ** Parameters: the_string - the string we set it to
 **
 ***********************************************************************/

void Ability::set_fail_trig(char *the_string)
{
   failed_trigger = the_string;
}


/***********************************************************************
 ** get_attempt_trig - gets the attempt trigger name
 **
 ** Returns: pointer to the string for success, NULL for failure
 **
 ***********************************************************************/

char *Ability::get_attempt_trig()
{
   return attempt_trigger.str_show();
}


/***********************************************************************
 ** set_attempt_trig - sets the attempt trigger name
 **
 ** Parameters: the_string - the string we set it to
 **
 ***********************************************************************/

void Ability::set_attempt_trig(char *the_string)
{
   attempt_trigger = the_string;
}


/***********************************************************************
 ** set_drain - sets the amount this ability drains from the user
 **
 ** Parameters: new_drain - the amount to drain
 **
 ***********************************************************************/

void Ability::set_drain(int new_drain)
{
   if (new_drain >= 0)
      drain = new_drain;
}


/***********************************************************************
 ** get_drain - returns the amount this ability drains the user
 **
 ***********************************************************************/

int Ability::get_drain()
{
   return drain;
}


/***********************************************************************
 ** set_attrib_acting - for set attrib command, sets the acting attributes
 **
 ** Parameters: message_to - who the message is for
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_attrib_acting(abil_str_type message_to, Builder *the_builder)
{
   Strings *tmp_str;

   switch(message_to) 
   {
      case Room:
         tmp_str = &acting_room;
         break;

      case Actor:
         tmp_str = &acting_actor;
         break;

      case Target:
         tmp_str = &acting_target;
         break;

      default:
         return -1;
         break;
   }

   if (the_builder->get_long_input(tmp_str) < 0)
   {
      the_builder->send_bldr(_("Error reading in input, failed!\n"));
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** set_attrib_success - for set attrib command, sets the success 
 **                      attributes
 **
 ** Parameters: message_to - who the message is for
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_attrib_success(abil_str_type message_to, 
                                                       Builder *the_builder)
{
   Strings *tmp_str;

   switch(message_to) 
   {
      case Room:
         tmp_str = &success_room;
         break;

      case Actor:
         tmp_str = &success_actor;
         break;

      case Target:
         tmp_str = &success_target;
         break;

      default:
         return -1;
         break;
   }

   if (the_builder->get_long_input(tmp_str) < 0)
   {
      the_builder->send_bldr(_("Error reading in input, failed!\n"));
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** set_attrib_failure - for set attrib command, sets the failure
 **                      attributes
 **
 ** Parameters: message_to - who the message is for
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_attrib_failure(abil_str_type message_to, 
                                                        Builder *the_builder)
{
   Strings *tmp_str;

   switch(message_to) 
   {
      case Room:
         tmp_str = &failure_room;
         break;

      case Actor:
         tmp_str = &failure_actor;
         break;

      case Target:
         tmp_str = &failure_target;
         break;

      default:
         return -1;
         break;
   }

   if (the_builder->get_long_input(tmp_str) < 0)
   {
      the_builder->send_bldr(_("Error reading in input, failed!\n"));
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** set_attrib_drain - for set attribute command, sets the drain attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_attrib_drain(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr(_("You need to specify a number.\n"));
      return -1;
   }
 
   if (!isdigit(*(the_parsed->get_speech()))) 
   {
      the_builder->send_bldr(_("You need to specify a number as drain.\n"));
      return -1;
   }

   set_drain(atoi(the_parsed->get_speech()));
   the_builder->send_bldr(_("Drain set to %d on ability object %s.\n"),
                                                  get_drain(), get_name());
   return 1;
}


/***********************************************************************
 ** set_attrib_special - for set_attrib, sets the special attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_attrib_specials(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr(_("You need to specify specials to set to.\n"));
      return -1;
   }
   set_special_name(the_parsed->get_speech());
   the_builder->send_bldr(_("Specials on %s set to: %s\n"), get_name(), 
                                                        get_special_name());
   set_modified(1);
   return 1;
}


/***********************************************************************
 ** set_attrib_succtrig - for set_attrib, sets the success trigger
 **                       attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_attrib_succtrig(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr(_("You need to specify succtrig to set to.\n"));
      return -1;
   }
   set_succ_trig(the_parsed->get_speech());
   the_builder->send_bldr(_("SuccTrig on %s set to: %s\n"), get_name(), 
                                                        get_succ_trig());
   set_modified(1);
   return 1;
}


/***********************************************************************
 ** set_attrib_failtrig - for set_attrib, sets the failure trigger
 **                       attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_attrib_failtrig(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr(_("You need to specify failtrig to set to.\n"));
      return -1;
   }
   set_fail_trig(the_parsed->get_speech());
   the_builder->send_bldr(_("FailTrig on %s set to: %s\n"), get_name(), 
                                                        get_fail_trig());
   set_modified(1);
   return 1;
}


/***********************************************************************
 ** set_attrib_attempttrig - for set_attrib, sets the attempt trigger
 **                          attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::set_attrib_attempttrig(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr(_("You need to specify attempttrig to set to.\n"));
      return -1;
   }
   set_attempt_trig(the_parsed->get_speech());
   the_builder->send_bldr(_("AttemptTrig on %s set to: %s\n"), get_name(), 
                                                     get_attempt_trig());
   set_modified(1);
   return 1;
}


/***********************************************************************
 ** read_ability_attrib - reads in ability attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **             build_format - loads the zone in builder format
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Ability::read_ability_attrib(FILE *read_file, ErrLog *error_log, 
                                                     int build_format)
{
   token_record *the_token;
   char         *temp_desc;
   Strings      holder;
   int          deptype;
   int          depintvalue;
   Strings      depstrvalue;

   /* if this is in builder format, then read in the modified attrib */
   if (build_format)
   {
      the_token = get_token(read_file, '\0');
     
      if (the_token->token_type != T_NUMERICAL)
      {
		error_log->invalid_format(get_name(), "abilities", "read_ability_attrib");
        return -1;
      }
  
      set_modified(atoi(the_token->the_string));
   }

   /* get the acting room string */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;

   set_acting_str(Room, temp_desc);
   delete temp_desc;

   /* get the acting actor string */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_acting_str(Actor, temp_desc);
   delete temp_desc;

   /* get the acting target string */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_acting_str(Target, temp_desc);
   delete temp_desc;

   /* get the success room string */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_success_str(Room, temp_desc);
   delete temp_desc;

   /* get the success actor string */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_success_str(Actor, temp_desc);
   delete temp_desc;

   /* get the success target string */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_success_str(Target, temp_desc);
   delete temp_desc;

   /* get the failure room string */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_failure_str(Room, temp_desc);
   delete temp_desc;

   /* get the failure actor string */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_failure_str(Actor, temp_desc);
   delete temp_desc;

   /* get the failure target string */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_failure_str(Target, temp_desc);
   delete temp_desc;

   // We keep this in for now but need to remove it before the next version release
   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "abilities", 
				    "items needed", "read_ability_attrib");
     return -1;
   }

   /* get the items needed string */
   the_token = get_token(read_file, '^');
   // REmoved

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "abilities", 
				    "special name", "read_ability_attrib");
     return -1;
   }

   /* get the special name string */
   the_token = get_token(read_file, '^');
   set_special_name(the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "abilities", 
				    "succtrig", "read_ability_attrib");
     return -1;
   }

   /* get the succtrig string */
   the_token = get_token(read_file, '^');
   set_succ_trig(the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "abilities", 
				    "failtrig", "read_ability_attrib");
     return -1;
   }

   /* get the failtrig string */
   the_token = get_token(read_file, '^');
   set_fail_trig(the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "abilities", 
				    "attempttrig", "read_ability_attrib");
     return -1;
   }

   /* get the attempttrig string */
   the_token = get_token(read_file, '^');
   set_attempt_trig(the_token->the_string);

   /* Set drain value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
     error_log->invalid_attr_format(get_name(), "abilities", 
				    "drain", "read_ability_attrib");
     return -1;
   }
   set_drain(atoi(the_token->the_string));

   // Now read in the dependencies
   the_token = get_token(read_file, '\0');     
   while (the_token->token_type != T_POUND)
   {
	  if (the_token->token_type != T_PLUS)
	  {
	     error_log->invalid_attr_format(get_name(), "abilities", 
				    "dependencies", "read_ability_attrib");
	     return -1;
	  }

	  /* Set deptype value */
          the_token = get_token(read_file, '\0');     
	  if (the_token->token_type != T_NUMERICAL)
	  {
             error_log->invalid_attr_format(get_name(), "abilities", 
				    "deptype", "read_ability_attrib");
             return -1;
	  }
          deptype = atoi(the_token->the_string);

	  /* get the next item, it should be a '^', if not, raise error */
	  the_token = get_token(read_file, '\0');
          if (the_token->token_type != T_CARROT)
	  {
             error_log->invalid_attr_format(get_name(), "abilities", 
				    "depstr", "read_ability_attrib");
             return -1;
	  }

	  /* get the depstrtype string */
	  the_token = get_token(read_file, '^');
          depstrvalue = the_token->the_string;

	  /* Set deptype value */
          the_token = get_token(read_file, '\0');     
	  if (the_token->token_type != T_NUMERICAL)
	  {
             error_log->invalid_attr_format(get_name(), "abilities", 
				    "deptype", "read_ability_attrib");
             return -1;
	  }
          depintvalue = atoi(the_token->the_string);

	  if (((depend_type) deptype) == MustHaveItem)
		  add_dependency((depend_type) deptype, depstrvalue.str_show());
	  else
		  add_dependency((depend_type) deptype, depintvalue);

      the_token = get_token(read_file, '\0');
   }

   return 1;
}


/***********************************************************************
 ** write_ability_attrib - writes the ability attributes to a specified 
 **                        file
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Ability::write_ability_attrib(FILE *the_file)
{
   dependency *tmp_dep;

   fprintf(the_file, "^%s^\n", (get_acting_str(Room) == NULL) ? "" : 
                                                   get_acting_str(Room));
   fprintf(the_file, "^%s^\n", (get_acting_str(Actor) == NULL) ? "" : 
                                                   get_acting_str(Actor));
   fprintf(the_file, "^%s^\n", (get_acting_str(Target) == NULL) ? "" : 
                                                   get_acting_str(Target));
   fprintf(the_file, "^%s^\n", (get_success_str(Room) == NULL) ? "" : 
                                                   get_success_str(Room));
   fprintf(the_file, "^%s^\n", (get_success_str(Actor) == NULL) ? "" : 
                                                   get_success_str(Actor));
   fprintf(the_file, "^%s^\n", (get_success_str(Target) == NULL) ? "" : 
                                                   get_success_str(Target));
   fprintf(the_file, "^%s^\n", (get_failure_str(Room) == NULL) ? "" : 
                                                   get_failure_str(Room));
   fprintf(the_file, "^%s^\n", (get_failure_str(Actor) == NULL) ? "" : 
                                                   get_failure_str(Actor));
   fprintf(the_file, "^%s^\n", (get_failure_str(Target) == NULL) ? "" : 
                                                   get_failure_str(Target));
   fprintf(the_file, "^^\n");  // Remove later
//   fprintf(the_file, "^%s^\n", (get_items_needed() == NULL) ? "" : 
//                                                     get_items_needed());
   fprintf(the_file, "^%s^\n", (get_special_name() == NULL) ? "" : 
                                                     get_special_name());
   fprintf(the_file, "^%s^\n", (get_succ_trig() == NULL) ? "" :
                                                     get_succ_trig());
   fprintf(the_file, "^%s^\n", (get_fail_trig() == NULL) ? "" :
                                                     get_fail_trig());
   fprintf(the_file, "^%s^\n", (get_attempt_trig() == NULL) ? "" :
                                                     get_attempt_trig());
   fprintf(the_file, "%d\n", get_drain());

   tmp_dep = depend_list;
   while (tmp_dep != NULL)
   {
		fprintf(the_file, "+\n");
		fprintf(the_file, "%d\n", (int) tmp_dep->the_type);
		fprintf(the_file, "^%s^\n", (tmp_dep->str_value.str_show() == NULL) ? "" : 
												tmp_dep->str_value.str_show());
		fprintf(the_file, "%d\n", tmp_dep->num_value);
		tmp_dep = tmp_dep->next_depend;
   }
   fprintf(the_file, "#\n");
   return 1;
}


/***********************************************************************
 ** copy_ability_attrib - copies all the ability attributes over from 
 **                       another ability
 **
 ** Parameters: copy_from - the object we copy attributes from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Ability::copy_ability_attrib(Ability *copy_from)
{
   set_acting_str(Room, copy_from->get_acting_str(Room));
   set_acting_str(Actor, copy_from->get_acting_str(Actor));
   set_acting_str(Target, copy_from->get_acting_str(Target));

   set_success_str(Room, copy_from->get_success_str(Room));
   set_success_str(Actor, copy_from->get_success_str(Actor));
   set_success_str(Target, copy_from->get_success_str(Target));

   set_failure_str(Room, copy_from->get_failure_str(Room));
   set_failure_str(Actor, copy_from->get_failure_str(Actor));
   set_failure_str(Target, copy_from->get_failure_str(Target));

   set_special_name(copy_from->get_special_name());
   set_drain(copy_from->get_drain());

   set_succ_trig(copy_from->get_succ_trig());
   set_fail_trig(copy_from->get_fail_trig());
   set_attempt_trig(copy_from->get_attempt_trig());

   copy_dependencies(copy_from->get_dependency_list());

   return 1;
}


/***********************************************************************
 ** test_success - tests for the success or failure of this skill/spell by
 **                returnining an index between 0 and 100 
 **
 ** Parameters: the_player - the player who is executing the ability
 **
 ** Returns: success index for success, -1 for failure
 **
 ***********************************************************************/

int Ability::test_success(Player *the_player, int difficulty)
{
   int the_rank;
   int succ_factor;
   float div_factor;
   Flags *skillflags = NULL;
   Flags *spellflags = NULL;

   the_rank = the_player->get_rank_number(get_name());

   /* first roll the dice */ 
   succ_factor = 1+ (int) (20.0*(rand()/(RAND_MAX+1.0)));

   /* factor in skill rank bonus */
   if (the_rank == 0)
      succ_factor += 10;
   else if (the_rank <= 10)
      succ_factor += ((the_rank*3)+10);
   else if (the_rank <= 20)
      succ_factor += (40 + ((the_rank - 10) * 2));
   else
      succ_factor += (60 + (the_rank - 20)); 

   if (get_type() == OBJ_TYPE_SPELL)
      spellflags = ((Spell *) this)->get_spellflags();
   else
      skillflags = ((Skill *) this)->get_skillflags();

   /* factor in stats bonus depending on what applies to this skill */
   {
      int num_apply = 0;
      int the_sum = 0;

      if (has_dependency(ReqIntel))
      {
         the_sum += get_stat_bonus(the_player->get_intel());
         num_apply++;
      }
      if (has_dependency(ReqStrength))
      {
         the_sum += get_stat_bonus(the_player->get_strength());
         num_apply++;
      }
      if (has_dependency(ReqDexterity))
      {
         the_sum += get_stat_bonus(the_player->get_dexterity());
         num_apply++;
      }
      if (has_dependency(ReqHealth))
      {
         the_sum += get_stat_bonus(the_player->get_health());
         num_apply++;
      }
      if (has_dependency(ReqConstitution))
      {
         the_sum += get_stat_bonus(the_player->get_constitution());
         num_apply++;
      }
      if (has_dependency(ReqEndurance))
      {
         the_sum += get_stat_bonus(the_player->get_endurance());
         num_apply++;
      }
      if (has_dependency(ReqMagic))
      {
         the_sum += get_stat_bonus(the_player->get_magic());
         num_apply++;
      }
      if (has_dependency(ReqWisdom))
      {
         the_sum += get_stat_bonus(the_player->get_wisdom());
         num_apply++;
      }
      if (has_dependency(ReqCharisma))
      {
         the_sum += get_stat_bonus(the_player->get_charisma());
         num_apply++;
      }

      if (num_apply > 0)
         succ_factor += (int) (the_sum / num_apply);
   }

   /* now we subtract stuff */

   /* first, is the difficulty factor of the wall */
   succ_factor -= (difficulty * 2);

   /* next we subtract a factor for the total weight carried */
   succ_factor -= (int) (the_player->get_weight_held() / 10);

   /* now we subtract for poor health */
   div_factor = ((float) the_player->get_health() / 
                                      (float) the_player->get_maxhealth());
   succ_factor -= (10 - ((int) (10.0 * div_factor)));

   /* now we subtract for exhaustion */
   div_factor = ((float) the_player->get_endurance() / 
                                   (float) the_player->get_max_endurance());

   succ_factor -= (10 - ((int) (10.0 * div_factor)));

   return succ_factor;
}


/***********************************************************************
 ** set_attrib_abil - allows the builder to set attributes common to both
 **                   spells and skills.  Parses user input
 **
 **	Parameters: the_builder - the builder who sent the input
 **                 the_parsed - the parsed input from the builder
 **
 ** Returns: 1 if success, -1 if failed
 **
 ***********************************************************************/

int Ability::set_attrib_abil(Builder *the_builder, Parse *the_parsed)
{
   if (!STRNCASECMP(the_parsed->get_target1(), "actingroom",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_acting(Room, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "actingactor",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_acting(Actor, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "actingtarget",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_acting(Target, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "successroom",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_success(Room, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "successactor",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_success(Actor, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "successtarget",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_success(Target, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "failroom",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_failure(Room, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "failactor",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_failure(Actor, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "failtarget",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_failure(Target, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "specials", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_specials(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "succtrig", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_succtrig(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "failtrig", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_failtrig(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "attempttrig", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_attempttrig(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "drain", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_drain(the_parsed, the_builder);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "depstring",
                               strlen(the_parsed->get_target1())))
   {
      Strings    deptype;
      Strings    the_string;
      int        results;

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("usage: set depstring <deptype> <string>\n");
         return -1;
      }

      deptype.assign_word(the_parsed->get_speech(), 1);
      if (deptype.str_show() == NULL)
      {
         the_builder->send_bldr("usage: set depstring <deptype> <string>\n");
         return -1;
      }
      
      the_string.assign_word(the_parsed->get_speech(), 2);
      if (the_string.str_show() == NULL)
      {
         the_builder->send_bldr("usage: set depstring <deptype> <string>\n");
         return -1;
      }

      results = set_dep_str_value(deptype.str_show(), the_string.str_show());
      if (results == 0)
      {
         the_builder->send_bldr("That deptype has not been created yet.\n"
				"Use 'new dependency <type>' to create one.\n");
         return -1;
      }
      else if (results == -1)
      {
	the_builder->send_bldr("The DepString can't be set on that dependency type.\n");
	return -1;
      }

      the_builder->send_bldr("DepString on dependency %s set to: %s\n", 
                        deptype.str_show(),  the_string.str_show());
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "depnumber",
                               strlen(the_parsed->get_target1())))
   {
      Strings    deptype;
      Strings    the_string;
      int        the_num;
      int        results;

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("usage: set depnumber <deptype> <number>\n");
         return -1;
      }

      deptype.assign_word(the_parsed->get_speech(), 1);
      if (deptype.str_show() == NULL)
      {
         the_builder->send_bldr("usage: set depnumber <deptype> <number>\n");
         return -1;
      }
      
      the_string.assign_word(the_parsed->get_speech(), 2);

      if ((the_string.str_show() == NULL) || (!atoi(the_string.str_show())))
      {
         the_builder->send_bldr("usage: set depnumber <deptype> <number>\n");
         return -1;
      }
      
      the_num = atoi(the_string.str_show());
      results = set_dep_int_value(deptype.str_show(), the_num);
      if (results == 0)
      {
         the_builder->send_bldr("That deptype has not been created yet.\n"
				"Use 'new dependency <type>' to create one.\n");
         return -1;
      }
      else if (results == -1)
      {
	the_builder->send_bldr("The DepNumber can't be set on that dependency type.\n");
	return -1;
      }

      the_builder->send_bldr("DepNumber on dependency %s set to: %d\n", 
                        deptype.str_show(),  the_num);
      set_modified(1);
      return 1;
   }
   return 0;
}


/***********************************************************************
 ** get_mem_size_ability - gets how much memory is being taken up by the
 **                        ability part of the object
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Ability::get_mem_size_ability()
{
   int size = 0;
   dependency *tmp_dep;

   size += acting_room.get_mem_size_dynamic();
   size += acting_actor.get_mem_size_dynamic();
   size += acting_target.get_mem_size_dynamic();

   size += success_room.get_mem_size_dynamic();
   size += success_actor.get_mem_size_dynamic();
   size += success_target.get_mem_size_dynamic();

   size += failure_room.get_mem_size_dynamic();
   size += failure_actor.get_mem_size_dynamic();
   size += failure_target.get_mem_size_dynamic();

   size += special_name.get_mem_size_dynamic();
   size += attempt_trigger.get_mem_size_dynamic();
   size += success_trigger.get_mem_size_dynamic();
   size += failed_trigger.get_mem_size_dynamic();

   tmp_dep = depend_list;
   while (tmp_dep != NULL)
   {
	  size += tmp_dep->str_value.get_mem_size_dynamic();
      tmp_dep = tmp_dep->next_depend;
   }
   return size;
}

/***********************************************************************
 ** add_dep_item - adds an already created dependency item to the list
 **
 **	Parameters: new_item - the new dependency to add
 **
 ** Returns: 1 for success, 0 for already found, -1 for error
 **
 ***********************************************************************/

int Ability::add_dep_item(dependency *new_item)
{
	dependency *tmp_dep = NULL;

	tmp_dep = depend_list;
	while ((tmp_dep != NULL) && (tmp_dep->next_depend != NULL) && 
												(tmp_dep->the_type != new_item->the_type))
		tmp_dep = tmp_dep->next_depend;

	if ((tmp_dep != NULL) && (tmp_dep->the_type == new_item->the_type))
		return 0;

	if (tmp_dep == NULL)
		depend_list = new_item;
	else
	{
		tmp_dep->next_depend = new_item;
	}
	return 1;
}


/***********************************************************************
 ** add_dependency - adds a dependency to the list
 **
 **	Parameters: the_type - the type of dependency to add
 **             the_val - the value of the dependency to assign
 **
 ** Returns: 1 for success, 0 for already found, -1 for error
 **
 ***********************************************************************/

int Ability::add_dependency(depend_type the_type, int the_val)
{
	dependency *new_dep;
	int results;

	new_dep = new dependency;
	new_dep->next_depend = NULL;

	if (the_type == MustHaveItem)
		return -1;

	new_dep->the_type = the_type;
	new_dep->num_value = the_val;

	if ((results = add_dep_item(new_dep)) <= 0)
	{
		delete new_dep;
	}
	return results;	
}


int Ability::add_dependency(char *the_type, int the_val)
{
   int i = 0;
   while ((depend_name[i] != NULL) && STRNCASECMP(depend_name[i], the_type, 
						  strlen(the_type)))
     i++;

   if (depend_name[i] == NULL)
     return -1;

   return add_dependency((depend_type) i, the_val);
}

/***********************************************************************
 ** add_dependency - adds a dependency to the list
 **
 **	Parameters: the_type - the type of dependency to add
 **             the_val - the value of the dependency to assign
 **
 ** Returns: 1 for success, 0 for already found, -1 for error
 **
 ***********************************************************************/

int Ability::add_dependency(depend_type the_type, char *the_val)
{
	dependency *new_dep;
	int results;

	new_dep = new dependency;
	new_dep->next_depend = NULL;

	if (the_type != MustHaveItem)
		return -1;

	new_dep->the_type = the_type;
	new_dep->str_value = the_val;

	if ((results = add_dep_item(new_dep)) <= 0)
	{
		delete new_dep;
	}
	return results;	
}


int Ability::add_dependency(char *the_type, char *the_val)
{
   int i = 0;
   while ((depend_name[i] != NULL) && STRNCASECMP(depend_name[i], the_type, 
						  strlen(the_type)))
     i++;

   if (depend_name[i] == NULL)
     return -1;

   return add_dependency((depend_type) i, the_val);
}


/***********************************************************************
 ** has_dependency - checks if a dependency exists
 **
 **	Parameters: the_type - the type of dependency to search for
 **
 ** Returns: 1 for found, 0 for not found
 **
 ***********************************************************************/

int Ability::has_dependency(depend_type the_type)
{
	dependency *tmp_dep;

	tmp_dep = depend_list;
	while ((tmp_dep != NULL) && (tmp_dep->the_type != the_type))
		tmp_dep = tmp_dep->next_depend;

	if (tmp_dep == NULL)
		return 0;
	return 1;
}


/***********************************************************************
 ** get_dep_str_value - gets the dependency string value for a specified type
 **
 **	Parameters: the_type - the type of dependency to search for
 **
 ** Returns: the value if found, NULL for not found
 **
 ***********************************************************************/

char *Ability::get_dep_str_value(depend_type the_type)
{
	dependency *tmp_dep;

	tmp_dep = depend_list;
	while ((tmp_dep != NULL) && (tmp_dep->the_type != the_type))
		tmp_dep = tmp_dep->next_depend;

	if (tmp_dep == NULL)
		return NULL;
	return tmp_dep->str_value.str_show();
}


/***********************************************************************
 ** get_dep_int_value - gets the dependency string value for a specified type
 **
 **	Parameters: the_type - the type of dependency to search for
 **
 ** Returns: the value if found, NULL for not found
 **
 ***********************************************************************/

int Ability::get_dep_int_value(depend_type the_type)
{
   dependency *tmp_dep;

   tmp_dep = depend_list;
   while ((tmp_dep != NULL) && (tmp_dep->the_type != the_type))
      tmp_dep = tmp_dep->next_depend;

   if (tmp_dep == NULL)
      return -1;
   return tmp_dep->num_value;
}


/***********************************************************************
 ** set_dep_str_value - sets the dependency string value for a specified type
 **
 **	Parameters: the_type - the type of dependency to search for
 **                 the_str - the string to set it to
 **
 ** Returns: 1 for success, 0 for not found
 **
 ***********************************************************************/

int Ability::set_dep_str_value(depend_type the_type, char *the_str)
{
	dependency *tmp_dep;

	if (the_type != MustHaveItem)
	  return -1;

	tmp_dep = depend_list;
	while ((tmp_dep != NULL) && (tmp_dep->the_type != the_type))
		tmp_dep = tmp_dep->next_depend;

	if (tmp_dep == NULL)
		return 0;

	tmp_dep->str_value = the_str;
	return 1;
}

int Ability::set_dep_str_value(char *the_type, char *the_val)
{  
   int i = 0;
   while ((depend_name[i] != NULL) && STRNCASECMP(depend_name[i], the_type, 
						  strlen(the_type)))
     i++;

   if (depend_name[i] == NULL)
     return 0;

   return set_dep_str_value((depend_type) i, the_val);
}

/***********************************************************************
 ** set_dep_int_value - gets the dependency int value for a specified type
 **
 **	Parameters: the_type - the type of dependency to search for
 **                 the_val - the value to set it to
 **
 ** Returns: 1 for success, 0 for not found
 **
 ***********************************************************************/

int Ability::set_dep_int_value(depend_type the_type, int the_val)
{
   dependency *tmp_dep;

   if (the_type == MustHaveItem)
      return -1;

   tmp_dep = depend_list;
   while ((tmp_dep != NULL) && (tmp_dep->the_type != the_type))
      tmp_dep = tmp_dep->next_depend;

   if (tmp_dep == NULL)
      return 0;
   tmp_dep->num_value = the_val;
   return 1;
}

int Ability::set_dep_int_value(char *the_type, int the_val)
{  
   int i = 0;
   while ((depend_name[i] != NULL) && STRNCASECMP(depend_name[i], the_type, 
						  strlen(the_type)))
     i++;

   if (depend_name[i] == NULL)
     return 0;

   return set_dep_int_value((depend_type) i, the_val);
}

/***********************************************************************
 ** del_dependency - deletes the dependency with the indicated type
 **
 **	Parameters: the_type - the type of dependency to search for
 **
 ** Returns: 1 if removed, 0 if not found, NULL for not found
 **
 ***********************************************************************/

int Ability::del_dependency(depend_type the_type)
{
	dependency *tmp_dep;
	dependency *prev_dep = NULL;

	tmp_dep = depend_list;
	while ((tmp_dep != NULL) && (tmp_dep->the_type != the_type))
	{
		prev_dep = tmp_dep;
		tmp_dep = tmp_dep->next_depend;
	}

	if (tmp_dep == NULL)
		return 0;

	if (prev_dep == NULL)
	{
		depend_list = depend_list->next_depend;
	}
	else
		prev_dep->next_depend = tmp_dep->next_depend;

	delete tmp_dep;
	return 1;
}


int Ability::del_dependency(char *the_type)
{  
   int i = 0;
   while ((depend_name[i] != NULL) && STRNCASECMP(depend_name[i], the_type, 
						  strlen(the_type)))
     i++;

   if (depend_name[i] == NULL)
     return 0;

   return del_dependency((depend_type) i);
}

/***********************************************************************
 ** get_dependency_list - returns the list of dependencies
 **
 ***********************************************************************/

dependency *Ability::get_dependency_list()
{
	return depend_list;
}


/***********************************************************************
 ** copy_dependencies - copies the list of dependencies from a list passed
 **                     in
 **
 **	Parameters: the_list - the list to copy from
 **
 ** Returns: 1 if success, -1 if failed
 **
 ***********************************************************************/

int Ability::copy_dependencies(dependency *the_list)
{
	dependency *copy_dep = the_list;

	while (copy_dep != NULL)
	{
	   if (copy_dep->the_type == MustHaveItem)	
	      add_dependency(copy_dep->the_type, copy_dep->str_value.str_show());
	   else
	      add_dependency(copy_dep->the_type, copy_dep->num_value);
	   copy_dep = copy_dep->next_depend;
	}
	return 1;
}


#endif




