/**********************************************************************
 ** Entity class:  This class provides methods for all mudobjects,
 **                specials, and actions
 **
 **    Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate), Kelly Gerke
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef ENTITY_C
#define ENTITY_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "objtype.h"
#include "specials.h"
#include "builder.h"
#include "editor.h"
#include "global.h"
#include "utils.h"
#include "mudobject.h"
#include "book.h"
#include "door.h"
#include "location.h"
#include "mobile.h"
#include "moveable.h"
#include "marker.h"
#include "merger.h"
#include "money.h"
#include "weapon.h"
#include "wearable.h"
#include "key.h"
#include "food.h"
#include "action.h"
#include "text.h"
#include "newfuncts.h"
#include "rope.h"
#include "level.h"
#include "boat.h"
#include "entity.h"
#include "mask.h"
#include "chatline.h"
#include "inclination.h"
#include "talent.h"


/***********************************************************************
 ** Entity (constructor) - creates the entity
 **
 ** Parameters: the_name - the name of the object
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Entity::Entity()
{
   modified = 0;
}

/***********************************************************************
 ** ~Entity (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Entity::~Entity()
{
}


/***********************************************************************
 ** get_name - retrieves the name of the object
 **
 ** Parameters: None
 **
 ** Returns: pointer to the name character string
 **
 ***********************************************************************/

char *Entity::get_name(void)
{
   return name.str_show();
}



/***********************************************************************
 ** set_name - sets the objects name
 **
 ** Parameters: the_name - the name string to set it to
 **
 ** Returns: 1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Entity::set_name(char *the_name)
{
   if (the_name == NULL)
      return -1;

   if (strlen(the_name) > MAXNAMELEN)
      return -1;

   return name = the_name;
}


/***********************************************************************
 ** get_number - retrieves the number of the object
 **
 ** Parameters: None
 **
 ** Returns: returns the object number
 **
 ***********************************************************************/

int Entity::get_type() {
   return obj_type;
}

/***********************************************************************
 ** is_modified - gets if the object has been modified from the original
 **
 ** Parameters: None
 **
 ** Returns: 1 for modified, 0 for not
 **
 ***********************************************************************/

int Entity::is_modified(void)
{
   return modified;
}

/***********************************************************************
 ** set_modified - sets if the object has been modified or not
 **
 ** Parameters: the_val - the modified value
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Entity::set_modified(int the_val)
{
   modified = the_val;
}

/***********************************************************************
 ** trylock - tries to lock the object mutex, if it can't somebody already
 **           has it locked
 **
 ** Parameters: Nothing
 **
 ** Returns: the results, 0 for good, error code for not good
 **
 ***********************************************************************/

int Entity::trylock(void)
{
#ifdef USE_PTHREAD
   return pthread_mutex_trylock(&obj_mutex);
#else
   return 0;
#endif
}

/***********************************************************************
 ** unlock - unlock the mutex for others to use
 **
 ** Parameters: Nothing
 **
 ** Returns: the results, 0 for good, error code for not good
 **
 ***********************************************************************/

int Entity::unlock(void)
{
#ifdef USE_PTHREAD
   return pthread_mutex_unlock(&obj_mutex);
#else
   return 0;
#endif
}


/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Entity *Entity::operator = (Entity *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}


/***********************************************************************
 ** copy_object - copies attributes of another object to this one
 **
 ** Parameters: copy_obj - the object we won't copy cause it is an error
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Entity::copy_object(Entity *copy_obj)
{
   copy_obj = NULL;
   printf("Error!  Attempted to copy an entity object\n");
   return -1;
}


/***********************************************************************
 ** is_a_mudobject - is this entity a mudobject type
 **
 **
 ***********************************************************************/

int Entity::is_a_mudobject()
{
   switch(get_type())
   {
      case OBJ_TYPE_BOOK:
      case OBJ_TYPE_DOOR:
      case OBJ_TYPE_KEY:
      case OBJ_TYPE_LOCATION:
      case OBJ_TYPE_MARKER:
      case OBJ_TYPE_MOBILE:
      case OBJ_TYPE_MOVEABLE:
      case OBJ_TYPE_WEAPON:
      case OBJ_TYPE_MERGER:
      case OBJ_TYPE_MONEY:
      case OBJ_TYPE_WEARABLE:
      case OBJ_TYPE_FOOD:
      case OBJ_TYPE_PLAYER:
      case OBJ_TYPE_ITEM:
      case OBJ_TYPE_ROPE:
      case OBJ_TYPE_BOAT:

         return 1;
         break;

      default:
         return 0;
         break;
   }
   return -1;
}

/***********************************************************************
 ** is_a_moveable - is this object a moveable object
 **
 ** Parameters: Nothing
 **
 ** Returns: 1 if it is a moveable, 0 if not
 **
 ***********************************************************************/

int Entity::is_a_moveable()
{
   switch(obj_type)
   {
      case OBJ_TYPE_MOVEABLE:  
      case OBJ_TYPE_KEY:
      case OBJ_TYPE_BOAT: 
      case OBJ_TYPE_BOOK: 
      case OBJ_TYPE_MERGER:
      case OBJ_TYPE_MONEY:
      case OBJ_TYPE_WEAPON:
      case OBJ_TYPE_WEARABLE:
      case OBJ_TYPE_FOOD:
      case OBJ_TYPE_ROPE:
         return 1;
      default:
         return 0;
   }
}


/***********************************************************************
 ** is_an_item - is the object an item object
 **
 ** Parameters: Nothing
 **
 ** Returns: 1 if it is an item, 0 if not
 **
 ***********************************************************************/

int Entity::is_an_item()
{
   switch(obj_type)
   {
      case OBJ_TYPE_MARKER:
      case OBJ_TYPE_DOOR:
      case OBJ_TYPE_MOVEABLE:  
      case OBJ_TYPE_KEY:
      case OBJ_TYPE_BOAT: 
      case OBJ_TYPE_BOOK: 
      case OBJ_TYPE_MERGER:
      case OBJ_TYPE_MONEY:
      case OBJ_TYPE_WEAPON:
      case OBJ_TYPE_WEARABLE:
      case OBJ_TYPE_FOOD:
      case OBJ_TYPE_ROPE:
         return 1;
      default:
         return 0;
   }
}


/***********************************************************************
 ** is_an_ability - is the object an ability object
 **
 ** Parameters: Nothing
 **
 ** Returns: 1 if it is an ability, 0 if not
 **
 ***********************************************************************/

int Entity::is_an_ability()
{
   switch(obj_type)
   {
      case OBJ_TYPE_SPELL:
      case OBJ_TYPE_SKILL:
         return 1;
      default:
         return 0;
   }
}


/***********************************************************************
 ** is_an_individual - is the object an individual object
 **
 ** Parameters: Nothing
 **
 ** Returns: 1 if it is an individual, 0 if not
 **
 ***********************************************************************/

int Entity::is_an_individual()
{
   switch(obj_type)
   {
      case OBJ_TYPE_PLAYER:
      case OBJ_TYPE_MOBILE:
         return 1;
      default:
         return 0;
   }
}

/***********************************************************************
 ** is_merger - is this object a merger type
 **
 ** Parameters: None
 **
 ** Returns: 1 if it is a merger, 0 if not
 **
 ***********************************************************************/

int Entity::is_merger()
{
   switch(obj_type)
   {
      case OBJ_TYPE_MERGER:  
      case OBJ_TYPE_MONEY:
      case OBJ_TYPE_FOOD:
         return 1;
      default:
         return 0;
   }
}


/***********************************************************************
 ** get_area - retrieves the name of the object
 **
 ** Parameters: None
 **
 ** Returns: pointer to the name character string
 **
 ***********************************************************************/

char *Entity::get_area(void)
{
   return area.str_show();
}

/***********************************************************************
 ** set_area - sets the area of the object
 **
 ** Parameters: the_area - the name of the area
 **
 ** Returns:  1 for success
 **          -1 for failure
 **
 ***********************************************************************/

int Entity::set_area(char *the_area)
{
   /* if they sent us an empty area */
   if (the_area == NULL)
      return -1;

   /* copy the area into the string */
   area = the_area;

   /* if the string is too long, chop it off */
   if (area.str_len() > MAXNAMELEN)
      area.truncate(MAXNAMELEN-1);

   return 1;
}


/***********************************************************************
 ** copy_obj - copies an object to a different name
 **
 ** Parameters: new_name - the new name that the new object should have
 **
 ** Returns: a pointer to the new mudobject if it worked, NULL if failed
 **
 ***********************************************************************/

Entity *Entity::copy_obj(char *new_name)
{
   Entity *new_obj;

   if (new_name == NULL)
      return NULL;

   switch(get_type())
   {
      case OBJ_TYPE_BOOK:
      {  
         new_obj = (Entity *) new_Book(new_name, get_area());
         break;
      }      
      case OBJ_TYPE_DOOR:
      {
         new_obj = (Entity *) new_Door(new_name, get_area());
         break;
      }

      case OBJ_TYPE_KEY:
      {
         new_obj = (Entity *) new_Key(new_name, get_area());
         break;
      }

      case OBJ_TYPE_BOAT:
      {
         new_obj = (Entity *) new_Boat(new_name, get_area());
         break;
     }

      case OBJ_TYPE_LOCATION:
      {
         new_obj = (Entity *) new_Location(new_name, get_area());
         break;
      }

      case OBJ_TYPE_MARKER:
      {
         new_obj = (Entity *) new_Marker(new_name, get_area());
         break;
      }

      case OBJ_TYPE_MOBILE:
      {
         new_obj = (Entity *) new_Mobile(new_name, get_area());
         break;
      }

      case OBJ_TYPE_MOVEABLE:
      {
         new_obj = (Entity *) new_Moveable(new_name, get_area());
         break;
      }
      case OBJ_TYPE_WEARABLE:
      {
         new_obj = (Entity *) new_Wearable(new_name, get_area());
         break;
      }
      case OBJ_TYPE_WEAPON:
      {
         new_obj = (Entity *) new_Weapon(new_name, get_area());
         break;
      }
      case OBJ_TYPE_MERGER:
      {
         new_obj = (Entity *) new_Merger(new_name, get_area());
         break;
      }
      case OBJ_TYPE_MONEY:
      {
         new_obj = (Entity *) new_Money(new_name, get_area());
         break;
      }
      case OBJ_TYPE_FOOD:
      {
         new_obj = (Entity *) new_Food(new_name, get_area());
         break;
      }
      case OBJ_TYPE_ROPE:
      {
         new_obj = (Entity *) new_Rope(new_name, get_area());
         break;
      }
      case OBJ_TYPE_SPECIAL:
      {
         new_obj = (Entity *) new_Specials(new_name, get_area());
         break;
      }
      case OBJ_TYPE_ACTION:
      {
         new_obj = (Entity *) new_Action(new_name);
         break;
      }
      case OBJ_TYPE_TEXT:
      {
         new_obj = (Entity *) new_Text(new_name, 
                                         ((Text *) this)->get_text_type());
         break;
      }
      case OBJ_TYPE_LEVEL:
      {
         new_obj = (Entity *) new Level(new_name);
         break;
      }
      case OBJ_TYPE_BULLETIN:
      {
         new_obj = (Entity *) new_Bulletin(new_name, 
                                      ((Bulletin *)this)->is_build_port());
         break;
      }
      case OBJ_TYPE_RACE:
      {
         new_obj = (Entity *) new_Race(new_name);
         break;
      }
      case OBJ_TYPE_INCLINAT:
      {
         new_obj = (Entity *) new_Inclination(new_name);
         break;
      }
      case OBJ_TYPE_TALENT:
      {
         new_obj = (Entity *) new_Talent(new_name);
         break;
      }
      case OBJ_TYPE_MASK:
      {
         new_obj = (Entity *) new_Mask(new_name);
         break;
      }
      case OBJ_TYPE_CHATLINE:
      {
         new_obj = (Entity *) new_ChatLine(new_name);
         break;
      }
      case OBJ_TYPE_QUEST:
      {
         new_obj = (Entity *) new_Quest(new_name);
         break;
      }
      case OBJ_TYPE_SPELL:
      {
         new_obj = (Entity *) new_Spell(new_name);
         break;
      }
      case OBJ_TYPE_SKILL:
      {
         new_obj = (Entity *) new_Skill(new_name);
         break;
      }

      default:
         mainstruct->
             log_error("Attempt to copy an object of unrecognized type",
                                                       "copy_obj");
         return NULL; 
   }

   if (new_obj->copy_object(this) <= 0)
   {
#ifdef DEBUG_MEM
      printf("Warning!  Deleting object without logging it!");
#endif
      delete new_obj;
      return NULL;
   }
   return new_obj;
}


/***********************************************************************
 ** write_object - writes an entity to a file.  This write raises an
 **                error as we should not be writing just entities ever
 **
 ** Parameters: new_name - the new name that the new object should have
 **
 ** Returns: a pointer to the new mudobject if it worked, NULL if failed
 **
 ***********************************************************************/

void Entity::write_object(FILE *the_file, int build_format)
{
   build_format = 0;
   the_file = NULL;
   mainstruct->log_error("Tried to write entity to file.", "write_object");
}


/***********************************************************************
 ** describe - raises an error for trying to describe an entity
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Entity::describe(Player *the_player)
{
   the_player = NULL;
   mainstruct->log_error("Tried to describe an entity", "describe");
}


/***********************************************************************
 ** describe - raises an error for trying to describe an entity
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Entity::describe(Builder *the_builder)
{
   the_builder = NULL;
   mainstruct->log_error("Tried to describe an entity", "describe");
}


/***********************************************************************
 ** set_attrib - raises an error for trying to set attrib on an entity
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

int Entity::set_attrib(Builder *the_builder, Parse *the_parsed)
{
   the_builder = NULL;
   the_parsed = NULL;
   mainstruct->log_error("Tried to set attrib on an entity", "set_attrib");
   return -1;
}

/***********************************************************************
 ** get_mem_size_entity - gets how much memory is being taken up by the
 **                       entity part of this object
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Entity::get_mem_size_entity()
{
   int size = 0;

   size += name.get_mem_size_dynamic();
   size += area.get_mem_size_dynamic();
   return size;
}


/***********************************************************************
 ** get_mem_size - should not ever run this
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Entity::get_mem_size()
{
   return 0;
}

/***********************************************************************
 ** get_mem_size_dynamic - should not ever run this
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Entity::get_mem_size_dynamic()
{
   return 0;
}

#endif






