/**********************************************************************
 ** Main function: the main function that starts out the program
 **
 **    Reviewed through: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate), Ed Boraas
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef MAIN_C
#define MAIN_C

#include <locale.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include "config.h"
#include "sysdep.h"
#include "gettext.h"
#include "strings.h"
#include "configs.h"
#include "global.h"
#include "newfuncts.h"
#include "player.h"
#include "builder.h"
#include "utils.h"

void trap_signals();
void crash_on_signal(int sig);
void ignore_signals(int sig);
void signal_shutdown(int sig);
void disconnect_players(char *the_msg);
void disconnect_builders(char *the_msg);
int check_for_files(bool quiet_mode);
void check_directory(char *filename, char *dir_name);

int main(int argc, char *argv[]) {
   Strings holder;
   bool   quiet_mode = false;
   int    i = 0;   
   struct stat check_dir;
   char *poss_locale_locs[] = {
      ".." DSEP "locale",
      "." DSEP "locale",
      NULL
   };

   /* first find the locale directory */

   while ((poss_locale_locs[i] != NULL) && 
          (stat(poss_locale_locs[i], &check_dir) == -1))
   {
      i++;
   }

#ifdef HAVE_GETTEXT
   if ((poss_locale_locs[i] == NULL) ||
       (!S_ISDIR(check_dir.st_mode)))
   {
     sysmessage("Could not find locale directory.\n"
            "Locale (foreign language) support will not be available.\n");
   }
   else
   {
      /* set up the gettext utility */
      setlocale(LC_ALL, "");
      bindtextdomain(PACKAGE, poss_locale_locs[i]);
      textdomain(PACKAGE); 
   }
#endif

   for (i=1; i < argc; i++)
   {	   
      /* if we have an option here */
      if (argv[i][0] == '-')
      {  
         switch (argv[i][1]){
	 case 'q':
	 case 'Q':
	    quiet_mode = true;
            break;
         case 'f':
         case 'F':
            if ( argc > (i+1) )
	    { 
                the_config.conffile_loc = argv[i+1];
		break;
            }
	    sysmessage("Invalid format for -f option.\n");
	    //we fall though if there is no file name. 
	 default:
            sysmessage("Invalid option %s.\n", argv[i]);
	    sysmessage("Usage: %s [OPTIONS]\n"
		   "   Options are:\n"
		   "      -q -quiet: Starts the mud without any screen output\n"
	           "      -f <fileloc>: Allows one to indicate the location of aime.conf\n"                                               ,argv[0]);
	    
            exit(0);
         }
      }
   }

   if (!quiet_mode)
   {
      sysmessage(_(
  "\n\n----------------------------------------------------------------------\n"
  "| Advanced Interactive Mudding Environment version: %s       |\n"
  "| Copyright (C) 2000 George Noel (Slate), Ed Boraas, Kelly Gerke     |\n"
  "| Free Software licensed under the GNU General Public License        |\n"
  "----------------------------------------------------------------------\n\n"),
                                                    FULLVERSION);
   }

   mainstruct = NULL;

   if (!quiet_mode)
   {
      sysmessage(_("Reading in config file..."));
      fflush(stdout);
   }

   if (the_config.load_configs() <= 0)
   {
      sysmessage(_("Error in config file, failed.\n"));
      exit(0);
   }
   if (!quiet_mode)
      sysmessage(_("Done.\n"));


#if defined( AIME_DAEMONIZABLE )
   if (the_config.conf_flags->get_flag(CF_DAEMONIZE))
   {
      holder.sprintf("%s/%s/stderr.log", the_config.basedir.str_show(), 
                                               LOGDIR);
      int  fd = open(holder.str_show(),O_WRONLY|O_TRUNC|O_CREAT, S_IRUSR|S_IWUSR); 

      if (fd != 2)
      {   dup2(fd,2);
          close(fd);
      }
      fprintf(stderr,_("Stdout redirected to %s/%s/stderr.log.\n"),
              the_config.basedir.str_show(), LOGDIR);
   }
#endif // AIME_DAEMONIZABLE


   check_for_files(quiet_mode);

   mainstruct = NULL;
   mainstruct = new_Mud(quiet_mode);


#if defined( AIME_DAEMONIZABLE )
   if (the_config.conf_flags->get_flag(CF_DAEMONIZE))
   {

      if (!quiet_mode)
         sysmessage(_("Trapping Signals..."));
      trap_signals();
      if (!quiet_mode)
         sysmessage(_("Ok\n"));

      if (!quiet_mode)
         sysmessage(_("Daemonizing mud..."));
      fflush(stdout);
      pid_t the_pid;
      if (( the_pid = fork()) < 0)
      {   sysmessage(_("Fork failed, staying in foreground.\n"));
      }
      else 
      {   if (the_pid != 0)
          {   if (!quiet_mode)
                 sysmessage(_("Ok.\n"));
              return 0;
          }
          else
          {

              holder.sprintf("%s/%s/stdout.log", the_config.basedir.str_show(), 
                                               LOGDIR);
              int fd = open(holder.str_show(),O_WRONLY|O_TRUNC|O_CREAT, S_IRUSR|S_IWUSR); 

              if (fd != 1)
              {   dup2(fd,1);
                  close(fd);
              }

	      fprintf(stdout,_("Stdout redirected to %s/%s/stdout.log \n"),
			                    the_config.basedir.str_show(), LOGDIR);
	      

          }
      }
   }
#endif // AIME_DAEMONIZABLE


   mainstruct->start_timers();
   for (;;)
   {

      /* check the socket for a new connection */
      mainstruct->check_sockets();

      if (mainstruct->is_shutdown())
      {
         Strings holder;

         delete_Mud(mainstruct);
         sysmessage(_("Mud shut down successfully.\n"));
         return 1;
      }

      /* check the player list for any quitting players to remove */
      mainstruct->handle_quits();	      


      /* if we are single process, we need to go through all players and
         look for input */
      mainstruct->check_players();
      mainstruct->check_builders();

      mainstruct->check_events();
   }

}//end of main()



#if defined( AIME_SIGNALS )

/****************************************************************************
 ** trap_signals - sets up the signal handling so the mud wont be swept away
 **                from under our feet.
 ***************************************************************************/
void trap_signals()
{  
#if !defined( __CYGWIN__ )
   signal(SIGPWR,signal_shutdown);
#endif

   signal(SIGHUP,ignore_signals);
   signal(SIGCHLD,ignore_signals);
   signal(SIGPIPE,ignore_signals);
   signal(SIGINT,ignore_signals);
   signal(SIGSEGV,crash_on_signal);
   signal(SIGBUS,crash_on_signal);
   signal(SIGILL,crash_on_signal);
   signal(SIGFPE,crash_on_signal);
   signal(SIGQUIT,signal_shutdown);
   signal(SIGABRT,signal_shutdown);
   signal(SIGTERM,signal_shutdown);
}

void ignore_signals(int sig)
{   Strings the_msg;

/* we don't want this, just fills up the error logs */
//   the_msg.sprintf("Signal %d received, ignoring..\n",sig);
//   mainstruct->log_error(the_msg.str_show(),"signalhandling");
    signal(sig,ignore_signals);
}

void crash_on_signal(int sig)
{   Strings the_errormsg;

    the_errormsg.sprintf("*** Mud Received Signal %d, Shutting down server ***",sig);
    mainstruct->log_error(the_errormsg.str_show(),"signalhandling");

    /* Lose the players & builders */
    the_errormsg = "&+RSYSTEM ERROR: An internal error occured, shutting down the server for rescue operations.&*";

    disconnect_players(the_errormsg.str_show());
    disconnect_builders(the_errormsg.str_show());
    mainstruct->handle_quits();

    /* Shutdown of DB's and Calendar? */

    sysmessage("shutting down main mud.\n");
    mainstruct->shutdown();
    sysmessage("shutdown.\n");
}

void signal_shutdown(int sig)
{
   Strings the_msg;

#if defined( __CYGWIN__ )
   the_msg.sprintf("*** Shutting down Mud after receiving signal %d\n",sig);
   mainstruct->log_error(the_msg.str_show(),"signalhandling");
   the_msg = "&+RSYSTEM MESSAGE: Server is shutting down.&*\n";

#else
    switch (sig) {
    case SIGPWR:
	    mainstruct->log_error("*** Server lost power, shutting down Mud ***\n","signalhandling");
	    mainstruct->log_error("*** Disconnecting all users and shutting down normally.\n", "signalhandling");

	    the_msg = "&+RSYSTEM MESSAGE: The server has lost power, shutting down before it crashes.&*";
	    break;
    default:
	    the_msg.sprintf("*** Shutting down Mud after receiving signal %d\n",sig);
	    mainstruct->log_error(the_msg.str_show(),"signalhandling");
	    the_msg = "&+RSYSTEM MESSAGE: Server is shutting down.&*\n";
    }
#endif

    /* Lose the players & builders */
    disconnect_players(the_msg.str_show());
    disconnect_builders(the_msg.str_show());
    mainstruct->handle_quits();

    /* Shutdown of DB's and Calendar? */

    the_msg.sprintf("*** Killing Server on signal %d ***\n", sig);
    mainstruct->log_error(the_msg.str_show(),"signalhandling");
    mainstruct->shutdown();
}

#endif // AIME_SIGNALS



void disconnect_players(char *the_msg)
{   Player  *tmp_player;
    Strings the_logmsg;

    tmp_player = mainstruct->get_first_player();

    while (tmp_player != NULL)
    {  tmp_player->send_plr(the_msg);
       the_logmsg.sprintf("SHUTDOWN: Cutting connection to Player %s\n",tmp_player->get_name());
       mainstruct->log_error(the_logmsg.str_show(),"signalhandling");
       tmp_player->set_quitting();
       tmp_player->set_off();
       tmp_player = tmp_player->get_next_player();
    }
}

void disconnect_builders(char *the_msg)
{   Builder *tmp_builder;
    Strings the_logmsg;

    tmp_builder = mainstruct->get_first_builder();

    while (tmp_builder != NULL)
    {  
       tmp_builder->send_bldr(the_msg);

       the_logmsg.sprintf("SHUTDOWN: Cutting connection to Builder %s\n",tmp_builder->get_name());
       mainstruct->log_error(the_logmsg.str_show(),"signalhandling");
       tmp_builder->set_quitting();
       tmp_builder->set_off();
       tmp_builder = tmp_builder->get_next_builder();
    }
}


/****************************************************************************
 ** check_directory - checks to make sure the specified directory exits and
 **                   if not, asks the user if we should add it, then does it
 **
 ** Parameters: filename - the directory path to check for
 **             dir_name - a text description of the directory
 **
 ***************************************************************************/

#if defined( AIME_WIN32 )

void check_directory(char *base_path, char *dir_name)
{
   long lAttr;
   Strings fullpath;
   char user_input[11];

   fullpath = the_config.basedir.str_show();
   fullpath.str_cat("\\");
   fullpath.str_cat( base_path );

   lAttr = GetFileAttributes( fullpath.str_show( ) );
   if(    lAttr == 0xFFFFFFFF
       || ( lAttr & FILE_ATTRIBUTE_DIRECTORY ) == 0 )
   {
      sysmessage("%s directory missing.  "
             "Would you like to create it? (y/n) ", dir_name);
      fflush(stdout);

      fgets(user_input, 10, stdin);
      if (user_input[0] == 'n')
      {
         sysmessage("%s directory must exist.  Exiting.\n", dir_name);
         exit( 1 );
      }      

      if( CreateDirectory( fullpath.str_show( ), NULL ) == 0 )
      {
         sysmessage( "Error creating directory %s\n", dir_name );
         exit( 1 );
      }
   }
}

#else // AIME_WIN32

void check_directory(char *filename, char *dir_name)
{
   Strings the_filename;
   char user_input[11];
   struct stat check_file;

   the_filename.sprintf("%s/%s", the_config.basedir.str_show(), filename);
   if ((stat(the_filename.str_show(), &check_file) == -1) ||
       (!S_ISDIR(check_file.st_mode)))
  {
      sysmessage("%s directory missing.  "
             "Would you like to create it? (y/n) ", dir_name);
      fflush(stdout);

      fgets(user_input, 10, stdin);
      if (user_input[0] == 'n')
      {
         sysmessage("%s directory must exist.  Exiting.\n", dir_name);
         exit(0);
      }      

      the_filename.sprintf("%s/%s", the_config.basedir.str_show(), 
                                                                 filename);
      mkdir(the_filename.str_show(), 0770);
   }
}

#endif // AIME_WIN32

/****************************************************************************
 ** check_for_files - checks to make sure all required directories and files
 **                   exist in the right places
 ***************************************************************************/

#if defined( AIME_WIN32 )

int check_for_files(bool quiet_mode)
{
   Strings user_dir;
   char    user_input[11];
   char    char_counter[ 2 ];
   int     usr_missing = 0;
   long    lAttr;

   if (!quiet_mode)
   {
      sysmessage(_("Looking for necessary directories..."));
      fflush(stdout);
   }

   check_directory(DATA_DIR, "Data");
   check_directory(AREADIR, "Area");
   check_directory(BLDRDIR, "Builder");
   check_directory(BULLETINDIR, "Bulletin");
   check_directory(LOGDIR, "Log");
   check_directory(LOGARCHIVE, "Log archive");
   check_directory(LOGDIR, "Log");
   check_directory(USERDATADIR, "Users");

   strcpy( char_counter, "a" );
   while( *char_counter <= (int)'z' )
   {
      user_dir = the_config.basedir.str_show();
      user_dir.str_cat("\\");
      user_dir.str_cat( USERDATADIR );
      user_dir.str_cat( "\\" );
      user_dir.str_cat( char_counter );

      lAttr = GetFileAttributes( user_dir.str_show( ) );
      if(    lAttr == 0xFFFFFFFF
          || ( lAttr & FILE_ATTRIBUTE_DIRECTORY ) == 0 ) {
         if( !usr_missing )
         {
            sysmessage(_("One or more user directories are missing.\n"
                   "Would you like to create them? (y/n) "));
            fflush(stdout);
	 
            fgets(user_input, 10, stdin);
            if (user_input[0] == 'n')
            {
               sysmessage(_("All user directory must exist.  Exiting.\n"));
               exit( 1 );
            }
            usr_missing = 1;
         }      


         if( CreateDirectory( user_dir.str_show( ), NULL ) == 0 )
         {
            sysmessage(_("Error creating directory %s\n"), user_dir.str_show( ) );
            exit( 1 );
         }
      }

      *char_counter = *char_counter + 1;
   }
   if (!quiet_mode)
      sysmessage(_("Done!\n"));
   return 1;
}

#else // AIME_WIN32

int check_for_files(bool quiet_mode)
{
   Strings the_filename;
   char    user_input[11];
   FILE    *script_file = NULL;
   char    char_counter;
   int     usr_missing = 0;
   struct  stat check_file;

   if (!quiet_mode)
   {
      sysmessage(_("Looking for necessary directories..."));
      fflush(stdout);
   }


   the_filename.sprintf("%s/%s", the_config.basedir.str_show(), DATA_DIR);
   if ((stat(the_filename.str_show(), &check_file) == -1) ||
       (!S_ISDIR(check_file.st_mode)))
   {
      sysmessage(_("Fatal error, data directory '%s/%s' missing.  "
             "Reinstall Mud engine.\n"), the_config.basedir.str_show(), DATA_DIR); 
      exit(0);
   }

   check_directory(AREADIR, "Area");
   check_directory(BLDRDIR, "Builder");
   check_directory(BULLETINDIR, "Bulletin");
   check_directory(LOGDIR, "Log");
   check_directory(LOGARCHIVE, "Log archive");
   check_directory(LOGDIR, "Log");
   check_directory(USERDATADIR, "Users");
   check_directory(LOCALE_DIR, "Locale");

   char_counter = 'a';
   while (char_counter <= ((int) 'z'))
   {
      the_filename.sprintf("%s/%s/%c", the_config.basedir.str_show(), USERDATADIR, 
                                                                   char_counter);
      if ((stat(the_filename.str_show(), &check_file) == -1) ||
          (!S_ISDIR(check_file.st_mode)))
      {
         if (!usr_missing)
	 {
            sysmessage(_("One or more user directories are missing.\n"
                "Would you like to create them? (y/n) "));
            fflush(stdout);
	 
            fgets(user_input, 10, stdin);
            if (user_input[0] == 'n')
            {
               sysmessage(_("All user directory must exist.  Exiting.\n"));
               exit(0);
            }
            usr_missing = 1;
	 }      
         
         the_filename.sprintf("%s/%s/%c", the_config.basedir.str_show(),
                                                   USERDATADIR, char_counter);
	 mkdir(the_filename.str_show(), 0770);

      }
      char_counter++;
   }
   if (!quiet_mode)
      sysmessage(_("Done!\n"));
   return 1;
}

#endif // AIME_WIN32

#endif // MAIN_C

