/**********************************************************************
 ** Spell - contains the attributes needed to store info on a spell
 **
 **    
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef SPELL_C
#define SPELL_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "spell.h"
#include "objtype.h"
#include "newfuncts.h"
#include "flags.h"
#include "spellflags.h"
#include "player.h"

extern char *depend_name[];

/***********************************************************************
 ** Spell (constructor) - loads spell name and initializes attributes
 **
 ** Parameters: spell_name - this spell name 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Spell::Spell(char *spell_name)
{
   set_name(spell_name);
   set_area("ability");

   obj_type = OBJ_TYPE_SPELL;

   min_intel = 0;
   spell_flags = new_Flags(1);
}


/***********************************************************************
 ** ~Spell (destructor) - cleans up for destruction
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Spell::~Spell(void)
{
   delete_Flags(spell_flags);
}

/***********************************************************************
 ** set_min_intel - sets the minimum intelligence the caster must have
 **
 ** Parameters: new_min_intel - the new value
 **
 ***********************************************************************/

void Spell::set_min_intel(int new_min_intel)
{
   if (new_min_intel >= 0)
      min_intel = new_min_intel;
}


/***********************************************************************
 ** get_min_intel - returns the minimum intelligence the caster must have
 **
 ***********************************************************************/
   
int Spell::get_min_intel()
{
   return min_intel;
}


/***********************************************************************
 ** get_spellflags - returns a pointer to the spell flags
 **
 ***********************************************************************/

Flags *Spell::get_spellflags()
{
   return spell_flags;
}


/***********************************************************************
 ** read_spell_attrib - reads in spell attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Spell::read_spell_attrib(FILE *read_file, ErrLog *error_log)
{
   Strings      holder;

// This is now defunct
#if 0
   /* Set min_intel value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute MinInt for spell %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "read_spell_attrib");
      return -1;
   }
   set_min_intel(atoi(the_token->the_string));
#endif

   /* read in the spell flags */
   if (spell_flags->read_flags(read_file, error_log) <= 0)
   {
      holder.sprintf("Error reading spellflags for spell '%s@%s'", get_name(),
                                                                 get_area()); 
      error_log->log_err(holder.str_show(), "read_spell_attrib");
      return -1;
   }   

   return 1;
}


/***********************************************************************
 ** describe - describes the spell to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Spell::describe(Builder *the_builder)
{
  dependency *tmp_dep;

   the_builder->send_bldr("\n&+GSpell: \t\t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GActingRoom:&*\n%s\n", 
        (get_acting_str(Room) == NULL) ? "" : get_acting_str(Room));
   the_builder->send_bldr("&+GActingActor:&*\n%s\n",
        (get_acting_str(Actor) == NULL) ? "" : get_acting_str(Actor));
   the_builder->send_bldr("&+GActingTarget:&*\n%s\n",
        (get_acting_str(Target) == NULL) ? "" : get_acting_str(Target));
   the_builder->send_bldr("&+GSuccessRoom:&*\n%s\n",
        (get_success_str(Room) == NULL) ? "" : get_success_str(Room));
   the_builder->send_bldr("&+GSuccessActor:&*\n%s\n",
        (get_success_str(Actor) == NULL) ? "" : get_success_str(Actor));
   the_builder->send_bldr("&+GSuccessTarget:&*\n%s\n", 
        (get_success_str(Target) == NULL) ? "" : get_success_str(Target));
   the_builder->send_bldr("&+GFailRoom:&*\n%s\n",
        (get_failure_str(Room) == NULL) ? "" : get_failure_str(Room));
   the_builder->send_bldr("&+GFailActor:&*\n%s\n",
        (get_failure_str(Actor) == NULL) ? "" : get_failure_str(Actor));
   the_builder->send_bldr("&+GFailTarget:&*\n%s\n",
        (get_failure_str(Target) == NULL) ? "" : get_failure_str(Target));
   the_builder->send_bldr("&+GSpecials: \t&+g%s&*\n",
                  (get_special_name() == NULL) ? "" : get_special_name());
   the_builder->send_bldr("&+GSuccTrig: \t&+g%s&*\n",
                  (get_succ_trig() == NULL) ? "" : get_succ_trig());
   the_builder->send_bldr("&+GFailTrig: \t&+g%s&*\n",
                  (get_fail_trig() == NULL) ? "" : get_fail_trig());
   the_builder->send_bldr("&+GAttemptTrig: \t&+g%s&*\n",
                  (get_attempt_trig() == NULL) ? "" : get_attempt_trig());

   the_builder->send_bldr("&+GDrain: \t\t&+w%d&*\n", get_drain());
   the_builder->send_bldr("&+GMinInt: \t&+w%d&*\n\n", get_min_intel());

   the_builder->send_bldr("&+CDependencies:&*\n");
   tmp_dep = depend_list;
   while (tmp_dep != NULL)
   {
     the_builder->send_bldr("&+GType: \t\t&+g%s&*\n", depend_name[tmp_dep->the_type]);
     if (tmp_dep->the_type == MustHaveItem)
       the_builder->send_bldr("&+GDepString: \t&+w%s&*\n", 
			      tmp_dep->str_value.str_show());
     else
       the_builder->send_bldr("&+GDepNumber: \t&+w%d&*\n", tmp_dep->num_value);
     tmp_dep = tmp_dep->next_depend;
   }
   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the spell to a player
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Spell::describe(Player *the_player)
{
   dependency *tmp_dep;

   the_player->send_plr("\n&+GSpell: \t\t&+M%s&*\n", get_name());
   the_player->send_plr("&+GActingRoom:&*\n%s\n", 
        (get_acting_str(Room) == NULL) ? "" : get_acting_str(Room));
   the_player->send_plr("&+GActingActor:&*\n%s\n",
        (get_acting_str(Actor) == NULL) ? "" : get_acting_str(Actor));
   the_player->send_plr("&+GActingTarget:&*\n%s\n",
        (get_acting_str(Target) == NULL) ? "" : get_acting_str(Target));
   the_player->send_plr("&+GSuccessRoom:&*\n%s\n",
        (get_success_str(Room) == NULL) ? "" : get_success_str(Room));
   the_player->send_plr("&+GSuccessActor:&*\n%s\n",
        (get_success_str(Actor) == NULL) ? "" : get_success_str(Actor));
   the_player->send_plr("&+GSuccessTarget:&*\n%s\n", 
        (get_success_str(Target) == NULL) ? "" : get_success_str(Target));
   the_player->send_plr("&+GFailRoom:&*\n%s\n",
        (get_failure_str(Room) == NULL) ? "" : get_failure_str(Room));
   the_player->send_plr("&+GFailActor:&*\n%s\n",
        (get_failure_str(Actor) == NULL) ? "" : get_failure_str(Actor));
   the_player->send_plr("&+GFailTarget:&*\n%s\n",
        (get_failure_str(Target) == NULL) ? "" : get_failure_str(Target));
   the_player->send_plr("&+GSpecials: \t&+g%s&*\n",
                  (get_special_name() == NULL) ? "" : get_special_name());
   the_player->send_plr("&+GSuccTrig: \t&+g%s&*\n",
                  (get_succ_trig() == NULL) ? "" : get_succ_trig());
   the_player->send_plr("&+GFailTrig: \t&+g%s&*\n",
                  (get_fail_trig() == NULL) ? "" : get_fail_trig());
   the_player->send_plr("&+GAttemptTrig: \t&+g%s&*\n",
                  (get_attempt_trig() == NULL) ? "" : get_attempt_trig());
   the_player->send_plr("&+GDrain: \t\t&+w%d&*\n", get_drain());
   the_player->send_plr("&+GMinInt: \t&+w%d&*\n\n", get_min_intel());

   the_player->send_plr("&+CDependencies:&*\n");
   tmp_dep = depend_list;
   while (tmp_dep != NULL)
   {
     the_player->send_plr("&+GType: \t\t&+g%s&*\n", depend_name[tmp_dep->the_type]);
     if (tmp_dep->the_type == MustHaveItem)
       the_player->send_plr("&+GDepString: \t&+w%s&*\n", 
			      tmp_dep->str_value.str_show());
     else
       the_player->send_plr("&+GDepNumber: \t&+w%d&*\n", tmp_dep->num_value);
     tmp_dep = tmp_dep->next_depend;
   }
   the_player->send_plr("&+YSize: \t\t\t&+W%d&*\n", get_mem_size());
   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib_spellflags - for set attribute, sets the spellflags for any
 **                         spell
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Spell::set_attrib_spellflags(Parse *the_parsed, Builder *the_builder)
{
   Flags *tmp_spellflags;
   int flagnum;
   Strings holder;

   tmp_spellflags = get_spellflags();
      
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("Set which spell flag?\n");
      return -1;
   }
   holder.assign_word(the_parsed->get_speech(), 1);
     
   if ((flagnum = 
       tmp_spellflags->get_by_name(holder.str_show(), spellflagnames)) == -1)
   {
      the_builder->send_bldr("That is not an spell flag.\n");
      return -1;
   }

   holder.assign_word(the_parsed->get_speech(), 2);
   if (holder.str_show() == NULL)
   {
      the_builder->send_bldr("Set that flag to what?\n"
                                "Valid choices are: On or Off\n");
      return -1;
   }

   if (holder.str_n_cmp("On", holder.str_len()))
      tmp_spellflags->set_flag(flagnum);
   else if (holder.str_n_cmp("Off", holder.str_len()))
      tmp_spellflags->clr_flag(flagnum);
   else
   {
      the_builder->send_bldr("That is not a valid setting.\n"
                             "Valid choices are: On or Off\n");
      return -1;
   }
   the_builder->send_bldr("Flag &+M%s&* has been set to: %s\n",
       spellflagnames[flagnum], (tmp_spellflags->get_flag(flagnum)) ? 
			     "&+GOn&*" : "&+ROff&*");
   return 1;
 
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Spell::set_attrib(Builder *the_builder, Parse *the_parsed){
   int results = 0;
   if (the_parsed->get_target1() == NULL)
   {   the_builder->
            send_bldr("You can set the following attributes on a spell.\n"
               "   actingroom, actingactor, actingtarget, successroom,\n"
               "   successactor, successtarget, failroom, failactor,\n"
               "   failtarget, specials, drain, and minint\n");
       return -1;
   }

   if ((results = set_attrib_abil(the_builder, the_parsed)) == 1)
     return results;

   if (results != 0)
     return results;

   if (!STRNCASECMP(the_parsed->get_target1(), "spellflags",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_spellflags(the_parsed, the_builder);
   }

   the_builder->send_bldr("The attribute '%s' is not a spell attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** write_object - writes the spell to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Spell::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nspell %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   write_ability_attrib(the_file);

   spell_flags->write_flag(the_file);
}


/***********************************************************************
 ** copy_object - copies the spell to a spell of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Spell::copy_object(Entity *copy_obj)
{
   Spell *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_SPELL)
      return 0;

   copy_from = (Spell *) copy_obj;

   copy_ability_attrib((Ability *) copy_from);

   set_min_intel(copy_from->get_min_intel());   
   spell_flags->copy_flags(copy_from->get_spellflags());

   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Spell::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Spell::get_mem_size_dynamic()
{
   dependency *tmp_dep = depend_list;

   int  size = 0;
   
   size += spell_flags->get_mem_size();   
   size += get_mem_size_ability();
   size += get_mem_size_entity();

   while (tmp_dep != NULL)
   {
     size += sizeof(tmp_dep);
     size += tmp_dep->str_value.get_mem_size_dynamic();
     tmp_dep = tmp_dep->next_depend;
   }

   return size;
}

#endif






