/**********************************************************************
 ** Commands: This is a listing of all the commands that a player can
 **           call in the game. The functions here must pass back void
 **           and must accept a Parse pointer and a Player pointer, like
 **           this: void chatcom(Parse *this_parse, MudObject *this_user)
 **
 **
 ** Reviewed through:
 **
 **
 ** Copyright (C) 2001 George Noel (Slate), Ed Boraas, Kelly Gerke
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the
 **   License, or any later version.
 **
 **   This program is distributed in the hope that it will be useful, but
 **   WITHOUT ANY WARRANTY; without even the implied warranty of
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 **   General Public License for more details.
 **
 **   You should have received a copy of the GNU General Public License
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 **
 **********************************************************************/

#ifndef COMMANDS_C
#define COMMANDS_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "global.h"
#include "mudobject.h"
#include "objtype.h"
#include "indflags.h"
#include "locflags.h"
#include "gameflags.h"
#include "adminflags.h"
#include "comflags.h"
#include "spellflags.h"
#include "skillflags.h"
#include "bullflags.h"
#include "money.h"
#include "shop.h"
#include "mobile.h"
#include "boat.h"
#include "door.h"
#include "book.h"
#include "input.h"
#include "spell.h"
#include "skill.h"
#include "food.h"
#include "utils.h"
#include "specials.h"
#include "code.h"
#include "newfuncts.h"
#include "memchk.h"
#include "inp_funct.h"
#include "commands.h"
#include "wearable.h"
#include "access_list.h"
#include "user_dbase.h"
#include "data_log.h"
#include "syslog.h"
#include "errlog.h"
#include "verb_list.h"
#include "inclination.h"
#include "talent.h"
#include "chatline.h"
#include "doorflags.h"
#include "linkedlist.h"


/***********************************************************************
 ** shoutcom - sends a message to all who don't have noshout set
 **
 **
 ***********************************************************************/

void shoutcom(Parse *the_parsed, MudObject *the_user) {

   Strings holder;
   Player *the_player;
   Strings formatted;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;


   /* if there is nothing after shout, send a message to the player */
   if ((the_parsed->get_speech() == NULL)||
       (strlen(the_parsed->get_speech()) <= 0))
   {
      the_player->send_plr(_("Format: shout <phrase>\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You shout on the MUD in your dreams!\n"));
      return;
   }

   /* send a message to the player that chatted */
   formatted.format_for_comm(the_parsed->get_speech(), 11,
                                                  the_config.max_line_len);
   the_player->send_plr(_("&+MYou&+w shout &+W'&+M%s&+W'&*\n"),
                                                   formatted.str_show());

   /* send a message to the player that chatted */
   formatted.format_for_comm(the_parsed->get_speech(),
                  (strlen(the_player->get_name()), + 9), the_config.max_line_len);
   holder.sprintf(_("&+M%s &+wshouts &+W'&+M%s&+W'&*\n"),
                      the_player->get_name(), formatted.str_show());

   mainstruct->send_all_players(holder.str_show(), the_player);

}

/***********************************************************************
 ** saycom - sends a message to all in the room the player is in
 **
 **
 ***********************************************************************/

void saycom(Parse *the_parsed, MudObject *the_user) 
{    
    Location *this_loc;
    Strings holder, holder2;
    Player *the_player;
    Strings formatted;
    
    if (the_user->get_type() != OBJ_TYPE_PLAYER)
        return;
 
    the_player = (Player *) the_user;
    
    // if there is nothing after chat, send a message to the player
    if ((the_parsed->get_speech() == NULL) || (strlen(the_parsed->get_speech()) <= 0))
    {
        the_player->send_plr(_("Format: say <phrase>\n"));
        return;
    }
    
    if (the_player->is_asleep())
    {
        the_player->send_plr(_("You hold a discussion with your dreams.\n"));
        return;
    }
    
    if (((the_player->get_loc())->check_contained_spec(_("pre_say"), the_player, NULL, the_player, 0)) == 3)
        return;
    
    // send a message to the player that chatted
    formatted.format_for_comm(the_parsed->get_speech(), 9, the_config.max_line_len);
    the_player->send_plr(_("You say &+C'%s'&*\n"), formatted.str_show());
    
    // send a message to the player that chatted
    formatted.format_for_comm(the_parsed->get_speech(), (strlen(the_player->get_title()) + 7), the_config.max_line_len);
    // create the string for sending to other players
    formatted.sanitize_input();
    holder.sprintf(_("%s says '&+C%s&+C'&*\n"), the_player->get_name(), formatted.str_show());
    
    formatted.format_for_comm(the_parsed->get_speech(), 14, the_config.max_line_len);
    holder2.sprintf(_("Someone says '&+C%s&*'\n"), formatted.str_show());
    
    this_loc = the_player->get_loc();
    this_loc->send_location(holder.str_show(), the_player, CAN_SEE);
    this_loc->send_location(holder2.str_show(), the_player, CANT_SEE);
    
    if (((the_player->get_loc())->check_contained_spec(_("post_say"), the_player,
        NULL, the_player, 0)) == 3)
        return;    
}

/***********************************************************************
 **  incognitocom - sets a player's vislvl to VIS_INCOG
 **
 ***********************************************************************/

void incognitocom(Parse *the_parsed, MudObject *the_user)
{
    if (the_user->get_type() != OBJ_TYPE_PLAYER)
        return;

    ((Player *)the_user)->set_vislvl(VIS_INCOG);
    ((Player *)the_user)->send_plr("You are now incognito.\n");

    return;
}


/***********************************************************************
 **  gocom - tries to go a direction
 **
 ***********************************************************************/

void gocom(Parse *the_parsed, MudObject *the_user) {
   char *direction;
   Player *the_player;
   int flee = 0;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;


   direction = the_parsed->get_verb();
   if (!STRCASECMP(direction, _("flee")))
   {
     direction = the_parsed->get_target1();
     flee = 1;
   }
   else if (!STRCASECMP(direction, _("go")))
   {
     direction = the_parsed->get_target1();
     if (direction == NULL)
     {
        the_player->send_plr(_("Go where?\n"));
        return;
     }
   }
   
   the_player->move_dir(direction, flee);
   return;

}

/***********************************************************************
 **  lookcom - looks at a room/objects or in a container
 **
 ***********************************************************************/

void lookcom(Parse *the_parsed, MudObject *the_user) {
   Location *this_loc;
   Player *the_player;
   MudObject *the_obj;
   int       the_num = 0;
   Flags     *tmp_admflags;
   char      *live_or_not = _("the ");
   Player    *target_player = NULL;
   Strings   holder;
   Strings   target;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;

   the_player = (Player *) the_user;

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to do that.\n"));
      return;
   }

   this_loc = the_player->get_loc();

   /* if they don't give anything after look, then it is an examine of the room
      so we show the room desc to them */
   if (the_parsed->get_preposition() == NULL)
   {
      if (!STRCASECMP(the_parsed->get_verb(), _("examine")))
      {
         the_player->send_plr(_("Examine what?\n"));
         return;
      }

      if (the_player->is_asleep())
      {
          the_player->send_plr(
              _("Good heavens!!!  You seen a horrible drooling monster!\n"
              "Oh wait, this is just a dream.  You see your eyelids.\n"));
          return;
      }

      if (check_specials(_("pre_look"), this_loc, NULL, NULL,
                                                 this_loc, the_player) == 3)
         return;

      this_loc->show_location(the_player, 0);

      if (check_specials(_("post_look"), this_loc, NULL, NULL,
                                                 this_loc, the_player) == 3)
         return;

      return;
   }

   if (!this_loc->is_lit(the_player))
   {
      the_player->send_plr(_("There is not enough light to examine anything.\n"));
      return;
   }

   /* they are trying to look in something, show them the contents */
   if (!STRNCASECMP(the_parsed->get_preposition(), _("in"),
    		       strlen(the_parsed->get_preposition())))
   {
     if (the_player->is_asleep())
     {
       the_player->send_plr(_("Wake first (stand).\n"));
       return;
     }

     /* look in the location for the object */
     if (the_parsed->get_target1() == NULL)
     {
       the_player->send_plr(_("Look in what?\n"));
       return;
     }

     if ((the_obj = this_loc->find_contained(the_parsed->get_target1(),
					     &the_num)) == NULL)
     {
       /* look in the players inventory goes here */
       if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
						 &the_num)) == NULL)
       {  
	 the_player->send_plr(_("That is not here.\n"));
	 return;
       }
     }

     if (check_specials(_("pre_examine"), the_obj, NULL, NULL,
			the_obj, the_player) == 3)
       return;
     
     tmp_admflags = the_player->get_admflags();
     if (tmp_admflags->get_flag(ADMINFLAG_SHOWNAMES))
     {
       the_player->send_plr(_("&+gName: &+M%s&*\n"), the_obj->get_name());
     }
     
     if (the_obj->is_an_item())
     {
       Item       *the_item;
       Flags      *tmp_itemflags;
       Inventory  *the_list;
       
       the_item = (Item *) the_obj;
       tmp_itemflags = the_item->get_itemflags();
       
       if (tmp_itemflags->get_flag(ITEMFLAG_CONTAINER))
       {
	 if (tmp_itemflags->get_flag(ITEMFLAG_LOCKED))
	 {
	   the_player->send_plr(_("The %s is locked.\n"), the_item->get_title());
	 }
	 else if (tmp_itemflags->get_flag(ITEMFLAG_CLOSED))
	 {
	   the_player->send_plr(_("The %s is closed.\n"), the_item->get_title());
	 }
	 else
	 {
	   the_player->send_plr(_("The %s contains:"), the_item->get_title());
	   
	   the_list = the_item->get_inventory();
	   if (!the_list->display_list(the_player))
	     the_player->send_plr(_("Nothing.\n\n"));
	 }
       }
       else
       {
	 the_player->send_plr(_("That is not a container.\n"));
	 return;
       }
     }
     else
     {
       the_player->send_plr(_("You see nothing.\n"));
       return;
     }
   }
   else
   {
     if (STRCASECMP(the_parsed->get_preposition(), _("at")))
     {
       target = the_parsed->get_preposition();
     }
     else
     {
       target = the_parsed->get_target1();
     }
     
     if (the_player->is_asleep())
     {
       the_player->send_plr(
	       _("Argh!!!  You seen a horrible drooling monster!\n"
		 "Oh wait, this is just a dream.  You see your eyelids.\n"));
       return;
     }

     /* first check if the player wants to examine themselves */
     if (!STRCASECMP(target.str_show(), _("me")))
       the_obj = the_player;
     
     /* look in the location for the object */
     else if ((the_obj = this_loc->find_contained(target.str_show(),
						  &the_num)) == NULL)
     {
       /* look in the players inventory goes here */
       if ((the_obj = the_player->find_contained(target.str_show(),
						 &the_num)) == NULL)
       {
	 the_player->send_plr(_("That is not here.\n"));
	 return;
       }
     }
     
     if (check_specials(_("pre_examine"), the_obj, NULL, NULL,
			the_obj, the_player) == 3)
       return;
     
     tmp_admflags = the_player->get_admflags();
     if (tmp_admflags->get_flag(ADMINFLAG_SHOWNAMES))
     {
       the_player->send_plr(_("&+gName: &+M%s&*\n"), the_obj->get_name());
     }
     
     if (the_obj->get_type() == OBJ_TYPE_DOOR)
     {
       Door *the_door = (Door *) the_obj;
       
       // If the outside is supposed to be invisible to the player
       if (!the_door->is_visible(the_player->get_loc()))
       {
	 the_player->send_plr(_("That is not here.\n"));
	 return;
       }
       the_player->send_plr("%s", the_door->get_door_desc(this_loc));
     }
     else
     {
       if (the_obj->is_an_individual())
       {
	 Individual *tmp_ind;
	 
	 tmp_ind = (Individual *) the_obj;
	 if (((tmp_ind->get_indflags())->get_flag(INDFLAG_CORPSE)) &&
	     (tmp_ind->get_type() == OBJ_TYPE_MOBILE))
	 {
	   ((Mobile *) tmp_ind)->load_decay_str(&holder);
	   the_player->send_plr("%s\n", holder.str_show());
	 }
	 else
	   the_player->send_plr("%s", the_obj->get_desc());
       }
       else
	 the_player->send_plr("%s", the_obj->get_desc());
     }
     
     if ((the_obj->get_type() == OBJ_TYPE_PLAYER) ||
	 (the_obj->get_type() == OBJ_TYPE_MOBILE))
     {
       Individual *the_ind;
       Moveable *the_moveable;
       char *which_hand;
       char *a_or_an;
       char *sex;
       int both = 0;
       
       the_ind = (Individual *) the_obj;
       
       the_ind->display_top_worn(the_player);
       
       live_or_not = "";
       

	 sex = the_ind->get_sex_possessive_str();

       
       if ((the_moveable = the_ind->get_wielded(Left)) != NULL)
       {
	 if ((*(the_moveable->get_title()) == 'a') ||
	     (*(the_moveable->get_title()) == 'e') ||
	     (*(the_moveable->get_title()) == 'i') ||
	     (*(the_moveable->get_title()) == 'o') ||
	     (*(the_moveable->get_title()) == 'u'))
	 {
	   a_or_an = _("an");
	 }
	 else
	   a_or_an = _("a");
	 
	 if (the_moveable == the_ind->get_wielded(Right))
	 {
	   which_hand = _("both hands");
	   sex = "";
	   both = 1;
	 }
	 else
	   which_hand = _("left hand");
	 
	 the_player->send_plr(_("%s wields %s %s in %s%s.\n"),
		   the_ind->get_title(), a_or_an, the_moveable->get_title(),
		   sex, which_hand);
       }
       if ((!both) && ((the_moveable = the_ind->get_wielded(Right)) != NULL))
       {
	 if ((*(the_moveable->get_title()) == 'a') ||
	     (*(the_moveable->get_title()) == 'e') ||
	     (*(the_moveable->get_title()) == 'i') ||
	     (*(the_moveable->get_title()) == 'o') ||
	     (*(the_moveable->get_title()) == 'u'))
	 {
	   a_or_an = _("an");
	 }
	 else
	   a_or_an = _("a");
	 which_hand = _("right hand");
	 
	 the_player->send_plr(_("%s wields %s %s in %s%s.\n"),
		  the_ind->get_title(), a_or_an, the_moveable->get_title(),
			      sex, which_hand);
       }
       if ((!(the_ind->get_indflags())->get_flag(INDFLAG_CORPSE)) &&
	   (!(the_ind->get_indflags())->get_flag(INDFLAG_GHOST)))
	 the_ind->display_health(the_player);
       
       if (the_ind->get_type() == OBJ_TYPE_PLAYER)
       {
	 target_player = (Player *) the_ind;
	 if (target_player != the_player)
	   target_player->send_plr(_("%s examines you carefully.\n"),
				   the_player->get_title());
       }
     }
     else if (the_obj->is_an_item())
     {
       Item       *the_item;
       Flags      *tmp_itemflags;
       
       the_item = (Item *) the_obj;
       tmp_itemflags = the_item->get_itemflags();
       if (tmp_itemflags->get_flag(ITEMFLAG_LIT))
       {
	 the_player->send_plr(_("The %s is &+Ylit&*.\n"), the_item->get_title());
       }
     }
     
     holder.sprintf(_("%s examines %s%s carefully.\n"), the_player->get_name(),
		    live_or_not, the_obj->get_title());
     
     (the_player->get_loc())->send_location(holder.str_show(), the_player,
					    target_player);
   }

   if (check_specials(_("post_examine"), the_obj, NULL, NULL, the_obj, the_player)
       == 3)
     return;
   the_parsed = NULL;
}

/***********************************************************************
 **  exitscom - Gives all exits
 **
 ***********************************************************************/

void exitscom(Parse *the_parsed, MudObject *the_user) {
   Location *this_loc;
   Player *the_player;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You can exit your dream by waking (stand).\n"));
      return;
   }
   this_loc = the_player->get_loc();
   this_loc->show_exits(the_player);
   the_parsed = NULL;
}


/***********************************************************************
 **  socketcom - lists the current sockets open and who is populating
 **              them along with their hosts
 **
 ***********************************************************************/

void socketcom(Parse *the_parsed, MudObject *the_user) {
    Player  *the_list;
    Builder *bldr_list;
    int     count = 0;    /* the number of players online */
    Player  *the_player;
    struct  tm *time_struct;
    char    buff[30];
    time_t  idle_time;
    Connection *the_conn;
    int     is_game = 1;
    
    if (the_user->get_type() != OBJ_TYPE_PLAYER)
        return;
    the_player = (Player *) the_user;
    
    
    if (the_parsed->get_target1() != NULL)
    {
        if (!STRNCASECMP(_("game"), the_parsed->get_target1(),
            strlen(the_parsed->get_target1())))
        {
            is_game = 1;
        }
        else if (!STRNCASECMP(_("builder"), the_parsed->get_target1(),
            strlen(the_parsed->get_target1())))
        {
            is_game = 0;
        }
        else
        {
            the_player->send_plr(_("The '%s' port does not exist.\n"),
                the_parsed->get_target1());
            return;
        }
    }
    
    /* display the header */
    the_player->send_plr(_("&+WSocket List for %s&*\n\n"), is_game ? _("Game Port") :
    _("Builder Port"));
    
    the_player->send_plr(_("Socket      Name            Idle      Host&*\n"));
    the_player->send_plr(
        "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=&*\n");
    
    
    if (is_game)
    {
        the_list = mainstruct->get_first_player();
        while (the_list != NULL)
        {
            if (!((Player *)the_user)->can_see_ind(the_list))
            {
                the_list = the_list->get_next_player();
                continue;
            }

            idle_time = the_list->get_idle();
            time_struct = (struct tm *) gmtime((const time_t *) &idle_time);
            
            /* prepare idle string */
            BZERO(buff, 30);
            strftime (buff, 10, "%H:%M:%S", time_struct);
            
            the_conn = the_list->get_connection();
            the_player->send_plr("  &+W%-8d  &+M%-13s &+W%10s  &+m%s\n",
                the_conn->get_socknum(), the_list->get_title(), buff,
                (the_conn->get_host_addr() == NULL) ? the_conn->get_ip_addr() :
            the_conn->get_host_addr());
            count++;
            
            the_list = the_list->get_next_player();
        }
    } else {
        bldr_list = mainstruct->get_first_builder();
        while (bldr_list != NULL)
        {
            if (!((Builder *)the_user)->can_see_bldr(bldr_list))
            {
                bldr_list = bldr_list->get_next_builder();
                continue;
            }
            
            idle_time = bldr_list->get_idle();
            time_struct = (struct tm *) gmtime((const time_t *) &idle_time);
            
            /* prepare idle string */
            BZERO(buff, 30);
            strftime (buff, 10, "%H:%M:%S", time_struct);
            
            the_conn = bldr_list->get_connection();
            the_player->send_plr("  &+W%-8d  &+M%-13s &+W%10s  &+m%s\n",
                the_conn->get_socknum(), bldr_list->get_title(), buff,
                (the_conn->get_host_addr() == NULL) ? the_conn->get_ip_addr() :
            the_conn->get_host_addr());
            count++;
            
            bldr_list = bldr_list->get_next_builder();
        }
        
    }
    
    the_player->send_plr(
        "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=&*\n");
    
    the_player->send_plr(_("There %s &+W%d&* %s open.\n"),
        ((count == 1) ? _("is") : _("are")), count,
        ((count == 1) ? _("socket") : _("sockets")));
    the_parsed = NULL;
}


/***********************************************************************
 **  opencom - opens an openable object like a door
 **
 ***********************************************************************/

void opencom(Parse *the_parsed, MudObject *the_user) {
   Player      *the_player;
   MudObject   *the_obj;
   Door        *the_door;
   int         state;
   Location    *the_loc;
   Strings     holder, holder2;
   int         the_num = 0;
   Mobile      *guard;
   Location    *test_loc;
   MudObject   *loc_obj;
   Object_List *the_dbase;
   Flags       *tmp_itemflags = NULL;
   Flags       *tmp_doorflags = NULL;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   /* look in the location for the object */
   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Open what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to open your eyes first.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy.\n"));
      return;
   }

   the_loc = the_player->get_loc();

   if (!the_loc->is_lit(the_player))
   {
      the_player->send_plr(_("You can't see anything to open in the dark.\n"));
      return;
   }

   if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                       &the_num)) == NULL)
   {
      if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                       &the_num)) == NULL)
      {
         the_player->send_plr(_("That is not here.\n"));
         return;
      }
   }

   if (check_specials(_("pre_open"), the_obj, NULL, NULL,
                                 the_obj, the_player) == 3)
      return;

   if (the_obj->is_an_item())
      tmp_itemflags = ((Item *) the_obj)->get_itemflags();

   if (the_obj->get_type() != OBJ_TYPE_DOOR)
   {
      Item *the_item;


      if ((!the_obj->is_an_item()) ||
          ((!tmp_itemflags->get_flag(ITEMFLAG_LOCKABLE)) &&
           (!tmp_itemflags->get_flag(ITEMFLAG_CLOSEABLE))))
      {
         the_player->send_plr(_("That will not open.\n"));
         return;
      }

      the_item = (Item *) the_obj;

      if (!tmp_itemflags->get_flag(ITEMFLAG_CLOSED))
      {
         the_player->send_plr(_("The container is already open.\n"));
         return;
      }
      else if (tmp_itemflags->get_flag(ITEMFLAG_LOCKED))
      {
         the_player->send_plr(_("It is locked.\n"));
         return;
      }

      the_player->send_plr(_("You open the %s.\n"), the_item->get_title());
      holder.sprintf(_("%s opens the %s.\n"), the_player->get_name(),
                                              the_item->get_title());
      the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
      the_loc->send_location(_("You hear the sound of something being opened.\n"),
                                                 the_player, CANT_SEE);
      tmp_itemflags->clr_flag(ITEMFLAG_CLOSED);
      if (check_specials(_("post_open"), the_obj, NULL, NULL,
                                               the_obj, the_player) == 3)
         return;
      return;
   }

   the_door = (Door *) the_obj;

   the_dbase = mainstruct->get_dbase();

   // If they can't see it, don't let them know it is here.
   if (!the_door->is_visible(the_player->get_loc()))
   {
     the_player->send_plr(_("That is not here.\n"));
     return;
   }

   tmp_doorflags = the_door->get_doorflags();
   if ((tmp_doorflags->get_flag(DOORFLAG_ROPETIE)) ||
       (tmp_doorflags->get_flag(DOORFLAG_SPECIALDOOR)))
   {
      the_player->send_plr(_("That will not open.\n"));
      return;
   }

   loc_obj = the_dbase->get_mudobject(the_door->get_area(),
                                               the_door->get_outside_loc());
   if ((loc_obj == NULL) || (loc_obj->get_type() != OBJ_TYPE_LOCATION))
   {
      the_player->send_plr(_("Error! This door is missing a valid side!\n"));
      return;
   }

   test_loc = (Location *) loc_obj;

   if ((guard = the_door->is_guarded(test_loc)) == NULL)
   {

      loc_obj = the_dbase->get_mudobject(the_door->get_area(),
                                               the_door->get_location());

      if ((loc_obj == NULL) || (loc_obj->get_type() != OBJ_TYPE_LOCATION))
      {
         the_player->send_plr(_("Error! This door is missing a valid side!\n"));
         return;
      }
      test_loc = (Location *) loc_obj;
      guard = the_door->is_guarded(test_loc);
   }

   if (guard != NULL)
   {
      if ((test_loc == the_player->get_loc()) &&
          !(guard->get_indflags())->get_flag(INDFLAG_CORPSE))
      {
         the_player->
            send_plr(_("&+B%s&* tells &+Byou &+G'Leave that alone!'&*\n"),
                                                      guard->get_title());
         return;
      }
      else
      {
         the_player->send_plr(_("&+BSomeone &*shouts through the %s "
                       "&+m'Leave that alone!'&*\n"), the_door->get_title());
         return;
      }
   }

   state = the_door->get_door_state();
   switch(state) {
      case DOOR_OPEN:
         the_player->send_plr(_("The %s is already open.\n"),
                                                   the_door->get_title());
         return;

      case DOOR_CLOSED:
         the_player->send_plr(_("You open the %s.\n"), the_door->get_title());
         holder.sprintf(_("%s opens the %s.\n"), the_player->get_name(),
                                              the_door->get_title());
         the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
         the_loc->send_location(
            _("You hear the sound of something being opened.\n"),
                                                         the_player, CANT_SEE);
         the_door->set_door_state(DOOR_OPEN);

         if (check_specials(_("post_open"), the_obj, NULL, NULL,
                                               the_obj, the_player) == 3)
            return;

         return;

      case DOOR_LOCKED:
         the_player->send_plr(_("It appears to be locked.\n"));
         holder.sprintf(_("%s tries to open the %s and fails.\n"),
                          the_player->get_name(), the_door->get_title());
         the_loc->send_location(holder.str_show(), the_player);
         return;

      case DOOR_MLOCKED:
         the_player->send_plr(_("It appears to be magically locked.\n"));
         holder.sprintf(_("%s tries to open the %s and fails.\n"),
                          the_player->get_name(), the_door->get_title());
         the_loc->send_location(holder.str_show(), the_player);
         return;
      default:
         mainstruct->log_error(_("Invalid door state.\n"), "opencom");
         return;
   }
   return;
}

/***********************************************************************
 **  closecom - closes a closeable object like a door
 **
 ***********************************************************************/

void closecom(Parse *the_parsed, MudObject *the_user) {
   Player      *the_player;
   MudObject   *the_obj;
   Door        *the_door;
   int         state;
   Location    *the_loc;
   Strings     holder;
   int         the_num = 0;
   Mobile      *guard;
   Location    *test_loc;
   MudObject   *loc_obj;
   Object_List *the_dbase;
   Flags       *tmp_itemflags = NULL;
   Flags       *tmp_doorflags = NULL;


   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   /* look in the location for the object */
   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Close what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("Your eyes are already closed (wake up!).\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy.\n"));
      return;
   }

   the_loc = the_player->get_loc();

   if (!the_loc->is_lit(the_player))
   {
      the_player->send_plr(_("You can't see anything to close in the dark.\n"));
      return;
   }

   if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                      &the_num)) == NULL)
   {
      if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                      &the_num)) == NULL)
      {
         the_player->send_plr(_("That is not here.\n"));
         return;
      }
   }

   if (check_specials(_("pre_close"), the_obj, NULL, NULL,
                                 the_obj, the_player) == 3)
      return;


   if (the_obj->is_an_item())
      tmp_itemflags = ((Item *) the_obj)->get_itemflags();

   if (the_obj->get_type() != OBJ_TYPE_DOOR)
   {
      Item *the_item;

      if ((!the_obj->is_an_item()) ||
          ((!tmp_itemflags->get_flag(ITEMFLAG_LOCKABLE)) &&
           (!tmp_itemflags->get_flag(ITEMFLAG_CLOSEABLE))))
      {
         the_player->send_plr(_("You can't close that.\n"));
         return;
      }

      the_item = (Item *) the_obj;

      if ((tmp_itemflags->get_flag(ITEMFLAG_LOCKED)) ||
          (tmp_itemflags->get_flag(ITEMFLAG_CLOSED)))
      {
         the_player->send_plr(_("The container is already closed.\n"));
         return;
      }

      the_player->send_plr(_("You close the %s.\n"), the_item->get_title());
      holder.sprintf(_("%s closes the %s.\n"), the_player->get_name(),
                                              the_item->get_title());
      the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
         the_loc->send_location(
            _("You hear the sound of something being closed.\n"),
                                                         the_player, CANT_SEE);
      tmp_itemflags->set_flag(ITEMFLAG_CLOSED);
      if (check_specials(_("post_close"), the_obj, NULL, NULL,
                                               the_obj, the_player) == 3)
         return;
      return;
   }

   the_door = (Door *) the_obj;

   tmp_doorflags = the_door->get_doorflags();

   // If they can't see it, don't let them know it is here.
   if (!the_door->is_visible(the_player->get_loc()))
   {
     the_player->send_plr(_("That is not here.\n"));
     return;
   }

   if ((tmp_doorflags->get_flag(DOORFLAG_ROPETIE)) ||
       (tmp_doorflags->get_flag(DOORFLAG_SPECIALDOOR)))
   {
      the_player->send_plr(_("That will not close.\n"));
      return;
   }

   the_dbase = mainstruct->get_dbase();


   loc_obj = the_dbase->get_mudobject(the_door->get_area(),
                                               the_door->get_outside_loc());
   if ((loc_obj == NULL) || (loc_obj->get_type() != OBJ_TYPE_LOCATION))
   {
      the_player->send_plr(_("Error! This door is missing a valid side!\n"));
      return;
   }

   test_loc = (Location *) loc_obj;

   if ((guard = the_door->is_guarded(test_loc)) == NULL)
   {

      loc_obj = the_dbase->get_mudobject(the_door->get_area(),
                                               the_door->get_location());

      if ((loc_obj == NULL) || (loc_obj->get_type() != OBJ_TYPE_LOCATION))
      {
         the_player->send_plr(_("Error! This door is missing a valid side!\n"));
         return;
      }
      test_loc = (Location *) loc_obj;
      guard = the_door->is_guarded(test_loc);
   }

   if (guard != NULL)
   {
      if (test_loc == the_player->get_loc())
      {
         the_player->
            send_plr(_("&+B%s&* tells &+Byou &+G'Leave that alone!'&*\n"),
                                                      guard->get_title());
         return;
      }
      else
      {
	the_player->send_plr(_("&+BSomeone &*shouts through the %s "
                       "&+m'Leave that alone!'&*\n"), the_door->get_title());
         return;
      }
   }

   state = the_door->get_door_state();
   switch(state) {
      case DOOR_OPEN:
         the_player->send_plr(_("You close the %s.\n"), the_door->get_title());
         holder.sprintf(_("%s closes the %s.\n"), the_player->get_name(),
                                              the_door->get_title());
         the_loc->send_location(
            _("You hear the sound of something being closed.\n"),
                                                         the_player, CANT_SEE);
         the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
         the_door->set_door_state(DOOR_CLOSED);
         if (check_specials(_("post_close"), the_obj, NULL, NULL,
                                               the_obj, the_player) == 3)
            return;

         return;

      case DOOR_CLOSED:
      case DOOR_LOCKED:
      case DOOR_MLOCKED:
         the_player->send_plr(_("The %s is already closed.\n"),
                                                   the_door->get_title());
         return;

      default:
         mainstruct->log_error(_("Invalid door state.\n"), "closecom");
         return;
   }
   return;
}


/***********************************************************************
 **  tiecom - ties a rope, and in the process "opens" the "door"
 **
 ***********************************************************************/

void tiecom(Parse *the_parsed, MudObject *the_user) {
   Player      *the_player;
   MudObject   *the_obj;
   MudObject   *loc_obj;
   Rope        *rope_obj;
   Door        *the_door;
   int         state;
   Location    *the_loc;
   Strings     holder;
   int         the_num = 0;
   Mobile      *guard;
   Location    *inside_loc = NULL;
   Location    *outside_loc;
   Location    *guard_loc = NULL;
   Object_List *the_dbase;
   Flags       *tmp_doorflags;
   Strings     tmp_name;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   /* look in the location for the rope object firfst*/
   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Tie what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You tie a perfect square knot in your dreams.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to tie anything.\n"));
      return;
   }

   the_loc = the_player->get_loc();

   if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                       &the_num)) == NULL)
   {
      if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                       &the_num)) == NULL)
      {
         the_player->send_plr(_("That is not here.\n"));
         return;
      }
   }

   if (the_obj->get_type() != OBJ_TYPE_ROPE)
   {
      the_player->send_plr(_("You can't tie that, it is not a rope object.\n"));
      return;
   }

   rope_obj = (Rope *) the_obj;


   /* look in the location for the door to tie to */
   if (the_parsed->get_preposition() == NULL)
   {
      the_player->send_plr(_("Tie to what?\n"));
      return;
   }

   tmp_name = the_parsed->get_preposition();
   if (!STRCASECMP(tmp_name.str_show(), _("to")))
   {
      if (the_parsed->get_target2() == NULL)
      {
         the_player->send_plr(_("Tie %s to what?"), the_parsed->get_target1());
         return;
      }

      tmp_name = the_parsed->get_target2();
   }

   if ((the_obj = the_loc->find_contained(tmp_name.str_show(),
                                                       &the_num)) == NULL)
   {
      if ((the_obj = the_player->find_contained(tmp_name.str_show(),
                                                       &the_num)) == NULL)
      {
         the_player->send_plr(_("The '%s' is not here.\n"),
                                                    tmp_name.str_show());
         return;
      }
   }

   if (the_obj->get_type() != OBJ_TYPE_DOOR)
   {
      the_player->send_plr(_("You can't tie to that.\n"));
      return;
   }

   the_door = (Door *) the_obj;

   tmp_doorflags = the_door->get_doorflags();
   the_dbase = mainstruct->get_dbase();

   tmp_doorflags = the_door->get_doorflags();
   if (!tmp_doorflags->get_flag(DOORFLAG_ROPETIE))
   {
      the_player->send_plr(_("You can't tie to that.\n"));
      return;
   }

   loc_obj = the_dbase->get_mudobject(the_door->get_area(),
                                               the_door->get_outside_loc());
   if ((loc_obj == NULL) || (loc_obj->get_type() != OBJ_TYPE_LOCATION))
   {
      the_player->send_plr(_("Error! This rope door is missing a valid side!\n"));
      return;
   }

   outside_loc = (Location *) loc_obj;

   if ((guard = the_door->is_guarded(outside_loc)) == NULL)
   {

      loc_obj = the_dbase->get_mudobject(the_door->get_area(),
                                               the_door->get_location());

      if ((loc_obj == NULL) || (loc_obj->get_type() != OBJ_TYPE_LOCATION))
      {
         the_player->send_plr(_("Error! This door is missing a valid side!\n"));
         return;
      }
      inside_loc = (Location *) loc_obj;
      guard = the_door->is_guarded(inside_loc);
   }
   else
      guard_loc = outside_loc;

   if (guard != NULL)
   {
      if (guard_loc == NULL)
         guard_loc = inside_loc;

      if (guard_loc == the_player->get_loc())
      {
         the_player->
            send_plr(_("&+B%s&* tells &+Byou &+G'Leave that alone!'&*\n"),
                                                      guard->get_title());
         return;
      }
   }

   tmp_doorflags = the_door->get_doorflags();
   if (((the_loc == inside_loc) &&
        (!tmp_doorflags->get_flag(DOORFLAG_TIEINSIDE))) ||
       ((the_loc == outside_loc) &&
        (!tmp_doorflags->get_flag(DOORFLAG_TIEOUTSIDE))))
   {
      the_player->send_plr(_("You have to tie this from the other side.\n"));
      return;
   }

   if (check_specials(_("pre_tie"), the_door, rope_obj, NULL,
                                 the_door, the_player) == 3)
      return;

   state = the_door->get_door_state();
   switch(state) {
      case DOOR_OPEN:
      case DOOR_LOCKED:
         the_player->send_plr(_("There is already a rope tied here.\n"));
         return;

      case DOOR_CLOSED:
         the_player->send_plr(_("You tie the %s to the %s.\n"),
                              rope_obj->get_title(), the_door->get_title());
         holder.sprintf(_("%s ties the %s to the %s.\n"),
                the_player->get_name(), rope_obj->get_title(),
                                              the_door->get_title());
         the_loc->send_location(holder.str_show(), the_player);

         if (the_player->find_contained(rope_obj))
            the_player->remove_contained(rope_obj);

         if (the_door->get_distance() > rope_obj->get_rope_len())
	 {
            the_player->send_plr(_("It does not appear to be long enough.\n"));
            the_door->set_door_state(DOOR_LOCKED);
            the_loc->add_contained(rope_obj);
         }
         else
	 {
            the_door->set_door_state(DOOR_OPEN);
            rope_obj->set_dual_loc(the_door->get_location(),
                       the_door->get_outside_loc(), mainstruct->get_dbase());
         }

         rope_obj->set_tied();

         tmp_name.sprintf("%s@%s", rope_obj->get_name(),
                                                     rope_obj->get_area());
         the_door->set_rope_name(tmp_name.str_show());

         if (check_specials(_("post_tie"), the_door, rope_obj, NULL,
                                               the_door, the_player) == 3)
            return;

         return;

      default:
         mainstruct->log_error(_("Invalid rope door state.\n"), "tiecom");
         return;
   }
   return;
}

/***********************************************************************
 **  untiecom - unties a rope and in the process, "closes" the "door"
 **
 ***********************************************************************/

void untiecom(Parse *the_parsed, MudObject *the_user) {
   Player      *the_player;
   MudObject   *the_obj;
   MudObject   *loc_obj;
   MudObject   *tmp_obj;
   Door        *the_door = NULL;
   Rope        *the_rope;

   int         state;
   Location    *the_loc;
   Strings     holder;
   int         the_num = 0;
   Mobile      *guard;
   Location    *inside_loc = NULL;
   Location    *outside_loc;
   Location    *guard_loc = NULL;
   Object_List *the_dbase;
   Flags       *tmp_itemflags;
   Flags       *tmp_doorflags;
   Inventory   *the_inv;


   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   /* look in the location for the object */
   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Untie what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You untie a complex knot in your dream.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to untie anything.\n"));
      return;
   }

   the_loc = the_player->get_loc();

   if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                      &the_num)) == NULL)
   {
      if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                      &the_num)) == NULL)
      {
         the_player->send_plr(_("That is not here.\n"));
         return;
      }
   }

   if (the_obj->get_type() != OBJ_TYPE_ROPE)
   {
      the_player->send_plr(_("You can't untie that.\n"));
      return;
   }

   the_rope = (Rope *) the_obj;
   tmp_itemflags = the_rope->get_itemflags();

   if (!tmp_itemflags->get_flag(ITEMFLAG_ISTIED))
   {
      the_player->send_plr(_("The rope is not tied to anything.\n"));
      return;
   }

   holder.sprintf("%s@%s", the_rope->get_name(), the_rope->get_area());
   the_inv = the_loc->get_inventory();
   the_inv->reset_current();
   tmp_obj = the_inv->get_next();
   while (tmp_obj != NULL)
   {
      if (tmp_obj->get_type() == OBJ_TYPE_DOOR)
      {
         if ((!STRCASECMP(holder.str_show(),
                                    ((Door *)tmp_obj)->get_rope_name())) &&
             ((((Door*)tmp_obj)->get_doorflags())->
                                               get_flag(DOORFLAG_ROPETIE)))

	 {
            the_door = (Door *) tmp_obj;
            break;
	 }
      }
      tmp_obj = the_inv->get_next();
   }

   if (the_door == NULL)
   {
      the_player->send_plr(_("Error, door this rope is tied to not here!\n"));
      return;
   }

   the_dbase = mainstruct->get_dbase();


   loc_obj = the_dbase->get_mudobject(the_door->get_area(),
                                               the_door->get_outside_loc());
   if ((loc_obj == NULL) || (loc_obj->get_type() != OBJ_TYPE_LOCATION))
   {
      the_player->send_plr(_("Error! This door is missing a valid side!\n"));
      return;
   }

   outside_loc = (Location *) loc_obj;

   if ((guard = the_door->is_guarded(outside_loc)) == NULL)
   {

      loc_obj = the_dbase->get_mudobject(the_door->get_area(),
                                               the_door->get_location());

      if ((loc_obj == NULL) || (loc_obj->get_type() != OBJ_TYPE_LOCATION))
      {
         the_player->send_plr(_("Error! This door is missing a valid side!\n"));
         return;
      }
      inside_loc = (Location *) loc_obj;
      guard = the_door->is_guarded(inside_loc);
   }
   else
      guard_loc = outside_loc;

   if (guard != NULL)
   {
      if (guard_loc == NULL)
         guard_loc = inside_loc;

      if (guard_loc == the_player->get_loc())
      {
         the_player->
            send_plr(_("&+B%s&* tells &+Byou &+G'Leave that alone!'&*\n"),
                                                      guard->get_title());
         return;
      }
      else
      {
         the_player->send_plr(_("&+BSomeone &*shouts through the %s "
                       "&+m'Leave that alone!'&*\n"), the_door->get_title());
         return;
      }
   }

   tmp_doorflags = the_door->get_doorflags();
   if (((the_loc == inside_loc) &&
        (!tmp_doorflags->get_flag(DOORFLAG_UNTIEINSIDE))) ||
       ((the_loc == outside_loc) &&
        (!tmp_doorflags->get_flag(DOORFLAG_UNTIEOUTSIDE))))
   {
      the_player->send_plr(_("It is out of reach, try from the other side.\n"));
      return;
   }

   if (check_specials(_("pre_untie"), the_obj, NULL, NULL,
                                 the_obj, the_player) == 3)
      return;

   state = the_door->get_door_state();
   switch(state) {
      case DOOR_OPEN:
      case DOOR_LOCKED:
         the_player->send_plr(_("You untie the %s from the %s.\n"),
                  the_rope->get_title(), the_door->get_title());
         holder.sprintf(_("%s unties the %s from the %s.\n"),
                         the_player->get_name(), the_rope->get_title(),
                                                 the_door->get_title());
         the_loc->send_location(holder.str_show(), the_player);
         the_door->set_door_state(DOOR_CLOSED);

	 // Remove the rope from both locations, just to play it safe
         outside_loc->remove_contained(the_rope);
         inside_loc->remove_contained(the_rope);

	 // Now add it to the player's location
         if (the_loc == inside_loc)
	 {
            holder.sprintf("%s@%s", inside_loc->get_name(),
                                                   inside_loc->get_area());
	    inside_loc->add_contained(the_rope);
	 }
         else
	 {
            holder.sprintf("%s@%s", outside_loc->get_name(),
                                                   outside_loc->get_area());
	    outside_loc->add_contained(the_rope);
	 }
         the_rope->set_loc_str(holder.str_show());
         the_rope->set_untied();

         if (check_specials(_("post_untie"), the_obj, NULL, NULL,
                                               the_obj, the_player) == 3)
            return;

         return;

      case DOOR_CLOSED:
      case DOOR_MLOCKED:
      default:
         mainstruct->log_error(_("Invalid door state to have "
                              "gotten this far in the code."), "untiecom");
         return;
   }
   return;
}


/***********************************************************************
 **  readcom - Reads a book.
 **
 ***********************************************************************/
void readcom(Parse *the_parsed, MudObject *the_user)
{  MudObject *the_obj;
   Book      *the_book;
   Player    *the_plr;
   Location  *the_loc;
   int       the_num = 0;
   Bulletin  *the_board = NULL;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
       return;

   the_plr = (Player *) the_user;

   if (the_parsed->get_target1() == NULL)
   {  the_plr->send_plr(_("Read what?\n"));
      return;
   }

   if (the_plr->is_asleep())
   {
      the_plr->send_plr(_("You find it hard to read with your eyes shut.\n"));
      return;
   }

   if (the_plr->is_busy())
   {
      the_plr->send_plr(_("You are too busy to take the time to read.\n"));
      return;
   }

   the_loc = the_plr->get_loc();

   if (!the_loc->is_lit(the_plr))
   {
      the_plr->send_plr(_("It is too dark to read anything.\n"));
      return;
   }

   if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                         &the_num)) == NULL)
   {   /* try inventory */
       the_obj = the_plr->find_contained(the_parsed->get_target1(), &the_num);
   }

   if (the_obj == NULL)
   {   the_plr->send_plr(_("You can not find it.\n"));
       return;
   }

   if (the_obj->get_type() == OBJ_TYPE_BOOK)
   {  the_book = (Book *) the_obj;

      if (check_specials(_("pre_read"), the_book, NULL, NULL,
                                               the_book, the_plr) == 3)
            return;

      the_book->read_book(the_user);

      if (check_specials(_("post_read"), the_book, NULL, NULL,
                                               the_book, the_plr) == 3)
            return;

   }
   else if ((the_board = mainstruct->has_bulletin(the_obj)) != NULL)
   {
      Strings     *prompt;
      Inp_Handler *tmp_handler;

      tmp_handler = the_plr->get_input_handler();

      /* now enter them into the bulletin board handler */
      the_plr->send_plr(
         _("You read the list of bulletin entries on the %s.\n"),
                                                       the_obj->get_title());
      the_board->list_entries(the_plr);
      prompt = new Strings(_("&+gBoard&+W>&* "));
      /* enter the bulletin board here */
      tmp_handler->push_input_handler(bulletin_command_handler,
                                  prompt, 1, HANDLER_DATA_BULLETIN, 0);
      tmp_handler->set_data(the_board);
      return;
   }
   else
      the_plr->send_plr(_("You can not read that.\n"));
}

/***********************************************************************
 **  inventorycom - shows a player's inventory.. perhaps add so you
 **                 can show whats inside an object as well?
 **
 ***********************************************************************/
void inventorycom(Parse *the_parsed, MudObject *the_user)
{  Player     *the_player;
   Inventory  *the_list;
   Location   *the_loc;
   Strings    holder;

   the_player = (Player *) the_user;

   the_player->send_plr(_("You're carrying:"));

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("   Nothing!!!.\n\n"
               "Oh wait, nevermind, you are just dreaming.\n"));
      return;
   }

   if (check_specials(_("pre_inventory"), the_player, NULL, NULL,
                                               the_player, the_player) == 3)
            return;

   if (!(the_player->get_loc())->is_lit(the_player))
   {
      the_player->send_plr(
              _("\n   You can't see what is in your inventory in the dark.\n"));
      return;
   }

   the_list = the_player->get_inventory();
   if (the_list->display_list(the_player) == 0)
      the_player->send_plr(_("Nothing!\n\n"));
   else
   {
      Moveable *the_moveable;
      char *a_or_an;
      char *which_hand;
      int both = 0;

      the_player->display_all_worn();

      if ((the_moveable = the_player->get_wielded(Left)) != NULL)
      {
         if ((*(the_moveable->get_title()) == 'a') ||
             (*(the_moveable->get_title()) == 'e') ||
             (*(the_moveable->get_title()) == 'i') ||
             (*(the_moveable->get_title()) == 'o') ||
             (*(the_moveable->get_title()) == 'u'))
	 {
            a_or_an = _("an");
         }
         else
            a_or_an = _("a");
         if (the_moveable == the_player->get_wielded(Right))
	 {
            which_hand = _("both hands");
            both = 1;
         }
         else
            which_hand = _("your left hand");

         the_player->send_plr(_("\nYou wield %s %s in %s.\n"),
                      a_or_an, the_moveable->get_title(), which_hand);
      }
      if ((!both) &&
         ((the_moveable = the_player->get_wielded(Right)) != NULL))
      {
         if ((*(the_moveable->get_title()) == 'a') ||
             (*(the_moveable->get_title()) == 'e') ||
             (*(the_moveable->get_title()) == 'i') ||
             (*(the_moveable->get_title()) == 'o') ||
             (*(the_moveable->get_title()) == 'u'))
	 {
            a_or_an = _("an");
         }
         else
            a_or_an = _("a");
         which_hand = _("your right hand");

         the_player->send_plr(_("\nYou wield %s %s in %s.\n"),
                    a_or_an, the_moveable->get_title(), which_hand);
      }

      the_player->send_plr("\n");
   }

   the_loc = the_player->get_loc();
   holder.sprintf(_("%s rummages through %s belongings.\n"),
     the_player->get_title(), the_player->get_sex_possessive_str());
   the_loc->send_location(holder.str_show(), the_player, CAN_SEE);

   if (check_specials(_("post_inventory"), the_player, NULL, NULL,
                                               the_player, the_player) == 3)
            return;

   /* To remove the warning about parsed not being used. */
   the_parsed = the_parsed;
}

/***********************************************************************
 **  dropcom - gets an object
 **
 ***********************************************************************/
void dropcom(Parse *the_parsed, MudObject *the_user) {
  Player    *the_player = (Player *)the_user;
  MudObject *the_obj;
  Location  *the_loc;
  Strings   holder;
  int       the_num = 0;

  if (the_player == NULL)
     return;

  the_loc = the_player->get_loc();

  if (the_parsed->get_target1() == NULL)
  {   the_player->send_plr(_("What do you want to drop?\n"));
      return;
  }
  else if (the_player->is_asleep())
  {
     the_player->send_plr(_("You need to wake first (stand).\n"));
     return;
  }
  else if (the_player->is_busy())
  {
     the_player->send_plr(_("You are too busy to drop anything.\n"));
     return;
  }

  else
  {
      if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
      {   if (STRCASECMP(the_parsed->get_target1(),_("all")) == 0)
          {   Inventory *the_inventory;

              the_inventory = the_player->get_inventory();
              if (the_inventory == NULL)
              {    mainstruct->log_error(
                          _("Request for inventory returned NULL!"), "dropcom");
                   return;
              }

              the_inventory->reset_current();
              the_obj = the_inventory->get_next();
              while (the_obj != NULL)
              {  if (the_obj->is_a_moveable())
                 {
                    if (check_specials(_("pre_drop"), the_obj, NULL, NULL,
                                               the_obj, the_player) == 3)
                       return;

                    the_player->send_plr(_("You drop the %s.\n"),
                                               the_obj->get_title());
                    holder.sprintf(_("%s drops the %s.\n"),
                            the_player->get_name(), the_obj->get_title());
                    the_loc->send_location(holder.str_show(),the_player,
                                                                      CAN_SEE);
                    the_loc->send_location(
                            _("You hear a thud as something hits the ground.\n"),
                                                    NULL, CANT_SEE);
                    if (the_obj->get_type() == OBJ_TYPE_WEARABLE)
                       the_player->remove_item((Wearable *) the_obj);
                    the_player->remove_contained(the_obj);
                    the_loc->add_contained(the_obj);
                    the_inventory->reset_current();
                    if (check_specials(_("post_drop"), the_obj, NULL, NULL,
                                               the_obj, the_player) == 3)
                       return;
                 }
                 the_obj = the_inventory->get_next();
              }
          }
          else
              the_player->send_plr(_("You're not carrying a %s\n"),
                                                the_parsed->get_target1());

          return;
      }

      if (the_obj->is_a_moveable())
      {
         if (check_specials(_("pre_drop"), the_obj, NULL, NULL,
                                               the_obj, the_player) == 3)
                       return;

         /* Check if can carry */
         /* move to players inventory */
         if (the_parsed->get_number() > 0)
         {
            Merger *the_merger;
            Merger *new_merger;
            MudObject *parent_merger;
            int    new_num;

            if (!the_obj->is_merger())
            {
               if (the_obj->get_type() == OBJ_TYPE_WEARABLE)
                  the_player->remove_item((Wearable *) the_obj);
               the_player->remove_contained(the_obj);
            }
            else
            {
               the_merger = (Merger *) the_obj;

               if (the_parsed->get_number() > the_merger->get_number_of())
	       {
                  the_player->send_plr(_("You don't have that many.\n"));
                  return;
               }
               if (the_parsed->get_number() == the_merger->get_number_of())
	       {
                  the_player->remove_contained(the_obj);
               }
               else
	       {
                  if (the_merger->get_parent() != NULL)
	          {
                     parent_merger =
                           mainstruct->get_object(the_merger->get_parent());
                     new_merger = (Merger *)
                        (mainstruct->get_dbase())->clone_object(parent_merger);
                  }
                  else
                     new_merger = (Merger *)
                          (mainstruct->get_dbase())->clone_object(the_obj);
                  new_merger->set_number_of(the_parsed->get_number());
                  the_obj = (MudObject *) new_merger;
                  new_num =
                      the_merger->get_number_of() - the_parsed->get_number();
                  the_merger->set_number_of(new_num);
               }
            }
         }
         else
	 {
            if (the_obj->get_type() == OBJ_TYPE_WEARABLE)
               the_player->remove_item((Wearable *) the_obj);
            the_player->remove_contained(the_obj);
         }

         the_player->send_plr(_("You drop the %s.\n"),the_obj->get_title());
         holder.sprintf(_("%s drops the %s.\n"),the_player->get_name(),
                                                       the_obj->get_title());
         the_loc->send_location(holder.str_show(),the_player, CAN_SEE);
         the_loc->send_location(
            _("You hear a thud as something hits the ground.\n"), NULL, CANT_SEE);

         the_loc->add_contained(the_obj);

         if (check_specials(_("post_drop"), the_obj, NULL, NULL,
                                               the_obj, the_player) == 3)
                       return;

      }
      else
         the_player->send_plr(_("You cannot drop the %s\n"),the_obj->get_title());

   }
}

/***********************************************************************
 **  getcom - gets an object
 **
 ***********************************************************************/

void getcom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Location *the_loc;
   MudObject *the_obj;
   MudObject *from_loc;
   Strings   holder;
   char      *fromptr;
   Strings   fromobj_str;
   Strings   obj_str;
   int      the_num = 0;
   int      player_holds = 1;
   Individual *target_ind = NULL;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {   the_player->send_plr(_("format: get [<number>] <object> [from] <object>\n"));
       return;
   }

   the_loc = the_player->get_loc();

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to wake first (stand).\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to grab something.\n"));
      return;
   }

   if (!the_loc->is_lit(the_player))
   {
      the_player->send_plr(_("You fumble around in the dark but find nothing.\n"));

      holder.sprintf(_("%s fumbles around blindly looking for something.\n"),
                                                   the_player->get_title());
      the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
      the_loc->send_location(_("The sound of commotion springs up nearby.\n"),
                                                 the_player, CANT_SEE);
      return;
   }

   if (the_parsed->get_preposition() != NULL)
   {
      Flags *from_itemflags;

      if (STRNCASECMP(the_parsed->get_preposition(), _("from"),
                                   strlen(the_parsed->get_preposition())))
      {
         fromptr = the_parsed->get_preposition();
      }
      else
      {
         fromptr = the_parsed->get_target2();
      }

      if (fromptr == NULL)
      {
         the_player->
              send_plr(_("Get %s from what?\n"), the_parsed->get_target1());
         return;
      }

      the_player->check_pronouns(fromptr, &fromobj_str);

      if ((from_loc = the_player->find_contained(fromobj_str.str_show(),
                                                          &the_num)) == NULL)
      {
         player_holds = 0;
         if ((from_loc = the_loc->find_contained(fromobj_str.str_show(),
                                                          &the_num)) == NULL)
	 {
            the_player->send_plr(_("I can't seem to find the %s\n"),
                                               fromobj_str.str_show());
            return;
         }
      }

      if ((!from_loc->is_an_item()) &&
          (!from_loc->is_an_individual()))
      {
         printf(_("That can't contain anything.\n"));
         return;
      }

      /* if they are trying to take from an individual (for corpses) */
      if (from_loc->is_an_individual())
      {
         target_ind = (Individual *) from_loc;

         if (!(target_ind->get_indflags())->get_flag(INDFLAG_CORPSE))
	 {
            the_player->send_plr(_("%s is alive and won't let you take from them.\n"),
                                         target_ind->get_title());
            return;
	 }

      }
      else
      {
         from_itemflags = ((Item *) from_loc)->get_itemflags();

         if (from_itemflags->get_flag(ITEMFLAG_CLOSED))
         {
            the_player->send_plr(_("The %s is closed.\n"), from_loc->get_title());
            return;
         }
      }
   }
   else
   {
      player_holds = 0;
      from_loc = (MudObject *) the_loc;
   }

   the_player->check_pronouns(the_parsed->get_target1(), &obj_str);

   if (STRCASECMP(obj_str.str_show(), _("all")) == 0)
   {   Inventory *the_inventory;
       int results;

       the_inventory = from_loc->get_inventory();
       if (the_inventory == NULL)
       {    mainstruct->log_error(_("Request for inventory returned NULL!"),
                                                                 "getcom");
            return;
       }

       the_inventory->reset_current();
       the_obj = the_inventory->get_next();
       while (the_obj != NULL)
       {  if (the_obj->is_a_moveable())
          {
             if ((results = the_player->
                  game_get_obj(the_obj, from_loc, 0, player_holds)) > 0)
                the_inventory->reset_current();
             else
                return;
          }
          the_obj = the_inventory->get_next();
       }
       return;
   }

   if ((the_obj = from_loc->find_contained(obj_str.str_show(),
                                                          &the_num)) == NULL)
   {
      the_player->send_plr(_("I can't seem to find the %s.\n"),
                                               obj_str.str_show());
      return;
   }

   if (the_obj == NULL || from_loc == NULL)
   {   mainstruct->log_error(_("Error in getcom, the_obj or from_loc was "
                             "NULL while a target was expected.\n"), "getcom");
       return;
   }

   the_player->game_get_obj(the_obj, from_loc, the_parsed->get_number(),
                                                            player_holds);
   return;

}


/***********************************************************************
 **  shutdowncom - shuts down the mud nicely
 **
 ***********************************************************************/

void shutdowncom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;


   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   the_player->send_plr(_("Shutting down the mud.\n"));
   mainstruct->shutdown();

   the_parsed = NULL;
}


/***********************************************************************
 ** bhostscom - the hosts allowed to connect to the builder port
 ***********************************************************************/

void bhostscom(Parse *the_parsed, MudObject *the_user) {
   Player      *the_player;
   Strings     tmp_word;
   int         count = 0;
   Access_List *tmp_access_list;
   Strings     holder, holder2;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   tmp_access_list = mainstruct->get_bldr_access_list();

   if ((the_parsed->get_target1() == NULL) ||
      (!STRNCASECMP(the_parsed->get_target1(), _("list"),
                               strlen(the_parsed->get_target1()))))
   {
      the_player->
         send_plr(_("&+WUsers '%s' by default unless host is listed here:\n"),
                       the_config.conf_flags->get_flag(CF_BLDRALLOW) ?
                                            _("&+GAllowed&*") : _("&+RDenied&*"));
      the_player->send_plr("&+M--------------------------------------&*\n\n");
      count = tmp_access_list->display_list(the_player);
      the_player->send_plr("\n&+M--------------------------------------&*\n");
      the_player->send_plr(_("&+WNumber of entries: &+Y%d&*\n\n"), count);
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("add"),
                               strlen(the_parsed->get_target1())))
   {
      tmp_word = the_parsed->get_speech();
      if (tmp_word.str_show() == NULL)
      {
         the_player->send_plr(_("You need to specify what to add.\n"));
         return;
      }

      if (tmp_access_list->add_entry(tmp_word.str_show()) < 0)
      {
         the_player->send_plr(_("Error adding entry, not added.\n"));
         return;
      }

      holder.sprintf(_("&+m%s&* added bhost '&+c%s&*'"),
                the_player->get_title(), tmp_word.str_show());

      /* now log their death if appropriate */
      if (the_config.conf_flags->set_flag(CF_LOGBHOSTS))
      {
         mainstruct->log_event(holder.str_show());
      }

      holder2.sprintf("&+Y[&*%s&+Y]&*\n", holder.str_show());
      mainstruct->send_all_players(holder2.str_show(), the_player,
                                                        ADMINFLAG_SEEBHOSTS);

      the_player->send_plr(_("Entry added successfully.\n"));
      return;
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("delete"),
                               strlen(the_parsed->get_target1())))
   {
      tmp_word = the_parsed->get_speech();
      if (tmp_word.str_show() == NULL)
      {
         the_player->send_plr(_("You need to specify what to delete.\n"));
         return;
      }

      if (tmp_access_list->del_entry(tmp_word.str_show()) <= 0)
      {
         the_player->send_plr(_("Couldn't find entry '%s'; not deleted.\n"),
                                                         tmp_word.str_show());
         return;
      }

      holder.sprintf(_("&+m%s&* deleted bhost '&+c%s&*'"),
                the_player->get_title(), tmp_word.str_show());

      /* now log their death if appropriate */
      if (the_config.conf_flags->set_flag(CF_LOGBHOSTS))
      {
         mainstruct->log_event(holder.str_show());
      }

      holder2.sprintf("&+Y[&*%s&+Y]&*\n", holder.str_show());
      mainstruct->send_all_players(holder2.str_show(), the_player,
                                                        ADMINFLAG_SEEBHOSTS);

      the_player->send_plr(_("Entry deleted successfully.\n"));
      return;

   }
   else
   {
      the_player->send_plr(_("The command '%s' is not a bhosts command.\n"),
                                        the_parsed->get_target1());
      return;
   }
}

/***********************************************************************
 ** banhostscom - the hosts denied from the game port
 ***********************************************************************/

void banhostscom(Parse *the_parsed, MudObject *the_user) {
   Player      *the_player;
   Strings     tmp_word;
   int         count = 0;
   Access_List *tmp_access_list;
   Strings     holder, holder2;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   tmp_access_list = mainstruct->get_game_access_list();

   if ((the_parsed->get_target1() == NULL) ||
      (!STRNCASECMP(the_parsed->get_target1(), _("display"),
                               strlen(the_parsed->get_target1()))))
   {
      the_player->
         send_plr(_("&+WUsers '%s' by default unless host is listed here:\n"),
                       the_config.conf_flags->get_flag(CF_GAMEALLOW) ?
                                           _("&+GAllowed&*") : _("&+RDenied&*"));
      the_player->send_plr("&+M--------------------------------------&*\n\n");
      count = tmp_access_list->display_list(the_player);
      the_player->send_plr("\n&+M--------------------------------------&*\n");
      the_player->send_plr(_("&+WNumber of entries: &+Y%d&*\n\n"), count);
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("add"),
                               strlen(the_parsed->get_target1())))
   {
      tmp_word = the_parsed->get_speech();
      if (tmp_word.str_show() == NULL)
      {
         the_player->send_plr(_("You need to specify what to add.\n"));
         return;
      }

      if (tmp_access_list->add_entry(tmp_word.str_show()) < 0)
      {
         the_player->send_plr(_("Error adding entry, not added.\n"));
         return;
      }

      holder.sprintf(_("&+m%s&* added banhost '&+c%s&*'"),
                the_player->get_title(), tmp_word.str_show());

      /* now log their death if appropriate */
      if (the_config.conf_flags->set_flag(CF_LOGBANHOSTS))
      {
         mainstruct->log_event(holder.str_show());
      }

      holder2.sprintf("&+Y[&*%s&+Y]&*\n", holder.str_show());
      mainstruct->send_all_players(holder2.str_show(), the_player,
                                                        ADMINFLAG_SEEBANHOSTS);

      the_player->send_plr(_("Entry added successfully.\n"));
      return;
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("delete"),
                               strlen(the_parsed->get_target1())))
   {
      tmp_word = the_parsed->get_speech();
      if (tmp_word.str_show() == NULL)
      {
         the_player->send_plr(_("You need to specify what to delete.\n"));
         return;
      }

      if (tmp_access_list->del_entry(tmp_word.str_show()) <= 0)
      {
         the_player->send_plr(_("Couldn't find entry '%s'; not deleted.\n"),
                                                         tmp_word.str_show());
         return;
      }
      the_player->send_plr(_("Entry deleted successfully.\n"));
      return;

   }
   else
   {
      the_player->send_plr(_("The command '%s' is not a banhosts command.\n"),
                                        the_parsed->get_target1());
      return;
   }
}

/***********************************************************************
 ** flagscom - sets, clears, and displays flags for individuals
 ***********************************************************************/

void flagscom(Parse *the_parsed, MudObject *the_user) {
   Player      *the_player = NULL;
   Player      *target_player = NULL;
   Builder     *target_builder = NULL;
   User_Dbase  *tmp_dbase = NULL;
   Flags       *tmp_flags = NULL;
   Flags       *tmp_bldr_flags = NULL;
   Flags       flag_holder;
   Strings     the_name;
   Strings     holder;
   Strings     holder2;
   int         flagnum;
   int         for_all = 0;
   int         flagtype;  // 1 = comflags, 2 = adminflags, 3 = gameflags
   char        *flagnames;
   char        **flagnamelist;
   int         not_on = 0;
   int         flag_on = 0;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (!STRCASECMP(the_parsed->get_verb(), _("comflags")))
   {
      flagtype = 1;
      flagnames = _("ComFlags");
      flagnamelist = comflagnames;
   }
   else if (!STRCASECMP(the_parsed->get_verb(), _("adminflags")))
   {
      flagtype = 2;
      flagnames = _("AdminFlags");
      flagnamelist = admflagnames;
   }
   else if (!STRCASECMP(the_parsed->get_verb(), _("gameflags")))
   {
      flagtype = 3;
      flagnames = _("GameFlags");
      flagnamelist = gameflagnames;
   }
   else
   {
      the_player->send_plr(_("Invalid verb given.\n"));
      return;
   }

   if ((the_parsed->get_target1() == NULL) ||
       (!STRCASECMP(the_parsed->get_target1(), _("me"))))
      the_name = the_player->get_name();
   else
      the_name = the_parsed->get_target1();

   /* if the builder is logged on, we change him too */
   if ((target_builder = mainstruct->get_builder(the_name.str_show())) != NULL)
   {
      if (flagtype == 1)
         tmp_bldr_flags = target_builder->get_comflags();
      else if (flagtype == 2)
         tmp_bldr_flags = target_builder->get_adminflags();
      else if (flagtype == 3)
         tmp_bldr_flags = target_builder->get_gameflags();
   }

   if ((target_player = mainstruct->get_player(the_name.str_show()))
                                                                  == NULL)
   {

      the_name.upper(0);
      tmp_dbase = mainstruct->get_user_database();

      target_player = new_Player(the_name.str_show());

      if (tmp_dbase->read_user(the_name.str_show(), target_player) < 1)
      {
         the_player->send_plr(_("That user does not exist.\n"));
         return;
      }

      if (flagtype == 1)
         tmp_flags = target_player->get_comflags();
      else if (flagtype == 2)
         tmp_flags = target_player->get_admflags();
      else if (flagtype == 3)
         tmp_flags = target_player->get_gameflags();
      not_on = 1;
   }
   else
   {
      if (flagtype == 1)
         tmp_flags = target_player->get_comflags();
      else if (flagtype == 2)
         tmp_flags = target_player->get_admflags();
      else if (flagtype == 3)
         tmp_flags = target_player->get_gameflags();

   }

   if (the_parsed->get_preposition() == NULL)
   {
      tmp_flags->display_flags(the_player, the_name.str_show(), flagnamelist,
                                                 flagnames, null_frozen);
      return;
   }

   if ((flagnum = tmp_flags->get_by_name(the_parsed->get_preposition(),
                                        flagnamelist)) == -1)
   {
      if (!STRCASECMP(the_parsed->get_preposition(), _("all")))
         for_all = 1;
      else
      {
         the_player->send_plr(_("I don't know the '%s' flag.\n"),
                                              the_parsed->get_preposition());
         return;
      }
   }

   if (the_parsed->get_target2() != NULL)
   {

      if (!STRNCASECMP(the_parsed->get_target2(), _("off"),
                                  strlen(the_parsed->get_target2())))
      {
         if (for_all)
	 {
            tmp_flags->clr_all(flagnamelist);

            if (tmp_bldr_flags != NULL)
               tmp_bldr_flags->clr_all(flagnamelist);

            the_player->send_plr(_("All %s for %s have been cleared.\n"),
                                            flagnames, the_name.str_show());
         }
         else
	 {
            if (tmp_bldr_flags != NULL)
               tmp_bldr_flags->clr_flag(flagnum);

            tmp_flags->clr_flag(flagnum);
         }
      }
      else if (!STRCASECMP(the_parsed->get_target2(), _("on")))
      {
         if (for_all)
	 {
            if (tmp_bldr_flags != NULL)
               tmp_bldr_flags->set_all(flagnamelist);

            tmp_flags->set_all(flagnamelist);
            the_player->send_plr(_("All %s for %s have been set.\n"),
                                            flagnames, the_name.str_show());
         }
         else
	 {
            if (tmp_bldr_flags != NULL)
               tmp_bldr_flags->set_flag(flagnum);

            tmp_flags->set_flag(flagnum);
         }
      }
      else
      {
	the_player->send_plr(_("'%s' is not a valid %s setting. "
                              "Valid settings are 'On' and 'Off'.\n"),
                                  flagnames, the_parsed->get_target2());
         return;
      }
   }

   flag_on = tmp_flags->get_flag(flagnum);

   if (not_on)
   {
      if (tmp_dbase->write_user(target_player) <= 0)
      {
         mainstruct->log_error(_("Error storing user record to database"),
                                                            "flagscom");
         return;
      }
      delete_Player(target_player);
   }

   if (for_all)
      return;

   holder.sprintf(_("&+m%s&* set %s '&+C%s&*' on &+m%s&* to %s"),
                                                the_player->get_title(),
           flagnames, flagnamelist[flagnum], the_name.str_show(),
           (flag_on ? _("&+GOn&*") : _("&+ROff&*")));

   /* now log their death if appropriate */
   if (the_config.conf_flags->set_flag(CF_LOGFLAGS))
   {
      mainstruct->log_event(holder.str_show());
   }

   holder2.sprintf("&+Y[&*%s&+Y]&*\n", holder.str_show());
   mainstruct->send_all_players(holder2.str_show(), the_player,
                                                        ADMINFLAG_SEEFLAGS);

   the_player->send_plr(_("The %s '&+c%s&*' for %s is: %s\n"),
                flagnames, flagnamelist[flagnum], the_name.str_show(),
	        (flag_on ? _("&+GOn&*") : _("&+ROff&*")));
   return;

}

/***********************************************************************
 ** savecom - saves a player
 ***********************************************************************/

void savecom(Parse *the_parsed, MudObject *the_user) {
   Player      *the_player;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   mainstruct->save_player(the_player);

   the_player->send_plr(_("Successfully saved.\n"));

   the_parsed = NULL;
   return;
}


/***********************************************************************
 ** reloadcom - reloads a zone for using
 ***********************************************************************/

void reloadcom(Parse *the_parsed, MudObject *the_user)
{
   Player  *the_player;
   Strings holder;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Reload what?\n"));
      return;
   }

   if (!STRCASECMP(the_parsed->get_target1(), _("all")))
   {
      mainstruct->send_all_players(_("&+RReloading all zones!&*\n"));
      mainstruct->flush_players();

      mainstruct->reload_all();

      mainstruct->send_all_players(_("&+RZones reloaded successfully.&*\n"));

      holder.sprintf(_("&+Y[&*All zones have been reloaded by &+M%s&*&+Y]&*\n"),
                                                the_player->get_title());

      mainstruct->send_all_players(holder.str_show(), the_player,
                                                      ADMINFLAG_SEERELOADS);

      return;

   }
   else if (!STRCASECMP(the_parsed->get_target1(), _("verbs")))
   {
      mainstruct->send_all_players(_("&+RReloading action from file.&*\n"));
      mainstruct->flush_players();

      mainstruct->reload_verbs();

      mainstruct->send_all_players(_("&+RActions reloaded successfully.&*\n"));
      holder.sprintf(_("&+Y[&*Verbs have been reloaded by &+M%s&*&+Y]&*\n"),
                                            the_player->get_title());

      mainstruct->send_all_players(holder.str_show(), the_player,
                                                      ADMINFLAG_SEERELOADS);

      return;

   }
   else if (!STRCASECMP(the_parsed->get_target1(), _("abilities")))
   {
      mainstruct->send_all_players(_("&+RReloading abilities from file.&*\n"));
      mainstruct->flush_players();

      mainstruct->reload_abilities();

      mainstruct->send_all_players(_("&+RAbilities reloaded successfully.&*\n"));
      holder.sprintf(_("&+Y[&*Abilities have been reloaded by &+M%s&*&+Y]&*\n"),
                                            the_player->get_title());

      mainstruct->send_all_players(holder.str_show(), the_player,
                                                      ADMINFLAG_SEERELOADS);

      return;

   }
   else if (!STRCASECMP(the_parsed->get_target1(), _("bulletins")))
   {
      mainstruct->send_all_players(_("&+RReloading bulletins from file.&*\n"));
      mainstruct->flush_players();

      mainstruct->reload_bulletins();

      mainstruct->send_all_players(_("&+RBulletins reloaded successfully.&*\n"));
      holder.sprintf(_("&+Y[&*Bulletins have been reloaded by &+M%s&*&+Y]&*\n"),
                                            the_player->get_title());

      mainstruct->send_all_players(holder.str_show(), the_player,
                                                      ADMINFLAG_SEERELOADS);

      return;

   }
   else if (!STRCASECMP(the_parsed->get_target1(), _("masks")))
   {
      mainstruct->send_all_players(_("&+RReloading masks from file.&*\n"));
      mainstruct->flush_players();

      mainstruct->reload_masks();

      mainstruct->send_all_players(_("&+RMasks reloaded successfully.&*\n"));
      holder.sprintf(_("&+Y[&*Masks have been reloaded by &+M%s&*&+Y]&*\n"),
                                            the_player->get_title());

      mainstruct->send_all_players(holder.str_show(), the_player,
                                                      ADMINFLAG_SEERELOADS);

      return;

   }
   else if (!STRCASECMP(the_parsed->get_target1(), _("races")))
   {
      mainstruct->send_all_players(_("&+RReloading races from file.&*\n"));
      mainstruct->flush_players();

      mainstruct->reload_races();

      mainstruct->send_all_players(_("&+RRaces reloaded successfully.&*\n"));
      holder.sprintf(_("&+Y[&*Races have been reloaded by &+M%s&*&+Y]&*\n"),
                                            the_player->get_title());

      mainstruct->send_all_players(holder.str_show(), the_player,
                                                      ADMINFLAG_SEERELOADS);
      return;

   }
   else if (!STRCASECMP(the_parsed->get_target1(), _("inclinations")))
   {
      mainstruct->send_all_players(_("&+RReloading inclinations from file.&*\n"));
      mainstruct->flush_players();

      mainstruct->reload_inclinations();

      mainstruct->send_all_players(_("&+RInclinations reloaded successfully.&*\n"));
      holder.sprintf(_("&+Y[&*Inclinations have been reloaded by &+M%s&*&+Y]&*\n"),
                                            the_player->get_title());

      mainstruct->send_all_players(holder.str_show(), the_player,
                                                      ADMINFLAG_SEERELOADS);

      return;
   }
   else if (!STRCASECMP(the_parsed->get_target1(), _("talents")))
   {
      mainstruct->send_all_players(_("&+RReloading talents from file.&*\n"));
      mainstruct->flush_players();

      mainstruct->reload_talents();

      mainstruct->send_all_players(_("&+RTalents reloaded successfully.&*\n"));
      holder.sprintf(_("&+Y[&*Talents have been reloaded by &+M%s&*&+Y]&*\n"),
                                            the_player->get_title());

      mainstruct->send_all_players(holder.str_show(), the_player,
                                                      ADMINFLAG_SEERELOADS);

      return;

   }
   else
   {
      Strings areaname;

      if (the_parsed->get_target1() == NULL)
      {
         the_player->send_plr(_("Reload what area?\n"));
         return;
      }

      areaname = the_parsed->get_target1();
      areaname.lowercase();

      if (mainstruct->reload_area(areaname.str_show()) <= 0)
      {
         the_player->send_plr(_("That area does not exist.\n"));
         return;
      }

      the_player->send_plr(_("&+RArea '&+M%s&+R' reloaded.\n"),
                                            areaname.str_show());
      holder.sprintf(_("&+Y[&*Area &+y%s&* reloaded by &+M%s&*&+Y]&*\n"),
                       areaname.str_show(), the_player->get_title());

      mainstruct->send_all_players(holder.str_show(), the_player,
                                                      ADMINFLAG_SEERELOADS);

      return;
   }
}


/***********************************************************************
 ** lockcom - locks a door or object
 ***********************************************************************/

void doorlockcom(Parse *the_parsed, MudObject *the_user)
{
   Player      *the_player;
   Location    *the_loc;
   MudObject   *the_obj;
   Door        *the_door;
   Item        *the_container = NULL;
   MudObject   *key_obj;
   Strings     holder;
   int         the_num = 0;
   Flags       *tmp_itemflags = NULL;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Lock what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to open your eyes first (stand).\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to lock something.\n"));
      return;
   }

   the_loc = the_player->get_loc();

   if (!the_loc->is_lit(the_player))
   {
      the_player->send_plr(_("You can't see a lock in the dark.\n"));
      return;
   }

   if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                        &the_num)) == NULL)
   {
      the_player->send_plr(_("The '%s' does not appear to be here.\n"),
                                                  the_parsed->get_target1());
      return;
   }

   if (the_obj->get_type() != OBJ_TYPE_DOOR)
   {
      if (the_obj->is_an_item())
      {
         the_container = (Item *) the_obj;
         tmp_itemflags = the_container->get_itemflags();
      }

      if ((!the_obj->is_an_item()) ||
          (!tmp_itemflags->get_flag(ITEMFLAG_LOCKABLE)))
      {
         the_player->send_plr(_("You can't lock that.\n"));
         return;
      }


      if (tmp_itemflags->get_flag(ITEMFLAG_LOCKED))
      {
         the_player->send_plr(_("The container is already locked.\n"));
         return;
      }

      if (!tmp_itemflags->get_flag(ITEMFLAG_CLOSED))
      {
         the_player->send_plr(_("It needs to be closed first.\n"));
         return;
      }

      the_door = NULL;
   }
   else
      the_door = (Door *) the_obj;

   if (the_door != NULL)
   {
      switch(the_door->get_door_state())
      {
         case 0:
            the_player->send_plr(_("It is open, you can't lock it.\n"));
            return;
         case 2:
            the_player->send_plr(_("It is already locked.\n"));
            return;
         case 3:
            the_player->
               send_plr(_("It is already magically locked.\n"));
            return;
         default: break;
      }
   }

   /* if they dont supply anything to lock with, check their inventory */
   if (the_parsed->get_preposition() == NULL)
   {
      Inventory *the_inv;
      MudObject  *tmp_obj;

      the_inv = the_player->get_inventory();
      the_inv->reset_current();
      tmp_obj = the_inv->get_next();

      while (tmp_obj != NULL)
      {
         if (tmp_obj->get_type() == OBJ_TYPE_KEY)
         {
            if (the_door != NULL)
	    {
               holder.sprintf("%s@%s", tmp_obj->get_name(), tmp_obj->get_area());
               if (the_door->toggle_lock(holder.str_show()) > 0)
               {
                  the_player->send_plr(_("You lock the %s with the %s.\n"),
                                the_door->get_title(), tmp_obj->get_title());
                  holder.sprintf(_("%s locks the %s with %s %s.\n"),
                        the_player->get_title(), the_door->get_title(),
	            the_player->get_sex_possessive_str(), tmp_obj->get_title());
                  the_loc->send_location(holder.str_show(), the_player,
                                                                     CAN_SEE);
                  the_loc->send_location(
                       _("You hear the sound of a key turning in a lock.\n"),
                                                     the_player, CANT_SEE);
                  return;
               }
	    }
            else
	    {
               if (check_specials(_("pre_lock"), the_container, tmp_obj, NULL,
                                               the_container, the_player) == 3)
                  return;

               the_player->send_plr(_("You lock the %s.\n"),
                                                  the_container->get_title());
               holder.sprintf(_("%s locks the %s.\n"), the_player->get_title(),
                                              the_container->get_title());
               the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
               the_loc->send_location(
                       _("You hear the sound of a key turning in a lock.\n"),
                                                     the_player, CANT_SEE);
               tmp_itemflags->set_flag(ITEMFLAG_LOCKED);
               if (check_specials(_("post_lock"), the_container, tmp_obj, NULL,
                                               the_container, the_player) == 3)
                  return;
               return;
	    }
         }
         tmp_obj = the_inv->get_next();
      }
      the_player->
       send_plr(_("You don't seem to have a key that will work in the %s.\n"),
                                                 the_obj->get_title());
      return;
   }

   if (STRCASECMP(the_parsed->get_preposition(), _("with")))
   {
      the_player->
        send_plr(_("The preposition '%s' is not one you can use with lock\n"));
      return;
   }

   if (the_parsed->get_target2() == NULL)
   {
      the_player->send_plr(_("Lock the %s with what?\n"), the_obj->get_title());
      return;
   }

   if ((key_obj =
           the_player->find_contained(the_parsed->get_target2(),
                                                          &the_num)) == NULL)
   {
      the_player->send_plr(_("You don't seem to be carrying the %s\n"),
                                                   the_parsed->get_target2());
      return;
   }
   else if (the_door != NULL)
   {
      if (check_specials(_("pre_lock"), the_door, key_obj, NULL,
                                               the_door, the_player) == 3)
         return;

      holder.sprintf("%s@%s", key_obj->get_name(), key_obj->get_area());
      if (the_door->toggle_lock(holder.str_show()) > 0)
      {
         the_player->send_plr(_("You lock the %s with the %s.\n"),
                                   the_obj->get_title(), key_obj->get_title());
         holder.sprintf(_("%s locks the %s with %s %s.\n"),
                        the_player->get_title(), the_door->get_title(),
	            the_player->get_sex_possessive_str(), key_obj->get_title());
         the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
         the_loc->send_location(
                       _("You hear the sound of a key turning in a lock.\n"),
                                                     the_player, CANT_SEE);
         check_specials(_("post_lock"), the_door, key_obj, NULL,
                                               the_door, the_player);
         return;

      }
      else
      {
         the_player->send_plr(_("The %s will not work in the lock.\n"),
                                                      key_obj->get_title());
         return;
      }
   }
   else
   {
      if (check_specials(_("pre_lock"), the_container, key_obj, NULL,
                                               the_container, the_player) == 3)
         return;

      the_player->send_plr(_("You lock the %s.\n"), the_container->get_title());
      holder.sprintf(_("%s locks the %s.\n"), the_player->get_title(),
                                              the_container->get_title());
      the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
      the_loc->send_location(
                       _("You hear the sound of a key turning in a lock.\n"),
                                                     the_player, CANT_SEE);

      tmp_itemflags->set_flag(ITEMFLAG_LOCKED);
      if (check_specials(_("post_lock"), the_container, key_obj, NULL,
                                               the_container, the_player) == 3)
         return;
   }
   return;
}


/***********************************************************************
 ** unlockcom - unlocks a door or object
 ***********************************************************************/

void doorunlockcom(Parse *the_parsed, MudObject *the_user)
{
   Player      *the_player;
   Location    *the_loc;
   MudObject   *the_obj;
   Door        *the_door;
   MudObject   *key_obj;
   Strings     holder;
   int         the_num = 0;
   Flags       *tmp_itemflags = NULL;
   Item        *the_container = NULL;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Unlock what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to wake up first (stand).\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to unlock something.\n"));
      return;
   }

   the_loc = the_player->get_loc();

   if (!the_loc->is_lit(the_player))
   {
      the_player->send_plr(_("You can't see a lock in the dark.\n"));
      return;
   }

   if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                           &the_num)) == NULL)
   {
      the_player->send_plr(_("The '%s' does not appear to be here.\n"),
                                                  the_parsed->get_target1());
      return;
   }

   if (the_obj->get_type() != OBJ_TYPE_DOOR)
   {
      if (the_obj->is_an_item())
      {
         the_container = (Item *) the_obj;
         tmp_itemflags = the_container->get_itemflags();
      }

      if ((!the_obj->is_an_item()) ||
          (!tmp_itemflags->get_flag(ITEMFLAG_LOCKABLE)))
      {
         the_player->send_plr(_("You can't unlock that.\n"));
         return;
      }


      if (!tmp_itemflags->get_flag(ITEMFLAG_LOCKED))
      {
         the_player->send_plr(_("The container is not locked.\n"));
         return;
      }

      the_door = NULL;
   }
   else
      the_door = (Door *) the_obj;

   if (the_door != NULL)
   {
      switch(the_door->get_door_state())
      {
         case 0:
         case 1:
            the_player->send_plr(_("It is already unlocked.\n"));
            return;
         case 3:
            the_player->
               send_plr(_("It is magically locked; no key will help you.\n"));
            return;
         default: break;
      }
   }

   /* if they dont supply anything to unlock with, check their inventory */
   if (the_parsed->get_preposition() == NULL)
   {
      Inventory  *the_inv;
      MudObject  *tmp_obj;

      the_inv = the_player->get_inventory();
      the_inv->reset_current();
      tmp_obj = the_inv->get_next();

      while (tmp_obj != NULL)
      {
         if (tmp_obj->get_type() == OBJ_TYPE_KEY)
         {
            if (the_door != NULL)
	    {
               holder.sprintf("%s@%s", tmp_obj->get_name(), tmp_obj->get_area());
               if (the_door->toggle_lock(holder.str_show()) > 0)
               {
                  the_player->send_plr(_("You unlock the %s with the %s.\n"),
                                the_door->get_title(), tmp_obj->get_title());
                  holder.sprintf(_("%s unlocks the %s with %s %s.\n"),
                        the_player->get_title(), the_door->get_title(),
	            the_player->get_sex_possessive_str(), tmp_obj->get_title());
                  the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
                  the_loc->send_location(
                       _("You hear the sound of a key turning in a lock.\n"),
                                                     the_player, CANT_SEE);

                  return;
               }
	    }
            else
	    {
               if (check_specials(_("pre_unlock"), the_container, tmp_obj, NULL,
                                                the_container, the_player) == 3)
                  return;

               the_player->send_plr(_("You unlock the %s.\n"),
                                                  the_container->get_title());
               holder.sprintf(_("%s unlocks the %s.\n"), the_player->get_title(),
                                              the_container->get_title());
               the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
               the_loc->send_location(
                       _("You hear the sound of a key turning in a lock.\n"),
                                                     the_player, CANT_SEE);

               tmp_itemflags->clr_flag(ITEMFLAG_LOCKED);
               if (check_specials(_("post_unlock"), the_container, tmp_obj, NULL,
                                               the_container, the_player) == 3)
                  return;
               return;
	    }

         }
         tmp_obj = the_inv->get_next();
      }
      the_player->
       send_plr(_("You don't seem to have a key that will work in the %s.\n"),
                                                 the_obj->get_title());
      return;
   }

   if (STRCASECMP(the_parsed->get_preposition(), _("with")))
   {
      the_player->
        send_plr(_("The preposition '%s' is not one you can use with unlock\n"));
      return;
   }

   if (the_parsed->get_target2() == NULL)
   {
      the_player->send_plr(_("Unlock the %s with what?\n"), the_obj->get_title());
      return;
   }

   if ((key_obj =
              the_player->find_contained(the_parsed->get_target2(),
                                                           &the_num)) == NULL)
   {
      the_player->send_plr(_("You don't seem to be carrying the %s\n"),
                                                   the_parsed->get_target2());
      return;
   }
   else
   {
      if (the_door != NULL)
      {
         if (check_specials(_("pre_unlock"), the_container, key_obj, NULL,
                                               the_container, the_player) == 3)
            return;

         holder.sprintf("%s@%s", key_obj->get_name(), key_obj->get_area());
         if (the_door->toggle_lock(holder.str_show()) > 0)
         {
            the_player->send_plr(_("You unlock the %s with the %s.\n"),
                                   the_obj->get_title(), key_obj->get_title());

            holder.sprintf(_("%s locks the %s with %s %s.\n"),
                        the_player->get_title(), the_door->get_title(),
	            the_player->get_sex_possessive_str(), key_obj->get_title());
            the_loc->send_location(holder.str_show(), the_player, CAN_SEE);

            the_loc->send_location(
                       _("You hear the sound of a key turning in a lock.\n"),
                                                     the_player, CANT_SEE);
            check_specials(_("pre_unlock"), the_container, key_obj, NULL,
                                               the_container, the_player);
            return;

         }
         else
         {
            the_player->send_plr(_("The %s will not work in the lock.\n"),
                                                      key_obj->get_title());
            return;
         }
      }
      else
      {
         if (check_specials(_("pre_unlock"), the_container, key_obj, NULL,
                                               the_container, the_player) == 3)
            return;

         the_player->send_plr(_("You unlock the %s.\n"),
                                                 the_container->get_title());
         holder.sprintf(_("%s unlocks the %s.\n"), the_player->get_title(),
                                              the_container->get_title());
         the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
         the_loc->send_location(
                       _("You hear the sound of a key turning in a lock.\n"),
                                                     the_player, CANT_SEE);
         tmp_itemflags->clr_flag(ITEMFLAG_LOCKED);
         if (check_specials(_("post_unlock"), the_container, key_obj, NULL,
                                               the_container, the_player) == 3)
            return;
      }
   }
}


/***********************************************************************
 ** tellcom - tells a person in the same room something
 ***********************************************************************/

void tellcom(Parse *the_parsed, MudObject *the_user)
{
    Strings   holder;
    Player    *the_player;
    MudObject *the_obj;
    Location  *this_loc;
    int       the_num = 0;
    if (the_user->get_type() != OBJ_TYPE_PLAYER)
        return;
   
    the_player = (Player *) the_user;

    // they need to specify a player or mobile to tell to
    if (the_parsed->get_target1() == NULL)
    {
        the_player->send_plr(_("Tell who?\n"));
        return;
    }
    
    if (the_player->is_asleep())
    {
        the_player->send_plr(_("You hold a discussion in your dream.\n"));
        return;
    }
    
    // if there is nothing after tell <who>, send a message to the player
    if ((the_parsed->get_speech() == NULL) ||
        (strlen(the_parsed->get_speech()) <= 0))
    {
        the_player->send_plr(_("Format: tell <individual> <phrase>\n"));
        return;
    }
    
    this_loc = the_player->get_loc();
    
    if (the_config.conf_flags->get_flag(CF_DISTANCETELL))
    {
        if ((the_obj = mainstruct->get_player(the_parsed->get_target1())) == NULL)
        {
            if ((the_obj = this_loc->find_contained(the_parsed->get_target1(),&the_num)) == NULL)
            {
                the_player->send_plr(_("They are not here.\n"));
                return;
            }
        }
    }
    else
    {
        if (!(the_player->get_loc())->is_lit(the_player))
        {
            the_player->send_plr(_("It is too dark to see who you are talking to.\n"));
            return;
        }
        
        if ((the_obj = this_loc->find_contained(the_parsed->get_target1(), &the_num)) == NULL)
        {
            the_player->send_plr(_("They are not here.\n"));
            return;
        }
    }
    if (!the_player->can_see_ind((Individual *)the_obj))
    {
        the_player->send_plr(_("They are not here.\n"));
        return;
    }
    
    if (the_obj->get_type() == OBJ_TYPE_PLAYER)
    {
        Player *target_plr;
        
        target_plr = (Player *) the_obj;
        
        if (target_plr->is_asleep())
        {
            the_player->send_plr(_("They are asleep and can't hear you.\n"));
            return;
        }
        
        if (check_specials(_("pre_tell"), target_plr, NULL, NULL,
            target_plr, the_player) == 3)
            return;
        
        the_player->tell_ind(target_plr, the_parsed->get_speech());
        
        if (check_specials(_("post_tell"), target_plr, NULL, NULL,
            target_plr, the_player) == 3)
            return;
        
        return;
    }
    else if (the_obj->get_type() == OBJ_TYPE_MOBILE)
    {
        Mobile  *target_mob;
        Strings word_holder;
        Strings formatted;
        int     i = 1;
        char    *reply_holder = NULL;
        Shop    *tmp_shop;
        int     not_found = 1;
        
        target_mob = (Mobile *) the_obj;
        
        if (check_specials(_("pre_tell"), target_mob, NULL, NULL,
            target_mob, the_player) == 3)
            return;
        
        the_player->tell_ind(target_mob, the_parsed->get_speech());
        if ((target_mob->get_indflags())->get_flag(INDFLAG_CORPSE))
            return;
        
        if ((target_mob->get_indflags())->get_flag(INDFLAG_GHOST))
        {
            the_player->send_plr(_("The ghost can't respond.\n"));
            return;
        }
        
        // first, pass in a string and see if it has keywords in it
        while (word_holder.assign_word(the_parsed->get_speech(), i) != -1)
        {
            reply_holder = target_mob->find_reply(word_holder.str_show());
            if (reply_holder != NULL)
            {
                formatted.format_for_comm(reply_holder,
                    (strlen(target_mob->get_title()) + 10), the_config.max_line_len);
                formatted.sanitize_input();
                the_player->send_plr(_("&+B%s&+w replies &+W'&+G%s&+W'&*\n"),
                    target_mob->get_title(), formatted.str_show());
                not_found = 0;
                break;
            }
            else if ((tmp_shop = target_mob->get_shop()) != NULL)
            {
                if ((!STRCASECMP(word_holder.str_show(), _("buy"))) ||
                    (!STRCASECMP(word_holder.str_show(), _("list"))) ||
                    (!STRCASECMP(word_holder.str_show(), _("items"))) ||
                    (!STRCASECMP(word_holder.str_show(), _("train"))) ||
                    (!STRCASECMP(word_holder.str_show(), _("training"))))
                {
                    Strings next_word;
                    int num_of = 1;
                    int offset = 1;
                    
                    next_word.assign_word(the_parsed->get_speech(), i+offset);
                    if (next_word.str_show() != NULL)
                    {
                        // we will account for "buy a sword" here
                        if (next_word.str_cmp(_("a")))
                        {
                            offset++;
                            next_word.assign_word(the_parsed->get_speech(),
                                i+offset);
                        }
                        
                        // here we account for "buy 20 torches"
                        if ((next_word.str_show() != NULL) &&
                            (isdigit(*(next_word.str_show()))))
                        {
                            offset++;
                            num_of = atoi(next_word.str_show());
                            if (num_of <= 0)
                                num_of = 1;
                            
                            next_word.assign_word(the_parsed->get_speech(),
                                i+offset);
                        }
                        
                        // here we finally get the item
                        if (next_word.str_show() != NULL)
                        {
                            if (tmp_shop->add_request(the_player,
                                next_word.str_show(), num_of) > 0)
                            {
                                int cost;
                                Flags *gameflags;
                                
                                cost = (tmp_shop->get_request_cost(the_player)) /
                                    (tmp_shop->get_currency_ptr())->get_money_index();
                                if ((cost *
                                    (tmp_shop->get_currency_ptr())->get_money_index()) <
                                    tmp_shop->get_request_cost(the_player))
                                    cost++;
                                
                                target_mob->tell_ind(the_player,
                                    tmp_shop->get_desc(next_word.str_show()));
                                holder.sprintf(_("That will cost you %d %s"),
                                    cost, (tmp_shop->get_currency_ptr())->get_title());
                                target_mob->tell_ind(the_player, holder.str_show());
                                
                                gameflags = the_player->get_gameflags();
                                if (gameflags->get_flag(GAMEFLAG_AUTOPAY))
                                {
                                    Input      fake_input;
                                    Strings    command_str;
                                    Inventory  *the_inv;
                                    MudObject  *the_obj;
                                    MudObject  *check_obj;
                                    Money      *the_money = NULL;
                                    int        the_offset = 1;
                                    int        value;
                                    
                                    the_inv = the_player->get_inventory();
                                    
                                    // first find a money object large enough to pay
                                    the_inv->reset_current();
                                    the_obj = the_inv->get_next();
                                    while (the_obj != NULL)
                                    {
                                        if (the_obj->get_type() == OBJ_TYPE_MONEY)
                                        {
                                            the_money = (Money *) the_obj;
                                            value = the_money->get_money_index() *
                                                the_money->get_number_of();
                                            if (value >
                                                tmp_shop->get_request_cost(the_player))
                                                break;
                                        }
                                        the_obj = the_inv->get_next();
                                    }
                                    
                                    // we didnt find anything to pay with
                                    if (the_obj == NULL)
                                    {
                                        the_player->send_plr(
                                            _("Autopay failed, you don't have enough money.\n"));
                                        return;
                                    }
                                    
                                    // now we check to see if the pay object is a few
                                    // objects in, so it needs gold2 or gold3
                                    the_inv->reset_current();
                                    check_obj = the_inv->get_next();
                                    while (check_obj != NULL)
                                    {
                                        if (check_obj->get_type() == OBJ_TYPE_MONEY)
                                        {
                                            if ((!STRCASECMP(check_obj->get_title(),
                                                the_obj->get_title())) &&
                                                (check_obj != the_obj))
                                                the_offset++;
                                            if (the_obj == check_obj)
                                                break;
                                        }
                                        check_obj = the_inv->get_next();
                                    }
                                    
                                    if (the_offset > 1)
                                        command_str.sprintf(_("give %s%d to %s"),
                                        the_money->get_title(), the_offset,
                                        target_mob->get_title());
                                    else
                                        command_str.sprintf(_("give %s to %s"),
                                        the_money->get_title(),
                                        target_mob->get_title());
                                    fake_input.handle_input(&command_str, the_player);
                                    return;
                                }
                                
                                not_found = 0;
                            }
                        }
               }
               
               if (not_found)
               {
                   formatted.format_for_comm(_("&+GYou've come to the very best!"),
                       (strlen(target_mob->get_title()) + 10),
                       the_config.max_line_len);
                   // reply with the shop list of for sale items
                   the_player->send_plr(_("&+B%s&+w replies &+W'%s&+W'&*\n\n"),
                       target_mob->get_title(), formatted.str_show());
                   tmp_shop->display_wares(the_player);
                   not_found = 0;
                   break;
               }
            }
            else if (!STRCASECMP(word_holder.str_show(), _("sell")))
            {
                formatted.format_for_comm(
                    _("&+GWe offer the best prices for old items!"),
                    (strlen(target_mob->get_title()) + 10),
                    the_config.max_line_len);
                the_player->send_plr(_("&+B%s&+w replies &+W'%s&+W'&*\n\n"),
                    target_mob->get_title(), formatted.str_show());
                
                // reply with the list of items in the player's inventory that
                // the shopkeeper will buy, and the price
                tmp_shop->display_will_buy(the_player);
                not_found = 0;
            }
         }
         
         i++;
      }
      if (not_found)
      {
          the_player->
              send_plr(_("&+B%s&+w replies &+W'&+GI know nothing about that&+W'&*\n"),
              target_mob->get_title());
      }
      
      if (check_specials(_("post_tell"), target_mob, NULL, NULL,
          target_mob, the_player) == 3)
          return;
      
      return;
      
   }
   else
   {
       the_player->send_plr(_("The %s is an excellent listener!\n"),
           the_obj->get_name());
       holder.sprintf(_("%s holds a conversation with the %s.\n"),
           the_player->get_title(), the_obj->get_title());
       this_loc->send_location(holder.str_show(), the_player, CAN_SEE);
       this_loc->send_location(_("You hear the sounds of a conversation.\n"),
           the_player, CANT_SEE);
       return;
   }
   
}


/***********************************************************************
 ** movecom - a specials action, looks for special code
 ***********************************************************************/

void movecom(Parse *the_parsed, MudObject *the_user)
{
   Player *the_player;
   Location *this_loc;
   MudObject *the_obj;
   int       the_num = 0;
   int       results;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   /* they need to specify an object to move */
   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Move what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You move a big dragon in your dream.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to move something.\n"));
      return;
   }

   this_loc = the_player->get_loc();

   if (!this_loc->is_lit(the_player))
   {
      the_player->send_plr(_("You can't find anything to move in the darkness.\n"));
      return;
   }

   if ((the_obj = this_loc->find_contained(the_parsed->get_target1(),
                                                          &the_num)) == NULL)
   {
      the_player->send_plr(_("That is not here.\n"));
      return;
   }

   if (the_obj->get_type() == OBJ_TYPE_DOOR)
   {
     Door *the_door = (Door *) the_obj;
     
     // If the outside is supposed to be invisible to the player
     if (!the_door->is_visible(the_player->get_loc()))
     {
       the_player->send_plr(_("That is not here.\n"));
       return;
     }
   }

   if ((results = check_specials(_("premove"), the_obj, NULL, NULL,
                                 the_obj, the_player)) == 3)
                return;

   the_player->send_plr(_("It does not budge.\n"));
   return;
}


/***********************************************************************
 **  gotocom - transes to a location
 **
 ***********************************************************************/

void gotocom(Parse *the_parsed, MudObject *the_user) {
   Location *cur_loc;
   MudObject *next_mudobj;
   Location *next_loc;
   Strings holder;
   Player *the_player;
   Player *tmp_player = NULL;
   Flags  *tmp_gameflags;
   Flags  *tmp_indflags;
   Inventory *the_inventory;
   MudObject *tmp_obj;
   Mobile    *tmp_mob;
   Location  *tmp_loc;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("Ummm...wake first please?.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("Not while marked as busy.\n"));
      return;
   }

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("You must specify a location to goto.\n"));
      return;
   }

   cur_loc = the_player->get_loc();

   holder = the_parsed->get_target1();
   if ((holder.num_char('@') != 1) &&
       ((tmp_player = mainstruct->get_player(holder.str_show())) == NULL))
   {
      the_player->send_plr(_("'%s' is not a valid location.\n"
                           "usage: goto <locname>@<areaname>\n"),
                                                       holder.str_show());
      return;
   }

   if (tmp_player != NULL)
   {
      next_mudobj = (MudObject *) (tmp_player->get_loc());
   }
   else if ((next_mudobj = mainstruct->get_object(holder.str_show()))
                                                                   == NULL)
   {
      the_player->send_plr(_("That object does not exist.\n"));
      return;
   }

   while (next_mudobj->get_type() != OBJ_TYPE_LOCATION)
   {
      Strings new_loc;
      MudObject *old_obj;

      old_obj = next_mudobj;

      new_loc = old_obj->get_location();
      if (new_loc.num_char('@') == 0)
         new_loc.sprintf("%s@%s", next_mudobj->get_location(),
                                               next_mudobj->get_area());
      if ((next_mudobj =
             mainstruct->get_object(new_loc.str_show())) == NULL)
      {
         the_player->send_plr(_("Error in command.\n"));
         holder.sprintf(_("Object location for %s is invalid.\n"),
                                                  old_obj->get_name());
         mainstruct->log_error(holder.str_show(), "gotocom");
         return;
      }
   }

   next_loc = (Location *) next_mudobj;

   tmp_gameflags = the_player->get_gameflags();
   next_loc->show_location(the_player, tmp_gameflags->get_flag(GAMEFLAG_BRIEF));

   tmp_indflags = the_player->get_indflags();

   /* say that we are leaving the room */
   if (tmp_indflags->get_flag(INDFLAG_FIGHTING))
   {
      tmp_indflags->clr_flag(INDFLAG_FIGHTING);
      mainstruct->remove_fighter(the_player);
      mainstruct->clear_fighter(the_player);
      holder.sprintf(_("%s has suddenly vanished from the fight!\n"),
                                                the_player->get_title());
      cur_loc->send_location(holder.str_show(), the_player);
   }
   else
   {
      holder.sprintf(_("%s vanishes into thin air!\n"), the_player->get_title());
      cur_loc->send_location(holder.str_show(), the_player, CAN_SEE);
   }

   /* tell the target room that we have arrived */
   holder.sprintf(_("%s appears out of thin air!\n"), the_player->get_name());
   next_loc->send_location(holder.str_show(), the_player, CAN_SEE);

   if (tmp_indflags->get_flag(INDFLAG_INBOAT))
   {
      Boat *the_boat;

      the_boat = (Boat *) the_player->get_contained_by();
      the_boat->exit_boat(the_player);
   }

   if (the_player->set_location(next_loc) == -1)
   {
      mainstruct->log_error(_("Could not set player location!"), "gotocom");
      return;
   }

   the_inventory = next_loc->get_inventory();
   the_inventory->reset_current();
   tmp_obj = the_inventory->get_next();
   while (tmp_obj != NULL)
   {
      if (tmp_obj->get_type() == OBJ_TYPE_MOBILE)
      {
         tmp_mob = (Mobile *) tmp_obj;

         if ((tmp_mob->get_indflags())->get_flag(INDFLAG_GHOST))
	 {
	    /* if this is their ghost, merge them back together */
            if (!STRCASECMP(tmp_mob->get_clones(), the_player->get_name()))
	    {
               tmp_loc = tmp_mob->get_loc();
               tmp_loc->remove_contained(tmp_mob);
               (mainstruct->get_dbase())->delete_obj(tmp_mob);
               tmp_obj = the_inventory->get_next();
               holder.sprintf(_("The ghost merges effortlessly into %s's body.\n"),
                                          the_player->get_title());
               tmp_loc->send_location(holder.str_show(), the_player, CAN_SEE);
               the_player->send_plr(
                _("The ghost merges into your body.  You now feel whole again.\n"));
               continue;
	    }
	 }
      }
      tmp_obj = the_inventory->get_next();
   }

   return;
}



/***********************************************************************
 **  givecom - gives an object to another individual
 **
 ***********************************************************************/

void givecom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   MudObject *target_ind;
   Location *the_loc;
   Moveable *the_moveable;
   MudObject *the_obj;
   Strings   holder;
   Strings   target;
   Strings   object;
   int      the_num = 0;
   int      delete_obj = 0;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {   the_player->send_plr(_("Give what to whom?\n"));
       return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to wake up first.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy.\n"));
      return;
   }

   object = the_parsed->get_target1();
   the_loc = the_player->get_loc();

   if (!the_loc->is_lit(the_player))
   {
      the_player->send_plr(_("You can't give to what you can't see.\n"));
      return;
   }

   /* if there is no preposition, then expect the target */
   if ((the_parsed->get_preposition() != NULL) &&
       (STRCASECMP(the_parsed->get_preposition(), _("to"))))
   {
      target = the_parsed->get_preposition();
   }

   /* else get the target after ignoring the preposition */
   else
   {
      if (the_parsed->get_target2() == NULL)
      {
         the_player->send_plr(_("Give to who?\n"));
         return;
      }
      target = the_parsed->get_target2();
   }

   /* if we can't find the object we are giving */
   if (((target_ind = the_loc->find_contained(target.str_show(),
                                                   &the_num)) == NULL) ||
     ((target_ind->get_type() != OBJ_TYPE_MOBILE) &&
      (target_ind->get_type() != OBJ_TYPE_PLAYER)))
   {

      /* if we can't find the target ind, raise an error, otherwise, we
         should swap the two */
      if (((target_ind = the_loc->find_contained(object.str_show(),
                                                   &the_num)) == NULL) ||
         ((target_ind->get_type() != OBJ_TYPE_MOBILE) &&
          (target_ind->get_type() != OBJ_TYPE_PLAYER)))
      {
         the_player->send_plr(_("That individual is not here.\n"));
         return;
      }
      else
      {
         Strings tmp_str;

         tmp_str = target.str_show();
         target = object.str_show();
         object = tmp_str.str_show();
      }
   }

   if (!STRCASECMP(target_ind->get_name(), the_player->get_name()))
   {
      the_player->send_plr(_("You already have them!\n"));
      return;
   }

   if ((((Individual *)target_ind)->get_indflags())->get_flag(INDFLAG_CORPSE))
   {
      the_player->send_plr(_("Strange, the corpse doesn't seem to want it.\n"));
      return;
   }

   if ((((Individual *)target_ind)->get_indflags())->get_flag(INDFLAG_GHOST))
   {
      the_player->send_plr(_("The ghost is ethereal and can't hold things.\n"));
      return;
   }

   /* if they want to give all */
   if (STRCASECMP(object.str_show(),_("all")) == 0)
   {   Inventory *the_inventory;

       the_inventory = the_player->get_inventory();
       if (the_inventory == NULL)
       {    mainstruct->log_error(_("Request for inventory returned NULL!"),
                                                                 "givecom");
            return;
       }

       the_inventory->reset_current();
       the_obj = the_inventory->get_next();
       while (the_obj != NULL)
       {  if (the_obj->is_a_moveable())
          {
             if (check_specials(_("pre_give"), the_obj, target_ind, NULL,
                                 the_obj, the_player) == 3)
                return;

             if (check_specials(_("pre_give"), target_ind, target_ind, NULL,
                                 the_obj, the_player) == 3)
                return;

             the_player->send_plr(_("You give the %s to %s.\n"),
                             the_obj->get_title(), target_ind->get_title());
             holder.sprintf(_("%s gives the %s to %s.\n"), the_player->get_title(),
                               the_obj->get_title(), target_ind->get_title());
             if (target_ind->get_type() == OBJ_TYPE_PLAYER)
	     {
                the_loc->send_location(holder.str_show(), the_player,
                                          (Player *) target_ind, CAN_SEE);
                ((Player *) target_ind)->send_plr(_("%s gives you the %s.\n"),
                      (the_loc->is_lit((Player *) target_ind)) ?
                     the_player->get_name() : _("Someone"), the_obj->get_title());
             }
             else
	     {
                the_loc->send_location(holder.str_show(),the_player, CAN_SEE);
             }
             if (the_obj->get_type() == OBJ_TYPE_WEARABLE)
                the_player->remove_item((Wearable *) the_obj);
             the_player->remove_contained(the_obj);
             target_ind->add_contained(the_obj);
             the_inventory->reset_current();

             if (check_specials(_("post_give"), the_obj, target_ind, NULL,
                                 the_obj, the_player) == 3)
                return;

             if (check_specials(_("post_give"), target_ind, target_ind, NULL,
                                 the_obj, the_player) == 3)
                return;


          }
          the_obj = the_inventory->get_next();
       }
       return;
   }

   if ((the_obj = the_player->find_contained(object.str_show(),
                                                          &the_num)) == NULL)
   {   /* try inventory */
      the_player->send_plr(_("The %s is not here.\n"), object.str_show());
      return;

   }

   if (the_obj == NULL)
   {   mainstruct->log_error(_("The_obj they attempted to give was "
                             "NULL while a target was expected.\n"), "givecom");
       return;
   }

   if (the_obj->is_a_moveable())
      the_moveable = (Moveable *)the_obj;

   if (check_specials(_("pre_give"), the_obj, target_ind, NULL,
                                 the_obj, the_player) == 3)
      return;
   if (check_specials(_("pre_give"), target_ind, target_ind, NULL,
                                 the_obj, the_player) == 3)
      return;

   /* Check if can carry */
   /* move to players inventory */
   if (the_parsed->get_number() > 0)
   {
      Merger *the_merger;
      int    results;
      Strings parentname;

      if (!the_obj->is_merger())
      {
         if (the_obj->get_type() == OBJ_TYPE_WEARABLE)
            the_player->remove_item((Wearable *) the_obj);
         the_player->remove_contained(the_obj);
      }
      else
      {
         the_merger = (Merger *) the_obj;

         results = the_player->move_merger((Individual *) target_ind,
                                     &the_merger, the_parsed->get_number());

         if (results == -2)
	 {
            the_player->send_plr(_("You do not have enough.\n"));
            return;
         }
	 else if (results == 2)
	    delete_obj = 1;

         the_obj = (MudObject *) the_merger;
      }
   }
   else
   {
      if (the_obj->get_type() == OBJ_TYPE_WEARABLE)
         the_player->remove_item((Wearable *) the_obj);
      the_player->remove_contained(the_obj);

      if (target_ind->add_contained(the_obj) == 2)
      {
         delete_obj = 1;
      }
   }

  the_player->send_plr(_("You give the %s to %s.\n"),
                            the_obj->get_title(), target_ind->get_title());
  holder.sprintf(_("%s gives the %s to %s.\n"), the_player->get_name(),
                               the_obj->get_title(), target_ind->get_title());

  if (target_ind->get_type() == OBJ_TYPE_PLAYER)
  {
     the_loc->send_location(holder.str_show(), the_player,
                                          (Player *) target_ind, CAN_SEE);
     ((Player *) target_ind)->send_plr(_("%s gives you the %s.\n"),
	    the_loc->is_lit((Player *) target_ind) ? the_player->get_name() :
                             _("Someone"), the_obj->get_title());
  }
  else
  {
    the_loc->send_location(holder.str_show(),the_player, CAN_SEE);
  }

   if (target_ind->get_type() == OBJ_TYPE_MOBILE)
   {
      Mobile *the_mobile;
      Shop *the_shop;
      int results;

      the_mobile = (Mobile *) target_ind;

      if ((the_shop = the_mobile->get_shop()) != NULL)
      {
         if (the_obj->get_type() == OBJ_TYPE_MONEY)
            results = the_shop->complete_transaction(the_player, the_mobile,
                                                      (Money *) the_obj);
         else
	 {
            results = the_shop->complete_sell(the_player, the_mobile, the_obj);
         }
         if (results == -2)
            delete_obj = 0;
      }
   }

   if (check_specials(_("post_give"), the_obj, target_ind, NULL,
                                 the_obj, the_player) == 3)
      return;

   if (check_specials(_("post_give"), target_ind, target_ind, NULL,
                                 the_obj, the_player) == 3)
      return;

  if (delete_obj)
      mainstruct->delete_object(the_obj);
}


/***********************************************************************
 **  autoexits - toggles autoexits for this player on and off
 **
 ***********************************************************************/

void autoexitscom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Flags    *game_flags;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   game_flags = the_player->get_gameflags();

   if (game_flags->get_flag(GAMEFLAG_AUTOEXITS))
   {
      game_flags->clr_flag(GAMEFLAG_AUTOEXITS);
      the_player->send_plr(_("Autoexits has been turned off.\n"));
   }
   else
   {
      game_flags->set_flag(GAMEFLAG_AUTOEXITS);
      the_player->send_plr(_("Autoexits has been turned on.\n"));
   }
   return;
   the_parsed = NULL;
}


/***********************************************************************
 **  briefcom - toggles brief descriptions for this player on and off
 **
 ***********************************************************************/

void briefcom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Flags    *game_flags;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   game_flags = the_player->get_gameflags();

   if (game_flags->get_flag(GAMEFLAG_BRIEF))
   {
      game_flags->clr_flag(GAMEFLAG_BRIEF);
      the_player->send_plr(_("Brief descriptions has been turned &+Roff&*.\n"));
   }
   else
   {
      game_flags->set_flag(GAMEFLAG_BRIEF);
      the_player->send_plr(_("Brief descriptions has been turned &+Gon&*.\n"));
   }
   return;
   the_parsed = NULL;
}


/***********************************************************************
 **  autopaycom - toggles autopay for this player on and off
 **
 ***********************************************************************/

void autopaycom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Flags    *game_flags;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   game_flags = the_player->get_gameflags();

   if (game_flags->get_flag(GAMEFLAG_AUTOPAY))
   {
      game_flags->clr_flag(GAMEFLAG_AUTOPAY);
      the_player->send_plr(_("AutoPay has been turned &+Roff&*.\n"));
   }
   else
   {
      game_flags->set_flag(GAMEFLAG_AUTOPAY);
      the_player->send_plr(_("AutoPay has been turned &+Gon&*.\n"));
   }
   return;
   the_parsed = NULL;
}


/***********************************************************************
 **  colorcom - toggles color for this player on and off
 **
 ***********************************************************************/

void colorcom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Flags    *game_flags;
   Builder  *the_builder;
   Flags    *bldr_gameflags;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   game_flags = the_player->get_gameflags();

   if ((the_builder = mainstruct->get_builder(the_player->get_name())) != NULL)
      bldr_gameflags = the_builder->get_gameflags();

   if (game_flags->get_flag(GAMEFLAG_COLOR))
   {
      game_flags->clr_flag(GAMEFLAG_COLOR);
      (the_player->get_connection())->set_color(0);

      if (the_builder != NULL)
      {
         (the_builder->get_connection())->set_color(0);
         bldr_gameflags->clr_flag(GAMEFLAG_COLOR);
      }

      the_player->send_plr(_("Color has been turned off.\n"));
   }
   else
   {
      game_flags->set_flag(GAMEFLAG_COLOR);
      (the_player->get_connection())->set_color(1);

      if (the_builder != NULL)
      {
         (the_builder->get_connection())->set_color(1);
         bldr_gameflags->set_flag(GAMEFLAG_COLOR);
      }

      the_player->send_plr(_("Color has been turned on.\n"));
   }
   return;
   the_parsed = NULL;
}


/***********************************************************************
 **  disconnectcom - disconnects a player from the mud
 **
 ***********************************************************************/

void disconnectcom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Player   *target_player;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   /* they need to specify a player or mobile to tell to */
   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Disconnect who?\n"));
      return;
   }

   if (!STRCASECMP(the_parsed->get_target1(), "all"))
   {
     the_player->send_plr("Disconnecting all players\n");
     the_player->flush_player();
     target_player = mainstruct->get_first_player();
     while (target_player != NULL)
     {
       // We don't want to disconnect ourselves
       if (target_player != the_player)
       {
	 target_player->set_off();
	 target_player->send_plr(_("You have been disconnected.\n"));
       }
       target_player = target_player->get_next_player();
     }
     the_player->send_plr("All players disconnected.\n");
     return;
   }

   if ((target_player = mainstruct->get_player(the_parsed->get_target1()))
                                                                     == NULL)
   {
      the_player->send_plr(_("That player is not connected.\n"));
      return;
   }

   target_player->set_off();
   target_player->send_plr(_("You have been disconnected.\n"));
   the_player->send_plr(_("Player disconnected.\n"));
   return;

}

/***********************************************************************
 **  wieldcom - allows a player to wield a weapon
 **
 ***********************************************************************/

void wieldcom(Parse *the_parsed, MudObject *the_user) {
   Player    *the_player;
   MudObject *the_obj;
   Weapon    *the_weapon;
   Strings   obj_str;
   int       the_num = 0;
   Strings   holder;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("What do you want to wield?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You wield a large bananna in your dream!\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to wield anything.\n"));
      return;
   }

   if (!(the_player->get_loc())->is_lit(the_player))
   {
      the_player->send_plr(_("It is too dark to see what to wield.\n"));
      return;
   }

   the_player->check_pronouns(the_parsed->get_target1(), &obj_str);

   if ((the_obj = the_player->find_contained(obj_str.str_show(),
                                                     &the_num)) == NULL)
   {
      the_player->send_plr(_("You don't have that weapon.\n"));
      return;
   }

   if (the_obj->get_type() != OBJ_TYPE_WEAPON)
   {
      the_player->send_plr(_("That is not a weapon.\n"));
      return;
   }

   the_weapon = (Weapon *) the_obj;

   if (check_specials(_("pre_wield"), the_weapon, NULL, NULL,
                                 the_weapon, the_player) == 3)
      return;

   if (the_player->wield_moveable((Moveable *) the_weapon) == -2)
   {
      the_player->send_plr(_("You do not have enough free hands.\n"));
      return;
   }

   the_player->send_plr(_("You wield the %s.\n"), the_weapon->get_title());

   holder.sprintf(_("%s wields %s %s menacingly.\n"), the_player->get_title(), 
     the_player->get_sex_possessive_str(), the_weapon->get_title());
   (the_player->get_loc())->send_location(holder.str_show(), the_player, 
                                                                   CAN_SEE);

   if (check_specials(_("post_wield"), the_weapon, NULL, NULL,
                                 the_weapon, the_player) == 3)
      return;

   return;
}


/***********************************************************************
 **  unwieldcom - allows a player to unwield a weapon
 **
 ***********************************************************************/

void unwieldcom(Parse *the_parsed, MudObject *the_user) {
   Player     *the_player;
   Moveable   *the_moveable;
   playerhand the_hand;
   Strings    holder;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("What do you want to unwield?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You unwield a large bananna in your dream!\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to unwield anything.\n"));
      return;
   }

   if (((the_moveable = the_player->get_wielded(Left)) == NULL) ||
        (STRCASECMP(the_moveable->get_title(), the_parsed->get_target1())))
   {
      if (((the_moveable = the_player->get_wielded(Right)) == NULL) ||
          (STRCASECMP(the_moveable->get_title(), the_parsed->get_target1())))
      {
         the_player->send_plr(_("You are not wielding that.\n"));
         return;
      }
      the_hand = Right;
   }
   else
      the_hand = Left;

   if (check_specials(_("pre_unwield"), the_moveable, NULL, NULL,
                                 the_moveable, the_player) == 3)
      return;

   the_player->unwield_moveable(the_hand);

   the_player->send_plr(_("You unwield the %s\n"), the_moveable->get_title());

   holder.sprintf(_("%s unwields %s %s.\n"), the_player->get_title(), 
     the_player->get_sex_possessive_str(), the_moveable->get_title());
   (the_player->get_loc())->send_location(holder.str_show(), the_player, 
                                                                   CAN_SEE);

   if (check_specials(_("post_unwield"), the_moveable, NULL, NULL, 
                                 the_moveable, the_player) == 3)
      return;

   return;
}


/***********************************************************************
 **  putcom - allows a player to put an object in another object
 **
 ***********************************************************************/

void putcom(Parse *the_parsed, MudObject *the_user) {
   Player         *the_player;
   MudObject      *the_obj = NULL;
   MudObject      *second_obj;
   int            num_obj = 0;
   Location       *plr_loc;
   int            plr_has_first = 1;
   int            plr_has_second = 1;
   Flags          *tmp_itemflags;
   Strings        holder;
   Strings        target2;
   int            results;


   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   plr_loc = the_player->get_loc();

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("format: put [<number>] <object> [in] <object>\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to wake up first.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to put anything anywhere.\n"));
      return;
   }

   if (!plr_loc->is_lit(the_player))
   {
      the_player->send_plr(_("It is too dark to put anything anywhere.\n"));
      return;
   }

   if (STRNCASECMP(the_parsed->get_target1(), _("all"),
                                          strlen(the_parsed->get_target1())))
   {
      if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                &num_obj)) == NULL)
      {
         plr_has_first = 0;
         if ((the_obj = plr_loc->find_contained(the_parsed->get_target1(),
                                                &num_obj)) == NULL)
         {
            the_player->send_plr(_("I can't seem to find the %s\n"),
                                               the_parsed->get_target1());
            return;
         }
      }
   }

   if ( (the_parsed->get_preposition() == NULL) ||
       ((!STRCASECMP(the_parsed->get_preposition(), _("in"))) &&
         (the_parsed->get_target2() == NULL)) )
   {
      if (!STRNCASECMP(the_parsed->get_target1(), _("all"),
                strlen(the_parsed->get_target1())))
      {
         the_player->send_plr(_("Put it where exactly?\n"));
         return;
      }
      else
      {
         the_player->send_plr(_("Put the %s in what?\n"), the_obj->get_title());
         return;
      }
   }

   if (STRCASECMP(the_parsed->get_preposition(), _("in")))
   {
      target2 = the_parsed->get_preposition();
   }
   else
   {
      target2 = the_parsed->get_target2();
   }

   if ((second_obj = the_player->find_contained(target2.str_show(),
                                             &num_obj)) == NULL)
   {
      plr_has_second = 0;
      if ((second_obj = plr_loc->find_contained(target2.str_show(),
                                             &num_obj)) == NULL)
      {
         the_player->send_plr(_("I can't seem to find the %s\n"),
			      target2.str_show());
         return;
      }
   }

   if (!second_obj->is_an_item())
   {
      the_player->send_plr(_("You can't put anything in that!\n"));
      return;
   }

   tmp_itemflags = ((Item *) second_obj)->get_itemflags();

   if (!tmp_itemflags->get_flag(ITEMFLAG_CONTAINER))
   {
      the_player->send_plr(_("The %s is not a container.\n"), second_obj->get_title());
      return;
   }

   if (tmp_itemflags->get_flag(ITEMFLAG_CLOSED))
   {
      the_player->send_plr(_("The %s is closed.\n"), second_obj->get_title());
      return;
   }


   if (!STRNCASECMP(the_parsed->get_target1(), _("all"),
                                     strlen(the_parsed->get_target1())))
   {
      Inventory *the_inventory;

      the_inventory = the_player->get_inventory();
      if (the_inventory == NULL)
      {    mainstruct->log_error(_("Request for inventory returned NULL!"),
                                                                 "putcom");
            return;
      }

      the_inventory->reset_current();
      the_obj = the_inventory->get_next();

      if (the_obj == NULL)
      {
         the_player->send_plr(_("You are not carrying anything.\n"));
      }

      while (the_obj != NULL)
      {
         if ((the_obj->is_a_moveable()) && (the_obj != second_obj))
         {
            if (((results = check_specials(_("preput_target"), the_obj,
              second_obj, NULL, the_obj, the_player)) == 3) || (results == 2))
               return;

            if (((results = check_specials(_("preput_receptacle"), second_obj,
              the_obj, NULL, second_obj, the_player) == 3)) || (results == 2))
               return;

            the_player->game_put_obj(the_obj, second_obj, plr_has_second,
                                                               0, 0);

            if (((results = check_specials(_("postput_target"), the_obj,
              second_obj, NULL, the_obj, the_player)) == 3) || (results == 2))
               return;

            if (((results = check_specials(_("postput_receptacle"), second_obj,
              the_obj, NULL, second_obj, the_player) == 3)) || (results == 2))
               return;
            the_inventory->reset_current();
            the_obj = the_inventory->get_next();

         }
         the_obj = the_inventory->get_next();
      }
      return;
   }

   if (the_obj == second_obj)
   {
      the_player->send_plr(_("That would be a neat trick!\n"));
      return;
   }

   if (!the_obj->is_a_moveable())
   {
      the_player->send_plr(_("You can't move that!\n"));
      return;
   }

   if (((results = check_specials(_("preput_target"), the_obj, second_obj, NULL,
                   the_obj, the_player)) == 3) || (results == 2))
      return;

   if (((results = check_specials(_("preput_receptacle"), second_obj, the_obj,
                 NULL, second_obj, the_player) == 3)) || (results == 2))
      return;

   the_player->game_put_obj(the_obj, second_obj, plr_has_second,
                       the_parsed->get_number(), !plr_has_first);

   if (((results = check_specials(_("postput_target"), the_obj, second_obj, NULL,
                   the_obj, the_player)) == 3) || (results == 2))
      return;

   if (((results = check_specials(_("postput_receptacle"), second_obj, the_obj,
                 NULL, second_obj, the_player) == 3)) || (results == 2))
      return;

   return;
}


/***********************************************************************
 **  wearcom - wears a wearable item
 **
 ***********************************************************************/

void wearcom(Parse *the_parsed, MudObject *the_user) {
   Player     *the_player;
   MudObject  *the_obj;
   Wearable   *the_wearable;
   int        results;
   int        num_obj = 0;
   Strings    holder;
   Location   *the_loc;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Wear what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to wake up first.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to wear something.\n"));
      return;
   }

   the_loc = the_player->get_loc();

   if (!the_loc->is_lit(the_player))
   {
      the_player->send_plr(_("It is too dark to see anything to wear.\n"));
      return;
   }

   if (STRCASECMP(the_parsed->get_target1(),_("all")) == 0)
   {   Inventory *the_inventory;
       int count = 0;

       the_inventory = the_player->get_inventory();
       if (the_inventory == NULL)
       {    mainstruct->log_error(_("Request for inventory returned NULL!"),
                                                              "wearcom");
            return;
       }

       the_inventory->reset_current();
       the_obj = the_inventory->get_next();
       while (the_obj != NULL)
       {  if (the_obj->get_type() == OBJ_TYPE_WEARABLE)
          {
             the_wearable = (Wearable *) the_obj;

             if (check_specials(_("pre_wear"), the_wearable, NULL, NULL,
                                 the_wearable, the_player) == 3)
                return;

             if (the_player->wear_item(the_wearable) > 0)
	     {
                the_player->
                        send_plr(_("You wear the %s\n"), the_obj->get_title());
                holder.sprintf(_("%s wears %s %s.\n"), the_player->get_title(), 
                    the_player->get_sex_possessive_str(), the_obj->get_title());
                the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
                the_loc->send_location(_("You hear the rustling of clothing.\n"),
                                                      the_player, CANT_SEE);
                count++;
	     }

             if (check_specials(_("post_wear"), the_wearable, NULL, NULL,
                                 the_wearable, the_player) == 3)
                return;

          }
          the_obj = the_inventory->get_next();
       }
       if (count == 0)
          the_player->send_plr(_("Nothing left to wear.\n"));
       return;
   }

   if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                     &num_obj)) == NULL)
   {
      the_player->send_plr(_("I can't seem to find the %s\n"),
                                            the_parsed->get_target1());
      return;
   }


   if (the_obj->get_type() != OBJ_TYPE_WEARABLE)
   {
      the_player->send_plr(_("You can't wear that!\n"));
      return;
   }

   the_wearable = (Wearable *) the_obj;

   if (check_specials(_("pre_wear"), the_wearable, NULL, NULL,
                                 the_wearable, the_player) == 3)
      return;

   if ((results = the_player->wear_item(the_wearable)) < 0)
   {
      if (results == -1)
      {
         the_player->send_plr(_("Error in mudcode\n"));
      }
      else if (results == -2)
      {
         the_player->send_plr(_("It won't fit over your other clothes!\n"));
      }
      else if (results == -3)
      {
         the_player->send_plr(_("You are already wearing it!\n"));
      }
      else if (results == -4)
      {
         the_player->send_plr(_("You have no free hands!\n"));
      }
      return;
   }
   the_player->send_plr(_("You wear the %s\n"), the_obj->get_title());

   holder.sprintf(_("%s wears %s %s.\n"), the_player->get_title(), 
       the_player->get_sex_possessive_str(), the_obj->get_title());
   the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
   the_loc->send_location(_("You hear the rustling of clothing.\n"),
                                                      the_player, CANT_SEE);

   if (check_specials(_("post_wear"), the_wearable, NULL, NULL,
                                 the_wearable, the_player) == 3)
      return;

   return;
}


/***********************************************************************
 **  removecom - removes a worn item
 **
 ***********************************************************************/

void removecom(Parse *the_parsed, MudObject *the_user) {
   Player     *the_player;
   MudObject  *the_obj;
   Individual *target_obj = NULL;
   int        num_obj = 0;
   Strings    target;
   Strings    holder;
   Location   *the_loc = NULL;
   Mobile     *ghost_mob;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Remove what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to wake up first.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to remove anything.\n"));
      return;
   }

   the_loc = the_player->get_loc();

   /* if they are trying to remove an article from a corpse */
   if ((the_parsed->get_preposition() != NULL) &&
       (strlen(the_parsed->get_preposition()) > 0))
   {
      /* allow for 'remove chainmail from corpse' or 'remove chainmail corpse' */
      if (!STRCASECMP(the_parsed->get_preposition(), _("from")))
      {
         target = the_parsed->get_target2();
      }
      else
      {
         if ((the_parsed->get_target2() == NULL) ||
             (strlen(the_parsed->get_target2()) == 0))
	 {
            target = the_parsed->get_preposition();
	 }
         else
            target = the_parsed->get_target2();
      }

      /* find the corpse in the room */
      if ((the_obj = the_loc->find_contained(target.str_show(), &num_obj)) == NULL)
      {
         the_player->send_plr(_("I can't seem to find the %s\n"), target.str_show());
         return;
      }

      if (!the_obj->is_an_individual())
      {
         the_player->send_plr(_("The %s is not an individual.\n"), target.str_show());
         return;
      }

      target_obj = (Individual *) the_obj;

      if (!(target_obj->get_indflags())->get_flag(INDFLAG_CORPSE))
      {
         the_player->send_plr(
            _("%s is alive and won't let you remove their clothes.\n"),
                                                    the_obj->get_title());
         return;
      }
   }
   else
   {
      target_obj = the_player;
   }

   if ((the_obj = target_obj->find_contained(the_parsed->get_target1(),
                                                     &num_obj)) == NULL)
   {
      if (target_obj == the_player)
         the_player->send_plr(_("I can't seem to find the %s.\n"),
                                            the_parsed->get_target1());
      else
         the_player->send_plr(_("They are not carrying the %s.\n"),
                                            the_parsed->get_target1());
      return;
   }

   num_obj = 0;

   if (check_specials(_("pre_remove"), the_obj, target_obj, NULL,
                                 the_obj, the_player) == 3)
      return;

   if ((the_obj->get_type() != OBJ_TYPE_WEARABLE) ||
       (!target_obj->remove_item((Wearable *) the_obj)))
   {
      if (target_obj == the_player)
         the_player->send_plr(_("You're not wearing it.\n"));
      else
         the_player->send_plr(_("The corpse is not wearing it.\n"));
      return;
   }

   if (target_obj == the_player)
   {
      the_player->send_plr(_("You remove the %s\n"), the_obj->get_title());

      holder.sprintf(_("%s removes %s %s.\n"), the_player->get_title(), 
       the_player->get_sex_possessive_str(), the_obj->get_title());
      the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
      the_loc->send_location(_("You hear the rustling of clothing.\n"),
                                                      the_player, CANT_SEE);
   }
   else
   {
      target_obj->remove_contained(the_obj);
      the_player->add_contained(the_obj);
      the_player->send_plr(_("You remove the %s from the corpse.\n"),
                                                  the_obj->get_title());
      holder.sprintf(_("%s removes the %s from the corpse.\n"),
                                the_player->get_title(), the_obj->get_title());

      if (the_loc == NULL)
      {
         printf(_("Loc should have been null here, fatal error.\n"));
         RAISE(11);
      }
      the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
      the_loc->send_location(_("You hear the rustling of clothing.\n"), 
                                                     the_player, CANT_SEE);

      /* lets see if a ghost is guarding this corpse and if so, curses are
         in order */
      holder.sprintf("%s@%s", target_obj->get_clones(),
                              the_config.hardcoded_areaname.str_show());
      if (((ghost_mob = (Mobile *) mainstruct->get_object(holder.str_show())) !=
                                                                       NULL) &&
           (the_loc == ghost_mob->get_loc()))
      {
         the_player->send_plr(
           _("The ghost's eyes flash red as it reaches out to you with a wispy\n"
           "finger.  It touches your forehead, sending a strong shiver through\n"
           "your entire body.\n"));
         holder.sprintf(
           _("The ghost's eyes flash red as it reaches a wispy finger towards\n"
           "%s and touches %s forehead.  %s shivers.\n"),
                         the_player->get_title(),
          the_player->get_sex_possessive_str(), the_player->get_title());
         the_loc->send_location(holder.str_show(), the_player);
         (the_player->get_gameflags())->set_flag(GAMEFLAG_CURSED);
      }
   }

   if (check_specials(_("post_remove"), the_obj, target_obj, NULL,
                                 the_obj, the_player) == 3)
      return;

   return;
}


/***********************************************************************
 **  statscom - gets statistics on a certain topic specified by the user
 **
 ***********************************************************************/

void statscom(Parse *the_parsed, MudObject *the_user) {
   Player     *the_player;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("&+WStatistics topics&*\n"));
      the_player->send_plr("&+B------------------------------&*\n");
      the_player->send_plr(_("&+Mobjectdb \t&+C- "
                           "&+wMudObject Database statistics\n"));
      the_player->send_plr(_("&+Mplrverbdb \t&+C- "
                           "&+wPlayer Verbs Database statistics\n"));
      the_player->send_plr(_("&+Mplrverbs \t&+C- "
                           "&+wStatistics on all Player verbs\n"));
      the_player->send_plr(_("&+Mbldrverbdb \t&+C- "
                           "&+wBuilder Verbs List statistics\n"));
      the_player->send_plr(_("&+Mglobal \t\t&+C- "
                           "&+wGlobal statistics on game performance\n"));
      the_player->send_plr(_("&+Mareas\t\t&+C- "
                           "&+wAll game area statistics\n"));
      the_player->send_plr(_("&+Mspecials\t&+C- "
                           "&+wTime stats on specials in an area\n"));
      the_player->send_plr(_("&+Mmem\t\t&+C- "
                           "&+wHow many of a particular data type exist\n"));
      the_player->send_plr(_("&+Mlevels\t\t&+C- "
                           "&+wCurrent level chains or levels in a chain\n"));
      the_player->send_plr("\n");
      return;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("objectdb"),
                                       strlen(the_parsed->get_target1())))
   {
      Object_List *the_dbase;

      the_dbase = mainstruct->get_dbase();
      the_dbase->display_data(the_player);
      return;
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("plrverbdb"),
                                       strlen(the_parsed->get_target1())))
   {
      Verb_List *the_verbs;

      the_verbs = mainstruct->get_player_verb_list();
      the_verbs->display_data(the_player);
      return;
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("plrverbs"),
                                       strlen(the_parsed->get_target1())))
   {
      Verb_List *the_verbs;

      the_verbs = mainstruct->get_player_verb_list();
      the_verbs->display_verb_stats(the_player);
      return;
   }

   else if (!STRNCASECMP(the_parsed->get_target1(), _("bldrverbdb"),
                                       strlen(the_parsed->get_target1())))
   {
      Verb_List *the_verbs;

      the_verbs = mainstruct->get_builder_verb_list();
      the_verbs->display_data(the_player);
      return;
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("global"),
                                       strlen(the_parsed->get_target1())))
   {
      mainstruct->display_mud_stats(the_player);
      return;
   }

   else if (!STRNCASECMP(the_parsed->get_target1(), _("areas"),
                                       strlen(the_parsed->get_target1())))
   {
      Object_List *object_db;

      object_db = mainstruct->get_dbase();
      object_db->display_list_data(the_player);
      return;
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("specials"),
                                       strlen(the_parsed->get_target1())))
   {
      Area_Dbase  *tmp_area;
      Object_List *tmp_dbase;
      Entity      *tmp_obj;
      int         counter = 0;
      Strings     holder;

      if (the_parsed->get_speech() != NULL)
         holder.assign_word(the_parsed->get_speech(), 1);

      if (holder.str_show() == NULL)
      {
         the_player->
            send_plr(_("You need to specify an area to list specials from.\n"));
         the_player->send_plr(_("  format: stats specials <areaname>\n"));
         return;
      }


      tmp_dbase = mainstruct->get_dbase();
      if ((tmp_area =
                  tmp_dbase->get_area(holder.str_show())) == NULL)
      {
         the_player->send_plr(_("Could not find the area %s.\n"),
                                                        holder.str_show());
         return;
      }

      the_player->send_plr(_("&+W    Name            Num Run  Min     Max     Avg&*\n"));
      the_player->send_plr("&+B--------------------------------------------------&*\n\n");
      tmp_area->reset_list();
      while ((tmp_obj = tmp_area->get_next_obj()) != NULL)
      {
         if (tmp_obj->get_type() == OBJ_TYPE_SPECIAL)
	 {
            ((Specials *)tmp_obj)->disp_time_stats(the_player);
            counter++;
	 }
      }
      the_player->send_plr("&+B--------------------------------------------------&*\n");
      the_player->send_plr(_("&+WSpecials Found: &*%d\n"), counter);
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("mem"),
                                       strlen(the_parsed->get_target1())))
   {
#ifdef DEBUG_MEM
      show_mem(the_player);
#else
      the_player->send_plr(
            _("\nDEBUG_MEM not defined during compile in config.h.\n"
            "This feature not available.\n\n"));
#endif
      return;
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("levels"),
                                       strlen(the_parsed->get_target1())))
   {
      Strings thetype;
      Level_List *level_dbase;

      level_dbase = mainstruct->get_levels();
      thetype.assign_word(the_parsed->get_speech(), 1);

      if ((thetype.str_show() == NULL) || (!STRNCASECMP(thetype.str_show(),
                                            _("chains"), thetype.str_len())))
      {
         level_dbase->show_chains(the_player);
         return;
      }
      else
      {
         level_dbase->show_levels(the_player, thetype.str_show());
         return;
      }
      return;
   }
   else
   {
      the_player->send_plr(_("I have no statistics on that.\n"));
      return;
   }

}


/***********************************************************************
 **  eatcom - eats a piece of food
 **
 ***********************************************************************/

void eatcom(Parse *the_parsed, MudObject *the_user) {
   Player     *the_player;
   Location   *the_loc;
   MudObject  *the_obj;
   int        player_has = 1;
   Strings    holder;
   Food       *the_food;
   char       *sex;
   int        the_num = 0;
   int        results;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Eat what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You eat a full steak dinner in your dream!\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to eat anything.\n"));
      return;
   }

   the_loc = the_player->get_loc();

   if (!the_loc->is_lit(the_player))
   {
      the_player->send_plr(_("You can't see anything to consume.\n"));
      return;
   }

   if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
   {
      player_has = 0;
      if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
      {
         the_player->send_plr(_("That is not here.\n"));
         return;
      }
   }

   if ((results = check_specials(_("pre_eat"), the_obj, NULL, NULL,
                                             the_obj, the_player)) == 3)
                return;

	 sex = the_player->get_sex_possessive_str();

   if (the_obj->get_type() != OBJ_TYPE_FOOD)
   {
      the_player->send_plr(_("You can't eat the %s!\n"), the_obj->get_title());
      holder.sprintf(_("%s attempts to stick %s %s in %s mouth.\n"),
                 the_player->get_name(), ((player_has) ? sex : _("the")),
                  the_obj->get_title(), sex);
      the_loc->send_location(holder.str_show(), the_player, CAN_SEE);
      return;
   }

   the_food = (Food *) the_obj;
   if (the_food->get_number_of() <= 0)
   {
      the_player->send_plr(_("Error, food item indicates 0 count.\n"));
      return;
   }


   the_player->send_plr(_("You consume the %s\n"), the_food->get_title());
   holder.sprintf(_("%s consumes %s %s\n"),
                  the_player->get_name(), (player_has) ? sex : _("the"),
                  the_obj->get_title());
   the_loc->send_location(holder.str_show(), the_player, CAN_SEE);

   the_food->set_number_of(the_food->get_number_of() - 1);

   if (the_food->get_number_of() == 0)
   {
      if (player_has)
         the_player->remove_contained(the_obj);
      else
         the_loc->remove_contained(the_obj);
      mainstruct->delete_object(the_obj);
      the_obj = NULL;
   }

   if ((the_obj != NULL) &&
       ((results = check_specials(_("post_eat"), the_obj, NULL, NULL,
                                             the_obj, the_player)) == 3))
                return;

   return;
}


/***********************************************************************
 **  lightcom - lights an object using another object
 **
 ***********************************************************************/

void lightcom(Parse *the_parsed, MudObject *the_user) {
   Player         *the_player;
   Location       *the_loc;
   MudObject      *the_obj;
   Strings        holder;
   int            has_lit_from = 1;
   int            has_lit = 1;
   int            the_num = 0;
   Item           *to_item;
   Item           *from_item;
   Flags          *tmp_itemflag;
   Individual     *guard;
   int            results;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Format: light <object> with <object>\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("Lighting things while sleeping is dangerous.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to light something.\n"));
      return;
   }

   the_loc = the_player->get_loc();
   if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
   {
      has_lit = 0;
      if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
      {
         the_player->send_plr(_("The %s is not here.\n"),
                                              the_parsed->get_target1());
         return;
      }
   }

   if (!the_obj->is_an_item())
   {
      the_player->send_plr(_("I'll bet you would like to burn %s.\n"),
                                           the_obj->get_title());
      return;
   }

   to_item = (Item *) the_obj;
   tmp_itemflag = to_item->get_itemflags();
   if (!tmp_itemflag->get_flag(ITEMFLAG_LIGHTABLE))
   {
      the_player->send_plr(_("It won't light.\n"));
      return;
   }
   if (tmp_itemflag->get_flag(ITEMFLAG_LIT))
   {
      the_player->send_plr(_("It is already lit.\n"));
      return;
   }

   the_obj = NULL;
   if (the_parsed->get_preposition() == NULL)
   {
      if ((the_obj = the_player->find_fire()) == NULL)
      {
         has_lit_from = 0;
         if ((the_obj = the_loc->find_fire()) == NULL)
         {
            the_player->send_plr(_("No fire here to light from.\n"));
            return;
         }
      }
   }
   else if (STRCASECMP(_("with"), the_parsed->get_preposition()))
   {
      the_player->send_plr(_("Format: light <object> with <object>\n"));
      return;
   }

   if ((the_obj == NULL) && ((the_parsed->get_target2() == NULL)))
   {
      the_player->send_plr(_("Format: light <object> with <object>\n"));
      return;
   }

   if ((the_obj == NULL) &&
       ((the_obj = the_player->find_contained(the_parsed->get_target2(),
                                                     &the_num)) == NULL))
   {
      has_lit_from = 0;
      if ((the_obj = the_loc->find_contained(the_parsed->get_target2(),
                                                     &the_num)) == NULL)
      {
         the_player->send_plr(_("The %s is not here.\n"),
                                             the_parsed->get_target2());
         return;
      }
   }

   if (!the_obj->is_an_item())
   {
      the_player->send_plr(_("The %s is not lit.\n"), the_obj->get_title());
      return;
   }

   from_item = (Item *) the_obj;
   tmp_itemflag = from_item->get_itemflags();
   if (!tmp_itemflag->get_flag(ITEMFLAG_LIT))
   {
      the_player->send_plr(_("The %s is not lit.\n"), the_obj->get_title());
      return;
   }

   if ((guard = to_item->is_guarded(the_player->get_loc())) != NULL)
   {
      guard->tell_ind(the_player, _("Keep that cold!"));
      return;
   }

   if ((results = check_specials(_("pre_light"), to_item, from_item, NULL,
                                             to_item, the_player)) == 3)
                return;

   tmp_itemflag = to_item->get_itemflags();
   tmp_itemflag->set_flag(ITEMFLAG_LIT);

   the_player->send_plr(_("You light %s %s with %s %s.\n"),
			(has_lit) ? _("your") : _("the"), to_item->get_title(),
                  (has_lit_from) ? _("your") : _("the"), from_item->get_title());

   holder.sprintf(_("%s lights %s %s with %s %s.\n"),
		  the_player->get_name(),
           (has_lit) ? the_player->get_sex_possessive_str() : _("the"),
                  to_item->get_title(),
         (has_lit_from) ? the_player->get_sex_possessive_str() : _("the"),
                  from_item->get_title());
   the_loc->send_location(holder.str_show(), the_player);

   if ((results = check_specials(_("post_light"), to_item, from_item, NULL,
                                             to_item, the_player)) == 3)
                return;

   return;

}


/***********************************************************************
 **  extinguishcom - puts out the fire!
 **
 ***********************************************************************/

void extinguishcom(Parse *the_parsed, MudObject *the_user) {
   Player     *the_player;
   Location   *the_loc;
   MudObject  *the_obj;
   Strings    holder;
   int        has_lit = 1;
   int        the_num = 0;
   Item       *the_item;
   Flags      *tmp_itemflag;
   Individual *guard;


   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Format: extinguish <object>\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to wake up first.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to extinguish anything.\n"));
      return;
   }

   the_loc = the_player->get_loc();
   if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
   {
      has_lit = 0;
      if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
      {
         the_player->send_plr(_("The %s is not here.\n"),
                                              the_parsed->get_target1());
         return;
      }
   }

   if (!the_obj->is_an_item())
   {
      the_player->send_plr(_("You can't put that out.\n"));
      return;
   }

   the_item = (Item *) the_obj;
   tmp_itemflag = the_item->get_itemflags();

   if (!tmp_itemflag->get_flag(ITEMFLAG_LIGHTABLE))
   {
      the_player->send_plr(_("Try as you might, you can't seem put it out.\n"));
      return;
   }

   if (!tmp_itemflag->get_flag(ITEMFLAG_LIT))
   {
      the_player->send_plr(_("It's not lit.\n"));
      return;
   }

   if ((guard = the_item->is_guarded(the_player->get_loc())) != NULL)
   {
      guard->tell_ind(the_player, _("That stays lit!"));
      return;
   }

   if (check_specials(_("pre_extinguish"), the_item, NULL, NULL,
                                             the_item, the_player) == 3)
                return;

   tmp_itemflag->clr_flag(ITEMFLAG_LIT);

   the_player->send_plr(_("You extinguish %s %s\n"),
			(has_lit) ? _("your") : _("the"), the_item->get_title());

   holder.sprintf(_("%s extinguishes %s %s.\n"),
		  the_player->get_name(),
           (has_lit) ? the_player->get_sex_possessive_str() : _("the"),
                  the_item->get_title());

   the_loc->send_location(holder.str_show(), the_player);

   if (check_specials(_("post_extinguish"), the_item, NULL, NULL,
                                             the_item, the_player) == 3)
                return;

   return;
}


/***********************************************************************
 **  killcom - starts a fight with an individual
 **
 ***********************************************************************/

void killcom(Parse *the_parsed, MudObject *the_user) {
   Player     *the_player;
   Location   *the_loc;
   MudObject  *the_obj;
   Strings    holder;
   Strings    target_str;
   Individual *the_target;
   char       *sex;
   int        the_num = 0;
   int        results;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Kill who?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You slaughter the pink pixies in your dream.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to kill.\n"));
      return;
   }

   the_player->check_pronouns(the_parsed->get_target1(), &target_str);

   the_loc = the_player->get_loc();

   if (!the_loc->is_lit(the_player))
   {
      the_player->send_plr(_("You can't see anyone to kill in the dark.\n"));
      return;
   }

   if ((the_obj = the_loc->find_contained(target_str.str_show(),
                                                     &the_num)) == NULL)
   {
      the_player->send_plr(_("They are not here.\n"));
      return;
   }

   if ((results = check_specials(_("pre_kill"), the_obj, the_player, NULL,
                                                the_obj, the_player)) == 3)
     return;
   else if (results == 2)
     return;

   if ((the_obj->get_type() != OBJ_TYPE_PLAYER) &&
       (the_obj->get_type() != OBJ_TYPE_MOBILE))
   {
      the_player->send_plr(_("That is not alive, you can't kill it.\n"));
      return;
   }

   if ((the_loc->get_locflags())->get_flag(LOCFLAG_PEACEFUL))
   {
      the_player->send_plr(
     _("The room radiates peace, prevents you from carrying out your attack.\n"));
      return;
   }

   the_target = (Individual *) the_obj;

   if (mainstruct->get_num_attacking(the_player) >= the_config.max_num_attacking)
   {
      the_player->send_plr(_("You can't fight with any more than you "
                                                       "already are.\n"));
      return;
   }

   sex = the_player->get_sex_possessive_str();

   if (the_target == (Individual *) the_player)
   {
      the_player->send_plr(_("Suicide does not become you.\n"));
      return;
   }

   if (the_player->get_target() == the_target)
   {
      the_player->send_plr(_("You are already fighting them.\n"));
      return;
   }

   if ((results = the_player->request_fight(the_target)) < 0)
   {
      /* too many attackers on them already */
      if (results == -2)
      {
         the_player->send_plr(_("You can't get at them with all the "
                              "others attacking them.\n"));
         return;
      }

      /* we are already fighting with them, shift our focus */
      if (results == -3)
      {
         the_player->switch_focus(the_target);
         the_player->send_plr(_("You switch your focus to %s!\n"),
                                              the_target->get_title());
         holder.sprintf(_("%s shifts %s focus to %s\n"),
                the_player->get_title(), sex, the_target->get_title());
         if (the_target->get_type() == OBJ_TYPE_PLAYER)
         {
            ((Player *) the_target)->
                                 send_plr(_("%s shifts %s focus to you.\n"),
                                 the_player->get_title(), sex);
            the_loc->send_location(holder.str_show(), the_player,
                                                   (Player *) the_target);
         }
         else
            the_loc->send_location(holder.str_show(), the_player);

         return;
      }
      if (results == -4)
      {
         the_player->send_plr(
             _("You want to attack the dead?  You really need to speak to someone\n"
             "about your excessive aggression.\n"));
         return;
      }
      return;
   }

   the_player->start_fight(the_target);
   the_player->send_plr(_("You attack %s!\n"), the_target->get_title());
   holder.sprintf(_("%s lunges at %s!\n"), the_player->get_title(),
                                               the_target->get_title());
   if (the_target->get_type() == OBJ_TYPE_PLAYER)
   {
      if (the_loc->is_lit((Player *) the_target))
         ((Player *) the_target)->send_plr(_("%s lunges at you!\n"),
                                                  the_player->get_title());
      the_loc->send_location(holder.str_show(), the_player,
                                           (Player *) the_target, CAN_SEE);
      the_loc->send_location(_("A growl emerges from the darkness!\n"), 
                                                   the_player, CANT_SEE);
   }
   else
      the_loc->send_location(holder.str_show(), the_player);

   if (check_specials(_("post_kill"), the_obj, the_player, NULL,
                                                the_target, the_player) == 3)
               return;

}


/***********************************************************************
 **  summoncom - summons a moveable object to the inventory of the player,
 **              or summons a non-moveable object to this location.  Does
 **              not summon locations
 **
 ***********************************************************************/

void summoncom(Parse *the_parsed, MudObject *the_user) {
   Location *cur_loc;
   MudObject *next_mudobj;
   MudObject *current_loc;
   Strings holder;
   Player *the_player;
   char   *the_or_no;
   Flags  *tmp_indflags;
   Player *tmp_player = NULL;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("You must specify a MudObject to summon.\n"));
      return;
   }

   cur_loc = the_player->get_loc();

   holder = the_parsed->get_target1();

   if ((next_mudobj = mainstruct->get_player(holder.str_show())) == NULL)
   {
      if (holder.num_char('@') != 1)
      {
         the_player->send_plr(_("'%s' is not a valid MudObject.\n"
                           "usage: summon <objname>@<areaname>\n"),
                                                       holder.str_show());
         return;
      }

      if ((next_mudobj = mainstruct->get_object(holder.str_show())) == NULL)
      {
         the_player->send_plr(_("That MudObject does not exist.\n"));
         return;
      }
   }

   if ((current_loc = next_mudobj->get_contained_by()) == NULL)
   {
      the_player->send_plr(_("You can't summon that object.  "
                           "It is not in the game.\n"));
      return;
   }

   if (next_mudobj->get_type() == OBJ_TYPE_LOCATION)
   {
      the_player->send_plr(_("You can't summon locations.\n"));
      return;
   }

   if (next_mudobj->is_a_moveable())
   {
      next_mudobj->set_location(the_player);
      the_or_no = _("the ");
   }
   else
   {
      if (next_mudobj->get_type() == OBJ_TYPE_PLAYER)
      {
         tmp_player = (Player *) next_mudobj;

         tmp_indflags = tmp_player->get_indflags();

         if (tmp_indflags->get_flag(INDFLAG_INBOAT))
         {
            Boat *the_boat;

            the_boat = (Boat *) tmp_player->get_contained_by();
            the_boat->exit_boat(tmp_player);
         }

         next_mudobj->set_location(the_player->get_loc());

         the_or_no = "";
      }
      else
      {
         next_mudobj->set_location(the_player->get_loc());
         the_or_no = _("the ");
	     
      }

   }

   if (next_mudobj->get_type() == OBJ_TYPE_PLAYER)
   {
     Player *trans_player = ((Player *) next_mudobj);

     trans_player->send_plr(
	 _("Your world folds in on itself and suddenly, you are elsewhere!\n"));
     trans_player->get_loc()->show_location(trans_player, 
		    trans_player->get_gameflags()->get_flag(GAMEFLAG_BRIEF));
   }

   the_player->send_plr(_("You summon %s%s out of thin air.\n"),
                                       the_or_no, next_mudobj->get_title());
   holder.sprintf(_("%s summons %s%s out of thin air.\n"),
           the_player->get_title(), the_or_no, next_mudobj->get_title());
   (the_player->get_loc())->send_location(holder.str_show(), the_player, tmp_player);


   return;
}


/***********************************************************************
 ** conditioncom - provides brief statistics on the player
 ***********************************************************************/

void conditioncom(Parse *the_parsed, MudObject *the_user)
{
   Player *the_player;
   char *color;
   int  percent;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   percent = (int) (((float) the_player->get_health() /
               (float) the_player->get_maxhealth()) * 100.0);

   if (percent == 100)
      color = "&+C";
   else if (percent > 80)
      color = "&+c";
   else if (percent > 60)
      color = "&+Y";
   else if (percent > 40)
      color = "&+Y";
   else if (percent > 20)
      color = "&+R";
   else
      color = "&+r";


   the_player->send_plr(_("\n\n&+GHealth:  \t%s%d&+W/&+C%d&*\n\n"), color,
                   the_player->get_health(), the_player->get_maxhealth());
   the_player->send_plr(_("&+GEndurance:  \t&+w%d&+W/&+c%d&*\n"),
         the_player->get_endurance(), the_player->get_max_endurance());
   the_player->send_plr(_("&+GMagic:  \t&+w%d&+W/&+c%d&*\n\n"),
                   the_player->get_magic(), the_player->get_max_magic());

   the_player->send_plr(_("&+GConstitution:  \t&+W%d&*\n"),
                                         the_player->get_constitution());
   the_player->send_plr(_("&+GStrength:  \t&+W%d&*\n"),
                                                the_player->get_strength());
   the_player->send_plr(_("&+GDexterity: \t&+W%d&*\n"),
                                               the_player->get_dexterity());
   the_player->send_plr(_("&+GIntelligence: \t&+W%d&*\n"),
                                                  the_player->get_intel());
   the_player->send_plr(_("&+GWisdom: \t&+W%d&*\n"),
                                                  the_player->get_wisdom());
   the_player->send_plr(_("&+GCharisma: \t&+W%d&*\n"),
                                                  the_player->get_charisma());
   the_player->send_plr(_("&+GWimpy: \t&+W%d&*\n\n"),
                                                  the_player->get_wimpy());

   the_player->send_plr(_("&+GExperience: \t&+W%d&*\n\n"), the_player->get_exp());

   the_player->send_plr(_("&+GLevels:&*\n"));
   the_player->show_levels(the_player);

   the_parsed = NULL;
}


/***********************************************************************
 ** emotecom - sends a message to all in the room the player is in with
 **            the player's name at the beginning of it
 **
 **
 ***********************************************************************/

void emotecom(Parse *the_parsed, MudObject *the_user) {

   Location *this_loc;
   Strings holder;
   Player *the_player;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;


   /* if there is nothing after chat, send a message to the player */
   if ((the_parsed->get_speech() == NULL) ||
       (strlen(the_parsed->get_speech()) <= 0))
   {
      the_player->send_plr(_("Format: emote <phrase>\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to open your eyes first.\n"));
      return;
   }
   holder.sprintf(_("%s %s\n"), the_player->get_name(),
                                  the_parsed->get_speech());
   this_loc = the_player->get_loc();
   this_loc->send_location(holder.str_show(), NULL);

}


/***********************************************************************
 ** abilitiescom - provides abilities this player has
 ***********************************************************************/

void abilitiescom(Parse *the_parsed, MudObject *the_user)
{
   Player *the_player;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   the_player->display_abilities();

   the_parsed = NULL;
}


/***********************************************************************
 ** weaponscom - provides the weapon proficiencies this player has
 ***********************************************************************/

void weaponscom(Parse *the_parsed, MudObject *the_user)
{
   Player *the_player;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   the_player->display_weapons();

   the_parsed = NULL;
}


/***********************************************************************
 **  castcom - casts a spell
 **     format: cast <spell> [<target>]
 **
 ***********************************************************************/

void castcom(Parse *the_parsed, MudObject *the_user) {
   Strings   targ_name;
   Strings   obj_name;
   Ability   *the_ability;
   MudObject *target = NULL;
   Player    *the_player;
   int       the_num = 0;
   Location  *cur_loc;

   /*
   Spell     *the_spell;
   Player    *the_player;
   Flags     *spellflags;

   Strings   formatted;
   int       succ_factor;
   int       results;
   */

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   cur_loc = the_player->get_loc();

   // First, get the ability and any possible targets
   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("You must specify a spell to cast.\n"));
      return;
   }

   if (((the_ability = mainstruct->get_ability(the_parsed->get_target1())) == NULL) ||
       (!the_player->find_ability(the_ability->get_name())))
   {
      the_player->send_plr(_("You don't know how to cast that.\n"));
      return;
   }

   if (the_ability->get_type() != OBJ_TYPE_SPELL)
   {
      the_player->send_plr(_("That is a skill; you must 'perform' skills.\n"));
      return;
   }

   if ((the_parsed->get_preposition() != NULL) &&
       (STRCASECMP(the_parsed->get_preposition(), _("at"))) &&
       (STRCASECMP(the_parsed->get_preposition(), _("on"))) &&
       (STRCASECMP(the_parsed->get_preposition(), _("in"))) &&
       (STRCASECMP(the_parsed->get_preposition(), _("the"))))

      obj_name = the_parsed->get_preposition();
   else if (the_parsed->get_preposition() != NULL)
      obj_name = the_parsed->get_target2();
   else
      obj_name.truncate(0);

   if (obj_name.str_show() != NULL)
      the_player->check_pronouns(obj_name.str_show(), &targ_name);
   else
      targ_name.truncate(0);


   if ((targ_name.str_show() != NULL) &&
       ((((Spell *)the_ability)->get_spellflags())->get_flag(SPELLFLAG_REQTARGET) ||
	(((Spell *)the_ability)->get_spellflags())->get_flag(SPELLFLAG_OPTTARGET)))

   {
      if ((target = cur_loc->find_contained(targ_name.str_show(),
                                                        &the_num)) == NULL)
      {
         the_player->send_plr(_("That is not here.\n"));
         return;
      }
   }

   the_ability->use_ability(the_player, target, targ_name.str_show());
   return;
}


/***********************************************************************
 **  performcom - perform a skill
 **     format: perform <skill> on [<target>]
 **
 ***********************************************************************/

void performcom(Parse *the_parsed, MudObject *the_user) {
   Strings   targ_name;
   Strings   obj_name;
   Ability   *the_ability;
   MudObject *target = NULL;
   Player    *the_player;
   int       the_num = 0;
   Location  *cur_loc;

   /*
   Spell     *the_spell;
   Strings   holder;
   Flags     *spellflags;

   int       the_num = 0;
   Strings   formatted;
   int       succ_factor;
   int       results;
   */

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   cur_loc = the_player->get_loc();

   // First, get the ability and any possible targets
   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("You must specify a skill to perform.\n"));
      return;
   }

   if (((the_ability = mainstruct->get_ability(the_parsed->get_target1())) == NULL) ||
       (!the_player->find_ability(the_ability->get_name())))
   {
      the_player->send_plr(_("You don't know how to perform that.\n"));
      return;
   }

   if (the_ability->get_type() != OBJ_TYPE_SKILL)
   {
      the_player->send_plr(_("That is a spell; you must 'cast' spells.\n"));
      return;
   }

   if ((the_parsed->get_preposition() != NULL) &&
       (STRCASECMP(the_parsed->get_preposition(), _("at"))) &&
       (STRCASECMP(the_parsed->get_preposition(), _("on"))) &&
       (STRCASECMP(the_parsed->get_preposition(), _("in"))) &&
       (STRCASECMP(the_parsed->get_preposition(), _("the"))))

      obj_name = the_parsed->get_preposition();
   else if (the_parsed->get_preposition() != NULL)
      obj_name = the_parsed->get_target2();
   else
      obj_name.truncate(0);

   if (obj_name.str_show() != NULL)
      the_player->check_pronouns(obj_name.str_show(), &targ_name);
   else
      targ_name.truncate(0);


   if ((targ_name.str_show() != NULL) &&
       ((((Skill *)the_ability)->get_skillflags())->get_flag(SKILLFLAG_REQTARGET) ||
	(((Skill *)the_ability)->get_skillflags())->get_flag(SKILLFLAG_OPTTARGET)))
   {
      if ((target = cur_loc->find_contained(targ_name.str_show(),
                                                        &the_num)) == NULL)
      {
         the_player->send_plr(_("That is not here.\n"));
         return;
      }
   }

   the_ability->use_ability(the_player, target, targ_name.str_show());
   return;
}


/***********************************************************************
 **  setcom - sets an attribute of this player or another player on the
 **           game
 **
 ***********************************************************************/

void setcom(Parse *the_parsed, MudObject *the_user)
{
   Player     *the_player;
   Player     *target_player = NULL;
   Entity     *target_obj = NULL;
   Flags      *admflags;
   Strings    holder;
   Strings    playername;
   User_Dbase *tmp_dbase;
   int        not_on = 0;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;

   the_player = (Player *) the_user;

   if ((the_parsed->get_target1() == NULL) ||
       (strlen(the_parsed->get_target1()) <= 0))
   {
      the_player->send_plr(_("Set what?\n"));
      return;
   }

   admflags = the_player->get_admflags();

   if (!STRCASECMP(the_parsed->get_target1(), _("time")))
   {
      char *tod[5] = {N_("night"), N_("dawn"), N_("morning"), N_("afternoon"), 
		      N_("dusk")};
      char *time_change[5] = {
          N_("Darkness suddenly envelops you as all light vanishes!\n"),
          N_("Suddenly, the sun peeks above the horizon.\n"),
          N_("The sun is suddenly above the horizon.\n"),
          N_("The sun is suddenly peaked in the sky.\n"),
          N_("The sun suddenly vanishes, leaving a barely lit sky.\n")};

      int i = 0;

      if (!admflags->get_flag(ADMINFLAG_SETENVIRON))
      {
         the_player->send_plr(_("You do not have permissions to set "
                                                         "the date.\n"));
         return;
      }

      if ((the_parsed->get_speech() == NULL) ||
          (strlen(the_parsed->get_speech()) <= 0))
      {
         the_player->send_plr(_("Please indicate a time to set it to.\n"
                "Valid times are: Night, Dawn, Morning, Afternoon, Dusk\n"));
         return;
      }

      /* find which time of day this is */
      while ((i < 5) && (STRCASECMP(the_parsed->get_speech(), _(tod[i]))))
         i++;

      if (i == 5)
      {
         the_player->send_plr(_("'%s' is not a valid time.\n"
                "Valid times are: Night, Dawn, Morning, Afternoon, Dusk\n"),
                                                  the_parsed->get_speech());
         return;
      }

      if (mainstruct->get_tod() == i)
      {
         the_player->send_plr(
             _("Time of day already set to that value, no changes made.\n"));
         return;
      }

      the_player->send_plr(_("Time of day set to: %s\n"), _(tod[i]));
      mainstruct->set_tod((TimeOfDay) i);
      mainstruct->send_all_outside(_(time_change[mainstruct->get_tod()]));

      return;
   }

   if (!STRCASECMP(the_parsed->get_target1(), _("defstart")))
   {
      if (!admflags->get_flag(ADMINFLAG_SETENVIRON))
      {
         the_player->send_plr(_("You do not have permissions to set "
                                          "the default start location.\n"));
         return;
      }

      if ((the_parsed->get_speech() == NULL) ||
          (strlen(the_parsed->get_speech()) <= 0))
      {
         the_player->send_plr(_("Please indicate a location to set it to\n"));
         return;
      }

      holder.assign_word(the_parsed->get_speech(), 1);
      mainstruct->set_def_start_loc(holder.str_show());

      the_player->send_plr(_("Default Start Location set to: %s\n"),
                                     mainstruct->get_def_start_loc());
      return;
   }

   if (the_parsed->get_speech() != NULL)
   {
      playername.assign_word(the_parsed->get_speech(), 1);
   }

   if ((playername.str_show() != NULL) &&
       (STRCASECMP(playername.str_show(), _("me"))))
   {
      Strings the_name;

      /* is the target a player in the game? */
      the_name = playername.str_show();
      if ((target_player = mainstruct->get_player(the_name.str_show()))
                                                                 == NULL)
      {

	 /* are they a player not on the game? */
         the_name.upper(0);
         tmp_dbase = mainstruct->get_user_database();

         target_player = new_Player(the_name.str_show());
         if (tmp_dbase->read_user(the_name.str_show(), target_player) < 1)
         {

	    /* finally, is it an object */
            if ((target_obj = mainstruct->get_object(playername.str_show()))
                                                                       == NULL)
	    {
               the_player->send_plr(_("That user or object does not exist.\n"));
               return;
	    }
         }
         not_on = 1;
      }

      if ((target_player != NULL) &&
          (!admflags->get_flag(ADMINFLAG_SETOTHERS)))
      {
            the_player->send_plr(_("You do not have permissions to set "
                                                  "other's attributes.\n"));
            return;
      }

      if ((target_obj != NULL) &&
          (!admflags->get_flag(ADMINFLAG_SETOBJECTS)))
      {
            the_player->send_plr(_("You do not have permissions to set "
                                                  "object attributes.\n"));
            return;
      }

   }
   else
   {
      target_player = the_player;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("description"),
                                         strlen(the_parsed->get_target1())))
   {
      if (target_player != the_player)
      {
         the_player->
               send_plr(_("Sorry, you can't change other's descriptions.\n"));
         return;
      }

      the_player->set_attrib_desc(target_player);
   }

   else if (!STRNCASECMP(the_parsed->get_target1(), _("prompt"),
                                         strlen(the_parsed->get_target1())))
   {
      Strings new_prompt;

      new_prompt.assign_phrase(the_parsed->get_speech(), 1);

      if (new_prompt.str_show() == NULL)
      {
         the_player->send_plr(_("Format: set prompt <player|me> <prompt>\n"));
         return;
      }

      target_player->set_prompt(new_prompt.str_show());
      return;
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("ability"),
                                         strlen(the_parsed->get_target1())))
   {
      Strings abil_name;
      Ability *the_abil;

      if (!admflags->get_flag(ADMINFLAG_SETPROTECTED))
      {
         the_player->send_plr(_("You do not have permissions to set "
                                                         "abilities.\n"));
         return;
      }

      abil_name.assign_word(the_parsed->get_speech(), 2);
    	  
      if (abil_name.str_show() == NULL)
      {
		the_player->send_plr(_("Format: set ability <player> <abilityname>\n"));
        return;
      }

      if ((the_abil = mainstruct->get_ability(abil_name.str_show())) == NULL)
      {
	 the_player->send_plr(_("The %s ability does not seem to exist.\n"), 
			     abil_name.str_show());
	 return;
      }
   
      if (target_player->find_ability(the_abil->get_name()))
      {
     	the_player->send_plr(_("They already have that ability.\n"));
        return;
      }

      target_player->add_rank(the_abil->get_name(), the_abil->get_type(), 0, 0);
      the_player->send_plr(_("The %s ability has been added to %s.\n"), 
			   the_abil->get_name(), target_player->get_title());
      return;
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("chatline"),
                                         strlen(the_parsed->get_target1())))
   {
      Strings chatline_name;
      ChatLine *the_chatline;
      Entity *tmp_entity;
      Verb_List *tmp_verblist;
      Strings plr_chatline;

      if (!admflags->get_flag(ADMINFLAG_SETPROTECTED))
      {
         the_player->send_plr(_("You do not have permissions to set "
                                                         "chatlines.\n"));
         return;
      }

      chatline_name.assign_word(the_parsed->get_speech(), 2);
     
      if (chatline_name.str_show() == NULL)
      {
	    the_player->send_plr(_("Format: set chatline <player> <chatlinename>\n"));
        return;
      }

      tmp_verblist = mainstruct->get_player_verb_list();
      if ((tmp_entity = tmp_verblist->get_verb(chatline_name.str_show())) 
                                                                         == NULL)
      {
	     tmp_verblist = mainstruct->get_builder_verb_list();
         if ((tmp_entity = tmp_verblist->get_verb(chatline_name.str_show())) 
                                                                         == NULL)
         {
	        the_player->send_plr(_("The %s chatline does not seem to exist.\n"), 
			     chatline_name.str_show());
	        return;
         }
      }

      if (tmp_entity->get_type() != OBJ_TYPE_CHATLINE)
      {
	    the_player->send_plr(_("The %s chatline does not seem to exist.\n"), 
			     chatline_name.str_show());
	    return;
      }

      the_chatline = (ChatLine *) tmp_entity;
      plr_chatline = target_player->get_chatlines();
      if (plr_chatline.find_in_str(chatline_name.str_show()))
      {
	     the_player->send_plr(_("They already have that chatline.\n"));
         return;
      }

      plr_chatline.str_cat(the_chatline->get_name());
      plr_chatline.str_cat(" ");

      target_player->set_chatlines(plr_chatline.str_show());

      the_player->send_plr(_("The %s chatline has been added to %s.\n"), 
			   the_chatline->get_name(), target_player->get_title());
      return;
   }
   else
   {
      int vartype;
      int max_val;
      int new_val;
      Strings attrib_name;
      Strings new_number;

      if (!admflags->get_flag(ADMINFLAG_SETPROTECTED))
      {
         the_player->send_plr(_("You do not have permissions to set "
                                                         "that attribute.\n"));
         return;
      }

      if (!STRNCASECMP(the_parsed->get_target1(), _("wimpy"),
		                          strlen(the_parsed->get_target1())))
      {
		 vartype = ATTRIB_WIMPY;
         max_val = 1000;
         attrib_name = _("Wimpy");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("health"),
		              	          strlen(the_parsed->get_target1())))
      {
	 vartype = ATTRIB_HEALTH;
         max_val = 10000;
         attrib_name = _("Health");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("strength"),
		              	          strlen(the_parsed->get_target1())))
      {
	 vartype = ATTRIB_STRENGTH;
         max_val = 100;
         attrib_name = _("Strength");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("dexterity"),
		              	          strlen(the_parsed->get_target1())))
      {
	 vartype = ATTRIB_DEXTERITY;
         max_val = 100;
         attrib_name = _("Dexterity");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("intelligence"),
		              	          strlen(the_parsed->get_target1())))
      {
	 vartype = ATTRIB_INTEL;
         max_val = 100;
         attrib_name = _("Intelligence");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("constitution"),
		              	          strlen(the_parsed->get_target1())))
      {
	 vartype = ATTRIB_CONSTITUTION;
         max_val = 100;
         attrib_name = _("Constitution");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("magic"),
			    strlen(the_parsed->get_target1())))
      {
	 vartype = ATTRIB_MAGIC;
         max_val = 10000;
         attrib_name = _("Magic");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("endurance"),
		              	          strlen(the_parsed->get_target1())))
      {
	 vartype = ATTRIB_ENDURANCE;
         max_val = 10000;
         attrib_name = _("Endurance");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("maxmagic"),
		              	          strlen(the_parsed->get_target1())))
      {
	 vartype = ATTRIB_MAXMAGIC;
         max_val = 10000;
         attrib_name = _("MaxMagic");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("maxendurance"),
		              	          strlen(the_parsed->get_target1())))
      {
		 vartype = ATTRIB_MAXENDUR;
         max_val = 10000;
         attrib_name = _("MaxEndurance");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("charisma"),
                                        strlen(the_parsed->get_target1())))
      {
         vartype = ATTRIB_CHARISMA;
         max_val = 100;
         attrib_name = _("Charisma");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("wisdom"),
                                        strlen(the_parsed->get_target1())))
      {
         vartype = ATTRIB_WISDOM;
         max_val = 100;
         attrib_name = _("Wisdom");
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("sex"), strlen(the_parsed->get_target1())))
      {
          vartype = ATTRIB_SEX;
          max_val = 2;
          attrib_name = _("Sex");
      }
      else
      {
         the_player->send_plr(_("That value is not currently settable.\n"));
         return;
      }

      new_number.assign_word(the_parsed->get_speech(), 2);

      if ((new_number.str_show() == NULL) ||
          (!isdigit(*(new_number.str_show()))))
      {
         the_player->send_plr(_("You must specify a number as %s.\n"), 
                                                   attrib_name.str_show());
         return;
      }

      new_val = atoi(new_number.str_show());

      if ((new_val < 0) || (new_val > max_val))
      {
         the_player->send_plr(_("Invalid %s value.\n"), attrib_name.str_show());
         return;
      }

      if (vartype == ATTRIB_WIMPY)
         target_player->set_wimpy(new_val);
      else if (vartype == ATTRIB_HEALTH)
	target_player->set_health(new_val);
      else if (vartype == ATTRIB_STRENGTH)
        target_player->set_strength(new_val);
      else if (vartype == ATTRIB_DEXTERITY)
        target_player->set_dexterity(new_val);
      else if (vartype == ATTRIB_INTEL)
        target_player->set_intel(new_val);
      else if (vartype == ATTRIB_CONSTITUTION)
        target_player->set_constitution(new_val);
      else if (vartype == ATTRIB_MAGIC)
        target_player->set_magic(new_val);
      else if (vartype == ATTRIB_ENDURANCE)
        target_player->set_endurance(new_val);
      else if (vartype == ATTRIB_MAXENDUR)
        target_player->set_max_endurance(new_val);
      else if (vartype == ATTRIB_MAXMAGIC)
        target_player->set_max_magic(new_val);
      else if (vartype == ATTRIB_WISDOM)
        target_player->set_wisdom(new_val);
      else if (vartype == ATTRIB_CHARISMA)
        target_player->set_charisma(new_val);
      else if (vartype == ATTRIB_SEX)
          target_player->set_sex(new_val);
      else
        fault("Error, no variable type defined in code\n");

      the_player->send_plr(_("%s value on %s set to %d\n"),
                  attrib_name.str_show(), target_player->get_title(), new_val);
      return;
   }

}


/***********************************************************************
 **  repeatcom - repeats the last command executed
 **
 ***********************************************************************/

void repeatcom(Parse *the_parsed, MudObject *the_user) {
   Player *the_player;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   the_player->repeat_last_command();

   the_parsed = NULL;
}



/***********************************************************************
 **  suggestcom - allows the player to offer suggestions for improvement
 **               of the mud.  Stores them in a log file
 **
 ***********************************************************************/

void suggestcom(Parse *the_parsed, MudObject *the_user) {
   Player  *the_player;
   Data_Log *suggestlog;
   Strings  tmp_str;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   suggestlog = mainstruct->get_suggestlog();

   if (the_parsed->get_speech() != NULL)
   {
      tmp_str = the_parsed->get_speech();
      tmp_str.str_cat("\n");
      suggestlog->log_data(the_player->get_name(), the_player->get_location(),
                                             tmp_str.str_show());
      the_player->send_plr(_("The suggestion has been logged.  Thanks!\n"));
   }
   else
   {

      the_player->send_plr(
       _("We appreciate your ideas!  Please input your feedback in the editor\n"
       "and they will be looked at soon.\n\n"));

      suggestlog->user_add_data(the_player);
   }

   the_parsed = NULL;
}



/***********************************************************************
 **  bugcom - allows the player to report bugs that they find in the
 **           mud
 **
 ***********************************************************************/

void bugcom(Parse *the_parsed, MudObject *the_user) {
   Player  *the_player;
   Data_Log *buglog;
   Strings tmp_str;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   buglog = mainstruct->get_buglog();

   if (the_parsed->get_speech() != NULL)
   {
      tmp_str = the_parsed->get_speech();
      tmp_str.str_cat("\n");
      buglog->log_data(the_player->get_name(), the_player->get_location(),
                                             tmp_str.str_show());
      the_player->send_plr(_("The bug has been logged.\n"));
   }
   else
   {
      the_player->send_plr(
         _("Ooh!  Found a bug eh?  Please provide as much detail as you can,\n"
         "such as what you were doing when you found it, what happened,\n"
         "where you were, and anything else you think may help in tracking\n"
         "and squashing this bug.  Thanks for your help!\n\n"));
      buglog->user_add_data(the_player);

   }

   the_parsed = NULL;
}


/***********************************************************************
 **  typocom - allows the player to report typos that they find in the
 **            mud
 **
 ***********************************************************************/

void typocom(Parse *the_parsed, MudObject *the_user) {
   Player  *the_player;
   Data_Log *typolog;
   Strings tmp_str;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   typolog = mainstruct->get_typolog();

   if (the_parsed->get_speech() != NULL)
   {
      tmp_str = the_parsed->get_speech();
      tmp_str.str_cat("\n");
      typolog->log_data(the_player->get_name(), the_player->get_location(),
                                             tmp_str.str_show());
      the_player->send_plr(_("The typo has been logged.\n"));
   }
   else
   {
      the_player->send_plr(
         _("Please provide the object name, where you found it, and the\n"
         "typo so we can fix it.  Thanks!!\n\n"));
      typolog->user_add_data(the_player);

   }

   the_parsed = NULL;
}


/***********************************************************************
 **  timecom - displays the real time and date, as well as the game time
 **            and date
 **
 ***********************************************************************/

void timecom(Parse *the_parsed, MudObject *the_user) {
   Player  *the_player;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   the_player->send_plr(_("&+GCurrent time (&+g%s time zone&+G):&*\n"),
                                               the_config.timezone.str_show());
   the_player->send_plr("\t%s\n\n", get_time_str());


   the_player->send_plr(_("&+GGame time:&*\n"));
   the_player->send_plr("   ");
   mainstruct->show_tod(the_player);
   the_player->send_plr(
        _("   It is day &+g%d&* of year &+C%d&* in the Age of Heroes.\n"),
                               mainstruct->get_doy(), mainstruct->get_yoa());
   the_player->send_plr("\n\n");

   the_parsed = NULL;
}


/***********************************************************************
 **  considercom - Analyses how good the opponent is and gives a rough
 **                estimate on how tough they might be
 **
 ***********************************************************************/

void considercom(Parse *the_parsed, MudObject *the_user) {
   char *str_level[7] = {
      N_("Would have trouble lifting a bored pebble"),
      N_("Is of average strength"),
      N_("Is fairly strong, but no Hercules"),
      N_("Has rippling muscles easily visible"),
      N_("Could easily lift boulders with one arm!"),
      N_("The ultimate strength, crushes coal into diamonds for fun!"),
      N_("Someone has been cheating!")};
   int  str_vals[6] = {5, 20, 40, 60, 80, 100};

   char *dex_level[7] = {
      N_("Trips over flat ground."),
      N_("Occasionally stumbles over his own feet"),
      N_("Can hit most of what they would aim for"),
      N_("Could dance a jig around a wild boar"),
      N_("Fast and accurate, they could hit a flying bee a mile away!"),
      N_("Ultimate dexterity!  A miss is not in their vocabulary"),
      N_("Someone has been cheating!")};
   int  dex_vals[6] = {5, 20, 40, 60, 80, 100};

   char *int_level[7] = {
      N_("An amoeba is smarter"),
      N_("Not the brightest in the bunch"),
      N_("Can do some math; occasionally maintains concentration"),
      N_("Performs calculus for fun; concentration is unwaivering"),
      N_("A stunning intellect"),
      N_("Ultimate intelligence!  Could outsmart some of the best wizards"),
      N_("Someone has been cheating!")};
   int  int_vals[6] = {5, 20, 40, 60, 80, 100};

   char *health_level[7] = {
      N_("Would be hurt by a mosquito bite"),
      N_("Could only withstand a few blows"),
      N_("Can take quite a pounding"),
      N_("Is very sturdy and extremely durable"),
      N_("Would require quite a bit of pounding to take down"),
      N_("Ultimate health!  Nearly impossible to take down"),
      N_("Someone has been cheating!")};
   int  health_vals[6] = {30, 60, 150, 300, 600, 10000};

   char *wis_level[7] = {
      N_("An amoeba has better judgment"),
      N_("Occasionally makes the right call"),
      N_("Can compete in a philisophical debate"),
      N_("A truely sagacious person"),
      N_("A revered Nestor"),
      N_("Ultimate wisdom!  A veritable Oracle"),
      N_("Someone has been cheating!")};
   int  wis_vals[6] = {5, 20, 40, 60, 80, 100};

   char *cha_level[7] = {
      N_("A troll is more charismatic"),
      N_("Not the most diplomatic"),
      N_("Shows leadership qualities"),
      N_("A true leader of men"),
      N_("Followers just flock to them"),
      N_("Ultimate Charisma!  Could persuade anyone of anything"),
      N_("Someone has been cheating!")};
   int  cha_vals[6] = {5, 20, 40, 60, 80, 100};

   char *con_level[7] = {
      N_(""),
      N_(""),
      N_(""),
      N_(""),
      N_(""),
      N_(""),
      N_("Someone has been cheating!")};
   int  con_vals[6] = {5, 20, 40, 60, 80, 100};

   char *overall_level[7] = {
      N_("A child could defeat this"),
      N_("A newbie may have trouble"),
      N_("Some experience needed to defeat this one"),
      N_("A tough challenge even for seasoned veterans"),
      N_("A walking killing machine"),
      N_("Only the absolute best could defeat this!"),
      N_("Someone has been cheating!")};
   Location   *the_loc;
   MudObject  *the_obj;
   Player     *the_player;
   Individual *the_target;
   int        the_str = 6;
   int        the_dex = 6;
   int        the_int = 6;
   int        the_con = 6;
   int        the_wis = 6;
   int        the_cha = 6;
   int        the_health = 6;
   int        overall;
   int        the_num = 0;
   int        i;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Consider who?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to open your eyes first.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to consider anything.\n"));
      return;
   }

   the_loc = the_player->get_loc();
   if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
   {
      the_player->send_plr(_("They are not here.\n"));
      return;
   }

   if ((the_obj->get_type() != OBJ_TYPE_PLAYER) &&
       (the_obj->get_type() != OBJ_TYPE_MOBILE))
   {
      the_player->send_plr(_("That is not alive, you can't consider it.\n"));
      return;
   }

   the_target = (Individual *) the_obj;

   if ((the_target->get_indflags())->get_flag(INDFLAG_CORPSE))
   {
      the_player->send_plr(_("The corpse is quite dead.\n"));
      return;
   }

   if ((the_target->get_indflags())->get_flag(INDFLAG_GHOST))
   {
      the_player->send_plr(_("The ghost is quite dead.\n"));
      return;
   }

   the_player->send_plr(_("&+WConsider for: &+M%s\n"), the_target->get_title());
   the_player->send_plr("&+B------------------------------&*\n\n");

   for (i=0; i<6; i++)
   {
      if (the_target->get_strength() < str_vals[i])
      {
         the_str = i;
         break;
      }
   }

   for (i=0; i<6; i++)
   {
      if (the_target->get_dexterity() < dex_vals[i])
      {
         the_dex = i;
         break;
      }
   }

   for (i=0; i<6; i++)
   {
      if (the_target->get_intel() < int_vals[i])
      {
         the_int = i;
         break;
      }
   }

   for (i=0; i<6; i++)
   {
      if (the_target->get_health() < health_vals[i])
      {
         the_health = i;
         break;
      }
   }

   for (i=0; i<6; i++)
   {
      if (the_target->get_constitution() < con_vals[i])
      {
         the_con = i;
         break;
      }
   }

   for (i=0; i<6; i++)
   {
      if (the_target->get_wisdom() < wis_vals[i])
      {
         the_wis = i;
         break;
      }
   }
   for (i=0; i<6; i++)
   {
      if (the_target->get_charisma() < cha_vals[i])
      {
         the_cha = i;
         break;
      }
   }

   the_player->send_plr(_("&+GStrength: &*%s\n"), _(str_level[the_str]));
   the_player->send_plr(_("&+GDexterity: &*%s\n"), _(dex_level[the_dex]));
   the_player->send_plr(_("&+GIntel: &*%s\n"), _(int_level[the_int]));
   the_player->send_plr(_("&+GHealth: &*%s\n"), _(health_level[the_health]));
   the_player->send_plr(_("&+GConstitution: &*%s\n"), _(con_level[the_con]));
   the_player->send_plr(_("&+GWisdom: &*%s\n"), _(wis_level[the_wis]));
   the_player->send_plr(_("&+GCharisma: &*%s\n"), _(cha_level[the_cha]));

   overall = (int) ((the_str + the_dex + the_int + the_health + the_con + 
		     the_wis + the_cha) / 7);

   the_player->send_plr(_("&+COverall: &*%s\n"), _(overall_level[overall]));

   the_parsed = NULL;
}


/***********************************************************************
 **  diagnosecom - Determines how hurt the mobile is, like examine but
 **                without the examine
 **
 ***********************************************************************/

void diagnosecom(Parse *the_parsed, MudObject *the_user) {

   Location   *the_loc;
   MudObject  *the_obj;
   Player     *the_player;
   Individual *the_target;
   int        the_num = 0;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to open your eyes first.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to diagnose anyone.\n"));
      return;
   }

   the_loc = the_player->get_loc();
   if (!the_loc->is_lit(the_player))
   {
      the_player->send_plr(_("You can't diagnose that which you cannot see.\n"));
      return;
   }

   if (the_parsed->get_target1() == NULL)
   {
      if ((the_target = the_player->get_target()) == NULL)
      {
         the_player->send_plr(_("Diagnose who?\n"));
         return;
      }
   }
   else
   {
      if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                      &the_num)) == NULL)
      {
         the_player->send_plr(_("They are not here.\n"));
         return;
      }

      if ((the_obj->get_type() != OBJ_TYPE_PLAYER) &&
          (the_obj->get_type() != OBJ_TYPE_MOBILE))
      {
         the_player->send_plr(_("That is not alive, you can't diagnose it.\n"));
         return;
      }

      the_target = (Individual *) the_obj;
   }

   if ((the_target->get_indflags())->get_flag(INDFLAG_CORPSE))
   {
      the_player->send_plr(_("The corpse is quite dead.\n"));
      return;
   }

   if ((the_target->get_indflags())->get_flag(INDFLAG_GHOST))
   {
      the_player->send_plr(_("The ghost is quite dead.\n"));
      return;
   }

   the_target->display_health(the_player);
   the_parsed = NULL;
}


/***********************************************************************
 **  sitcom - sits down
 **
 ***********************************************************************/

void sitcom(Parse *the_parsed, MudObject *the_user) {
   Player     *the_player;
   Flags      *tmp_indflags;
   Strings    holder;
   Location   *the_loc;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_parsed->get_target1() != NULL)
   {
      holder.sprintf(_("enter %s"), the_parsed->get_target1());
      run_command(the_user, holder.str_show());
      return;
   }

   tmp_indflags = the_player->get_indflags();
   if (tmp_indflags->get_flag(INDFLAG_FIGHTING))
   {
      the_player->send_plr(_("Sitting while fighting is not recommended.\n"));
      return;
   }

   if (tmp_indflags->get_flag(INDFLAG_SITTING))
   {
      the_player->send_plr(_("You are already sitting.\n"));
      return;
   }
   if (tmp_indflags->get_flag(INDFLAG_SLEEPING))
   {
      the_player->send_plr(_("You are asleep and must wake (stand) first.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to sit.\n"));
      return;
   }

   if (check_specials(_("pre_sit"), the_player, NULL, NULL,
                                                the_player, the_player) == 3)
      return;

   the_loc = the_player->get_loc();

   tmp_indflags->set_flag(INDFLAG_SITTING);
   the_player->send_plr(_("You assume the sitting position.\n"));
   holder.sprintf(_("%s assumes the sitting position.\n"),
                                                 the_player->get_title());
   the_loc->send_location(holder.str_show(), the_player);

   if (check_specials(_("post_sit"), the_player, NULL, NULL,
                                                the_player, the_player) == 3)
      return;

   return;

   the_parsed = NULL;
}


/***********************************************************************
 **  sleepcom - sleeps, so the player heals faster but can't do some stuff
 **
 ***********************************************************************/

void sleepcom(Parse *the_parsed, MudObject *the_user) {
   Player     *the_player;
   Flags      *tmp_indflags;
   Strings    holder;
   Location   *the_loc;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   tmp_indflags = the_player->get_indflags();
   if (tmp_indflags->get_flag(INDFLAG_FIGHTING))
   {
      the_player->send_plr(_("Sleeping while fighting is not recommended.\n"));
      return;
   }

   if (tmp_indflags->get_flag(INDFLAG_SLEEPING))
   {
      the_player->send_plr(_("You are already sleeping.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to sleep.\n"));
      return;
   }

   if (tmp_indflags->get_flag(INDFLAG_SITTING))
   {
      tmp_indflags->clr_flag(INDFLAG_SITTING);
   }

   if (check_specials(_("pre_sleep"), the_player, NULL, NULL,
                                                the_player, the_player) == 3)
      return;

   the_loc = the_player->get_loc();

   tmp_indflags->set_flag(INDFLAG_SLEEPING);
   the_player->send_plr(
          _("You lie down, close your eyes, and drift off to sleep.\n"));
   holder.sprintf(_("%s lies down and starts snoring away.\n"),
                                                 the_player->get_title());
   the_loc->send_location(holder.str_show(), the_player);
   return;

   if (check_specials(_("post_sleep"), the_player, NULL, NULL,
                                                the_player, the_player) == 3)
      return;

   the_parsed = NULL;
}

/***********************************************************************
 **  standcom - sits down
 **
 ***********************************************************************/

void standcom(Parse *the_parsed, MudObject *the_user) {
   Player     *the_player;
   Flags      *tmp_indflags;
   Strings    holder;
   Location   *the_loc;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   tmp_indflags = the_player->get_indflags();

   if ((!tmp_indflags->get_flag(INDFLAG_SITTING)) &&
       (!tmp_indflags->get_flag(INDFLAG_SLEEPING)))
   {
      the_player->send_plr(_("You are not sitting or sleeping.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to stand.\n"));
      return;
   }

   the_loc = the_player->get_loc();

   if (check_specials(_("pre_stand"), the_player, NULL, NULL,
                                                the_player, the_player) == 3)
      return;

   if (tmp_indflags->get_flag(INDFLAG_SITTING))
   {
      tmp_indflags->clr_flag(INDFLAG_SITTING);
   }
   if (tmp_indflags->get_flag(INDFLAG_SLEEPING))
   {
      tmp_indflags->clr_flag(INDFLAG_SLEEPING);
   }

   the_player->send_plr(_("You climb to your feet.\n"));
   holder.sprintf(_("%s climbs to his feet\n"), the_player->get_title());
   the_loc->send_location(holder.str_show(), the_player);

   if (check_specials(_("post_stand"), the_player, NULL, NULL,
                                                the_player, the_player) == 3)
      return;

   return;

   the_parsed = NULL;
}


/***********************************************************************
 ** logcom - displays a specified log in specified ways
 **
 ***********************************************************************/
void logcom(Parse *the_parsed, MudObject *the_user) 
{
    Player *the_player = NULL;
    Builder *the_builder = NULL;
    Strings logname;
    Strings logfile;
    Logs    *the_log;
    Strings keywords;
    Strings word_holder;
    int     num_lines = 0;
    int     first_keyword = 0;
    int     i = 1;
    int     search_all = 0;
    int     archive = 0;
    Connection *the_conn;
    int     pager_lines;
    
    switch (the_user->get_type()) {
    case OBJ_TYPE_PLAYER:
        the_player = (Player *) the_user;
        break;
    case OBJ_TYPE_BUILDER:
        the_builder = (Builder *) the_user;
        break;
    default:
        return;
    }
    
    if (!STRCASECMP(the_parsed->get_verb(), _("syslog")))
    {
        logname = _("System Log");
        the_log = mainstruct->get_syslog();
        logfile = SYSLOGNAME;
    }
    else if (!STRCASECMP(the_parsed->get_verb(), _("errlog")))
    {
        logname = _("Error Log");
        the_log = mainstruct->get_log();
        logfile = ERRLOGNAME;
    }
    else if (!STRCASECMP(the_parsed->get_verb(), _("sugglog")))
    {
        logname = _("Suggestions Log");
        the_log = mainstruct->get_suggestlog();
        logfile = SUGGESTLOGNAME;
    }
    else if (!STRCASECMP(the_parsed->get_verb(), _("buglog")))
    {
        logname = _("Bug Log");
        the_log = mainstruct->get_buglog();
        logfile = BUGLOGNAME;
    }
    else if (!STRCASECMP(the_parsed->get_verb(), _("typolog")))
    {
        logname = _("Typo Log");
        the_log = mainstruct->get_typolog();
        logfile = TYPOLOGNAME;
    }
    
    else
    {
        if (the_player != NULL)
            the_player->send_plr(_("%s not supported yet.\n"),
            the_parsed->get_verb());
        else if (the_builder != NULL)
            the_builder->send_bldr(_("%s not supported yet.\n"),
            the_parsed->get_verb());
        return;
    }
    
    while (!first_keyword)
    {
        word_holder.assign_word(the_parsed->get_speech(), i);
        if (word_holder.str_show() != NULL)
        {
            if (word_holder.str_cmp(_("archive")))
                archive = 1;
            else if (word_holder.str_cmp(_("all")))
                search_all = 1;
            else if (isdigit(*word_holder.str_show()))
                num_lines = atoi(word_holder.str_show());
            else
            {
                first_keyword = 1;
                i--;
            }
        }
        else
            break;
        i++;
    }
    
    /* first, pass in a string and see if it has keywords in it */
    while ((first_keyword) &&
        (word_holder.assign_word(the_parsed->get_speech(), i) != -1))
    {
        keywords.str_cat(word_holder.str_show());
        keywords.str_cat(" ");
        i++;
    }
    
    if (the_player != NULL)
        the_conn = the_player->get_connection();
    else
        the_conn = the_builder->get_connection();
    
    if (archive)
    {
        if (!STRCASECMP(the_parsed->get_verb(), _("syslog")))
        {
            if (((SysLog *) the_log)->archive_syslog() == -1)
            {
                the_conn->send_to_socket(_("Archive failed.\n"));
                return;
            }
        }
        else if (!STRCASECMP(the_parsed->get_verb(), _("errlog")))
        {
            if (((ErrLog *) the_log)->archive_errlog() == -1)
            {
                the_conn->send_to_socket(_("Archive failed.\n"));
                return;
            }
        }
        else if ((!STRCASECMP(the_parsed->get_verb(), _("suggestlog"))) ||
            (!STRCASECMP(the_parsed->get_verb(), _("buglog"))) ||
            (!STRCASECMP(the_parsed->get_verb(), _("typolog"))))
        {
            if (((Data_Log *) the_log)->
                archive_datalog(logfile.str_show()) == -1)
            {
                the_conn->send_to_socket(_("Archive failed.\n"));
                return;
            }
        }
        else
        {
            if (the_player != NULL)
                the_player->send_plr(_("Unable to archive log %s"),
                logname.str_show());
            if (the_builder != NULL)
                the_builder->send_bldr(_("Unable to archive log %s"),
                logname.str_show());
        }
        the_conn->send_to_socket(_("Archive completed successfully.\n"));
        return;
    }
    
    if (the_player != NULL)
    {
        the_player->send_plr(_("Querying Log: %s\n\n"), logname.str_show());
        pager_lines = the_player->get_pager_lines();
    }
    else
    {
        the_builder->send_bldr(_("Querying Log: %s\n\n"), logname.str_show());
        pager_lines = the_builder->get_pager_lines();
    }
    
    the_log->display_log(the_conn, logfile.str_show(),
        num_lines, &keywords, search_all, pager_lines);
    
    return;
}



/***********************************************************************
 ** playcom - a specials action, looks for special code
 ***********************************************************************/

void playcom(Parse *the_parsed, MudObject *the_user)
{
   Player *the_player;
   Location *this_loc;
   MudObject *the_obj;
   int       the_num = 0;
   int       results;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   /* they need to specify an object to move */
   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Play what?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("Your fingers twitch as you dream of playing.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to play anything.\n"));
      return;
   }

   this_loc = the_player->get_loc();
   if ((the_obj = this_loc->find_contained(the_parsed->get_target1(),
                                                          &the_num)) == NULL)
   {
      if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                          &the_num)) == NULL)
      {
         the_player->send_plr(_("That is not here.\n"));
         return;
      }
   }

   if ((results = check_specials(_("pre_play"), the_obj, NULL, NULL,
                                 the_obj, the_player)) == 3)
                return;

   if (results == 0)
   {
      the_player->send_plr(_("It produces no sound.\n"));
      return;
   }
   return;
}


/***********************************************************************
 **  noslain - toggles noslain for this player on and off
 **
 ***********************************************************************/

void noslaincom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Flags    *admin_flags;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   admin_flags = the_player->get_admflags();

   if (admin_flags->get_flag(ADMINFLAG_SEESLAIN))
   {
      admin_flags->clr_flag(ADMINFLAG_SEESLAIN);
      the_player->send_plr(_("Slain messages have been turned off.\n"));
   }
   else
   {
      admin_flags->set_flag(ADMINFLAG_SEESLAIN);
      the_player->send_plr(_("Slain messages have been turned on.\n"));
   }
   return;
   the_parsed = NULL;
}


/***********************************************************************
 ** qinfocom - shows information on the quests available or a specific
 **            quest
 **
 ***********************************************************************/
void qinfocom(Parse *the_parsed, MudObject *the_user) {
   Player      *the_player = NULL;
   Object_List *the_dbase;
   Quest       *the_quest;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;

   the_player = (Player *) the_user;

   if (!the_config.conf_flags->get_flag(CF_USEQUESTS))
   {
      the_player->send_plr(_("Quests have been turned off.\n"));
      return;
   }

   the_dbase = mainstruct->get_dbase();
   if (the_parsed->get_target1() == NULL)
   {
      the_dbase->list_quests(the_player);
      return;
   }

   if ((the_quest =
           the_dbase->get_quest_obj(the_parsed->get_target1())) == NULL)
   {
      the_player->send_plr(_("That Quest does not exist.\n"));
      return;
   }

   the_quest->display_qinfo(the_player);
   return;

}


/***********************************************************************
 ** questscom - shows the completed and uncompleted quests for this
 **             player
 **
 ***********************************************************************/
void questscom(Parse *the_parsed, MudObject *the_user) {
   Player      *the_player = NULL;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;

   the_player = (Player *) the_user;

   if (!the_config.conf_flags->get_flag(CF_USEQUESTS))
   {
      the_player->send_plr(_("Quests have been turned off.\n"));
      return;
   }

   the_player->display_quests();
   return;

   the_parsed = NULL;
}


/***********************************************************************
 **  entercom - enters an object, like a boat
 **
 ***********************************************************************/

void entercom(Parse *the_parsed, MudObject *the_user) {
   Player         *the_player;
   Location       *the_loc;
   MudObject      *the_obj;
   Strings        holder;
   Item           *the_item;
   Individual     *guard;
   int            results;
   int            the_num = 0;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Format: enter <object>\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You enter into a fit of snoring.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to enter anything.\n"));
      return;
   }

   the_loc = the_player->get_loc();
   if ((the_obj = the_loc->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
   {
      if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
      {
         the_player->send_plr(_("The %s is not here.\n"),
                                              the_parsed->get_target1());
         return;
      }
      else
      {
         the_player->send_plr(_("You must drop it before you can enter it.\n"));
         return;
      }
   }

   if (!STRCASECMP(the_parsed->get_verb(), _("sit")))
   {

   }

   if (!the_obj->is_an_item())
   {
      the_player->send_plr(_("I'll bet you would like to do that.\n"));
      return;
   }

   if ((the_player->get_contained_by())->get_type() != OBJ_TYPE_LOCATION)
   {
     the_player->send_plr(_("You are already in something.  "
                                                     "Type exit first.\n"));
      return;
   }

   the_item = (Item *) the_obj;

   if ((guard = the_item->is_guarded(the_player->get_loc())) != NULL)
   {
      guard->tell_ind(the_player, _("Stay out of there!"));
      return;
   }

   if ((results = check_specials(_("pre_enter"), the_item, NULL, NULL,
                                             the_item, the_player)) == 3)
      return;

   if (the_item->get_type() == OBJ_TYPE_BOAT)
   {
      Boat *the_boat;

      the_boat = (Boat *) the_item;

      if (the_boat->enter_boat(the_player) == -2)
      {
         the_player->send_plr(_("Sorry, the %s is too full.\n"),
                                                     the_item->get_title());
         return;
      }
   }
   else
   {
      the_player->send_plr(_("You can't enter that.\n"));
      return;
   }

   the_player->send_plr(_("You enter the %s.\n"), the_item->get_title());

   holder.sprintf(_("%s enters the %s.\n"), the_player->get_name(),
                                                      the_item->get_title());
   the_loc->send_location(holder.str_show(), the_player);

   if ((results = check_specials(_("post_enter"), the_item, NULL, NULL,
                                             the_item, the_player)) == 3)
      return;

   return;
}


/***********************************************************************
 **  leavecom - exits an object, like a boat
 **
 ***********************************************************************/

void leavecom(Parse *the_parsed, MudObject *the_user) {
   Player         *the_player;
   MudObject      *tmp_loc;
   Location       *the_loc;
   MudObject      *the_obj;
   Flags          *tmp_locflags;
   Strings        holder;
   int            results;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_player->is_asleep())
   {
      the_player->
            send_plr(_("You exit from the Grand Palace (in your dreams).\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to leave anything.\n"));
      return;
   }

   the_obj = the_player->get_contained_by();

   if (the_obj->get_type() == OBJ_TYPE_LOCATION)
   {
      the_player->send_plr(_("You are not in anything.\n"));
      return;
   }

   tmp_loc = the_obj->get_contained_by();
   if (tmp_loc->get_type() != OBJ_TYPE_LOCATION)
   {
      mainstruct->log_error(_("Error, player in cascaded contained objects."),
                                                   "exitcom");
      return;
   }

   the_loc = (Location *) tmp_loc;

   if ((results = check_specials(_("pre_leave"), the_obj, NULL, NULL,
                                             the_obj, the_player)) == 3)
      return;

   tmp_locflags = the_loc->get_locflags();

   if (tmp_locflags->get_flag(LOCFLAG_AQUATIC))
   {
      the_player->send_plr(_("There is no land nearby!\n"));
      return;
   }

   if (the_obj->get_type() == OBJ_TYPE_BOAT)
   {
      Boat *the_boat;

      the_boat = (Boat *) the_obj;

      the_boat->exit_boat(the_player);
   }
   else
   {
      the_player->send_plr(_("Error, in invalid item!  Tell a power!\n"));
      return;
   }

   the_player->send_plr(_("You leave the %s\n"), the_obj->get_title());

   holder.sprintf(_("%s leaves the %s\n"), the_player->get_name(),
                                                      the_obj->get_title());

   if (the_loc->get_type() == OBJ_TYPE_LOCATION)
      ((Location *) the_loc)->send_location(holder.str_show(), the_player);

   if ((results = check_specials(_("post_leave"), the_obj, NULL, NULL,
                                             the_obj, the_player)) == 3)
      return;

   return;

   the_parsed = NULL;
}


/***********************************************************************
 **  fulleditcom - toggles the full features editor mode on and off
 **
 ***********************************************************************/

void fulleditcom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Flags    *game_flags;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   game_flags = the_player->get_gameflags();

   if (game_flags->get_flag(GAMEFLAG_FULLEDIT))
   {
      game_flags->clr_flag(GAMEFLAG_FULLEDIT);
      the_player->send_plr(_("Editor has been set to simple edit.\n"));
   }
   else
   {
      game_flags->set_flag(GAMEFLAG_FULLEDIT);
      the_player->send_plr(_("Editor has been set to full features edit.\n"));
   }
   return;
   the_parsed = NULL;
}


/***********************************************************************
 **  fullexitscom - toggles the full exits display on and off
 **
 ***********************************************************************/

void fullexitscom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Flags    *game_flags;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   game_flags = the_player->get_gameflags();

   if (game_flags->get_flag(GAMEFLAG_FULLEXITS))
   {
      game_flags->clr_flag(GAMEFLAG_FULLEXITS);
      the_player->send_plr(_("Exits will now display in short format.\n"));
   }
   else
   {
      game_flags->set_flag(GAMEFLAG_FULLEXITS);
      the_player->send_plr(_("Exits will now display in verbose format.\n"));
   }
   return;
   the_parsed = NULL;
}


/***********************************************************************
 ** mailcom - either enters the mailer or starts a new message to send
 **           to folks
 **
 **
 ***********************************************************************/

void mailcom(Parse *the_parsed, MudObject *the_user) {
   Strings     holder;
   Player      *the_player = NULL;
   Strings     formatted;
   Inp_Handler *tmp_handler;
   Letter      *new_ltr;
   Mailer      *the_mailer;
   Strings     *prompt;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_config.conf_flags->get_flag(CF_USEMAILBOXES))
   {
      /* if a mailbox must be in the room to use, look for one */
      {
         Inventory  *loc_list;
         Location   *the_loc;
         MudObject  *next_obj;
         Item       *an_item;
         Flags      *itemflags = NULL;
         int        found = 0;

         the_loc = the_player->get_loc();
         loc_list = the_loc->get_inventory();
         loc_list->reset_current();

         next_obj = loc_list->get_next();
         while (next_obj != NULL)
         {
            if ((next_obj->is_an_item()) && !(next_obj->is_an_individual()))
            {
               an_item = (Item *) next_obj;
               itemflags = an_item->get_itemflags();
               if (itemflags->get_flag(ITEMFLAG_MAILBOX))
                  found = 1;
            }
            next_obj = loc_list->get_next();
         }

         if (!found)
         {
            the_player->send_plr(_("There are no mailboxes here.\n"
                                 "You must find a mailbox to check mail.\n"));
            return;
         }

      }
   }

   tmp_handler = the_player->get_input_handler();

   the_mailer = the_player->get_mailer();

   /* if there is nothing after 'mail', they want to enter the mailer */
   if ((the_parsed->get_speech() == NULL) ||
       (strlen(the_parsed->get_speech()) <= 0))
   {
      the_player->send_plr(_("&+B[&+WEntering mailer&+b]&*\n"));
      the_mailer->load_mail(Full);
      the_mailer->list_mail(the_player);
      prompt = new Strings(_("&+gMailer&+W>&* "));
      /* enter the mailer here */
      tmp_handler->push_input_handler(mailer_command_handler,
                                  prompt, 1, HANDLER_DATA_MAILER, 0);
      tmp_handler->set_data(the_player->get_mailer());


      return;
   }

   /* if there is something after mail, we want to mail a message */
   new_ltr = new_Letter();
   new_ltr->set_to(the_parsed->get_speech());
   new_ltr->set_from(the_player->get_title());
   the_player->send_plr(_("\n&+gTo: &*%s\n"), the_parsed->get_speech());
   the_player->send_plr(_("&+gFrom: &*%s\n"), the_player->get_title());
   the_mailer->set_current_ltr(new_ltr);

   prompt = new Strings(_("&+gSubject:&* "));

   /* get the subject from the user */
   tmp_handler->push_input_handler(mailer_get_subject,
                                  prompt, 1, HANDLER_DATA_MAILER, 0);
   tmp_handler->set_data(the_mailer);


   return;
}


/***********************************************************************
 ** pronouncom - shows the current pronouns
 ***********************************************************************/

void pronounscom(Parse *the_parsed, MudObject *the_user)
{
   Player *the_player;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   the_player->show_pronouns(the_player);
   the_parsed = NULL;
}


/***********************************************************************
 ** pagercom - either displays or sets the pager value
 ***********************************************************************/

void pagercom(Parse *the_parsed, MudObject *the_user)
{
   Player *the_player;
   int    new_val;
   Builder *the_bldr;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Pager line number value set to: %d\n"),
                                            the_player->get_pager_lines());
      return;
   }

   if (!isdigit(*(the_parsed->get_target1())))
   {
      the_player->send_plr(_("Format: pager <number>\n"));
      return;
   }

   new_val = atoi(the_parsed->get_target1());

   if (new_val < 0)
   {
      the_player->send_plr(_("Pager value must be greater than 0.\n"));
      return;
   }

   the_player->set_pager_lines(new_val);
   if ((the_bldr = mainstruct->get_builder(the_player->get_name())) != NULL)
   {
      the_bldr->set_pager_lines(new_val);
   }

   the_player->send_plr(_("Pager lines value set to: %d\n"),
                                               the_player->get_pager_lines());
   the_parsed = NULL;
}


/***********************************************************************
 ** digcom - a specials action, looks for special code
 ***********************************************************************/

void digcom(Parse *the_parsed, MudObject *the_user)
{
   Player *the_player;
   Location *this_loc;
   MudObject *with_obj = NULL;
   Strings   prep;
   Strings   obj_name;
   int       the_num = 0;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You dig for pirate treasure in your dreams.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to dig.\n"));
      return;
   }

   prep.assign_word(the_parsed->get_speech(), 1);

   if (prep.str_show() != NULL)
   {
      if (STRCASECMP(prep.str_show(), _("with")))
      {
         the_player->send_plr(_("format: dig [with <object>]\n"));
         return;
      }
      obj_name.assign_word(the_parsed->get_speech(), 2);

      if (obj_name.str_show() == NULL)
      {
         the_player->send_plr(_("format: dig [with <object>]\n"));
         return;
      }

      if ((with_obj = the_player->find_contained(obj_name.str_show(), &the_num))
                                                                    == NULL)
      {
         the_player->send_plr(_("You do not seem to have the %s.\n"),
                                                       obj_name.str_show());
         return;
      }

      if (with_obj->get_type() != OBJ_TYPE_WEAPON)
      {
         the_player->send_plr(_("You can't dig with that!\n"));
         return;
      }
   }

   this_loc = the_player->get_loc();

   if (check_specials(_("ondig"), this_loc, with_obj, NULL,
                                                  the_player, the_player) == 3)
      return;

   if ((this_loc->check_contained_spec(_("ondig"), with_obj,
                                                 NULL, the_player, 0)) == 3)
      return;

   the_player->send_plr(_("You find nothing of interest.\n"));
   return;
}


/***********************************************************************
 **  listencom - listens in a room
 **
 ***********************************************************************/

void listencom(Parse *the_parsed, MudObject *the_user) {
   Location *this_loc;
   Player *the_player;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;
   if (the_player->is_asleep())
   {
      the_player->send_plr(
          _("You hear the sound of monsterous snoring coming from nearby.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to listen to anything.\n"));
      return;
   }

   this_loc = the_player->get_loc();


   if (check_specials(_("pre_listen"), this_loc, NULL, NULL,
                                             this_loc, the_player) == 3)
      return;

   if (this_loc->get_listen() == NULL)
   {
      the_player->send_plr(_("You hear nothing of real interest.\n"));
      return;
   }
   the_player->send_plr("%s\n", this_loc->get_listen());

   if (check_specials(_("post_listen"), this_loc, NULL, NULL,
                                             this_loc, the_player) == 3)
      return;

   the_parsed = NULL;
}


/***********************************************************************
 **  smellcom - smells in a room
 **
 ***********************************************************************/

void smellcom(Parse *the_parsed, MudObject *the_user) {
   Location *this_loc;
   Player *the_player;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;
   if (the_player->is_asleep())
   {
      the_player->send_plr(
          _("You smell festering ogre drool.  No, wait, that is your own.\n"
          "You are sleeping.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to smell anything.\n"));
      return;
   }

   this_loc = the_player->get_loc();

   if (check_specials(_("pre_smell"), this_loc, NULL, NULL,
                                             this_loc, the_player) == 3)
      return;

   if (this_loc->get_smell() == NULL)
   {
      the_player->send_plr(_("You smell nothing of real interest.\n"));
      return;
   }
   the_player->send_plr("%s\n", this_loc->get_smell());

   if (check_specials(_("post_smell"), this_loc, NULL, NULL,
                                             this_loc, the_player) == 3)
      return;

   the_parsed = NULL;
}


/***********************************************************************
 ** attributescom - provides full statistics on the player
 ***********************************************************************/

void attributescom(Parse *the_parsed, MudObject *the_user)
{
   Player *the_player;
   char tmpline[120];
   Strings holder1;
   Strings holder2;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;


   holder1 = the_player->get_name();
   if (holder1.str_len() > 15)
      holder1.truncate(25);

   holder2 = the_player->get_race();
   if (holder2.str_len() > 15)
      holder2.truncate(20);

   sprintf(tmpline,
        _("\n\n&+GName: &+M%-15s&+G Race: &+c%-15s &+GSex: &+c%s&*"),
        holder1.str_show(), holder2.str_show(), the_player->get_sex_str());
   the_player->send_plr("%s\n", tmpline);

   get_date_str(&holder1, the_player->get_birthday());
   sprintf(tmpline, _("&+GBirthday: &+M%-15s&*"), holder1.str_show());
   the_player->send_plr("%s\n\n", tmpline);

   the_player->send_plr("\t\n");

   the_parsed = NULL;
}

/***********************************************************************
 ** bulletincom - enters the bulletin board handler
 **
 **
 ***********************************************************************/

void bulletincom(Parse *the_parsed, MudObject *the_user) {
   Strings     holder;
   Player      *the_player = NULL;
   Entity      *tmp_obj;
   Bulletin    *the_board;
   Inp_Handler *tmp_handler;
   Area_Dbase  *bull_area;
   Object_List *tmp_dbase;
   int         counter = 0;
   Flags       *tmp_bullflags;
   Flags       *tmp_adminflags;
   Strings     *prompt;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();
   tmp_adminflags = the_player->get_admflags();

   tmp_dbase = mainstruct->get_dbase();
   if ((bull_area = tmp_dbase->get_area("bulletins")) == NULL)
   {
      the_player->send_plr(_("Error, bulletin boards don't exist.\n"));
      mainstruct->log_error(_("Bulletin board area does not exist."),
                                                     "bulletincom");
      return;
   }

   /* if they don't specify a board, list the current boards they have
      access to */
   if (the_parsed->get_target1() == NULL)
   {
      bull_area->reset_list();
      tmp_obj = bull_area->get_next_obj();
      while (tmp_obj != NULL)
      {
         if (tmp_obj->get_type() != OBJ_TYPE_BULLETIN)
	 {
            the_player->send_plr(_("Error found in bulletin board area.\n"));
            holder.sprintf(_("Object '%s' not bulletin object in bulletin area."),
                           tmp_obj->get_name());
            mainstruct->log_error(holder.str_show(), "bulletincom");
            return;
	 }
         the_board = (Bulletin *)tmp_obj;
         tmp_bullflags = the_board->get_bullflags();
         if ((tmp_bullflags->get_flag(BULLFLAG_MORTALS)) ||
             (tmp_adminflags->get_flag(ADMINFLAG_ADMINISTRATOR)) ||
             (!((tmp_bullflags->get_flag(BULLFLAG_LIMITED)) &&
                (!the_board->player_authorized(the_player)))))
	 {
	    /* only show global bulletins on this list */
            if (tmp_bullflags->get_flag(BULLFLAG_GLOBAL))
	    {
               if (counter == 0)
	       {
                  the_player->
                        send_plr(_("&+WAvailable Global Bulletin Boards&*\n"));
                  the_player->send_plr(
                        "&+B--------------------------------&*\n");
               }
               the_player->send_plr("&+G%-15s &+w%-30s&*\n",
                     the_board->get_name(), the_board->get_title());
               counter++;
	    }
	 }
         tmp_obj = bull_area->get_next_obj();
      }
      if (counter == 0)
      {
         the_player->
             send_plr(_("You have no bulletin boards available to you.\n"));
      }
      return;
   }

   if ((tmp_obj = bull_area->get_areaobject(the_parsed->get_target1()))
                                                                   == NULL)
   {
      the_player->send_plr(_("The '%s' board doesn't seem to exist globally.\n"),
                                               the_parsed->get_target1());
      return;
   }

   if (tmp_obj->get_type() != OBJ_TYPE_BULLETIN)
   {
      the_player->send_plr(_("Error found in bulletin board area.\n"));
      holder.sprintf(_("Object '%s' not bulletin object in bulletin area."),
                           tmp_obj->get_name());
      mainstruct->log_error(holder.str_show(), "bulletincom");
      return;
   }

   the_board = (Bulletin *) tmp_obj;
   tmp_bullflags = the_board->get_bullflags();

   if (!tmp_bullflags->get_flag(BULLFLAG_GLOBAL))
   {
      the_player->send_plr(_("The '%s' board doesn't seem to exist globally.\n"),
                                               the_parsed->get_target1());
      return;
   }

   /* now enter them into the bulletin board handler */
   the_player->send_plr(_("&+B[&+WEntering board '&+M%s&+W'&+b]&*\n"),
                                          the_board->get_name());
   the_board->list_entries(the_player);
   prompt = new Strings(_("&+gBoard&+W>&* "));
   /* enter the bulletin board here */
   tmp_handler->push_input_handler(bulletin_command_handler,
                                  prompt, 1, HANDLER_DATA_BULLETIN, 0);
   tmp_handler->set_data(the_board);
   return;
}

/***********************************************************************
 **  autodiagnosecom - toggles the auto-diagnosis in battle on and off
 **
 ***********************************************************************/

void autodiagnosecom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Flags    *game_flags;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   game_flags = the_player->get_gameflags();

   if (game_flags->get_flag(GAMEFLAG_AUTODIAGNOSE))
   {
      game_flags->clr_flag(GAMEFLAG_AUTODIAGNOSE);
      the_player->send_plr(_("Health of targets will no longer be shown automatically.\n"));
   }
   else
   {
      game_flags->set_flag(GAMEFLAG_AUTODIAGNOSE);
      the_player->send_plr(_("Health of targets will be shown in battle.\n"));
   }
   return;
   the_parsed = NULL;
}



/***********************************************************************
 **  maskcom - allows admins to set permission on players based on a
 **            certain mask
 **
 ***********************************************************************/

void maskcom(Parse *the_parsed, MudObject *the_user)
{
   Player      *the_player;
   Player      *target_player = NULL;
   Builder     *target_builder = NULL;
   Strings     holder;
   Strings     the_name;
   User_Dbase  *tmp_dbase;
   int         not_on = 0;
   Strings     tmp_word;
   Mask        *the_mask;
   Object_List *the_dbase;
   Flags       *usr_flags[2];
   Flags       *mask_flags[2];
   int         flagnums[2];
   int         done = 0;
   int         once = 0;
   int         i, j;
   int         flagnum;
   int         the_type;


   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   the_dbase = mainstruct->get_dbase();

   if ((the_parsed->get_target1() == NULL) ||
       (strlen(the_parsed->get_target1()) <= 0))
   {
      Area_Dbase *mask_area;
      Entity     *tmp_obj;

      the_player->send_plr(_("\n&+WMasks Available&*\n"));
      the_player->send_plr("&+B---------------------------&*\n");

      if ((mask_area = the_dbase->get_area("masks")) == NULL)
      {
         the_player->send_plr(_("Error, could not get mask area.\n"));
         return;
      }

      mask_area->reset_list();
      tmp_obj = mask_area->get_next_obj();
      while (tmp_obj != NULL)
      {
         if (tmp_obj->get_type() == OBJ_TYPE_MASK)
	 {
            the_mask = (Mask *) tmp_obj;
            the_player->send_plr("&+g%s&*\n", the_mask->get_name());
	 }
         tmp_obj = mask_area->get_next_obj();
      }
      the_player->send_plr("\n");
      return;
   }

   /* if they want to mask themselves, let them */
   if ((the_parsed->get_target1() == NULL) ||
       (!STRCASECMP(the_parsed->get_target1(), _("me"))))
      the_name = the_player->get_name();
   else
      the_name = the_parsed->get_target1();

   /* if the builder is logged on, we change him too */
   target_builder = mainstruct->get_builder(the_name.str_show());

   tmp_dbase = mainstruct->get_user_database();

   /* if the player is not logged in, read him from file */
   if ((target_player = mainstruct->get_player(the_name.str_show()))
                                                                  == NULL)
   {
      the_name.upper(0);

      target_player = new_Player(the_name.str_show());
      if (tmp_dbase->read_user(the_name.str_show(), target_player) < 1)
      {
         the_player->send_plr(_("That user does not exist.\n"));
         return;
      }
      not_on = 1;
   }

   tmp_word.assign_word(the_parsed->get_speech(), 1);

   if (tmp_word.str_show() == NULL)
   {
      the_player->send_plr(_("You need to supply a mask name.\n"
                           "Format: mask <player> <maskname> <action>\n"));
      if (not_on)
         delete_Player(target_player);
      return;
   }

   if ((the_mask = the_dbase->get_mask_obj(tmp_word.str_show())) == NULL)
   {
      the_player->send_plr(_("I can't find the '%s' mask.\n"),
                                                       tmp_word.str_show());
      if (not_on)
         delete_Player(target_player);
      return;
   }


   // Here is where we will do the permissions setting
   tmp_word.assign_word(the_parsed->get_speech(), 2);
   if (tmp_word.str_show() == NULL)
   {
      the_player->send_plr(_("You must supply an action.\n"
                             "Format: mask <player> <maskname> <action>\n"));
      if (not_on)
         delete_Player(target_player);
      return;
   }

   if (!STRCASECMP(tmp_word.str_show(), _("set")))
   {
      the_type = 0;
   }
   else if (!STRCASECMP(tmp_word.str_show(), _("clear")))
   {
      the_type = 1;
   }
   else if (!STRCASECMP(tmp_word.str_show(), _("setonly")))
   {
      the_type = 2;
   }
   else
   {
      the_player->send_plr(_("That is not a valid action.\n"
                           "Valid actions are: set, clear, setonly\n"));
      if (not_on)
         delete_Player(target_player);
      return;
   }


   mask_flags[0] = the_mask->get_comflags();
   mask_flags[1] = the_mask->get_admflags();

   usr_flags[0] = target_player->get_comflags();
   usr_flags[1] = target_player->get_admflags();

   i=0;
   while (comflagnames[i] != NULL)
      i++;
   flagnums[0] = i;

   i=0;
   while (admflagnames[i] != NULL)
      i++;
   flagnums[1] = i;


   while (!done)
   {
      for (i=0; i<2; i++)
      {
         flagnum = 0;
         for (j=0; j<flagnums[i]; j++)
	 {
            switch(the_type)
	    {
	       case 0:
                  if (mask_flags[i]->get_flag(j))
                     usr_flags[i]->set_flag(j);
                  break;

	       case 1:
                  if (mask_flags[i]->get_flag(j))
                     usr_flags[i]->clr_flag(j);
                  break;

	       case 2:
                  if (mask_flags[i]->get_flag(j))
                     usr_flags[i]->set_flag(j);
                  else
                     usr_flags[i]->clr_flag(j);
                  break;

	       default:
                  fault("unexpected flag operator.\n");
                  break;
	    }

	 }
      }
      if ((target_builder != NULL) && (!once))
      {
         usr_flags[0] = target_builder->get_comflags();
         usr_flags[1] = target_builder->get_adminflags();
      }
      else
         done = 1;
      once = 1;
   }

   the_player->send_plr(_("All permissions on %s have been masked.\n"),
                                             target_player->get_title());

   if (not_on)
   {
      if (tmp_dbase->write_user(target_player) <= 0)
      {
         mainstruct->log_error(_("Error storing user record to database"),
                                                            "flagscom");
         return;
      }
      delete_Player(target_player);
   }

   return;


}

/***********************************************************************
 **  slaycom - allows an Immort to annhilate an individual w/o save.
 **
 ***********************************************************************/

void slaycom(Parse *the_parsed, MudObject *the_user)
{
   Player     *the_player;
   Location   *the_loc;
   MudObject  *the_obj;
   Strings    holder, holder2;
   Strings    target_str;
   Individual *the_target;
   char       *sex;
   int        the_num = 0;
// int        results;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Slay whom?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You dream yourself power almighty and smite your enemies! (...but you're asleep!)\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to slay anything.\n"));
      return;
   }

   the_player->check_pronouns(the_parsed->get_target1(), &target_str);

   the_loc = the_player->get_loc();
   if ((the_obj = the_loc->find_contained(target_str.str_show(),
                                                     &the_num)) == NULL)
   {
      the_player->send_plr(_("They are not here.\n"));
      return;
   }

   if ((the_obj->get_type() != OBJ_TYPE_PLAYER) &&
       (the_obj->get_type() != OBJ_TYPE_MOBILE))
   {
      the_player->send_plr(_("That is not alive, you can't kill it.\n"));
      return;
   }

   the_target = (Individual *) the_obj;

   sex = the_player->get_sex_possessive_str();

   if (the_target == (Individual *) the_player)
   {
      the_player->send_plr(_("Suicide does not become you.\n"));
      return;
   }

//  The meat of the Slaying Goes here:

   the_player->send_plr(_("You call upon your almighty powers and smite %s!\n"),
                                                        the_target->get_title());

   holder.sprintf(_("%s is suddenly surrounded by a radiant halo and points at %s!\n"),                           the_player->get_title(), the_target->get_title());
   holder2.sprintf(_("%s's body becomes suffused with a crimson glow that "
                   "radiates from within,\n"
                   "it flares up...then fades away, leaving a fine mist that "
                   "dispurses quickly.\n"), the_target->get_title());
   if (the_target->get_type() == OBJ_TYPE_PLAYER)
   {
      ((Player *) the_target)->send_plr(_("%s is suddenly surrounded by a "
                                    "blindingly radiant halo and points at you!\n"),
                                                  the_player->get_title());
      ((Player *) the_target)->send_plr(_("You are suddenly filled with an abysmal "
                  "chill as your body is annhilated"
                  " and the cold hand of death closes around your soul.\n"));
      the_loc->send_location(holder.str_show(), the_player,
                                                   (Player *) the_target);
      the_loc->send_location(holder2.str_show(), the_player,
                                                   (Player *) the_target);
   }
   else
   {
      the_loc->send_location(holder.str_show(), the_player);
      the_loc->send_location(holder2.str_show(), the_player);
   }

   holder.sprintf(_("&+m%s&* has slain '&+m%s&*'"),
                the_player->get_title(), the_target->get_title());

   /* now log their death if appropriate */
   if (the_config.conf_flags->set_flag(CF_LOGSLAY))
   {
      mainstruct->log_event(holder.str_show());
   }

   // And now ... young Skywalker ... you will die.
   the_target->death(DEATHFLAG_SLAY, (Individual *) the_player);
}




/***********************************************************************
 **  deleteuser_confirm - confirmation prompt for delete user, this
 **                       does the real work
 ***********************************************************************/

int deleteuser_confirm(MudObject *the_user, char *the_input)
{
   Player         *oPlr;
   Player         *oPlrOnline   = NULL;
   Builder        *oBuiOnline   = NULL;
   User_Dbase     *oTmpDBase;
   Inp_Handler    *oTmpHandler;
   Flags          *oTmpFlags;
   Strings         strUser;
   Strings         strPasswd;
   Strings         strEncrypted;
   char            bSelf        = 1;
   Strings         holder, holder2;

   // grab the player object
   oPlr = ( Player * )the_user;

   // pop the old input handler
   oTmpHandler = oPlr->get_input_handler( );
   oTmpHandler->pop_input_handler( );

   oPlr->send_plr( "\n" );

   // check the user's password
   strPasswd = the_input;
   strPasswd.remove_newline( );
   encrypt_passwd( &strPasswd, &strEncrypted );

   if( !strEncrypted.str_cmp( oPlr->get_passwd( ) ) )
   {
      oPlr->send_plr( _("Invalid password.  User not deleted.\n") );
      return 1;
   }


   // get the user to delete, player DB, and see if the person is deleting
   //   one's self
   strUser = oPlr->get_misc_string( );
   oTmpDBase = mainstruct->get_user_database();
   bSelf = oPlr->get_misc_string( )->str_cmp( oPlr->get_name( ) );

   // kick off the builder if it is online
   if( ( oBuiOnline = mainstruct->get_builder( strUser.str_show( ) ) ) != NULL )
   {
      oBuiOnline->set_off( );
      if( bSelf )
      {
         oBuiOnline->send_bldr( _("You have been deleted.\n") );
      }
      else
      {
         oBuiOnline->send_bldr( _("You have been deleted by '%s'.\n"),
                                oPlr->get_name( ) );
      }
   }

   // handle the player if it is online
   if( ( oPlrOnline = mainstruct->get_player( strUser.str_show( ) ) ) != NULL )
   {
      oTmpFlags = oPlrOnline->get_gameflags();
      oTmpFlags->set_flag( GAMEFLAG_DELETED );

      oPlrOnline->set_off( );

      if( bSelf )
      {
         oPlrOnline->send_plr( _("You have been deleted.\n") );
      }
      else
      {
         oPlrOnline->send_plr( _("You have been deleted by '%s'.\n"),
                               oPlr->get_name( ) );
      }

      if( oTmpDBase->write_user( oPlrOnline ) <= 0 )
      {
         mainstruct->log_error( _("Error storing user record to database (1)"),
                                "deleteuser" );
      }
   }

   // player offline, load and set flag
   else
   {
      strUser.upper( 0 );

      oPlrOnline = new_Player(strUser.str_show());
      if ( oTmpDBase->read_user( strUser.str_show( ), oPlrOnline ) < 1 )
      {
         oPlr->send_plr( _("User '%s' does not exist.\n"), strUser.str_show( ) );
         delete_Player( oPlrOnline );
         return 1;
      }

      oTmpFlags = oPlrOnline->get_gameflags();
      oTmpFlags->set_flag( GAMEFLAG_DELETED );

      if( oTmpDBase->write_user( oPlrOnline ) <= 0 )
      {
         mainstruct->log_error( _("Error storing user record to database (2)"),
                                "deleteuser" );
      }

      delete_Player( oPlrOnline );
   }

   if ( bSelf )
      holder.sprintf(_("&+M%s&* has deleted themselves"), oPlr->get_title());
   else
      holder.sprintf(_("&+M%s&* has deleted '&+M%s&*'"),
                                 oPlr->get_title(), strUser.str_show());

   /* now log the delete if appropriate */
   if (the_config.conf_flags->set_flag(CF_LOGDELETE))
   {
      mainstruct->log_event(holder.str_show());
   }

   holder2.sprintf("&+Y[&*%s&+Y]&*\n", holder.str_show());
   mainstruct->send_all_players(holder2.str_show(), oPlr,
                                                        ADMINFLAG_SEEDELETES);

   // report back to the user if it wasn't deleting itself
   if( !bSelf )
   {
      oPlr->send_plr( _("You have deleted '%s'.\n"), strUser.str_show( ) );
   }

   return 1;
}


/***********************************************************************
 **  deleteuser - allows a user to delete one's self or a god to
 **               delete anyone
 ***********************************************************************/

void deleteuser(Parse *the_parsed, MudObject *the_user)
{
   Player         *oPlr;
   Inp_Handler    *oTmpHandler;
   Strings         strUser;


   if( the_user->get_type( ) != OBJ_TYPE_PLAYER )
      return;

   oPlr = ( Player * )the_user;
   if( the_parsed->get_target1( ) != '\0' )
   {
      if( oPlr->get_admflags( )->get_flag( ADMINFLAG_DELETEUSER ) )
      {
         strUser = the_parsed->get_target1( );
      }
      else
      {
         oPlr->send_plr( _("You do not have permission to delete others.\n") );
         return;
      }
   }
   else
   {
      strUser = oPlr->get_name( );
   }


   strUser.lowercase( );
   oPlr->get_misc_string( )->str_copy( strUser.str_show( ) );

   oTmpHandler = oPlr->get_input_handler( );
   oTmpHandler->push_input_handler( deleteuser_confirm,
                                    _("Your password: "),
                                    0 );
}


/***********************************************************************
 **  clonecom -  creates a copy of an object and places it in an appropriate
 **              location.  Moveable objects, places in the inventory of
 **              the cloner.  Locations, places in the area of the parent.
 **              Non-moveable objects, places in the location of the cloner.
 **
 ***********************************************************************/

void clonecom(Parse *the_parsed, MudObject *the_user) {
   Location *cur_loc;
   MudObject *the_mudobj;
   MudObject *clone_obj;
   Strings holder, holder2;
   Player *the_player;
   int    number = 1;
   int    num_clones = 1;
   int    i;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("You must specify a MudObject to clone.\n"));
      return;
   }

   cur_loc = the_player->get_loc();

   holder = the_parsed->get_target1();
   if (holder.num_char('@') != 1)
   {
      the_player->send_plr(_("'%s' is not a valid MudObject.\n"
                           "usage: clone <objname>@<areaname> [<number>]\n"), 
                                                       holder.str_show());
      return;
   }
 
   if (the_parsed->get_preposition() != NULL)
   {
      if (!isdigit(*(the_parsed->get_preposition())))
      {
         the_player->send_plr(_("'%s' is not a valid number.\n"
                           "usage: clone <objname>@<areaname> [<number>]\n"), 
                                       the_parsed->get_preposition());
         return;
      
      }

      number = atoi(the_parsed->get_preposition());

      if (number < 1)
      {
         the_player->send_plr(_("That is not a valid number.\n"));
         return;
      }
   }

   if ((the_mudobj = mainstruct->get_object(holder.str_show())) == NULL)
   {
      the_player->send_plr(_("That MudObject does not exist.\n"));
      return;
   }

   if (!the_mudobj->is_merger())
      num_clones = number;

   for (i=0; i<num_clones; i++)
   {

      if ((clone_obj = (mainstruct->get_dbase())->clone_object(the_mudobj)) 
                                                                      == NULL)
      {
         the_player->send_plr(_("Error, the clone failed.  Talk to a coder.\n"));
         holder.sprintf(_("Clone of object '%s@%s' failed."), 
                               the_mudobj->get_name(), the_mudobj->get_area());
         mainstruct->log_error(holder.str_show(), "clonecom");
         return;
      }

      if (clone_obj->is_a_moveable())
      {
	((Moveable *) clone_obj)->set_moved();
      }

      if (clone_obj->get_type() != OBJ_TYPE_LOCATION)
      {
	if (clone_obj->is_a_moveable())
	  clone_obj->set_location(the_player);
	else
	  clone_obj->set_location(the_player->get_loc());
      }

   }

   if (clone_obj->is_merger())
      ((Merger *) clone_obj)->set_number_of(number);

   if (clone_obj->get_type() == OBJ_TYPE_LOCATION)
   {
      the_player->send_plr(_("Location cloned as '%s@%s'\n"),
                              clone_obj->get_name(), clone_obj->get_area());
      return;
   }
   else if (!clone_obj->is_a_moveable())
   {
      the_player->send_plr(
                 _("Object cloned as '%s@%s' and placed in current location.\n"),
                              clone_obj->get_name(), clone_obj->get_area());
      return;
   }
   else
   {
      the_player->send_plr(
                 _("Object cloned as '%s@%s' and placed in your inventory.\n"),
                              clone_obj->get_name(), clone_obj->get_area());
      return;
   }

   holder.sprintf(_("'&+M%s&*' cloned %d '&+c%s@%s&*'"), 
       the_player->get_title(), number, the_mudobj->get_name(), 
                                                 the_mudobj->get_area());
 
   /* now log the clone if appropriate */
   if (the_config.conf_flags->set_flag(CF_LOGCLONE))
   {
      mainstruct->log_event(holder.str_show());
   }

   holder2.sprintf("&+Y[&*%s&+Y]&*\n", holder.str_show());
   mainstruct->send_all_players(holder2.str_show(), the_player, 
                                                      ADMINFLAG_SEECLONES);

   return;
}


/***********************************************************************
 **  forcecom - forces a player to execute the command specified
 **
 ***********************************************************************/

void forcecom(Parse *the_parsed, MudObject *the_user) {
   Player *the_forced;
   Strings holder, holder2;
   Strings command;
   Player *the_player;
   Input  *forced_input;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("You must specify a Player to force.\n"));
      return;
   }

   if ((the_forced = mainstruct->get_player(the_parsed->get_target1())) == NULL)
   {
      the_player->send_plr(_("That player is not currently on.\n"));
      return;
   }

   command = the_parsed->get_speech();

   holder.sprintf(_("'&+M%s&*' forced '&+M%s&*'&+W:&* &+c%s&*"),
      the_player->get_title(), the_forced->get_title(), command.str_show());

   /* now log their death if appropriate */
   if (the_config.conf_flags->set_flag(CF_LOGFORCE))
   {
      mainstruct->log_event(holder.str_show());
   }

   holder2.sprintf("&+Y[&*%s&+Y]&*\n", holder.str_show());
   mainstruct->send_all_players(holder2.str_show(), the_player,
                                                      ADMINFLAG_SEEFORCE);

   the_forced->send_plr(_("%s forces you to: &+c%s&*\n"), the_player->get_title(),
                                           command.str_show());
   forced_input = new_Input();
   forced_input->handle_input(&command, the_forced);

   if (forced_input != NULL)
      delete_Input(forced_input);

   return;
}


/***********************************************************************
 **  bootcom - violently kicks a player off the mud
 **
 ***********************************************************************/

void bootcom(Parse *the_parsed, MudObject *the_user) {
   Player   *the_player;
   Player   *target_player;
   Strings  holder, holder2;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
      the_player = (Player *) the_user;
   else
      return;

   /* they need to specify a player or mobile to give the boot to */
   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Boot who?\n"));
      return;
   }

   if ((target_player = mainstruct->get_player(the_parsed->get_target1()))
                                                                     == NULL)
   {
      the_player->send_plr(_("That player is not connected.\n"));
      return;
   }

   target_player->set_off();
   target_player->send_plr(
     _("A massive boot emerges from the sky and lands square on your backside.\n"
     "You go flying through the air and tear through the fabric of space,\n"
     "emerging into reality.\n"));


   the_player->send_plr(
     _("You send your massive boot into %s's backside, sending them soaring\n"
     "clear out of the mud!\n"), target_player->get_title());

   holder.sprintf(_("'&+M%s&*' booted '&+M%s&*'"),
                       the_player->get_title(), target_player->get_title());

   /* now log their death if appropriate */
   if (the_config.conf_flags->set_flag(CF_LOGBOOT))
   {
      mainstruct->log_event(holder.str_show());
   }

   holder2.sprintf("&+Y[&*%s&+Y]&*\n", holder.str_show());
   mainstruct->send_all_players(holder2.str_show(), the_player,
                                                      ADMINFLAG_SEEBOOTS);

   return;

}


/***********************************************************************
 **  listcom - does various lists of a specified area
 **
 ***********************************************************************/

void listcom(Parse *the_parsed, MudObject *the_user)
{
   Player      *the_player;
   Entity      *the_object;
   Area_Dbase  *tmp_area;
   int         count = 0;
   Connection  *the_connection;
   Object_List *the_dbase;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;

   the_player = (Player *) the_user;

   the_dbase = mainstruct->get_dbase();

   the_connection = the_player->get_connection();
   the_connection->start_paging();

   if ((the_parsed->get_target1() == NULL) ||
       (!STRNCASECMP(the_parsed->get_target1(), _("areas"),
                               strlen(the_parsed->get_target1()))))
   {
      the_player->send_plr(_("\n&+WAreas loaded&*\n"));
      the_player->send_plr("&+G________________________________"
                             "___________&*\n");

      the_dbase->reset_list();
      tmp_area = the_dbase->get_next_area();
      while (tmp_area != NULL)
      {
	the_player->send_plr("&+g%s&*\n", tmp_area->get_areaname());
        count++;
        tmp_area = the_dbase->get_next_area();
      }

      the_player->send_plr("&+G________________________________"
                          "___________&*\n");
      the_player->send_plr(_("&+GTotal Areas Listed: &+W%d\n\n"), count);

      the_connection->end_paging(the_player);
      return;
   }


   if ((tmp_area = the_dbase->get_area(the_parsed->get_target1())) == NULL)
   {
      the_player->send_plr(_("I can't seem to find the loaded area '%s'\n"),
                                     the_parsed->get_target1());
      the_connection->end_paging(the_player);
      return;
   }


   the_player->send_plr(_("\n&+WObjects in area: &+m%s.&*\n"),
                                                 tmp_area->get_areaname());

   the_player->send_plr(_("&+GName                            "
                          "&+GObject Type   Clone&*\n"));
   the_player->send_plr("&+G________________________________"
                          "____________________&*\n");

   tmp_area->reset_list();
   the_object = tmp_area->get_next_obj();
   while (the_object != NULL)
   {
      if (the_object->is_a_mudobject())
         the_player->send_plr("%-31s %-14s     %s\n", the_object->get_name(),
                              objtype_name[the_object->get_type()],
            ((((MudObject*) the_object)->get_parent() == NULL) ? "" : "&+G*&*"));
      else
         the_player->send_plr("%-31s %s\n", the_object->get_name(),
                              objtype_name[the_object->get_type()]);
      count++;
      the_object = tmp_area->get_next_obj();
   }
   the_player->send_plr("&+G________________________________"
                          "_____________________&*\n");
   the_player->send_plr(_("&+GTotal Objects Listed: &+W%d\n\n"), count);

   the_connection->end_paging(the_player);
   return;
}


/***********************************************************************
 **  classifycom - provides the weapon classification to the user
 **
 ***********************************************************************/

void classifycom(Parse *the_parsed, MudObject *the_user) {
   Location   *the_loc;
   MudObject  *the_obj;
   Player     *the_player;
   Weapon     *the_weapon;
   int        the_num = 0;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return;
   the_player = (Player *) the_user;

   if (the_parsed->get_target1() == NULL)
   {
      the_player->send_plr(_("Get the classification of what weapon?\n"));
      return;
   }

   if (the_player->is_asleep())
   {
      the_player->send_plr(_("You need to open your eyes first.\n"));
      return;
   }

   if (the_player->is_busy())
   {
      the_player->send_plr(_("You are too busy to classify anything.\n"));
      return;
   }

   the_loc = the_player->get_loc();
   if ((the_obj = the_player->find_contained(the_parsed->get_target1(),
                                                     &the_num)) == NULL)
   {
      if ((the_obj = the_loc->find_contained(the_parsed->get_target1(), 
                                                     &the_num)) == NULL)
      {
         the_player->send_plr(_("That weapon is not here.\n"));
         return;
      }
   }

   if (the_obj->get_type() != OBJ_TYPE_WEAPON)
   {
      the_player->send_plr(
           _("That is not a weapon and bears no classification.\n"));
      return;
   }

   the_weapon = (Weapon *) the_obj;

   if ((the_weapon->get_weapon_class() == NULL) ||
       (!STRCASECMP(the_weapon->get_weapon_class(), _("none"))))
      the_player->send_plr(_("The %s has no classification.\n"), 
                                                      the_weapon->get_title());
   else
      the_player->send_plr(_("The %s falls under the '%s' classification.\n"), 
                     the_weapon->get_title(), the_weapon->get_weapon_class());

   the_parsed = NULL;
}


#endif


