//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

//Systemwide includes
#include <qfile.h>
#include <qtextstream.h>
#include <qstringlist.h>

#include <qlabel.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qsizegrip.h>
#include <qkeysequence.h>

//Projectwide includes
#include "helpWindow.h"
#include "contents.h"
#include "whatsNew.h"
#include "importing.h"
#include "annotating.h"
#include "framing.h"
#include "enhancing.h"
#include "proTools.h"
#include "manipulating.h"
#include "loadSave.h"
#include "shortcuts.h"

#include "../ALabel.h"
#include "../../config.h"

//==============================================
HelpWindow::HelpWindow( QWidget* parent, const char* name ) : QDialog(parent,name)
{
  //determine necessary encoding for reading and writing to html files
  QTextStream::Encoding fileEncoding;
  QString savingCharSet;
  QString loadingCharSet;

  //Mac OSX -> Use UTF16
  #if defined(Q_OS_MACX)
  fileEncoding = QTextStream::Unicode;
  savingCharSet = "utf16";
  loadingCharSet = "UTF-16";
  
  //Other UNIX or Windows with Unicode support -> Use UTF8
  #elif !defined(Q_WS_WIN) || (defined(Q_WS_WIN) && defined(UNICODE))
  fileEncoding = QTextStream::UnicodeUTF8;
  savingCharSet = "utf8";
  loadingCharSet = "UTF-8";

  //Windows without Unicode support (Win95/98/ME) -> Use Latin-1
  #else
  fileEncoding = QTextStream::Latin1;
  savingCharSet = "latin-1";
  loadingCharSet = "latin-1";
  #endif
  //-------------------------------------------------------------
  //generate html pages
  WhatsNew::generateHTML    (fileEncoding, savingCharSet);
  Importing::generateHTML   (fileEncoding, savingCharSet);
  Annotating::generateHTML  (fileEncoding, savingCharSet);
  Framing::generateHTML     (fileEncoding, savingCharSet);
  Enhancing::generateHTML   (fileEncoding, savingCharSet);
  ProTools::generateHTML    (fileEncoding, savingCharSet);
  Manipulating::generateHTML(fileEncoding, savingCharSet);
  LoadSave::generateHTML    (fileEncoding, savingCharSet);
  Shortcuts::generateHTML   (fileEncoding, savingCharSet);
  
  resize( 800, 400 );
  setPaletteBackgroundColor( QColor(255,255,255) );

  //set window title
  setCaption( tr("Album Shaper Help"));
  //--
  //create billboard widget
  billboard = new ALabel( this, "helpBillboard", NULL,
                                  APPEAR_IMMEDIATELY, SLIDE_OUT_LEFT );
  billboard->setPixmap( QPixmap( QString(IMAGE_PATH)+"helpImages/helpBillboard.png") );
  currentPage = BILLBOARD;
  connect( billboard, SIGNAL(pixmapRemoved()),
           this, SLOT(showFirstSelection()) );

  //construct special mime source factory for loading html files for the contents and content frames
  loadingMimeSource = new QMimeSourceFactory();
  loadingMimeSource->setExtensionType("html",QString("text/html;charset=%1").arg(loadingCharSet) );

  //create contents widget
  Contents* contents = new Contents(fileEncoding, savingCharSet, loadingMimeSource, this);
  connect( contents, SIGNAL(setPage(HELP_PAGE)),
           this, SLOT(setPage(HELP_PAGE)) );

  //create widget for holding content
  content = new QTextBrowser( this );
  content->setHScrollBarMode( QScrollView::Auto );
  content->setVScrollBarMode( QScrollView::Auto );
  content->setFrameStyle( QFrame::NoFrame );
  content->setMimeSourceFactory( loadingMimeSource );
  
  //PLATFORM_SPECIFIC_CODE
  //mac os x puts in a size grip that can interfere with the updates icon, in order
  //to avoid this we manually place the size grip ourselves
  //windows users expect a grip too, but qt doesn't put one in by default. we'll add
  //it for them too. :-)
#if defined(Q_OS_MACX) || defined(Q_OS_WIN)
  content->setCornerWidget( new QSizeGrip(this) );
#endif
    
  content->hide();
  //--
  //place items in grid layout
  QGridLayout* grid = new QGridLayout( this, 4, 3, 0);
  grid->addMultiCellWidget( billboard, 0,2, 0,0, Qt::AlignHCenter | Qt::AlignTop );
  grid->addWidget( contents, 1,1 );
  grid->addMultiCellWidget( content, 0,2, 2,2 );

  grid->setRowSpacing( 0, QMAX(billboard->sizeHint().height() - 
                               contents->minimumSizeHint().height(), 0)/2 );
  grid->setColSpacing( 1, contents->minimumSizeHint().width() );
  grid->setRowStretch( 1, 1 );
  grid->setColStretch( 2, 1 );
  //--
  //PLATFORM_SPECIFIC_CODE - Close Button
#if (!defined(Q_OS_WIN) && !defined(Q_OS_MACX))
  QPushButton* closeButton = new QPushButton( tr("Close"), this );
  closeButton->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
  closeButton->setDefault(true);
  connect( closeButton, SIGNAL(clicked()), SLOT(close()) );
  grid->addMultiCellWidget( closeButton, 3,3, 0,2, Qt::AlignCenter );
#endif  
  //--
}
//==============================================
HelpWindow::~HelpWindow()
{
  delete loadingMimeSource;
  loadingMimeSource = NULL;
}
//==============================================
void HelpWindow::closeEvent( QCloseEvent* e)
{
  QWidget::closeEvent( e );
  emit closed();
}
//==============================================
void HelpWindow::reject()
{
  QDialog::reject();
  emit closed();
}
//==============================================
void HelpWindow::setPage(HELP_PAGE page)
{
  //if billboard stillshown first remove it.
  if( currentPage == BILLBOARD )
  {
    billboard->removePixmap();
    currentPage = page;
    
    //show page only once billboard has finished sliding away to the left
  }
  else
  {
    currentPage = page;
    showCurrentPage();
  }
}
//==============================================
void HelpWindow::showFirstSelection()
{
  content->show();
  showCurrentPage();
}
//==============================================
void HelpWindow::showCurrentPage()
{
  if( currentPage == KEYBOARD_SHORTCUTS )
    content->setSource( Shortcuts::filename() );
  else if( currentPage == WHATS_NEW )
    content->setSource( WhatsNew::filename() );

  else if( currentPage == IMPORTING_AND_ORGANIZING )
    content->setSource( Importing::filename() );
  else if( currentPage == ANNOTATING_ALBUMS )
    content->setSource( Annotating::filename() );
  else if( currentPage == FRAMING )
    content->setSource( Framing::filename() );
  else if( currentPage == ENHANCING )
    content->setSource( Enhancing::filename() );
  else if( currentPage == PRO_TOOLS )
    content->setSource( ProTools::filename() );
  else if( currentPage == MANIPULATING )
    content->setSource( Manipulating::filename() );
  else if( currentPage == SAVING_AND_LOADING )
    content->setSource( LoadSave::filename() );
  else
    content->setText("");
  
  content->setFocus();
}
//==============================================


