// ---------------------------------------------------------------------------
// - Localset.hpp                                                            -
// - aleph engine - local set class definition                               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_LOCALSET_HPP
#define  ALEPH_LOCALSET_HPP

#ifndef  ALEPH_NAMESET_HPP
#include "Nameset.hpp"
#endif

#ifndef  ALEPH_NAMETABLE_HPP
#include "NameTable.hpp"
#endif

namespace aleph {

  /// The Localset class is a nameset designed to handle a small set of
  /// objects. Each object is associated with a symbol. Creating by name
  /// automatically creates the symbol object. The evaluation automatically
  /// retreive the symbol by name and return the object.
  /// @author amaury darsch

  class Localset : public Nameset {
  private:
    /// the primary name table
    NameTable* p_ptbl;
    /// the secondary name table
    NameTable* p_stbl;

  public:
    /// create a new local set
    Localset (void);
    
    /// create a new local set with the primary table only
    /// @param lset the local nameset
    Localset (Localset* lset);

    /// destroy this local set
    ~Localset (void);

    /// @return the class name
    String repr (void) const;

    /// make this locaset a shared object - i.e throw an exception
    void mksho (void);

    /// clear this local set
    void clear (void);

    /// add a new object by quark
    /// @param quark the object quark
    /// @param object the object to bind
    void bind (const long quark, Object* object);

    /// @return true if the qaurk exists in this global set
    bool exists (const long quark) const;

    /// @return an object by quark but do not evaluate
    Object* find (const long quark);

    /// remove an object by quark in this nameset
    /// @param quark the binding to remove
    void remove (const long quark);

    /// set an object as a const object by quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param quark  the quark to bind this constant
    /// @param object the object to set
    Object* cdef (Runnable* robj, Nameset* nset, const long quark, 
		  Object* object);

    /// set an object to this object by quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param quark  the quark to bind this object
    /// @param object the object to set
    Object* vdef (Runnable* robj, Nameset* nset, const long quark, 
		  Object* object);

    /// evaluate an object in the current nameset by quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param quark  the quark to evaluate
    Object* eval (Runnable* robj, Nameset* nset, const long quark);

  private:
    // make the copy constructor private
    Localset (const Localset&);
    // make the assignment operator private
    Localset& operator = (const Localset&);
  };
}

#endif
