// ---------------------------------------------------------------------------
// - Promise.cpp                                                             -
// - aleph engine - promise class implementation                             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "String.hpp"
#include "Promise.hpp"
#include "Runnable.hpp"

namespace aleph {

  // create a new promise with a form

  Promise::Promise (Object* form) {
    p_form   = Object::iref (form);
    p_object = nilp;
    d_delay  = true;
  }

  // destroy this promise

  Promise::~Promise (void) {
    Object::dref (p_form);
    Object::dref (p_object);
  }

  // return the class name

  String Promise::repr (void) const {
    return "Promise";
  }

  // make this promise a shared object

  void Promise::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_form   != nilp) p_form->mksho   ();
    if (p_object != nilp) p_object->mksho ();
  }

  // force the evaluation of this promise

  Object* Promise::force (Runnable* robj, Nameset* nset) {
    wrlock ();
    try {
      if (d_delay == true) {
	p_object = (p_form == nilp) ? nilp : p_form->eval (robj, nset);
	Object::iref (p_object);
	d_delay  = false;
      }
      robj->post (p_object);
      unlock ();
      return p_object;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // evaluate this promise

  Object* Promise::eval (Runnable* robj, Nameset* nset) {
    rdlock ();
    try {
      Object* result = d_delay ? this : p_object;
      robj->post (result);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
}
