// ---------------------------------------------------------------------------
// - Record.cpp                                                              -
// - aleph:odb library - record class implementation                         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can rodbstribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Odbsid.hpp"
#include "Record.hpp"
#include "Method.hpp"
#include "Integer.hpp"
#include "Lexical.hpp"
#include "Runnable.hpp"
#include "Exception.hpp"

namespace aleph {

  // the record supported quarks
  static const long QUARK_ADD     = String::intern ("add");
  static const long QUARK_GET     = String::intern ("get");
  static const long QUARK_SET     = String::intern ("set");
  static const long QUARK_FIND    = String::intern ("find");
  static const long QUARK_LOOKUP  = String::intern ("lookup");
  static const long QUARK_LENGTH  = String::intern ("length");
  static const long QUARK_GETNAME = String::intern ("get-name");
  static const long QUARK_SETNAME = String::intern ("set-name");

  // this procedure returns a new record object for deserialization
  static Serial* mksob (void) {
    return new Record;
  }
  // register this cell serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_RECD_ID, mksob);

  // create a nil record

  Record::Record (void) {
    d_quark = 0;
  }

  // create a new record by name

  Record::Record (const String& name) {
    d_quark = name.toquark ();
  }

  // return the object name

  String Record::repr (void) const {
    return "Record";
  }

  // return the record serial id

  t_byte Record::serialid (void) const {
    return SERIAL_RECD_ID;
  }

  // serialize a record

  void Record::wrstream (Output& os) const {
    rdlock ();
    // save the record name
    const String& name = String::qmap (d_quark);
    name.wrstream (os);
    // save the vector
    d_vcell.wrstream (os);
    unlock ();
  }

  // deserialize this record

  void Record::rdstream (Input& is) {
    wrlock ();
    // get the record name
    String name;
    name.rdstream (is);
    d_quark = name.toquark ();
    // get the vector
    d_vcell.rdstream (is);
    unlock ();
  }

  // return the record name

  String Record::getname (void) const {
    rdlock ();
    String result = String::qmap (d_quark);
    unlock ();
    return result;
  }

  // set the record name

  void Record::setname (const String& name) {
    wrlock ();
    d_quark = name.toquark ();
    unlock ();
  }

  // set the cell name by index

  void Record::setname (const long index, const String& name) {
    wrlock ();
    try {
    Cell* cell = dynamic_cast <Cell*> (d_vcell.get (index));
    if (cell != nilp) cell->setname (name);
    unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the length of the record list

  long Record::length (void) const {
    rdlock ();
    long result = d_vcell.length ();
    unlock ();
    return result;
  }

  // add an object to this record

  void Record::add (Object* object) {
    if (object == nilp) {
      add (new Cell);
      return;
    }
    // check for a cell
    Cell* cell = dynamic_cast <Cell*> (object);
    if (cell != nilp) {
      add (cell);
      return;
    }
    // check for a literal
    Literal* lobj = dynamic_cast <Literal*> (object);
    if (lobj != nilp) {
      add (lobj);
      return;
    }
    // check for a lexical
    Lexical* lex = dynamic_cast <Lexical*> (object);
    if ((lex != nilp) && (lex->isnil () == true)) {
      add (new Cell);
      return;
    }
    throw Exception ("type-error", "invalid object to add in record",
		     Object::repr (object));
  }

  // add a literal in this record

  void Record::add (Literal* lobj) {
    wrlock ();
    try {
      add (new Cell (lobj));
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a cell in this record

  void Record::add (Cell* cell) {
    wrlock ();
    try {
      Cell* data = (cell == nilp) ? new Cell : cell;
      d_vcell.append (data);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get a cell by index

  Cell* Record::get (const long index) const {
    rdlock ();
    Cell* result = dynamic_cast <Cell*> (d_vcell.get (index));
    unlock ();
    return result;
  }

  // get a cell literal by index

  Literal* Record::getlobj (const long index) const {
    rdlock ();
    try {
      Cell* cell = get (index);
      Literal* lobj = (cell == nilp) ? nilp : cell->get ();
      unlock ();
      return lobj;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  // set a literal in this record by index

  void Record::set (const long index, Literal* lobj) {
    wrlock ();
    Cell* cell = new Cell (lobj);
    try {
      d_vcell.set (index, cell);
    } catch (...) {
      delete cell;
      unlock ();
      throw;
    }
    unlock ();
  }

  // set an object in this record by index

  void Record::set (const long index, Object* object) {
    if (object == nilp) return;
    // check for a cell
    Cell* cell = dynamic_cast <Cell*> (object);
    if (cell != nilp) {
      set (index, cell);
      return;
    }
    // check for a literal
    Literal* lobj = dynamic_cast <Literal*> (object);
    if (lobj != nilp) {
      set (index, lobj);
      return;
    }
    throw Exception ("type-error", "invalid object to set in record",
		     Object::repr (object));
  }

  // set a cell in this record by index

  void Record::set (const long index, Cell* cell) {
    wrlock ();
    d_vcell.set (index, cell);
    unlock ();
  }

  // find a cell by quark

  Cell* Record::find (const long quark) const {
    rdlock ();
    long rlen = d_vcell.length ();
    for (long i = 0; i < rlen; i++) {
      Cell* cell = dynamic_cast <Cell*> (d_vcell.get (i));
      if (cell == nilp) continue;
      if (*cell == quark) {
	unlock ();
	return cell;
      }
    }
    unlock ();
    return nilp;
  }

  // find a cell by name

  Cell* Record::find (const String& name) const {
    long quark = name.toquark ();
    return find (quark);
  }

  // find a cell by quark or throw an exception

  Cell* Record::lookup (const long quark) const {
    Cell* cell = find (quark);
    if (cell == nilp)
      throw Exception ("lookup-error", "cannot find cell", 
		       String::qmap (quark));
    return cell;
  }

  // find a cell by name or throw an exception

  Cell* Record::lookup (const String& name) const {
    long quark = name.toquark ();
    return lookup (quark);
  }
    
  // create a new record in a generic way

  Object* Record::mknew (Vector* argv) {
    // get number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new Record;
    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new Record (name);
    }
    throw Exception ("argument-error", "too many argument with record");
  }

  // evaluate a record object by quark
  
  Object* Record::eval (Runnable* robj, Nameset* nset, const long quark) {
    // find the cell by quark
    Cell* cell = find (quark);
    if (cell != nilp) {
      Object* result = cell->get ();
      robj->post (result);
      return result;
    }
    // make it a method
    return new Method (quark, this);
  }

  // apply a record method with a set of arguments and a quark

  Object* Record::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for generic quark
    if (quark == QUARK_ADD) {
      for (long i = 0; i < argc; i++) {
	add (argv->get (i));
      }
      return nilp;
    }
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LENGTH ) return new Integer (length  ());
      if (quark == QUARK_GETNAME) return new String  (getname ());
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETNAME) {
	String name = argv->getstring (0);
	setname (name);
	return nilp;
      }
      if (quark == QUARK_FIND) {
	String name = argv->getstring (0);
	rdlock ();
	Object* result = find (name);
	robj->post (result);
	unlock ();
	return result;
      }
      if (quark == QUARK_LOOKUP) {
	String name = argv->getstring (0);
	rdlock ();
	Object* result = lookup (name);
	robj->post (result);
	unlock ();
	return result;
      }
      if (quark == QUARK_GET) {
	long idx = argv->getint (0);
	rdlock ();
	try {
	  Object* result = get (idx);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }
    // dispatch 2 argument
    if (argc == 2) {
      if (quark == QUARK_SET) {
	long    idx = argv->getint (0);
	Object* obj = argv->get (1);
	set (idx, obj);
	return nilp;
      }
      if (quark == QUARK_SETNAME) {
	long    idx = argv->getint    (0);
	String name = argv->getstring (1);
	setname (idx, name);
	return nilp;
      }
    }
    // call the persist method
    return Persist::apply (robj, nset, quark, argv);
  }
}
