// ---------------------------------------------------------------------------
// - Record.hpp                                                              -
// - aleph:odb library - record class definition                             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can rodbstribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_RECORD_HPP
#define  ALEPH_RECORD_HPP

#ifndef  ALEPH_VECTOR_HPP
#include "Vector.hpp"
#endif

#ifndef  ALEPH_PERSIST_HPP
#include "Persist.hpp"
#endif

#ifndef  ALEPH_CELL_HPP
#include "Cell.hpp"
#endif

namespace aleph {

  /// The Record class is a collection of cell. Like a cell a record can be
  /// bound to a name. Adding cell to a record is a simple process, like
  /// adding object to a vector.
  /// @author amaury darsch

  class Record : public Persist {
  private:
    /// the record name
    long   d_quark;
    /// the cell vector
    Vector d_vcell;

  public:
    /// create an empty record
    Record (void);

    /// create a new record with a name
    /// @param name the record name
    Record (const String& name);

    /// @return the object name
    String repr (void) const;

    /// @return the record serial id
    t_byte serialid (void) const;

    /// serialize this record
    /// @param os the output stream
    void wrstream (Output& os) const;

    /// deserialize this record
    /// @param is the input stream
    void rdstream (Input& os);

    /// @return the record name
    String getname (void) const;

    /// set the record name
    /// @param name the name to set
    void setname (const String& name);

    /// set the a cell name by index
    /// @param name the name to set
    void setname (const long index, const String& name);

    /// @return the record length
    long length (void) const;

    /// add a new literal in this record
    /// @param lobj the literal to add
    void add (Literal* lobj);

    /// add a new cell in this record
    /// @param cell the cell to add
    void add (Cell* cell);

    /// add an object in this record
    /// @param object the object to add
    void add (Object* object);

    /// @return a cell by index
    Cell* get (const long index) const;

    /// @return a cell literal by index
    Literal* getlobj (const long index) const;

    /// set a literal in this record by index
    /// @param index the literal index
    /// @param lobj  the literal to set
    void set (const long index, Literal* lobj);

    /// set a cell in this record by index
    /// @param index the cell index
    /// @param cell  the cell to set
    void set (const long index, Cell* cell);

    /// set an object in this record by index
    /// @param index  the object index
    /// @param object the object to set
    void set (const long index, Object* object);

    /// @param find a cell by quark
    /// @param name the cell to find
    Cell* find (const long quark) const;

    /// @param find a cell by name
    /// @param name the cell to find
    Cell* find (const String& name) const;

    /// @param find a cell by quark or throw an exception
    /// @param name the cell to find
    Cell* lookup (const long quark) const;

    /// @param find a cell by name or throw an exception
    /// @param name the cell to find
    Cell* lookup (const String& name) const;
 
    /// generate a new record
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// evaluate this record with a quark
    /// @param robj  robj the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to evaluate
    Object* eval (Runnable* robj, Nameset* nset, const long quark);

    /// apply this record with a set of arguments and a quark
    /// @param robj  robj the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);

  private:
    // make the copy constructor private
    Record (const Record&);
    // make the assignment operator private
    Record& operator = (const Record&);
  };
}

#endif
