// ---------------------------------------------------------------------------
// - ccnf.cpp                                                                -
// - standard platform library - config file generator                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include <stdio.h>
#include <string.h>

// supported platforms
#define ALEPH_PLATFORM_UNKNOWN   0
#define ALEPH_PLATFORM_LINUX     1
#define ALEPH_PLATFORM_SOLARIS   2
#define ALEPH_PLATFORM_FREEBSD   3

// recognized platform name
#define ALEPH_PLATNAME_LINUX     "linux"
#define ALEPH_PLATNAME_SOLARIS   "solaris"
#define ALEPH_PLATNAME_FREEBSD   "freebsd"

// supported processors
#define ALEPH_PROCTYPE_UNKNOWN   0
#define ALEPH_PROCTYPE_IA        1
#define ALEPH_PROCTYPE_SPARC     2
#define ALEPH_PROCTYPE_ALPHA     3
#define ALEPH_PROCTYPE_ARM       4
#define ALEPH_PROCTYPE_PPC       5
#define ALEPH_PROCTYPE_M68K      6
#define ALEPH_PROCTYPE_MIPS      7
#define ALEPH_PROCTYPE_MIPSEL    8
#define ALEPH_PROCTYPE_PA64      9
#define ALEPH_PROCTYPE_IA64     10
#define ALEPH_PROCTYPE_S390     11

// recognized processor name
#define ALEPH_PROCNAME_IA        "ia"
#define ALEPH_PROCNAME_SPARC     "sparc"
#define ALEPH_PROCNAME_ALPHA     "alpha"
#define ALEPH_PROCNAME_ARM       "arm"
#define ALEPH_PROCNAME_PPC       "ppc"
#define ALEPH_PROCNAME_M68K      "m68k"
#define ALEPH_PROCNAME_MIPS      "mips"
#define ALEPH_PROCNAME_MIPSEL    "mipsel"
#define ALEPH_PROCNAME_PA64      "pa64"
#define ALEPH_PROCNAME_IA64      "ia64"
#define ALEPH_PROCNAME_S390      "s390"

// force size type with S390
#if defined(__s390__)
const bool ALEPH_FORCE_LONG = true;
#else
const bool ALEPH_FORCE_LONG = false;
#endif

// compute the platform id
static int get_platid (const char* plat) {
  if (!plat || (strlen (plat) == 0)) return ALEPH_PLATFORM_UNKNOWN;
  // linux
  if (strcmp (plat, ALEPH_PLATNAME_LINUX) == 0) 
    return ALEPH_PLATFORM_LINUX;
  // solaris
  if (strcmp (plat, ALEPH_PLATNAME_SOLARIS) == 0) 
    return ALEPH_PLATFORM_SOLARIS;
  // freebsd
  if (strcmp (plat, ALEPH_PLATNAME_FREEBSD) == 0) 
    return ALEPH_PLATFORM_FREEBSD;
  // unknown
  return ALEPH_PLATFORM_UNKNOWN; 
}

// compute the processor id
static int get_procid (const char* proc) {
  if (!proc || (strlen (proc) == 0)) return ALEPH_PROCTYPE_UNKNOWN;
  // ia
  if (strcmp (proc, ALEPH_PROCNAME_IA) == 0) 
    return ALEPH_PROCTYPE_IA;
  // sparc
  if (strcmp (proc, ALEPH_PROCNAME_SPARC) == 0) 
    return ALEPH_PROCTYPE_SPARC;
  // alpha
  if (strcmp (proc, ALEPH_PROCNAME_ALPHA) == 0) 
    return ALEPH_PROCTYPE_ALPHA;
  // arm
  if (strcmp (proc, ALEPH_PROCNAME_ARM) == 0) 
    return ALEPH_PROCTYPE_ARM;
  // ppc
  if (strcmp (proc, ALEPH_PROCNAME_PPC) == 0) 
    return ALEPH_PROCTYPE_PPC;
  // m68k
  if (strcmp (proc, ALEPH_PROCNAME_M68K) == 0) 
    return ALEPH_PROCTYPE_M68K;
  // mips
  if (strcmp (proc, ALEPH_PROCNAME_MIPS) == 0)
    return ALEPH_PROCTYPE_MIPS;
  // mipsel
  if (strcmp (proc, ALEPH_PROCNAME_MIPSEL) == 0)
    return ALEPH_PROCTYPE_MIPSEL;
  // pa64
  if (strcmp (proc, ALEPH_PROCNAME_PA64) == 0)
    return ALEPH_PROCTYPE_PA64;
  // ia64
  if (strcmp (proc, ALEPH_PROCNAME_IA64) == 0)
    return ALEPH_PROCTYPE_IA64;
  // s390
  if (strcmp (proc, ALEPH_PROCNAME_S390) == 0)
    return ALEPH_PROCTYPE_S390;
  // unknown
  return ALEPH_PROCTYPE_UNKNOWN;
}

// check for endian sex
static bool is_big_endian (void) {
  union {
    char bval[2];
    unsigned short ival;
  } data;
  data.ival = 0x1234;
  if ((data.bval[0] == 0x12) && (data.bval[1] == 0x34)) return true;
  return false;
}

const char* header = 
"// ------------------------------------------------------------------------\n"
"// - ccnf.hpp                                                             -\n"
"// - standard platform library - base type configuration                  -\n"
"// ------------------------------------------------------------------------\n"
"// - This program is free software; you can redistribute it and/or modify -\n"
"// - it provided that this copyright notice is kept intact.               -\n"
"// -                                                                      -\n"
"// - This program is distributed in the hope  that it will be useful, but -\n"
"// - without  any  warranty;  without  even  the  implied   warranty   of -\n"
"// - merchantability  or  fitness  for  a particular purpose. In no event -\n"
"// - shall the  copyright  holder  be  liable  for any  direct, indirect, -\n"
"// - incidental or special  damages  arising in any way out of the use of -\n"
"// - this software.                                                       -\n"
"// -                                                                      -\n"
"// - This file was automatically generated by ccnf                        -\n"
"// ------------------------------------------------------------------------\n"
"// - copyright (c) 1999-2003 amaury darsch                                -\n"
"// ------------------------------------------------------------------------";

static const char* nilptr =
"// the base nilp pointer definition\n"
"#ifndef nilp\n"
"#define nilp 0\n"
"#endif\n";

static const char* types = 
"  // fundamental types of the aleph system\n"
"  typedef unsigned char      t_byte;\n"
"  typedef unsigned short     t_word;\n"
"  typedef unsigned int       t_quad;\n"
"  typedef unsigned long long t_octa;\n"
"  typedef long long          t_long;\n"
"  typedef double             t_real;\n";

static const char* constants =
"  // fundamental constants of the aleph system\n"
"  const char nilc = '\\0';\n"
"  const char eofc = '\\4';\n"
"  const char eolc = '\\n';\n"
"  const char crlc = '\\r';\n"
"  const char tabc = '\\t';\n";

int main (int, char**) {
  // print the header
  fprintf (stdout, "%s\n\n", header);

  // install the control header
  fprintf (stdout, "#ifndef  ALEPH_CCNF_HPP\n");
  fprintf (stdout, "#define  ALEPH_CCNF_HPP\n\n");

  // define current revision
  fprintf (stdout, "// aleph revision\n");
  fprintf (stdout, "#define  ALEPH_VERSION_MAJOR      %s\n", PLTMAJOR);
  fprintf (stdout, "#define  ALEPH_VERSION_MINOR      %s\n", PLTMINOR);
  fprintf (stdout, "#define  ALEPH_VERSION_PATCH      %s\n", PLTPATCH);
  fprintf (stdout, "#define  ALEPH_VERSION_TOTAL      \"%s-%s-%s\"\n", 
	   PLTMAJOR, PLTMINOR, PLTPATCH);
  
  fprintf (stdout, "\n");

  // define the platform list
  fprintf (stdout, "// supported platforms\n");
  fprintf (stdout, "#define  ALEPH_PLATFORM_UNKNOWN   %d\n",
	   ALEPH_PLATFORM_UNKNOWN);
  fprintf (stdout, "#define  ALEPH_PLATFORM_LINUX     %d\n",
	   ALEPH_PLATFORM_LINUX);
  fprintf (stdout, "#define  ALEPH_PLATFORM_SOLARIS   %d\n",
	   ALEPH_PLATFORM_SOLARIS);
  fprintf (stdout, "#define  ALEPH_PLATFORM_FREEBSD   %d\n",
	   ALEPH_PLATFORM_FREEBSD);
  fprintf (stdout, "\n");

  // define the processor list
  fprintf (stdout, "// supported processors\n");
  fprintf (stdout, "#define  ALEPH_PROCTYPE_UNKNOWN   %d\n",
	   ALEPH_PROCTYPE_UNKNOWN);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_IA        %d\n",
	   ALEPH_PROCTYPE_IA);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_SPARC     %d\n",
	   ALEPH_PROCTYPE_SPARC);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_ALPHA     %d\n",
	   ALEPH_PROCTYPE_ALPHA);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_ARM       %d\n",
	   ALEPH_PROCTYPE_ARM);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_PPC       %d\n",
	   ALEPH_PROCTYPE_PPC);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_M68K      %d\n",
	   ALEPH_PROCTYPE_M68K);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_MIPS      %d\n",
	   ALEPH_PROCTYPE_MIPS);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_MIPSEL    %d\n",
	   ALEPH_PROCTYPE_MIPSEL);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_PA64      %d\n",
	   ALEPH_PROCTYPE_PA64);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_IA64      %d\n",
	   ALEPH_PROCTYPE_IA64);
  fprintf (stdout, "#define  ALEPH_PROCTYPE_S390      %d\n",
	   ALEPH_PROCTYPE_S390);
  fprintf (stdout, "\n");

  // install the config names
  fprintf (stdout, "// platform definitions\n");
  fprintf (stdout, "#define  ALEPH_PLATFORM_PLATID    %d\n", 
	   get_platid (PLATNAME));
  fprintf (stdout, "#define  ALEPH_PLATFORM_PROCID    %d\n", 
	   get_procid (PROCNAME));
  fprintf (stdout, "#define  ALEPH_PLATFORM_VERSION   %s\n",     PLATVERS);
  fprintf (stdout, "#define  ALEPH_PLATFORM_MAJOR     %s\n",     PLATVMAJ);
  fprintf (stdout, "#define  ALEPH_PLATFORM_MINOR     %s\n",     PLATVMIN);
  fprintf (stdout, "#define  ALEPH_PLATFORM_NAME      \"%s\"\n", PLATNAME);
  fprintf (stdout, "#define  ALEPH_PLATFORM_VINFO     \"%s\"\n", PLATVERS);
  fprintf (stdout, "#define  ALEPH_PLATFORM_PROC      \"%s\"\n", PROCNAME);
  fprintf (stdout, "#define  ALEPH_PLATFORM_PTYPE     \"%s\"\n", PROCTYPE);
  fprintf (stdout, "\n");

  // install processor type
  fprintf (stdout, "// processor mode\n");
  if (is_big_endian () == true) {
    fprintf (stdout, "#define  ALEPH_HAVE_BIG_ENDIAN\n");
  } else {
    fprintf (stdout, "#define  ALEPH_HAVE_LITTLE_ENDIAN\n");
  }
  fprintf (stdout, "\n");

  // install installation onfiguration
  fprintf (stdout, "// configuration definitions\n");
  if (strcmp (LINKTYPE, "static") == 0) {
    fprintf (stdout, "#define  ALEPH_REGISTER_STATIC\n");
    fprintf (stdout, "#define  ALEPH_LINKTYPE_STATIC    true\n");
  } else {
    fprintf (stdout, "#define  ALEPH_LINKTYPE_STATIC    false\n");
  }
  fprintf (stdout, "\n");

  // add the nil pointer definition
  fprintf (stdout, "%s", nilptr);
  fprintf (stdout, "\n");

  // start the namespace
  fprintf (stdout, "namespace aleph {\n");

  // install the types
  fprintf (stdout, "%s", types);
  if (ALEPH_FORCE_LONG == true)
    fprintf (stdout, "  typedef unsigned long      t_size;\n");
  else if (sizeof (void*) == 8 )
    fprintf (stdout, "  typedef unsigned long      t_size;\n");
  else
    fprintf (stdout, "  typedef unsigned int       t_size;\n");
  fprintf (stdout, "\n");
  
  // install the constants
  fprintf (stdout, "%s", constants);

  // install the footer
  fprintf (stdout, "}\n\n");
  fprintf (stdout, "#endif\n");

  return 0;
}
