// ---------------------------------------------------------------------------
// - Exception.cpp                                                           -
// - standard object library - exception class implementation                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Exception.hpp"

namespace aleph {
  // the exception supported quarks
  static const long QUARK_EID    = String::intern ("eid");
  static const long QUARK_REASON = String::intern ("reason");
  static const long QUARK_OBJECT = String::intern ("object");

  // create a exception with an id
  
  Exception::Exception (const String& eid) {
    d_eid    = eid;
    p_object = nilp;
    d_nlf    = false; 
    d_abf    = false; 
  }

  // create a exception with an id and a reason
  
  Exception::Exception (const String& eid, const String& reason) {
    d_eid    = eid;
    d_reason = reason;
    p_object = nilp;
    d_nlf    = false;
    d_abf    = false;
  }

  // create a exception with an eid, an indicator and a long reason
  
  Exception::Exception (const String& eid, const String& reason,
			const String& name) {
    d_eid    = eid;
    d_reason = reason + " " + name;
    p_object = nilp;
    d_nlf    = false;
    d_abf    = false;
  }

  // create a exception with an id and an object
  
  Exception::Exception (const String& eid, Object* object) {
    d_eid    = eid;
    p_object = Object::iref (object);
    d_nlf    = false;
    d_abf    = false;
  }

  // create a exception with a short type, a reason and an object
  
  Exception::Exception (const String& eid, const String& reason,
			Object* object) {
    d_eid    = eid;
    d_reason = reason;
    p_object = Object::iref (object);
    d_nlf    = false;
    d_abf    = false;
  }

  // copy constructor for this exception
  
  Exception::Exception (const Exception& that) {
    d_eid    = that.d_eid;
    d_reason = that.d_reason;
    p_object = Object::iref (that.p_object);
    d_nlf    = that.d_nlf;
    d_abf    = that.d_abf;
  }

  // destroy this exception

  Exception::~Exception (void) {
    Object::dref (p_object);
  }

  // return the class name

  String Exception::repr (void) const {
    return "Exception";
  }

  // set the new line flag

  void Exception::setnlf (const bool flag) {
    d_nlf = flag;
  }

  // set the abort flag

  void Exception::setabf (const bool flag) {
    d_abf = flag;
  }

  // set the exception file name

  void Exception::setfname (const String& fname) {
    d_fname = fname;
  }

  // set the exception line number

  void Exception::setlnum (const long lnum) {
    d_lnum = lnum;
  }

  // evaluate an exception member by quark

  Object* Exception::eval (Runnable* robj, Nameset* nset, const long quark) {
    // check the exception first
    if (quark == QUARK_EID)    return new String (d_eid);
    if (quark == QUARK_REASON) return new String (d_reason);
    if (quark == QUARK_OBJECT) return p_object;
        // evaluate in the object
    return Object::eval (robj, nset, quark);
  }
}
