// ---------------------------------------------------------------------------
// - Input.cpp                                                               -
// - standard object library - input stream class implementation             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Input.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "Character.hpp"
#include "Exception.hpp"
#include "ccnv.hpp"

namespace aleph {

  // the input supported quarks
  static const long QUARK_EOFP   = String::intern ("eof-p");
  static const long QUARK_READ   = String::intern ("read");
  static const long QUARK_PUSHB  = String::intern ("pushback");
  static const long QUARK_VALIDP = String::intern ("valid-p");
  static const long QUARK_READLN = String::intern ("readln");
  static const long QUARK_BUFLEN = String::intern ("get-buffer-length");

  // return the stream descriptor

  int Input::getsid (void) const {
    return -1;
  }

  // read a buffer by size

  Buffer* Input::read (const long size) {
    wrlock ();
    Buffer* result = new Buffer;
    try {
      for (long i = 0; i < size; i++) {
	if (valid (-1) == false) break;
	result->add (read ());
      }
    } catch (...) {
      delete result;
      unlock ();
      throw;
    }
    unlock ();
    return result;
  }

  // read a line in this input stream

  String Input::readln (void) {
    wrlock ();
    // create a buffer to accumulate characters
    Buffer buffer;
    bool   crflag = false;
    try {
      // read the character in the buffer
      while (valid (-1) == true) {
	char c = read ();
	if (c == crlc) {
	  crflag = true;
	  continue;
	}
	if (c == eolc) {
	  unlock ();
	  return buffer.tostring ();
	}
	if (crflag == true) {
	  buffer.add (crlc);
	  crflag = false;
	}
	buffer.add (c);
      }
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
    return buffer.tostring ();
  }

  // pushback a character

  void Input::pushback (const char value) {
    wrlock ();
    try {
      d_buffer.pushback (value);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // pushback a string on this stream
  
  void Input::pushback (const String& value) {
    wrlock ();
    try {
      d_buffer.pushback (value);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // pushback a buffer by size

  void Input::pushback (const char* buffer, const long size) {
    wrlock ();
    try {
      d_buffer.pushback (buffer, size);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // return the size of the input buffer

  long Input::buflen (void) const {
    rdlock ();
    long result = d_buffer.length ();
    unlock ();
    return result;
  }

  // apply this input class with a set of arguments and a quark

  Object* Input::apply (Runnable* robj, Nameset* nset, const long quark,
			Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_EOFP)   return new Boolean   (iseof  ());
      if (quark == QUARK_READ)   return new Character (read   ());
      if (quark == QUARK_VALIDP) return new Boolean   (valid  (-1));
      if (quark == QUARK_READLN) return new String    (readln ());
      if (quark == QUARK_BUFLEN) return new Integer   (buflen ());
    }

    // dispatch one argument
    if (argc == 1) {
      if (quark == QUARK_READ) {
	long size = argv->getint (0);
	return read (size);
      }
      if (quark == QUARK_VALIDP) {
	long tout = argv->getint (0);
	return new Boolean (valid (tout));
      }
      if (quark == QUARK_PUSHB) {
	Object* obj = argv->get (0);
	// check for a character
	Character* c = dynamic_cast <Character*> (obj);
	if (c != nilp) {
	  pushback (c->tochar ());
	  return nilp;
	}
	String* s = dynamic_cast <String*> (obj);
	if (s != nilp) {
	  pushback (*s);
	  return nilp;
	}
	throw Exception ("type-error", "invalid object with pushback method",
			 Object::repr (obj));
      }
    }
    // apply these arguments with the object
    return Object::apply (robj, nset, quark, argv);
  }
}
