/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include <cstdio>
#include <kglobal.h>
#include <klocale.h>
#include <limits.h>
#include <vector>
#include "mealMasterCompiler.hpp"
#include "unitMap.hpp"
#include "utils.hpp"

using namespace boost;
using namespace std;

int mealMasterlex(void);
int mealMasterrestart( FILE * );

long position;
int lineNo;
int errorLine;
int recipeStartPos;
int recipeEndPos;
int recipeStartLine;
int recipeEndLine;
const char *errorMessage;

MealMasterCompiler *MealMasterCompiler::current = NULL;
static vector< string > outputBuffer( 5 );
static istream *currentInput;

/* index:  
   0: main-buffer
   1: delayed first ingredient-column
   2: delayed second ingredient-column
   3: right-top continuation of ingredient (in two-column format)
   4: right-top continuation of preparation-method (in two-column format).*/
void writeData( int index, const char *buffer, int size )
{
  assert( MealMasterCompiler::current != NULL );
  assert( index >= 0 );
  assert( index < (signed)outputBuffer.size() );
  outputBuffer[ index ] += string( buffer, size );
}

void writeUnit( int index, const char *buffer, int size )
{
  string unit( buffer, size );
  assert( MealMasterCompiler::current != NULL );
  string inverseUnit( MealMasterCompiler::current->
                      inverseUnitMap[ unit.c_str() ] );
  assert( !inverseUnit.empty() );
#ifndef NDEBUG
  // cerr << "buffer[0] = " << buffer[0] << ' ' << (int)buffer[0] << endl
  //      << "buffer[1] = " << buffer[1] << ' ' << (int)buffer[1] << endl
  //      << "size = " << size << endl
  //      << "string( buffer, size ) = " << unit << endl
  //      << "inverseUnit = " << inverseUnit << endl;
#endif
   outputBuffer[ index ] += (const char *)i18n( inverseUnit.c_str() );
}

void moveData( int index )
{
  assert( index >= 1 );
  assert( index < (signed)outputBuffer.size() );
  outputBuffer[ 0 ] += outputBuffer[ index ];
  outputBuffer[ index ].clear();
}

int readData( char *buffer, int max_size )
{
  int result;
  assert( MealMasterCompiler::current != NULL );
  if ( currentInput == NULL ) {
    result = 0;
  } else {
    currentInput->read( buffer, max_size );
    if ( currentInput->eof() ) {
      // Get number of bytes read.
      result = currentInput->gcount();
      currentInput = NULL;
    } else {
      // The buffer was filled.
      result = max_size;
    };
  };
  // cerr << "Read " << result << '/' << max_size << " characters." << endl;
  return result;
}

MealMasterCompiler::MealMasterCompiler
  ( int _maxRecipes, ParseErrorHandlerPtr _parseErrorHandler ):
  inverseUnitMap( createInverseUnitMap() ), maxRecipes(_maxRecipes),
  parseErrorHandler(_parseErrorHandler)
{
  assert( current == NULL );
  current = this;

  // Restart line-number counting.
  lineNo = 1;
}

MealMasterCompiler::~MealMasterCompiler(void)
{
  current = NULL;
}

void MealMasterCompiler::translate( istream &inputStream,
                                    ostream &outputStream ) const
  throw (Error)
{
  currentInput = &inputStream;
  // Store current position.
  position = inputStream.tellg();
  // Clear any remaining input from buffer.
  mealMasterrestart( NULL );
  outputStream << "<?xml version='1.0' encoding='UTF-8'?>"
               << endl << "<insert xml:lang='"
               << anyMealLanguage() << "'>" << endl;
  int counter = maxRecipes;
  while ( counter != 0 ) {
    try {
      for ( int i=0; i<(signed)outputBuffer.size(); i++ )
        outputBuffer[ i ].clear();
      int x = mealMasterlex();
      if ( x ==  0 ) break;
      ERRORMACRO( x < +2, ParseError,
                  ( recipeStartLine, recipeEndLine, errorLine, &inputStream,
                    recipeStartPos, recipeEndPos ), errorMessage );
      assert( x == 1 );
      outputStream << outputBuffer[ 0 ];
    } catch ( ParseError &e ) {
#ifndef NDEBUG
      cerr << e.what() << endl;
#endif
      assert( parseErrorHandler );
      parseErrorHandler->error( e );
    };
    if ( counter > 0 ) counter--;
  };
  outputStream << "</insert>\n";
  // Seek to current scanner position.
  inputStream.seekg( position, ios::beg );
}
