/*
 *  apcserial.c -- Serial line functions
 *
 *  apcupsd.c	-- Simple Daemon to catch power failure signals from a
 *		   BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *		-- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick <andre@suse.com>
 *  All rights reserved.
 *
 */

/*
 *		       GNU GENERAL PUBLIC LICENSE
 *			  Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *			     675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

#include "apc.h"

#ifndef O_NDELAY
#define O_NDELAY 0
#endif


/*********************************************************************/
void setup_serial(UPSINFO *ups)
{
    static int has_done_setup = 0;
    int cmd;
    
    if (!has_done_setup) {
	/*
	 * Marko Sakari Asplund <Marko.Asplund@cern.ch>
	 *    prevents double init of serial port 9/25/98
	 */
	has_done_setup++;
    } else {
        Error_abort0(_("Serial port already initialized.\n"));
    }
    
    /* Open the serial port device */
    if ((ups->fd = open(ups->device, O_RDWR | O_NOCTTY | O_NDELAY)) < 0) {
        Error_abort2(_("Cannot open UPS tty %s: %s\n"),
			ups->device, strerror(errno));
    }
    /* Cancel the no delay we just set */
    cmd = fcntl(ups->fd, F_GETFL, 0);
    fcntl(ups->fd, F_SETFL, cmd & ~O_NDELAY);

    /*
     * next 5 lines by BSC
     *
     * If create_lockfile fails there's no need to delete_lockfile.
     * -RF
     */ 		    
    if (create_lockfile(ups) == LCKERROR) {
	close(ups->fd);
	ups->fd = -1;
        Error_abort0(_("Unable to create serial port lock file.\n"));
    }

    tcgetattr(ups->fd, &oldtio); /* Save old settings */

    signal(SIGHUP, terminate);
    signal(SIGINT, terminate);
    signal(SIGTERM, terminate);
    signal(SIGKILL, terminate); /* But I think this is not effective -RF */

    newtio.c_cflag = DEFAULT_SPEED | CS8 | CLOCAL | CREAD;
    newtio.c_iflag = IGNPAR;		 /* Ignore errors, raw input */
    newtio.c_oflag = 0; 		 /* Raw output */
    newtio.c_lflag = 0; 		 /* No local echo */

#if defined(HAVE_OPENBSD_OS) || defined(HAVE_FREEBSD_OS) || defined(HAVE_NETBSD_OS)
    newtio.c_ispeed = DEFAULT_SPEED;	 /* Set input speed */
    newtio.c_ospeed = DEFAULT_SPEED;	 /* Set output speed */
#endif /* __openbsd__ || __freebsd__ || __netbsd__  */

    /* w.p. This makes a non.blocking read() with 5 sec. timeout */
    newtio.c_cc[VMIN] = 0;
    newtio.c_cc[VTIME] = TIMER_SERIAL * 10;

#if defined(HAVE_CYGWIN) || defined(HAVE_OSF1_OS) || defined(HAVE_LINUX_OS)
    cfsetospeed(&newtio, DEFAULT_SPEED);
    cfsetispeed(&newtio, DEFAULT_SPEED);
#endif /* do it the POSIX way */

    tcflush(ups->fd, TCIFLUSH);
    tcsetattr(ups->fd, TCSANOW, &newtio);
    tcflush(ups->fd, TCIFLUSH);

    switch(ups->cable.type) {
    case APC_940_0119A:
    case APC_940_0127A:
    case APC_940_0128A:
	(void)ioctl(ups->fd, TIOCMBIC, &dtr_bit); /* clear shutdown bit */
	(void)ioctl(ups->fd, TIOCMBIS, &rts_bit); /* tell him we are ready */
	break;

    /* Have to clear RTS line to access the serial cable mode PnP */
    case APC_940_0095A:
    case APC_940_0095B:
    case APC_940_0095C:
	(void) ioctl(ups->fd, TIOCMBIC, &rts_bit);
    default:
	break;
    }

    if (ups->mode.type > SHAREBASIC) {
	int attempts;
        char a = 'Y';

	write(ups->fd, &a, 1);	      /* This one might not work, if UPS is */
	sleep(1);		      /* in an unstable communication state */
        tcflush(ups->fd, TCIOFLUSH);  /* Discard UPS's response, if any */

        /* Don't use smart_poll here because it may loop waiting
	 * on the serial port, and here we may be called before
	 * we are a deamon, so we want to error after a reasonable
	 * time.
	 */
	for (attempts=0; attempts < 5; attempts++) {
	    char answer[10];

	    *answer = 0; 
	    write(ups->fd, &a, 1); /* enter smart mode */
	    getline(answer, sizeof(answer), ups);
            if (strcmp("SM", answer) == 0)
		    goto out;
	    sleep(1);
	}
        Error_abort0(_("PANIC! Cannot communicate with UPS via serial port.\n"));
    }

out:
    save_dumb_status(ups);

    return;
}

/*********************************************************************/
void kill_power(UPSINFO *ups)
{
    char response[32];
    char a;
    FILE *pwdf;
    int errflag = 0;
    int killcount;
    int shutdown_delay;

    response[0] = '\0';

    if (ups->mode.type <= SHAREBASIC) {
	/* Make sure we are on battery !! */
	for (killcount=0; killcount<3; killcount++) {
	    ioctl(ups->fd, TIOCMGET, &ups->sp_flags);
	    sleep(1);
	    switch(ups->cable.type) {
	    case CUSTOM_SIMPLE:
		ups->OnBatt = (ups->sp_flags & cd_bit);
		break;
	    case APC_940_0119A:
	    case APC_940_0127A:
	    case APC_940_0128A:
	    case APC_940_0020B:
	    case APC_940_0020C:
		ups->OnBatt = (ups->sp_flags & cts_bit);
		break;
	    case APC_940_0023A:
		ups->OnBatt = (ups->sp_flags & cd_bit);
		break;
	    case APC_940_0095A:
	    case APC_940_0095B:
	    case APC_940_0095C:
		ups->OnBatt = (ups->sp_flags & rng_bit);
		break;
	    case CUSTOM_SMART:
	    case APC_940_0024B:
	    case APC_940_0024C:
	    case APC_940_1524C:
	    case APC_940_0024G:
	    case APC_NET:
		break;
	    default:
		ups->OnBatt = 1;
		break;
	    }
	}
    }
    /*
     *	****FIXME*****
     * We really need to find out if we are on batteries
     * and if not, delete the PWRFAIL file.  Note, the code
     * above only tests OnBatt for dumb UPSes.
     */
    if ((((pwdf = fopen(PWRFAIL, "r" )) == NULL) &&
	    (ups->mode.type != BK)) ||
            (((pwdf = fopen(PWRFAIL, "r" )) == NULL) &&
	    (ups->OnBatt != 0) && (ups->mode.type == BK))) {
	/*						  
	 * At this point, we really should not be attempting
	 * a kill power since either the powerfail file is
	 * not defined, or we are not on batteries.
	 */
	if (pwdf) {
	   fclose(pwdf);	      /* close the file if openned */
	}
	/*
	 * Now complain 
	 */
	log_event(ups, LOG_WARNING,
                 _("Cannot find " PWRFAIL " file.\n Killpower requested in \
non power fail condition or bug.\n Killpower request ignored at %s:%d\n"),
		 __FILE__, __LINE__);
	Error_abort2(
                _("Cannot find " PWRFAIL " file.\n Killpower requested in \
non power fail condition or bug.\n Killpower request ignored at %s:%d\n"),
		 __FILE__, __LINE__);
    } else {
	errflag=0;			       /* w.p. */
	if ((ups->class.type == SHAREMASTER) ||
	    (ups->class.type == SHARENETMASTER) ||
	    (ups->class.type == NETMASTER)) {
	    log_event(ups, LOG_WARNING,
                      _("Waiting 30 seconds for slave(s) to shutdown."));
	    sleep(30);
	}
	if (pwdf) {
	   fclose(pwdf);
	}

        log_event(ups, LOG_WARNING,_("Attempting to kill the power!"));

	if (ups->mode.type == BK) {
	    switch(ups->cable.type) {
	    case CUSTOM_SIMPLE: 	  /* killpwr_bit */
	    case APC_940_0095A:
	    case APC_940_0095B:
	    case APC_940_0095C: 	 /* killpwr_bit */
		(void) ioctl(ups->fd, TIOCMBIS, &rts_bit);
		(void) ioctl(ups->fd, TIOCMBIS, &rts_bit);
		(void) ioctl(ups->fd, TIOCMBIS, &rts_bit);
		(void) ioctl(ups->fd, TIOCMBIS, &st_bit);
		break;
	    case APC_940_0119A:
	    case APC_940_0127A:
	    case APC_940_0128A:
	    case APC_940_0020B: 	  /* killpwr_bit */
	    case APC_940_0020C:
		(void) ioctl(ups->fd, TIOCMBIS, &dtr_bit);
		(void) ioctl(ups->fd, TIOCMBIS, &dtr_bit);
		(void) ioctl(ups->fd, TIOCMBIS, &dtr_bit);
		break;
	    case APC_940_0023A: 	 /* killpwr_bit */
		break;
	    case CUSTOM_SMART:
	    case APC_940_0024B:
	    case APC_940_0024C:
	    case APC_940_1524C:
	    case APC_940_0024G:
	    case APC_NET:
	    default:
		break;
	    }
	    sleep(10);		      /* leave the bit set for at least 4 seconds */
	    close(ups->fd);
            log_event(ups, LOG_WARNING, "UPS will power off after the configured delay  ...\n");
            log_event(ups, LOG_WARNING,_("Please power off your UPS before rebooting your computer ...\n"));
	    return;
	} else if (ups->mode.type == SHAREBASIC) {
	    sleep(10);
	    close(ups->fd);
            log_event(ups, LOG_WARNING,_("Waiting For ShareUPS Master to shutdown"));
	    sleep(120);
            log_event(ups, LOG_WARNING, _("Failed to have power killed by Master!"));
            log_event(ups, LOG_WARNING,_("Please power off your UPS before rebooting your computer ...\n"));
	    return;
	} else {
	    a = ups->UPS_Cmd[CI_DSHUTD]; /* shutdown delay */
	    write(ups->fd, &a, 1);
	    getline(response, sizeof(response), ups);
	    shutdown_delay = atof(response);
            a = 'S';      /* ask for soft shutdown */
	    write(ups->fd, &a, 1);
	    /*	Check whether the UPS has acknowledged the power-off command.
	     *	This might not happen in rare cases, when mains-power returns
	     *	just after LINUX starts with the shutdown sequence. 
             *  If the UPS doesn't acknowledge the command, it will not
	     *	interrupt the ouput-power. So LINUX will not come up without
	     *	operator intervention.	w.p.
	     */
	    sleep(10);
	    getline(response, sizeof response, ups);
            if (strcmp(response, "OK") == 0) {
		if (shutdown_delay > 0)
                   log_event(ups, LOG_WARNING, "UPS will power off after %d seconds ...\n", shutdown_delay);
		else
                   log_event(ups, LOG_WARNING, "UPS will power off after the configured delay  ...\n");
                log_event(ups, LOG_WARNING,_("Please power off your UPS before rebooting your computer ...\n"));
            } else if (strcmp(response,"NA") == 0) {
		/*
		 *  w.p. experiments shows that UPS needs
		 *  w.p. delays between chars to accept
		 *  w.p. this command
		 */
                a = '@';      /* Shutdown now */
		write(ups->fd, &a, 1);
		sleep(1);
                a = '0';
		write(ups->fd, &a, 1); 
		sleep(1);
                a = '0';
		write(ups->fd, &a, 1);
		sleep(1); 
                a = '0';
		write(ups->fd, &a, 1);
		sleep(2);
		getline(response, sizeof response, ups);
                if ((strcmp(response, "OK") == 0) || (strcmp(response,"*") == 0)) {
		    if (shutdown_delay > 0)
                       log_event(ups, LOG_WARNING, "UPS will power off after %d seconds ...\n", shutdown_delay);
		    else
                       log_event(ups, LOG_WARNING, "UPS will power off after the configured delay  ...\n");
                    log_event(ups, LOG_WARNING,_("Please power off your UPS before rebooting your computer ...\n"));
		} else
		    errflag++;
	    } else {  /* neither OK nor NA, try alternate way */
                a = '@';
		write(ups->fd, &a, 1);
		sleep(1);
                a = '0';
		write(ups->fd, &a, 1);
		sleep(1);
                a = '0';
		write(ups->fd, &a, 1);
		sleep(1);
		if ((ups->mode.type == BKPRO) || (ups->mode.type == VS)) {
                    a = '1';
                    log_event(ups, LOG_WARNING,_("BackUPS Pro and SmartUPS v/s sleep for 6 min"));
		} else {
                    a = '0';
		}
		write(ups->fd, &a, 1);
		sleep(2);
		/* And yet another method !!! */
                a = 'K';
		write(ups->fd, &a, 1);
		sleep(2);
		write(ups->fd, &a, 1);
		getline(response, sizeof response, ups);
                if ((strcmp(response,"*") == 0) || (strcmp(response,"OK") == 0) ||
		    (ups->mode.type >= BKPRO)) {
			if (shutdown_delay > 0)
                           log_event(ups, LOG_WARNING, "UPS will power off after %d seconds ...\n", shutdown_delay);
			else
                           log_event(ups, LOG_WARNING, "UPS will power off after the configured delay  ...\n");
                        log_event(ups, LOG_WARNING,_("Please power off your UPS before rebooting your computer ...\n"));
		} else
			errflag++;
	    }
	    if (errflag) {
                log_event(ups, LOG_WARNING,_("Unexpected error!\n"));
                log_event(ups, LOG_WARNING,_("UPS in unstable state\n"));
                log_event(ups, LOG_WARNING,_("You MUST power off your UPS before rebooting!!!\n"));
	    }
	}
    }
    return;
}


/* 
 * 
 * For older models, we must do the ioctls then do them
 * a second time (here) to properly determine the status
 * of the UPS.
 * This code only applies to the non-Smart UPS models.
 *
 * According to Andre Hedrick:
 *
 * Since the "Simple Signal" beasts only can tell you of events at that
 * instant, you have to buffer the state changes.  The old status allows one
 * to verify that you have a true extended state change and thus you make a
 * decission.
 *
 */
void save_dumb_status(UPSINFO *ups)
{
    if (ups->mode.type <= SHAREBASIC) {
	switch(ups->cable.type) {
	case CUSTOM_SIMPLE:
	    (void) ioctl(ups->fd, TIOCMBIC, &dtr_bit);	/* power_bit */
	    (void) ioctl(ups->fd, TIOCMBIC, &rts_bit);	/* killpwr_bit */
	    break;
	case APC_940_0119A:
	case APC_940_0127A:
	case APC_940_0128A:
	case APC_940_0020B:
	case APC_940_0020C:
	    (void) ioctl(ups->fd, TIOCMBIC, &dtr_bit);	/* killpwr_bit */
	    break;
	case APC_940_0023A:
	    break;
	case APC_940_0095A:
	case APC_940_0095B:
	case APC_940_0095C:
	    (void) ioctl(ups->fd, TIOCMBIC, &rts_bit);	/* killpwr_bit */
	    (void) ioctl(ups->fd, TIOCMBIC, &cd_bit);	/* lowbatt_bit */
	    (void) ioctl(ups->fd, TIOCMBIC, &rts_bit);	/* killpwr_bit */
	    break;
	case CUSTOM_SMART:
	case APC_940_0024B:
	case APC_940_0024C:
	case APC_940_1524C:
	case APC_940_0024G:
	case APC_NET:
	default:
            log_event(ups, LOG_WARNING,_("Simple Cable Signal Lost!!!"));
	    terminate(0);
	}
    }
}

/********************************************************************* 
 * Check if the UPS has changed state
 *
 */
int check_serial(UPSINFO *ups)
{
    int stat = SUCCESS;

    if (ups->mode.type <= SHAREBASIC) {

	/*
	 * With ioctl we have no way to sleep like with select()
	 * so we _need_ to sleep explicitly.
	 */
	sleep(TIMER_SERIAL);

	read_andlock_shmarea(ups);
	ioctl(ups->fd, TIOCMGET, &ups->sp_flags);
	switch(ups->cable.type) {
	case CUSTOM_SIMPLE:
	    ups->OnBatt   = !!(ups->sp_flags & cd_bit);
	    ups->BattLow  = !(ups->sp_flags & cts_bit);
	    ups->LineDown = !!(ups->sp_flags & sr_bit);
	    break;
	case APC_940_0119A:
	case APC_940_0127A:
	case APC_940_0128A:
	case APC_940_0020B:
	case APC_940_0020C:
	    ups->OnBatt   = !!(ups->sp_flags & cts_bit);
	    ups->BattLow  = !!(ups->sp_flags & cd_bit);
	    ups->LineDown = 0;
	    break;
	case APC_940_0023A:
	    ups->OnBatt   = !!(ups->sp_flags & cd_bit);

	    /* BOGUS STUFF MUST FIX IF POSSIBLE */

	    ups->BattLow  = !!(ups->sp_flags & sr_bit);

	    /* BOGUS STUFF MUST FIX IF POSSIBLE */

	    ups->LineDown = 0;
	    break;
	case APC_940_0095A:
	case APC_940_0095C:
	    ups->OnBatt   = !!(ups->sp_flags & rng_bit);
	    ups->BattLow  = !!(ups->sp_flags & cd_bit);
	    ups->LineDown = 0;
	    break;
	case APC_940_0095B:
	    ups->OnBatt   = !!(ups->sp_flags & rng_bit);
	    ups->LineDown = 0;
	    break;
	case CUSTOM_SMART:
	case APC_940_0024B:
	case APC_940_0024C:
	case APC_940_1524C:
	case APC_940_0024G:
	case APC_NET:
	default:
	    stat = FAILURE;
	}
	write_andunlock_shmarea(ups);
    } else {
	/* Smart UPS. Information is returned as characters, so   
	* read the serial port for any changes. 
	*/
	stat = getline(NULL, 0, ups);
    }
    return stat;
}

/*********************************************************************/
void prep_serial(UPSINFO *ups)
{
    if (ups->mode.type > SHAREBASIC) {
       read_extended(ups);
    }
    /* If no UPS name found, use hostname, or "default" */
    if (ups->upsname[0] == 0) {       /* no name given */
	gethostname(ups->upsname, sizeof(ups->upsname)-1);
	if (ups->upsname[0] == 0) {   /* error */
            strcpy(ups->upsname, "default");
	}
    }
}

/********************************************************************  
 *
 * NOTE! This is the starting point for a separate process (thread).
 * 
 */
void do_serial(UPSINFO *ups)
{
    init_thread_signals();   

#ifdef HAVE_CYGWIN     /* needed for NT */
    attach_ipc(ups, 0);
#endif

    while (1) {
	/*
	 * Check serial port to see if the UPS has changed state.
	 *    This routine waits a reasonable time to prevent
	 *    consuming too much CPU time.
	 */
	check_serial(ups);

	do_action(ups, 1);    /* take event actions */
	
	/* Get all info available from UPS by asking it questions */
	fillUPS(ups);

	do_reports(ups);
    }
}
