//============================================================================
// Name        : acngfs.cpp
// Author      : Eduard Bloch
// Description : Simple FUSE-based filesystem for HTTP access (apt-cacher NG)
//============================================================================


#define LOCAL_DEBUG
#include "debug.h"

#include "meta.h"
#include "header.h"
#include "caddrinfo.h"
#include "sockio.h"
#include "acbuf.h"
#include "acfg.h"
#include "lockable.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/vfs.h>

#include <unistd.h>
#include <inttypes.h>
#include <stdint.h>
#include <pthread.h>
#include <errno.h>
#include <signal.h>

#include <cstdio>
#include <algorithm>
#include <iostream>
#include <list>

#define FUSE_USE_VERSION 25
#include <fuse.h>


#define HEADSZ 5000
#ifndef MIN
#define MIN(a,b) ( (a<=b)?a:b)
#endif

#define _cerr(x) cerr << x

#define POOLMAXSIZE 20 // max size
#define POOLMAXAGE 50 // seconds

using namespace std;

// keep reused code quiet
namespace aclog
{
void err(const char*, const char*)
{}
}

// some globals, set only once
static struct stat statTempl;
static struct statfs stfsTemp;
static tHttpUrl baseUrl, proxyUrl;

lockable mx;
list< pair<time_t, int> > fdpool;

struct tDlDesc
{
private:
	header h;
	int fd;
public:
	off_t seen_length;
	int seen_status;
	tDlDesc(): fd(-1), seen_length(0), seen_status(0)
	{};
	~tDlDesc() {
		if(fd>=0)
		{
			lockguard g(mx);
			fdpool.push_back(pair<time_t,int>(time(0),fd));
		}
	};
	
	acbuf databuf;
	
	
	bool CheckOrExplore(const char *path)
	{
		static lockable mxCache;
		static map<string, int> cache;
		{
			lockguard g(mxCache);
			map<string,int>::iterator it=cache.find(path);
			if(it!=cache.end())
			{
				seen_length=it->second;
				seen_status=200;
				_cerr("Using precached\n");
				return true;
			}
		}
		// ok, not cached, do the hard way
		bool res=Act(path, 0, 1, true);
		_cerr("Act done, result: "<<res<<endl);
		if(!res)
			return res;
		rechecks::eFileKind type=rechecks::GetFiletype(path);
		_cerr ( "type: " << type );
		if (type!=rechecks::FILE_INDEX) // not caching volatile stuff
		{
			lockguard g(mxCache);
			cache[path]=seen_length;
		}
		return true;
	}
	
	bool Act(const char *path, off_t pos, off_t len, bool bJustProbe=false)
	{
		_cerr( path << ", from: " << pos << " , " << len << "bytes\n");
		databuf.clear();
		h.clear();
		
		if(bJustProbe)
			len=100; // fake something more sensible for now
		
		if(!_Connect(false))
			return false;
		
		string req(len==0?"HEAD ":"GET ");
		
		_cerr( "reqtype: " << req);
		req+=baseUrl.ToURI()+path+" HTTP/1.1\r\nConnection: Keep-Alive\r\nUser-Agent: ACNGFS\r\nX-Original-Source: 42\r\n";
		if(!bJustProbe && len>0)
		{
			char buf[100];
			sprintf(buf, "Range: bytes=%lu-%lu\r\n", (long unsigned)pos, (long unsigned) (pos+len-1));
			req+=buf;
		}
		req+="\r\n";
		_cerr( "requesting: " << req );
		
		for(bool bRetried=false;!req.empty();)
		{
			int n=::send(fd, req.c_str(), req.size(), MSG_NOSIGNAL);
			if(n<0)
			{
				if(EAGAIN==errno)
					continue;
				return false;
			}
			if(n==0)
			{
				if(EINTR==errno)
					continue;
				
				// old connection timed out? reconnect, but only once
				
				if(bRetried)
					return false;
				
				bRetried=true;
				_ShutDown();
				
				if(!_Connect(true))
					return false;
				
				continue;
			}
			req.erase(0, n);
		}
		
		//_cerr( "init buf for bytes: " << len);
		databuf.init(len+HEADSZ);
		_cerr( "capa: " << databuf.freecapa());
		int64_t remaining(len+HEADSZ); // to be adapted when head arrived
		while(true)
		{
			_cerr( "remaining: " << remaining <<endl);
			if(remaining<=0)
				return true;
			
			// drop all stuff read during precaching
			if(bJustProbe && h.type!=h.INVALID)
				databuf.clear();
			
			int nToRead=MIN(remaining,databuf.freecapa());
      //assert(nToRead>0);
			int n=::recv(fd, databuf.wptr(), nToRead, 0);
			_cerr( "got: " << n << " errno: " << errno);
			if(n<=0)
			{ // no other toleratable errors should appear here
				if(EINTR==errno || EAGAIN==errno)
					continue;
				//perror("ERROR: got <=0 from read: ");
				_ShutDown();
				return false;
			}
			databuf.got(n);
			remaining-=n;
			
			if(h.type==h.INVALID)
			{
				_cerr( "parsing head, free? " << databuf.freecapa()<<endl );
				
				int r=h.LoadFromBuf(databuf.rptr(), databuf.size());
				_cerr( "bufptr: " << (size_t) databuf.rptr() << " result: " << r <<endl);
				cerr.flush();

#ifdef DEBUG
				if(r>0)
					write(fileno(stderr), databuf.rptr(), r);
#endif
				
				if(r>0)
					databuf.drop(r);
				
				if(0==r)
				{
					if(databuf.freecapa()<=0)
						return false; // monster head?
					continue; // read more
				}
				// catch bad or weird stuff
				if(r<0 || r>=HEADSZ || h.type==h.INVALID)
					return false;
				
				_cerr( "skipped head, " << r << " bytes,  now size: " << databuf.size() <<" rptr: " << size_t(databuf.rptr()) );
				// got head so far, process data
				
				if(! h.h[header::CONTENT_LENGTH] || ! *h.h[header::CONTENT_LENGTH])
					return false;
				long contlen=atol(h.h[header::CONTENT_LENGTH]);
				remaining = contlen - databuf.size();
				
				seen_status=h.getStatus();
				_cerr("HTTP status: " << seen_status <<endl);
				if(200==seen_status)
					seen_length=contlen;
				
				if(len==0) // just HEAD
					return true;
			}
			
		}
		return true;
	};
	inline void _ShutDown()
	{
		_cerr("Disconnecting...\n");
		if (fd>=0)
		{
			shutdown(fd, O_RDWR);
			close(fd);
		}
		fd=-1;
	};
	bool _Connect(bool bForceFresh)
	{
		if(fd>=0)
			return true;
		
		if(!bForceFresh)
		{
			lockguard g(mx);
			// also do some pool maintenance... too large or too old? nuke it
			if(fdpool.size()>POOLMAXSIZE || fdpool.back().first < (time(0)-POOLMAXAGE))
			{
				while(!fdpool.empty())
				{
					int zombie=fdpool.front().second;
					::shutdown(zombie, O_RDWR);
					::close(zombie);
					fdpool.pop_front();
				}
			}
			if(!fdpool.empty())
			{
				fd=fdpool.back().second;
				fdpool.pop_back();
				return true;
			}
		}
		
		signal(SIGPIPE, SIG_IGN);
		string sErr;
    CAddrInfo::SPtr dns=CAddrInfo::CachedResolve(proxyUrl.sHost, proxyUrl.sPort, sErr);
		if(!dns)
		{
			sErr="500 Cannot resolve remote address.";
			return false;
		}
		struct addrinfo *pInfo(dns->m_bestInfo);
		fd = socket(pInfo->ai_family, pInfo->ai_socktype, pInfo->ai_protocol);
		if(fd<0)
			return false;

#ifndef NO_TCP_TUNNING
		int dummy(1);
		setsockopt(fd,SOL_SOCKET,SO_REUSEADDR,&dummy,sizeof(dummy));
		setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, &dummy, sizeof(dummy));
#endif
		int r=connect(fd, pInfo->ai_addr, pInfo->ai_addrlen);
		if (r<0)
		{
			sErr="500 Connection failure";
			return false;
		}
		return true;
	}

};

bool GetFstat(const char *path, struct stat *stbuf, tDlDesc *d=NULL)
{
	memcpy(stbuf, &statTempl, sizeof(statTempl));
	stbuf->st_mode &= ~S_IFMT; // delete mode flags and set them as needed

	// assume that everything downloadable is a file and a directory is 404

	rechecks::eFileKind type=rechecks::GetFiletype(path);
	_cerr ( "type: " << type );
	if (type==rechecks::FILE_PKG || type==rechecks::FILE_INDEX)
	{
		//ldbg("Is a file!");
		stbuf->st_mode |= S_IFREG;
		stbuf->st_size=0;

		tDlDesc e;
		if (!d)
			d=&e;

		if (!d->CheckOrExplore(path))
			return false;
		if (d->seen_status!=200)
			return false;
		stbuf->st_size=d->seen_length;
	}
	else
	{
		//ldbg("Is a directory!");
		stbuf->st_mode |= S_IFDIR;
		stbuf->st_size=4;
	}
	return true;
	
}


/// If found as downloadable, present as a file, or become a directory otherwise.
static int acngfs_getattr(const char *path, struct stat *stbuf)
{
	if(!GetFstat(path, stbuf))
		return -EIO;
	
	return 0;
}

static int acngfs_fgetattr(const char *path, struct stat *stbuf,
      struct fuse_file_info *fi)
{
	// FIXME: reuse the con later? or better not, size might change during operation
	return acngfs_getattr(path, stbuf);
}

static int acngfs_access(const char *path, int mask)
{
	// non-zero (failure) when trying to write
   return mask&W_OK;
}

static int acngfs_readlink(const char *path, char *buf, size_t size)
{
   return -EINVAL;
}

static int acngfs_opendir(const char *path, struct fuse_file_info *fi)
{
	// let FUSE manage directories
	return 0;
}

static int acngfs_readdir(const char *path, void *buf, fuse_fill_dir_t filler,
      off_t offset, struct fuse_file_info *fi)
{   
	return -EPERM;
}

static int acngfs_releasedir(const char *path, struct fuse_file_info *fi)
{

   return 0;
}

static int acngfs_mknod(const char *path, mode_t mode, dev_t rdev)
{
	return -EROFS;
}

static int acngfs_mkdir(const char *path, mode_t mode)
{
   return -EROFS;
}

static int acngfs_unlink(const char *path)
{
   return -EROFS;
}

static int acngfs_rmdir(const char *path)
{
   return -EROFS;
}

static int acngfs_symlink(const char *from, const char *to)
{
  return -EROFS;
}

static int acngfs_rename(const char *from, const char *to)
{
  return -EROFS;
}

static int acngfs_link(const char *from, const char *to)
{
	return -EROFS;
}

static int acngfs_chmod(const char *path, mode_t mode)
{
   return -EROFS;
}

static int acngfs_chown(const char *path, uid_t uid, gid_t gid)
{
return -EROFS;
}

static int acngfs_truncate(const char *path, off_t size)
{
   return -EROFS;
}

static int acngfs_ftruncate(const char *path, off_t size,
      struct fuse_file_info *fi)
{
return -EROFS;
}

static int acngfs_utime(const char *path, struct utimbuf *buf)
{
 return -EROFS;
}

//lockable mxTest;

static int acngfs_open(const char *path, struct fuse_file_info *fi)
{
	//lockguard g(&mxTest);
	
	if (fi->flags & (O_WRONLY|O_RDWR|O_TRUNC|O_CREAT))
			return -EROFS;

	tDlDesc *p(NULL);
	MYTRY
	{
		p=new tDlDesc;
		if(!p) // running exception-free? 
			return -EIO;	
	}
	MYCATCH(...)
	{
		return -EIO;
	}
	
	if(!p->CheckOrExplore(path) ||  p->seen_status !=200)
	{
		delete p;
		return -EISDIR;
	}

	/*
	long size = p->h.h[header::CONTENT_LENGTH] ? atol(p->h.h[header::CONTENT_LENGTH]) : 0;  
	if(size) // precache
		p->Act(path, size-1, 1);
	*/
	
	fi->fh = (uintptr_t) p;
	return 0;
}


static int acngfs_read(const char *path, char *buf, size_t size, off_t offset,
      struct fuse_file_info *fi)
{
	tDlDesc *p=(tDlDesc*) fi->fh;
	_cerr( offset << ":"<<size<<":"<<p->seen_length);
	if( off_t(offset+size) > p->seen_length)
		size=p->seen_length-offset;
		
	if(!p->Act(path, offset, size))
		return -EIO;
	
	int s=p->seen_status;
	_cerr( "read, status: " << s);
	//if( ( 200==s && 0==offset) || (206==s && 0!=offset))

	// 200 is possible if file is smaller than the first range
	if(206==s || (200==s && 0==offset && off_t(size) < p->seen_length))
	{
		int len=MIN(size, p->databuf.size());
		_cerr( "ret, len: " << len );
		memcpy(buf, p->databuf.rptr(), len);
		return len;
	}
	else
		return -EIO; 
}

static int acngfs_write(const char *path, const char *buf, size_t size,
      off_t offset, struct fuse_file_info *fi)
{
	return -EBADF;
}

static int acngfs_statfs(const char *path, struct statvfs *stbuf)
{
   memcpy(stbuf, &stfsTemp, sizeof(stbuf));
	return 0;
}

static int acngfs_release(const char *path, struct fuse_file_info *fi)
{
	if(fi->fh)
		delete (tDlDesc*)fi->fh;
	return 0;
}

static int acngfs_fsync(const char *path, int isdatasync,
      struct fuse_file_info *fi)
{
   return 0;
}


struct fuse_operations acngfs_oper;

void _ExitUsage() {
   cerr << "USAGE: acngfs BaseURL ProxyHost MountPoint [FUSE Mount Options]\n"
   << "example: acngfs http://ftp.uni-kl.de/debian localhost:3142 /var/local/aptfs\n"
        << "valid FUSE Mount Options follow:\n";
    const char *argv[] = {"...", "-h"};
    fuse_main( 2, const_cast<char**>(argv), &acngfs_oper);
    exit(EXIT_FAILURE);
}

#define barf(x) { cerr << endl << "ERROR: " << x <<endl; exit(1); }
#define erUsage { _ExitUsage(); }

int main(int argc, char *argv[])
{

   memset(&acngfs_oper, 0, sizeof(acngfs_oper));

   acngfs_oper.getattr	= acngfs_getattr;
   acngfs_oper.fgetattr	= acngfs_fgetattr;
   acngfs_oper.access	= acngfs_access;
   acngfs_oper.readlink	= acngfs_readlink;
   acngfs_oper.opendir	= acngfs_opendir;
   acngfs_oper.readdir	= acngfs_readdir;
   acngfs_oper.releasedir	= acngfs_releasedir;
   acngfs_oper.mknod	= acngfs_mknod;
   acngfs_oper.mkdir	= acngfs_mkdir;
   acngfs_oper.symlink	= acngfs_symlink;
   acngfs_oper.unlink	= acngfs_unlink;
   acngfs_oper.rmdir	= acngfs_rmdir;
   acngfs_oper.rename	= acngfs_rename;
   acngfs_oper.link	= acngfs_link;
   acngfs_oper.chmod	= acngfs_chmod;
   acngfs_oper.chown	= acngfs_chown;
   acngfs_oper.truncate	= acngfs_truncate;
   acngfs_oper.ftruncate	= acngfs_ftruncate;
   acngfs_oper.utime	= acngfs_utime;
//   acngfs_oper.create	= acngfs_create;
   acngfs_oper.open	= acngfs_open;
   acngfs_oper.read	= acngfs_read;
   acngfs_oper.write	= acngfs_write;
   acngfs_oper.statfs	= acngfs_statfs;
   acngfs_oper.release	= acngfs_release;
   acngfs_oper.fsync	= acngfs_fsync;

   umask(0);

   for(int i = 1; i<argc; i++)
   	   if(argv[i] && 0==strcmp(argv[i], "--help"))
   		   erUsage;
   
   if(argc<4)
      barf("Not enough arguments, try --help.\n");
   
   if(argv[1] && baseUrl.SetHttpUrl(argv[1]))
   {
#ifdef VERBOSE
	   cout << "Base URL: " << baseUrl.ToString()<<endl;
#endif
   }
   else
   {
      cerr << "Invalid base URL, " << argv[1] <<endl;
      exit(EXIT_FAILURE);
   }
   // FUSE adds starting / already, drop ours if present
   trimBack(baseUrl.sPath, "/");
   
   if(argv[2] && proxyUrl.SetHttpUrl(argv[2]))
   {
	   if(proxyUrl.sPort.empty())
		   proxyUrl.sPort="3142";
   }
   else
   {
	   cerr << "Invalid proxy URL, " << argv[2] <<endl;
	   exit(EXIT_FAILURE);
   }
   
   
   if(stat(argv[3], &statTempl) || statfs(argv[3], &stfsTemp))
	   barf(endl << "Cannot access " << argv[3]);
   if(!S_ISDIR(statTempl.st_mode))
      barf(endl<< argv[3] << " is not a directory.");


   // skip our arguments, keep those for fuse
   argv[2]=argv[0]; // application path
   argv=&argv[2];
   argc-=2;
   return fuse_main(argc, argv, &acngfs_oper);
}

