
#define LOCAL_DEBUG
#include "debug.h"

#include "job.h"
#include <cstdio>
#include <stdexcept>
#include <limits>
using namespace MYSTD;

#include "conn.h"
#include "acfg.h"
#include "fileitem.h"
#include "dlcon.h"
#include "sockio.h"

#include "maintenance.h"

#ifdef HAVE_LINUX_SENDFILE
#include <sys/sendfile.h>
#endif

#include <errno.h>

#define REPLEVEL (m_type==rechecks::FILE_INDEX ? 4 : (m_type==rechecks::FILE_PKG ? 5 : SPAMLEVEL))

//void DispatchAndRunMaintTask(const MYSTD::string &, int, const MYSTD::string &);

class tPassThroughFitem : public fileitem
{
protected:

	const char *m_pData;
	size_t m_nConsumable, m_nConsumed;

public:
	tPassThroughFitem(MYSTD::string s) : fileitem(s),
	m_pData(NULL), m_nConsumable(0), m_nConsumed(0)
	{
		m_bAllowStoreData=false;
	};
	virtual FiStatus Setup(bool, bool bIgnoreCached=false)
	{
		return status = FIST_INITED;
	}
	virtual void Unreg() {}; // NOOP, since this item is not shared
	virtual int GetFileFd() { return 1; }; // something, don't care
	virtual bool StoreFileData(const char *data, unsigned int size)
	{
		setLockGuard;

		// something might care, most likely... also about BOUNCE action
		notifyAll();

		m_nIncommingCount += size;

		if (status >= FIST_ERROR || status < FIST_DLGOTHEAD)
			return false;

		if (size == 0)
		{
			status = FIST_COMPLETE;
			// we are done! Fix header from chunked transfers?
			if (m_filefd >= 0 && !m_head.h[header::CONTENT_LENGTH])
				m_head.set(header::CONTENT_LENGTH, m_nSizeChecked);
		}
		else
		{
			status = FIST_DLRECEIVING;
			m_nSizeChecked += size;
			m_pData = data;
			m_nConsumable=size;
			m_nConsumed=0;
			while(0 == m_nConsumed && status<FIST_ERROR)
				wait();

			// let the downloader abort?
			if(status >= FIST_ERROR)
				return false;

			m_nConsumable=0;
			m_pData=NULL;
			return m_nConsumed;
		}
		return true;
	}
	ssize_t SendData(int out_fd, int, off_t &nSendPos, size_t nMax2SendNow)
	{
		setLockGuard;
		while(0 == m_nConsumable && status<FIST_ERROR)
			wait();
		if (status >= FIST_ERROR || !m_pData)
			return -1;

		ssize_t r = write(out_fd, m_pData, min(nMax2SendNow, m_nConsumable));
		if (r < 0 && (errno == EAGAIN || errno == EINTR)) // harmless
			r = 0;
		if(r<0)
		{
			status=FIST_ERROR;
		}
		else if(r>0)
		{
			m_nConsumable-=r;
			m_pData+=r;
			m_nConsumed+=r;
			nSendPos+=r;
		}
		notify();
		return r;
	}
};

ssize_t sendfile_generic(int out_fd, int in_fd, off_t *offset, size_t count)
{
	LOGSTART("sendfile_generic (fallback)");
	char buf[4096];
	ssize_t totalcnt=0;
	
	if(!offset)
	{
		errno=EFAULT;
		return -1;
	}
	if(lseek(in_fd, *offset, SEEK_SET)== (off_t)-1)
		return -1;
	while(count>0)
	{
		ssize_t readcount=read(in_fd, buf, count>sizeof(buf)?sizeof(buf):count);
		if(readcount<=0)
		{
			if(errno==EINTR || errno==EAGAIN)
				continue;
			else
				return readcount;
		}
		
		*offset+=readcount;
		totalcnt+=readcount;
		count-=readcount;
		
		ssize_t nPos(0);
		while(nPos<readcount)
		{
			ssize_t r=write(out_fd, buf+nPos, readcount-nPos);
			if(r==0) continue; // not nice but needs to deliver it
			if(r<0)
			{
				if(errno==EAGAIN || errno==EINTR)
					continue;
				else
					return r;
			}
			nPos+=r;
		}
	}
	return totalcnt;
}

ssize_t fileitem::SendData(int out_fd, int in_fd, off_t &nSendPos, size_t count)
{
#ifndef HAVE_LINUX_SENDFILE
	return sendfile_generic(out_fd, in_fd, &nSendPos, count);
#else
	ssize_t r=sendfile(out_fd, in_fd, &nSendPos, count);

	if(r<0 && (errno==ENOSYS || errno==EINVAL))
		return sendfile_generic(out_fd, in_fd, &nSendPos, count);
	else
		return r;
#endif
}


job::job(header *h, con *pParent) :
	m_filefd(-1),
	m_pParentCon(pParent),
	m_bChunkMode(false),
	//m_bCloseCon(false),
	m_bIsHttp11(false),
	m_state(STATE_FRESH),
	m_pReqHead(h),
	m_nSendPos(0),
	m_nRangeLast(OFFT_MAX-1), // range size should never overflow
	m_nChunkRemainingBytes(0)
{
	LOGSTART2("job::job", "job creating, " << m_pReqHead->frontLine << " and this: " << this);
	m_nAllDataCount=0;
}

job::~job()
{
	LOGSTART("job::~job");
	m_pParentCon->LogDataCounts(m_sFileLoc,
			m_pReqHead->h[header::XFORWARDEDFOR],
			(m_pItem ? m_pItem->GetTransferCount() : 0),
			m_nAllDataCount);
	
	checkforceclose(m_filefd);

	if (m_pItem)
	{
		m_pItem->Unreg();
		m_pItem.reset();
	}
	
	if(m_pReqHead)
	{
		delete m_pReqHead;
		m_pReqHead=NULL;
	}
}

void replaceChars(string &s, const char *szBadChars, char goodChar)
{
	for(string::iterator p=s.begin();p!=s.end();p++)
		for(const char *b=szBadChars;*b;b++)
			if(*b==*p)
			{
				*p=goodChar;
				break;
			}
}

void job::PrepareDownload() {

    LOGSTART("job::PrepareDownload");
    
    string sRawUriPath, sPathResdiual;
    tHttpUrl tUrl; // parsed URL
        
    acfg::tHostiVec * pBackends(NULL); // appropriate backends
    const string * psVname(NULL); // virtual name for storage pool, if available
    
    FiStatus fistate(FIST_FRESH);
    bool bPtMode(false);
    bool bForceFreshnessChecks(false); // force update of the file, i.e. on dynamic index files?

    if(!m_sErrorMsgBuffer.empty())
       	return ;
    
    if(m_pReqHead->type!=header::GET && m_pReqHead->type!=header::HEAD)
    	goto report_invpath;
    
    m_bIsHttp11=(m_pReqHead->frontLine.length()>8 && 
    		0 == m_pReqHead->frontLine.compare(
					m_pReqHead->frontLine.length()-8, 8, "HTTP/1.1"));
    

	{
		/*
		// persistent connection?
	    const char *p=m_pReqHead->h[header::CONNECTION];
		if (!p) 
			m_bCloseCon = true; // enough! too many stupid clients out there; previous default: "close" IFF not http 1.1 !m_bIsHttp11; 
		else // normalize case
			m_bCloseCon = (0 == strcasecmp(p, "close"));
		*/
		
		tStrVec tmp;
		if(3 == Tokenize(m_pReqHead->frontLine, SPACECHARS, tmp))
			sRawUriPath=tmp[1];
		else
			goto report_invpath; // invalid uri
	}


    MYTRY
	{
    	LOG("raw uri: " << sRawUriPath);
    	// filesystem browsing attempt?
		if(stmiss != sRawUriPath.find("..") || stmiss != sRawUriPath.find("/_actmp"))
			goto report_notallowed;

		if (0==sRawUriPath.compare(0, 12, "apt-cacher?/"))
		sRawUriPath.erase(0, 12);
		if (0==sRawUriPath.compare(0, 11, "apt-cacher/"))
		sRawUriPath.erase(11);
		
		if(!tUrl.SetHttpUrl(sRawUriPath))
		{
			m_sMaintCmd="/";
			return;
		}
		LOG("refined path: " << tUrl.sPath);

		if(!tUrl.sPort.empty() && tUrl.sPort!="80")
		{
			//USRDBG(3, "illegal port " << tUrl.sPort << "?" << tUrl.sHost << tUrl.sPath);
			if(acfg::port == tUrl.sPort)
				goto report_doubleproxy;

			goto report_invport;
		}

		// make a shortcut
		string & sPath=tUrl.sPath;

		// kill multiple slashes
		for(tStrPos pos; stmiss != (pos = sPath.find("//")); )
			sPath.erase(pos, 1);

		bPtMode=rechecks::MatchUncacheableRequest(tUrl.ToURI());

		LOG("input uri: "<<tUrl.ToURI()<<" , dontcache-flag? " << bPtMode);

		tStrPos nRepNameLen=acfg::reportpage.size();
		if(nRepNameLen>0)
		{
			if(0==tUrl.sHost.compare(0, nRepNameLen, acfg::reportpage))
			{
				m_sMaintCmd=tUrl.sHost;
				return;
			}
			if (tUrl.sHost == "style.css")
			{
				LOG("support CSS style file");
				m_sMaintCmd = "/style.css";
				return;
			}
		}
		if(!sPath.empty() && endsWithSzAr(sPath, "/"))
		{
			LOG("generic user information page");
			m_sMaintCmd="/";
			return;
		}


		m_type = rechecks::GetFiletype(sPath);

		if ( m_type == rechecks::FILE_INVALID ) goto report_notallowed;
		
		// got something valid, has type now, trace it
		USRDBG(REPLEVEL, string("Processing new job, ")+m_pReqHead->frontLine);

		// resolve to an internal location
		psVname = acfg::GetRepNameAndPathResidual(tUrl, sPathResdiual);
		
		if(psVname)
			m_sFileLoc=*psVname+sPathSep+sPathResdiual;
		else
			m_sFileLoc=tUrl.sHost+tUrl.sPath;
		
		// FIXME: this sucks, belongs into the fileitem
		if(acfg::stupidfs)
		{
			// fix weird chars that we don't like in the filesystem
			replaceChars(tUrl.sPath, ENEMIESOFDOSFS, '_');
			replaceChars(tUrl.sHost, ENEMIESOFDOSFS, '_');
#ifdef WIN32
			replaceChars(tUrl.sPath, "/", '\\');
#endif
		}
	}
	MYCATCH(out_of_range) // better safe...
	{
    	goto report_invpath;
    }
    
    bForceFreshnessChecks = ( ! acfg::offlinemode && m_type==rechecks::FILE_INDEX);

#ifdef QUATSCH
    // some users want to not be most up-to-date
	if(bForceFreshnessChecks && acfg::updinterval > 0)
	{
		struct stat stbuf;
		bForceFreshnessChecks=(0 == ::stat(m_sFileLoc.c_str(), &stbuf)
				&& ( time(0) - (UINT) stbuf.st_ctime) < acfg::updinterval);
	}
#endif

	m_pItem=fileitem::GetFileItem(m_sFileLoc);
    
    if(!m_pItem)
    {
    	if(acfg::debug)
    		aclog::err(string("Error creating file item for ")+m_sFileLoc);
    	goto report_overload;
    }
    
    fistate = m_pItem->Setup(bForceFreshnessChecks);
	LOG("Got initial file status: " << fistate);

	if (bPtMode && fistate != FIST_COMPLETE)
		fistate = _SwitchToPtItem(m_sFileLoc);

    if(acfg::offlinemode) { // make sure there will be no problems later in SendData or prepare a user message
    	if(fistate==FIST_COMPLETE)
    		return; // perfect, done here
    	// error and/or needs download, but freshness check was disabled, so it's really not complete.
    	goto report_offlineconf;
    }
    dbgline;
    if( fistate < FIST_DLGOTHEAD) // needs a downloader 
    {
    	dbgline;
    	if(!m_pParentCon->SetupDownloader(m_pReqHead->h[header::XFORWARDEDFOR]))
    	{
    		if(acfg::debug)
    			aclog::err(string("Error creating download handler for ")+m_sFileLoc);
    		goto report_overload;
    	}
    	
    	dbgline;
    	MYTRY
		{
			if(psVname && NULL != (pBackends=acfg::GetBackendVec(psVname)))
			{
				LOG("Backends found, using them with " << sPathResdiual
						<< ", first backend: " <<pBackends->front().ToURI());

				if(! bPtMode && rechecks::MatchUncacheableTarget(pBackends->front().ToURI()+sPathResdiual))
					fistate=_SwitchToPtItem(m_sFileLoc);

				m_pParentCon->m_pDlClient->AddJob(m_pItem, pBackends, sPathResdiual);
			}
			else
			{
			    if(acfg::forcemanaged)
			    	goto report_notallowed;
			    LOG("Passing new job for " << tUrl.ToURI() << " to " << m_pParentCon->m_dlerthr);

				if(! bPtMode && rechecks::MatchUncacheableTarget(tUrl.ToURI()))
					fistate=_SwitchToPtItem(m_sFileLoc);

			    m_pParentCon->m_pDlClient->AddJob(m_pItem, tUrl);
			}
			ldbg("Download job enqueued for " << m_sFileLoc);
		}
		MYCATCH(std::bad_alloc) // OOM, may this ever happen here?
		{
			if(acfg::debug)
				aclog::err("Out of memory");			    		
			goto report_overload;
		};
	}
    
	return;
    
report_overload:
    m_sErrorMsgBuffer="503 Server overload, try later"; 
    return ;

report_notallowed:
    m_sErrorMsgBuffer="403 Forbidden file type or location";
    USRDBG(5, m_sErrorMsgBuffer+": "+sRawUriPath);
    return ;

report_offlineconf:
	m_sErrorMsgBuffer="503 Unable to download in offline mode";
	return;

report_invpath:
    m_sErrorMsgBuffer="403 Invalid path specification"; 
    return ;

report_invport:
    m_sErrorMsgBuffer="403 Invalid or prohibited port"; 
    return ;

report_doubleproxy:
    m_sErrorMsgBuffer="403 URL seems to be made for proxy but contains apt-cacher-ng port. "
    		"Inconsistent apt configuration?";
    return ;

}

#define THROW_ERROR(x) { m_sErrorMsgBuffer=x; goto THROWN_ERROR; }
job::eJobResult job::SendData(int confd, int nAllJobCount)
{
	LOGSTART("job::SendData");

	if(!m_sMaintCmd.empty())
	{
		DispatchAndRunMaintTask(m_sMaintCmd, confd, m_pReqHead->h[header::AUTHORIZATION]);
			return R_DISCON; // just stop and close connection
	}
	
	off_t nGooddataSize(0);
	FiStatus fistate(FIST_ERROR);

	if (confd<0|| m_state==STATE_FAILMODE)
		return R_DISCON; // shouldn't be here

	if (!m_sErrorMsgBuffer.empty()) 
	{  // should just report it
		m_state=STATE_FAILMODE;
	}
	else if (m_pItem)
	{
		lockguard g(*m_pItem);
		
		while (true)
		{
			fistate=m_pItem->GetStatusUnlocked(nGooddataSize);
			
			dbgline;
			if (fistate>=FIST_ERROR)
			{
				const string & sErLine = m_pItem->GetHeaderUnlocked()->frontLine;
				m_sErrorMsgBuffer = (sErLine.length()<10) ? "500 Unknown error" : sErLine.substr(9);
				return R_AGAIN; // to send the error response
			}
			
			dbgline;
			if(fistate==FIST_COMPLETE || (m_nSendPos < nGooddataSize && fistate>=FIST_DLGOTHEAD))
				break;
			
			dbgline;
			m_pItem->wait();
			
			dbgline;
		}
		
		// left loop with usefull state of fileitem, get its header once
		if(m_RespHead.type == header::INVALID)
			m_RespHead = *(m_pItem->GetHeaderUnlocked());

	}
	else if(m_state != STATE_SEND_BUFFER)
	{
		ASSERT(!"no FileItem assigned and no sensible way to continue");
		return R_DISCON;
	}

	while (true) // left by returning
	{
		MYTRY // for bad_alloc in members
		{
			switch(m_state)
			{
				case(STATE_FRESH):
				{
					ldbg("STATE_FRESH");
					const char *szTempErr(0);
					eJobResult tempRes = BuildAndEnqueHeader(szTempErr, fistate, nGooddataSize);
					if(tempRes!=R_UNDEFINED)
						return tempRes;
					if(szTempErr)
						THROW_ERROR(szTempErr);
					USRDBG(REPLEVEL, "Prepared response header for user: \n" + m_sPrependBuf);
					continue;
				}
				case(STATE_HEADER_SENT):
				{
					ldbg("STATE_HEADER_SENT");
					
					if( fistate < FIST_DLGOTHEAD)
					{
						ldbg("ERROR condition detected: starts activity while downloader not ready")
						return R_AGAIN;
					}

						m_filefd=m_pItem->GetFileFd();
						if(m_filefd<0)
							THROW_ERROR("503 IO error");

					m_state=m_bChunkMode ? STATE_SEND_CHUNK_HEADER : STATE_SEND_PLAIN_DATA;
					ldbg("next state will be: " << m_state);
					continue;
				}
				case(STATE_SEND_PLAIN_DATA):
				{
					ldbg("STATE_SEND_PLAIN_DATA, max. " << nGooddataSize);

					// eof?
#define GOTOENDE { m_state=STATE_FINISHJOB ; continue; }
					if(m_nSendPos>=nGooddataSize)
					{
						if(fistate>=FIST_COMPLETE)
							GOTOENDE;
						LOG("Cannot send more, not enough fresh data yet");
						return R_AGAIN;
					}

					size_t nMax2SendNow=min(nGooddataSize-m_nSendPos, m_nRangeLast+1-m_nSendPos);
					ldbg("~sendfile: on "<< m_nSendPos << " up to : " << nMax2SendNow);
					int n = m_pItem->SendData(confd, m_filefd, m_nSendPos, nMax2SendNow);
					ldbg("~sendfile: " << n << " new m_nSendPos: " << m_nSendPos);

					if(n>0)
						m_nAllDataCount+=n;

					// shortcuts
					if(m_nSendPos>m_nRangeLast || (fistate==FIST_COMPLETE && m_nSendPos==nGooddataSize))
						GOTOENDE;
					
					if(n<0)
						THROW_ERROR("400 Client error");
					
					
					return R_AGAIN;
				}
				case(STATE_SEND_CHUNK_HEADER):
				{

					m_nChunkRemainingBytes=nGooddataSize-m_nSendPos;

					ldbg("STATE_SEND_CHUNK_HEADER for " << m_nChunkRemainingBytes);
					// if not on EOF then the chunk must have remaining size (otherwise the state would have been changed)
					//ASSERT( 0==(s&FIST_EOF) || 0==m_nChunkRemainingBytes);

					char buf[23];
					snprintf(buf, 23, "%x\r\n", m_nChunkRemainingBytes);
					m_sPrependBuf=buf;

					if(m_nChunkRemainingBytes==0)
						m_sPrependBuf+="\r\n";

					m_state=STATE_SEND_BUFFER;
					m_backstate=STATE_SEND_CHUNK_DATA;
					continue;
				}
				case(STATE_SEND_CHUNK_DATA):
				{
					ldbg("STATE_SEND_CHUNK_DATA");

					if(m_nChunkRemainingBytes==0)
						GOTOENDE; // yeah...
					int n = m_pItem->SendData(confd, m_filefd, m_nSendPos, m_nChunkRemainingBytes);
					if(n<0)
						THROW_ERROR("400 Client error");
					m_nChunkRemainingBytes-=n;
					m_nAllDataCount+=n;
					if(m_nChunkRemainingBytes<=0)
					{ // append final newline
						m_sPrependBuf="\r\n";
						m_state=STATE_SEND_BUFFER;
						m_backstate=STATE_SEND_CHUNK_HEADER;
						continue;
					}
					return R_AGAIN;
				}
				case(STATE_SEND_BUFFER):
				{
          // Sends data in the local buffer, used for page or chunk headers
          // sending, also error header sending. -> DELICATE STATE, should not
          // be interrupted by exception or throw to another flow uncontrolled.
					MYTRY
					{
						ldbg("prebuf sende: "<< m_sPrependBuf);
						int r=send(confd, m_sPrependBuf.data(), m_sPrependBuf.length(), 
								(m_backstate == STATE_ERRORCONT || m_backstate == STATE_TODISCON) ? 0 : MSG_MORE);
						if (r<0)
						{
							if (errno==EAGAIN || errno==EINTR || errno == ENOBUFS)
							{
								return R_AGAIN;
							}
							return R_DISCON;
						}
						m_nAllDataCount+=r;
						m_sPrependBuf.erase(0, r);
						
						USRDBG(REPLEVEL, "Sent " << r 
								<< " bytes of header data, remaining contents\n" 
								<< m_sPrependBuf);
						
						if(m_sPrependBuf.empty())
						{
							USRDBG(REPLEVEL, "Returning to last state, " << m_backstate);
							m_state=m_backstate;
							continue;
						}
					}
					MYCATCH(...)
					{
						return R_DISCON;
					}
					return R_AGAIN;
				}
				
				case (STATE_FAILMODE):
					goto THROWN_ERROR;
				
				case(STATE_ALLDONE):
					LOG("State: STATE_ALLDONE?");
				case (STATE_ERRORCONT):
					LOG("or STATE_ERRORCONT?");
				case(STATE_FINISHJOB):
					LOG("or STATE_FINISHJOB");
					{
						bool bClose=true;
						if(m_pReqHead)
						{
							bool bhttp11 = m_pReqHead->frontLine.find("HTTP/1.1") != stmiss;
							if(m_pReqHead->h[header::CONNECTION])
								bClose = 0==strcasecmp(m_pReqHead->h[header::CONNECTION], "close");
							else
								bClose = !bhttp11;
						}

						if(bClose)
							return R_DISCON;
						LOG("Reporting job done")
						return R_DONE;
					}
					break;
				case(STATE_TODISCON):
				default:
					return R_DISCON;
			}
			
			continue;
			
			
	THROWN_ERROR:
			ldbg("Processing error: " << m_sErrorMsgBuffer);
			if(0==m_nAllDataCount) // don't send errors when data header is underway
			{
				m_sPrependBuf=string("HTTP/1.1 ")+m_sErrorMsgBuffer+"\r\n";
				
				// only respond if the client has set it explicitely
				if(m_pReqHead->h[header::CONNECTION])
				{
					// whatever the client wants
					m_sPrependBuf+="Connection: ";
					m_sPrependBuf+=m_pReqHead->h[header::CONNECTION];
					m_sPrependBuf+="\r\n";
				}

				m_sPrependBuf+=header::GetInfoHeaders();
						
				if(m_RespHead.h[header::XORIG])
				{
					m_sPrependBuf+="X-Original-Source: ";
					m_sPrependBuf+=m_RespHead.h[header::XORIG];
					m_sPrependBuf+="\r\n";
				}
				else if(!m_sFileLoc.empty())
				{
					m_sPrependBuf+="X-Original-Source: ";
					m_sPrependBuf+=m_sFileLoc+"\r\n";
				}
				
				m_sPrependBuf+="\r\n";
				m_backstate=STATE_ERRORCONT;
				m_sErrorMsgBuffer.clear();
				USRDBG(REPLEVEL, "Response error or message, will continue (for " << m_sFileLoc<<")");
				//USRDBG(REPLEVEL, h.as_string(true));
				m_state=STATE_SEND_BUFFER;

				continue;
			}
			ldbg("Headers already sent -> forcing abort")
			return R_DISCON;
		}
		MYCATCH(bad_alloc) {
			// TODO: report memory failure?
			return R_DISCON;
		}
		ASSERT(!"UNREACHED");
	}
}

#define REPORT_ERR(x) { szErrRet=x; return R_UNDEFINED; }
inline job::eJobResult job::BuildAndEnqueHeader(const char * &szErrRet, const FiStatus &fistate, const off_t &nGooddataSize)
{
	if(fistate<FIST_DLGOTHEAD) // be sure about that
		return R_AGAIN;

	m_state=STATE_SEND_BUFFER;
	m_backstate=STATE_HEADER_SENT; // might change without body

	if(m_RespHead.type != header::ANSWER)
		REPORT_ERR("500 Rotten Data");

	//h.del("Content-Length"); // TESTCASE: Force chunked mode

	m_sPrependBuf=m_RespHead.frontLine+"\r\n";

	bool bSendBody(true);
	// no data for error heads
	if(m_RespHead.getStatus() != 200)
		bSendBody=false;

	// make sure there is no junk inside
	// if missing completely, chunked mode is to be used below
	if(m_RespHead.h[header::CONTENT_LENGTH])
	{
		if(0==atoofft(m_RespHead.h[header::CONTENT_LENGTH]))
			bSendBody=false;

		/*
		if(!bSendBody && ! m_pReqHead->type==header::HEAD)
			m_RespHead.del(header::CONTENT_LENGTH);
			*/
	}

	if( ! bSendBody)
	{
		m_backstate=STATE_ALLDONE;
	}
	else
	{

		if( ! m_RespHead.h[header::CONTENT_LENGTH])
		{
			// unknown length but must have data, will have to improvise: prepare chunked transfer
			if(m_bIsHttp11 != 0)
			REPORT_ERR("505 HTTP version not supported for this file"); // you don't accept this, go away
			m_bChunkMode=true;
			m_RespHead.set(header::TRANSFER_ENCODING, "chunked");
		}
		else
		{
			// OK, has content length. Can do optimizations?
			// Handle If-Modified-Since and Range headers

			// we deal with them equally but need to know which to use
			const char *pIfmo = m_pReqHead->h[header::RANGE] ?
					m_pReqHead->h[header::IFRANGE] : m_pReqHead->h[header::IF_MODIFIED_SINCE];

			bool bGotLen(false);

			// consider contents "fresh" for non-volatile data in "special cases"
			bool bFreshnessForced=(m_type != rechecks::FILE_INDEX
					|| m_pReqHead->h[header::ACNGFSMARK]);

			// this plays well with lighttpd <-> Apt, but may not work with the alternative time format
			// TODO: add format tolerant time comparer
			bool bIfModSeenAndChecked=(pIfmo && m_RespHead.h[header::LAST_MODIFIED] &&
					0==strcmp(pIfmo, m_RespHead.h[header::LAST_MODIFIED]));

			// is it fresh? or is this relevant? or is range mode forced?
			if(  bFreshnessForced || bIfModSeenAndChecked)
			{
				off_t nContLen=atoofft(m_RespHead.h[header::CONTENT_LENGTH]);

				/*
				 * Range: bytes=453291-
				 * ...
				 * Content-Length: 7271829
				 * Content-Range: bytes 453291-7725119/7725120
				 */

				const char *pRange=m_pReqHead->h[header::RANGE];

				// working around a bug in old curl versions
				if(!pRange)
					pRange=m_pReqHead->h[header::CONTENT_RANGE];

				if(pRange)
				{
					off_t nRangeFrom(0), nRangeTo(0);
					int nRangeItems=sscanf(pRange, "bytes=" OFF_T_FMT
                            "-" OFF_T_FMT, &nRangeFrom, &nRangeTo);
                    // working around bad (old curl style) requests
                    if(nRangeItems <= 0)
                    {
                        nRangeItems=sscanf(pRange, "bytes "
                                OFF_T_FMT "-" OFF_T_FMT,
                                &nRangeFrom, &nRangeTo);
                    }
					if(nRangeItems == 1) // open-end? set the limit
						nRangeTo=nContLen-1;

					/*
					 * make sure that our client doesn't just
					 * hang here while the download thread is
					 * fetching from 0 to start position for
					 * many minutes
					 */

					if(nRangeItems>0
						&&
						(  fistate==FIST_COMPLETE
							// or can start sending within this range
							|| (
									fistate >= FIST_DLGOTHEAD
									&& fistate <= FIST_COMPLETE
									&& nGooddataSize>=nRangeFrom
								)
							// don't care, found special hint from acngfs (kludge...)
							|| m_pReqHead->h[header::ACNGFSMARK]
							)
						)
					{
						// detect errors, out-of-range case
						if(nRangeTo>=nContLen || nRangeFrom>=nContLen || nRangeTo<nRangeFrom)
							REPORT_ERR("416 Requested Range Not Satisfiable")

						m_nSendPos = nRangeFrom;
						m_nRangeLast = nRangeTo;
						m_sPrependBuf="HTTP/1.1 206 Partial Response\r\nContent-Length: ";
						char buf[200];
						sprintf(buf,
                                OFF_T_FMT "\r\n"
                                "Content-Range: bytes " OFF_T_FMT "-" OFF_T_FMT "/" OFF_T_FMT "\r\n",
								m_nRangeLast-m_nSendPos+1,
								m_nSendPos,
								m_nRangeLast,
								nContLen
								);
						m_sPrependBuf+=buf;
						bGotLen=true;
					}
				}
				else if(bIfModSeenAndChecked)
				{
					// file is fresh, and user sent if-mod-since -> fine
					REPORT_ERR("304 Not Modified");
				}
			}
			// has cont-len available but this header was not set yet in the code above
			if( !bGotLen)
			{
				m_sPrependBuf+=string("Content-Length: ")
				+m_RespHead.h[header::CONTENT_LENGTH]
				+"\r\n";
			}
		}
		if(m_RespHead.h[header::LAST_MODIFIED])
			m_sPrependBuf+=string("Last-Modified: ")
			+m_RespHead.h[header::LAST_MODIFIED]+"\r\n";
	}
	m_sPrependBuf+=header::GetInfoHeaders();

	if(bSendBody)
		m_sPrependBuf+="Content-Type: application/octet-stream\r\n";
	else if(m_RespHead.h[header::LOCATION])
	{
			m_sPrependBuf+="Location: ";
			m_sPrependBuf+=m_RespHead.h[header::LOCATION];
			m_sPrependBuf+="\r\n";
	}

	if(m_RespHead.h[header::XORIG])
		m_sPrependBuf+=string("X-Original-Source: ")+m_RespHead.h[header::XORIG]+"\r\n";

	// only respond if the client has set it explicitely
	if(m_pReqHead->h[header::CONNECTION])
	{
		// whatever the client wants
		m_sPrependBuf+="Connection: ";
		m_sPrependBuf+=m_pReqHead->h[header::CONNECTION];
		m_sPrependBuf+="\r\n";
	}

	m_sPrependBuf+="\r\n";

	if(m_pReqHead->type==header::HEAD)
		m_backstate=STATE_ALLDONE; // simulated head is prepared but don't send stuff


	if(m_nSendPos>0) // maybe needs to wait in the prerequisites check in the beginning
		return R_AGAIN;

	return R_UNDEFINED;
}

inline FiStatus job::_SwitchToPtItem(const MYSTD::string &fileLoc)
{
	LOGSTART("job::_SwitchToPtItem");
	LOG("Changing to local pass-through file item");
	m_pItem->Unreg();
	m_pItem.reset(new tPassThroughFitem(m_sFileLoc));
	return m_pItem->Setup(true);
}
