#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Tests the debconf forwarding"""

import logging
import os
import subprocess
import sys
import tempfile
import unittest

import apt
import apt_pkg

sys.path.insert(0, "../..")
import aptdaemon.worker
import aptdaemon.enums

DEBUG=False

class MockTrans(object):
    def __init__(self, packages):
        self.packages = packages
        self.role = aptdaemon.enums.ROLE_COMMIT_PACKAGES
        self.tid = "12123"

class FutureStatusTest(unittest.TestCase):

    def setUp(self):
        self.worker = aptdaemon.worker.AptWorker()
        self.worker._cache = apt.Cache()
        self.status_orig = apt_pkg.config.get("Dir::State::status")

    def testInstall(self):
        # Create a transaction which installs a package which has got
        # uninstalled dependencies
        for pkg in self.worker._cache:
            if not pkg.is_installed and pkg.candidate:
                deps = self._get_uninstalled_deps(pkg)
                if deps:
                    break
                try:
                    pkg.mark_install()
                except SystemError:
                    self.worker._cache.clear()
                    continue
        trans = MockTrans([[pkg.name], [], [], [], []])
        # The test
        deps, status, download, space = self.worker.simulate(trans)
        # Check if the package is installed in the new status file
        status_file, status_path = tempfile.mkstemp(prefix="future-status-")
        os.write(status_file, status)
        apt_pkg.config.set("Dir::State::status", status_path)
        apt_pkg.init_system()
        self.worker._cache.open()
        self.assertTrue(self.worker._cache[trans.packages[0][0]].is_installed,
                        "The package is not installed in the future")
        if self.worker._cache.broken_count:
            broken = [pkg for pkg in self.worker._cache if pkg.is_now_broken]
            self.fail("The following packages are broken: " % " ".join(broken))

    def _get_uninstalled_deps(self, pkg):
        deps = []
        for dep in pkg.candidate.dependencies:
            if len(dep.or_dependencies) > 1:
                return None
            for base_dep in dep.or_dependencies:
                try:
                    dep = self.worker._cache[base_dep.name]
                except KeyError:
                    return None
                if not dep.is_installed and dep.candidate and \
                   apt_pkg.CheckDep(dep.candidate.version,
                                    base_dep.relation,
                                    base_dep.version):
                    deps.append(dep)
                else:
                    return None
        return deps

    def tearDown(self):
        pass

if __name__ == "__main__":
    if DEBUG:
        logging.basicConfig(level=logging.DEBUG)
    unittest.main()

# vim: ts=4 et sts=4
