// pkg_view.cc
//
//  Copyright 2000-2002 Daniel Burrows

#include "pkg_view.h"

#include "aptitude.h"

#include "pkg_columnizer.h"
#include "edit_pkg_hier.h"

#include <vscreen/vs_label.h>
#include <vscreen/vs_multiplex.h>
#include <vscreen/vs_table.h>
#include <vscreen/vscreen_widget.h>
#include <vscreen/config/keybindings.h>

#include <generic/apt.h>
#include <generic/config_signal.h>

#include <apt-pkg/error.h>

#include <sigc++/bind.h>

#include <ctype.h>

using namespace std;

class pkg_handling_label:public vs_label
{
  column_definition_list *columns;

  bool have_pkg;
  pkgCache::PkgIterator pkg;
  pkgCache::VerIterator ver;

  void zap_package()
  {
    have_pkg=false;
  }

public:
  pkg_handling_label(column_definition_list *_columns)
    :vs_label(" "), columns(_columns), have_pkg(false)
  {
    cache_closed.connect(SigC::slot(this, &pkg_handling_label::zap_package));
  }

  ~pkg_handling_label() {delete columns;}

  size size_request() {return size(1,1);}

  void set_columns(column_definition_list *_columns)
  {
    delete columns;
    columns=_columns;
    vscreen_update();
  }

  void do_columnify(const pkgCache::PkgIterator &_pkg,
		    const pkgCache::VerIterator &_ver)
  {
    pkg=_pkg;
    ver=_ver;

    have_pkg=!pkg.end();

    vscreen_update();
  }

  void paint()
  {
    if(apt_cache_file)
      {
	// Needed to initialize translated widths and stuff.
	pkg_item::pkg_columnizer::setup_columns();

	if(!have_pkg)
	  {
	    pkg=pkgCache::PkgIterator();
	    // Reinitialize it all the time to avoid the "magic autochanging
	    // pointers" bug.
	    ver=pkgCache::VerIterator(*apt_cache_file);
	  }

	set_text(pkg_item::pkg_columnizer(pkg, ver, *columns, 0).layout_columns(getmaxx()));
      }
    else
      set_text("");

    vs_label::paint();
  }
};

static void do_set_column_format(string s, pkg_handling_label *l)
{
  column_definition_list *columns=parse_columns(s, pkg_item::pkg_columnizer::parse_column_type,
						pkg_item::pkg_columnizer::defaults);

  if(!columns)
    _error->Error("Couldn't parse column definition");
  else
    l->set_columns(columns);
}

class pkg_description_widget:public vscreen_widget
{
  string desc;

  int start_line;
public:
  pkg_description_widget():start_line(0) {}

  void set_description(string newdesc)
  {
    if(newdesc!=desc) // uhhhh..hack?
      start_line=0;

    desc=newdesc;

    vscreen_update();
  }

  void set_package(const pkgCache::PkgIterator &pkg,
		   const pkgCache::VerIterator &ver)
  {
    string newdesc;

    if(!pkg.end() && !ver.end())
      newdesc=apt_package_records->Lookup(ver.FileList()).LongDesc();
    else
      newdesc="";

    set_description(newdesc);
  }

  size size_request() {return size(1,1);}
  bool get_cursorvisible() {return true;}
  point get_cursorloc() {return point(0,0);}
  bool focus_me()
  {
    int nlines=get_lines(false);
    return nlines>getmaxy();
  }

  // FIXME: this is way too long
  //
  // Steps through the description and calculates the number of lines it will
  // expand to.  If do_paint is true, displays it at the same time.
  int get_lines(bool do_paint)
  {
    int y=-start_line;
    int maxy=getmaxy();
    string::size_type loc=0;
    int nlines=-1;

    // Skip the short description
    while(loc<desc.size() && desc[loc]!='\n')
      ++loc;

    if(loc<desc.size()) // Skip the '\n'
      ++loc;

    while(loc<desc.size() && ((!do_paint && maxy>0) || y<maxy))
      {
	// We're at the beginning of a line here.
	//
	// Find out what type of line it is first.  (be liberal -- lines
	// not starting with ' ' are formatted as part of the current
	// paragraph)
	if(desc[loc]==' ')
	  ++loc;

	// Now check whether it's part of a paragraph, a blank line,
	// or a preformatted line.
	switch(desc[loc])
	  {
	  case ' ':
	    {
	      ++loc;

	      int amt=0;
	      int display_amt;
	      while(loc+amt<desc.size() && desc[loc+amt]!='\n')
		++amt;

	      if(amt<getmaxx())
		display_amt=amt;
	      else
		display_amt=getmaxx();

	      if(y>=0 && do_paint)
		mvaddnstr(y, 0, desc.c_str()+loc, display_amt);
	      ++y;
	      ++nlines;

	      loc+=amt;
	      if(loc<desc.size())
		++loc;
	      break;
	    }
	  case '.':
	    {
	      ++y;
	      ++nlines;
	      while(loc<desc.size() && desc[loc]!='\n')
		++loc;

	      if(loc<desc.size())
		++loc;
	      break;
	    }
	  default:
	    // It's a paragraph.
	    {
	      bool cont=true;
	      string::size_type amt=0;
	      string par="";

	      do {
		amt=0;
		while(loc+amt<desc.size() && desc[loc+amt]!='\n')
		  ++amt;

		par=par+string(desc, loc, amt);

		loc+=amt;
		if(loc<desc.size())
		  ++loc;

		if(!(loc<desc.size() && desc[loc]!=' ' && desc[loc]!='.'))
		  cont=false;
	      } while(cont && loc<desc.size());

	      // Now display it word-wrapped, with a \n at the end.
	      string::size_type disploc=0;

	      while(disploc<par.size() && ((!do_paint && maxy>0) || y<maxy))
		{
		  // Try to display getmaxx() chars
		  amt=getmaxx();
		  while(disploc+amt<par.size() && par[disploc+amt]!=' ' && amt>0)
		    --amt;

		  // Not so good, we found a word longer than the
		  // screen.  Hack it..
		  if(amt==0)
		    amt=getmaxx();

		  if(y>=0 && do_paint)
		    mvaddnstr(y, 0, par.c_str()+disploc, amt);
		  ++y;
		  ++nlines;

		  disploc+=amt;
		  while(disploc<par.size() && par[disploc]==' ')
		    ++disploc;
		}
	    }
	  }
      }

    return nlines;
  }

#if 0
  int get_lines(bool do_paint)
  {
    int x=0;
    int y=-start_line;
    int maxy=getmaxy();
    string::size_type loc=0;

    // Skip the short description
    while(loc<desc.size() && desc[loc]!='\n')
      ++loc;

    // Declared here so that we can use it to decide whether to start new lines.
    int amt=0;

    // This is hairy.
    while(loc<desc.size() && ((!do_paint && maxy>0) || y<maxy))
      {
	// Remembers whether the last inserted character was a newline.
	// (if not, we need to add an extra newline when we see " .\n" in the
	// input stream)
	bool newline_was_most_recent=false;

	// Tracks whether the last line was explicitl formatted or
	// a blank line.
	bool last_line_was_special=false;

	// Display a single line, broken at whitespace:
	while(loc<desc.size() && desc[loc]!='\n')
	  {
	    amt=0;

	    while(x+amt<getmaxx() && loc+amt<desc.size() && desc[loc+amt]!='\n')
	      ++amt;

	    if(loc+amt<desc.size())
	      while(amt>0 && !isspace(desc[loc+amt]))
		--amt;

	    if(x==0 && amt==0)
	      // Oh dear.  We found a word that took more space than was
	      // available on a single line.  Just hack it and split the word
	      // between two lines.  (don't try to word-wrap)
	      {
		while(x+amt<getmaxx() && loc+amt<desc.size() && desc[loc+amt]!='\n')
		  ++amt;
	      }

	    if(y>=0 && do_paint)
	      mvaddnstr(y, x, desc.c_str()+loc, amt);

	    if(amt>0)
	      newline_was_most_recent=false;

	    loc+=amt;
	    x+=amt;

	    if(x>=getmaxx() || amt==0)
	      // If amt==0, there's a linebreak in the middle of a word.
	      {
		++y;
		x=0;
		newline_was_most_recent=true;

		while(loc<desc.size() && desc[loc]==' ')
		    ++loc;
	      }
	    else
	      {
		if(y>=0 && do_paint)
		  // Add a space between this and the next word (this may be
		  // unneeded if we break the line before the next word; sue me)
		  mvaddch(y, x, ' ');

		++x;
	      }
	  }

	// FIXME: people who start their long-descriptions with a blank line
	// will see a '.' in my display.
	//
	//  OTOH, anyone who does something so wacked out deserves weirdness ;-)
	// Now move to the next line and handle preformatted lines/blank lines
	if(loc<desc.size() && desc[loc]=='\n')
	  {
	    ++loc;

	    // This should always be true:
	    if(loc<desc.size() && desc[loc]==' ')
	      {
		++loc;
		// Is it a blank line?
		if(loc<desc.size() && desc[loc]=='.')
		  {
		    ++y;
		    x=0;
		    ++loc;

		    // If stuff was output but no newline was, add an extra
		    // newline and reset amt.
		    if(!newline_was_most_recent && !last_line_was_special)
		      {
			++y;
			amt=0;
			newline_was_most_recent=true;
		      }

		    last_line_was_special=true;

		    // There should be a \n, or behavior is "undefined";
		    // just snarf until we see it.
		    while(loc<desc.size() && desc[loc]!='\n')
		      ++loc;

		    // Now bounce through the loop again, this same logic will
		    // handle the next newline.
		  }
		else if(loc<desc.size() && desc[loc]==' ')
		  {
		    ++loc;

		    if(!newline_was_most_recent && !last_line_was_special)
		      {
			++y;
			x=0;

			newline_was_most_recent=true;
		      }

		    last_line_was_special=true;

		    // Now output the line (yes, we do it here!  This is easier
		    // than special-casing the main word-wrap loop)

		    // Yes, this variable name sucks.
		    int amt2=0;

		    while(loc+amt2<desc.size() && amt2<getmaxx() && desc[loc+amt2]!='\n')
		      ++amt2;

		    if(y>=0 && do_paint)
		      mvaddnstr(y, x, desc.c_str()+loc, amt2);

		    loc+=amt2;
		    ++y;

		    // Snarf up to the newline if we didn't.
		    while(loc<desc.size() && desc[loc]!='\n')
		      ++loc;

		    // As above, bounce through the loop again.
		  }
		else
		  last_line_was_special=false;
	      }
	  }
      }
    return y+start_line;
  }
#endif

  void paint()
  {
    get_lines(true);
  }

  void linedown()
  {
    int nlines=get_lines(false);
    if(start_line<=nlines-getmaxy())
      {
	++start_line;
	vscreen_update();
      }
  }

  void lineup()
  {
    if(start_line>0)
      --start_line;
    vscreen_update();
  }

  bool handle_char(chtype ch)
  {
    if(global_bindings.key_matches(ch, "Down"))
      linedown();
    else if(global_bindings.key_matches(ch, "Up"))
      lineup();
    else
      return vscreen_widget::handle_char(ch);

    return true;
  }
};

vscreen_widget *make_package_view(list<package_view_item> &format,
				  vscreen_widget *mainwidget,
				  pkg_signal *sig, desc_signal *desc_sig)
{
  bool found_mainwidget=false;

  vs_table *rval=new vs_table;

  assert(mainwidget);

  for(list<package_view_item>::iterator i=format.begin();
      i!=format.end();
      i++)
    {
      switch(i->type)
	{
	case PACKAGE_VIEW_MAINWIDGET:
	  if(found_mainwidget)
	    _error->Error(_("make_package_view: error in arguments -- two main widgets??"));
	  else
	    i->widget=mainwidget;
	  break;
	case PACKAGE_VIEW_STATIC:
	  if(!i->columns)
	    _error->Error(_("make_package_view: error in arguments -- bad column list for static item"));
	  else
	    {
	      pkg_handling_label *l=new pkg_handling_label(i->columns);
	      i->widget=l;

	      if(sig)
		sig->connect(slot(l, &pkg_handling_label::do_columnify));

	      if(!i->columns_cfg.empty())
		aptcfg->connect(i->columns_cfg,
				SigC::bind(SigC::slot(do_set_column_format),
					   l));
	    }
	  break;
	case PACKAGE_VIEW_DESCRIPTION:
	  // FIXME: this is automagically made into a multiplexer..
	  {
	    vs_multiplex *m=new vs_multiplex;
	    vs_hier_editor *e=new vs_hier_editor;
	    pkg_description_widget *w=new pkg_description_widget;

	    // HACK: speaks for itself
	    vs_tree *thetree=dynamic_cast<vs_tree *>(mainwidget);

	    i->widget=m;

	    if(sig)
	      {
		sig->connect(slot(w, &pkg_description_widget::set_package));
		sig->connect(slot(e, &vs_hier_editor::set_package));
	      }

	    if(desc_sig)
	      desc_sig->connect(slot(w, &pkg_description_widget::set_description));

	    mainwidget->connect_key("DescriptionDown", &global_bindings,
				    slot(w,
					 &pkg_description_widget::linedown));
	    mainwidget->connect_key("DescriptionUp", &global_bindings,
				    slot(w,
					 &pkg_description_widget::lineup));
	    mainwidget->connect_key("EditHier", &global_bindings,
				    slot(e, &vscreen_widget::show));
	    mainwidget->connect_key("EditHier", &global_bindings,
				    slot(m, &vscreen_widget::show));
	    mainwidget->connect_key("EditHier", &global_bindings,
				    bind(slot(rval, &vs_table::focus_widget),
					 m));

	    e->hidden_sig.connect(bind(slot(rval, &vs_table::focus_widget),
				       mainwidget));

	    if(thetree)
	      e->commit_changes.connect(slot(thetree, &vs_tree::line_down));

	    m->add_visible_widget(e, false);
	    m->add_visible_widget(w, true);

	    w->show();

	    // FIXME: this is a grotesque hack.
	    if(mainwidget->get_visible())
	      rval->focus_widget(mainwidget);
	  }
	  break;
	default:
	  _error->Error(_("make_package_view: bad argument!"));
	  break;
	}

      if(i->widget)
	{
	  rval->add_widget_opts(i->widget, i->row, i->col, i->h, i->w,
				i->xopts, i->yopts);

	  i->widget->set_bg(i->bg);

	  if(i->popupdownkey.size()>0)
	    rval->connect_key(i->popupdownkey,
			      &global_bindings,
			      slot(i->widget,
				   &vscreen_widget::toggle_visible));

	  if(i->visible)
	    i->widget->show();
	}
    }

  // Slow, but the list is (hopefully) << 10 elements or so.
  for(list<package_view_item>::iterator i=format.begin();
      i!=format.end();
      i++)
    if(i->popupdownlinked.size()>0)
      for(list<package_view_item>::iterator j=format.begin();
	  j!=format.end();
	  j++)
	{
	  if(!strcasecmp(j->name.c_str(), i->popupdownlinked.c_str()))
	    {
	      // Having to make two connections is annoying.
	      j->widget->shown_sig.connect(slot(i->widget, &vscreen_widget::show));
	      j->widget->hidden_sig.connect(slot(i->widget, &vscreen_widget::hide));
	      break;
	    }
	}

  if(!mainwidget)
    _error->Error(_("make_package_view: no main widget found"));

  return rval;
}
