// vs_pager.h       -*-c++-*-
//
//  Copyright 2000 Daniel Burrows
//
//  A vscreen which acts as a text pager.

#ifndef VS_PAGER_H
#define VS_PAGER_H

#include "vscreen_widget.h"

#include <string.h>
#include <string>

class keybindings;

class vs_pager:public vscreen_widget
{
  const char *text;
  // We don't use the STL string class because I want closer control over how
  // the text gets accessed.  However, we *don't* take over management of it
  // (ie, it doesn't get deleted when we do); you can derive from this if you
  // need to do that.
  int len;

  int start;
  // The offset into the buffer representing the upper-left corner of the
  // display.
  int horizontal_shift;
  // How far to the left or right the user scrolled.

  // We precalculate these, both to speed up other stuff and to allow for (eg)
  // quick generation of scrollbar info.  (note that this precludes arbitrary
  // modification of the memory area -- there'll be a method eventually that
  // you can call to explicitly recalculate these)
  int maxx, maxy;
  int cury; // (the current row, used for scrollbars and stuff)

  std::string last_search;
  // Handled the same way as in vs_tree.

  void do_signal();
  // Emits a signal describing how much of the text we've displayed (see below)
protected:
  const char *get_text() const {return text;}
  int get_len() const {return len;}
  int get_start() const {return start;}

public:
  vs_pager(const char *_text);
  vs_pager(const char *_text, int _len);

  virtual ~vs_pager();

  virtual void set_text(const char *_text, int _len);

  virtual void set_text(const char *_text)
  {
    set_text(_text, strlen(_text));
  }

  void scroll_up(int nlines);
  void scroll_down(int nlines);
  void scroll_top();
  void scroll_bottom();
  void search_for(std::string s);

  void calc_bounds();
  // Recalculates line numbers and so on -- useful if the text has been
  // modified.
  // Currently causes the pager to jump back to the first line of its displayed
  // text.

  // Hmmm..
  int get_maxy() {return maxy;}
  int get_cury() {return cury;}

  virtual bool handle_char(chtype ch);
  virtual bool focus_me() {return true;}
  virtual void paint();

  size size_request();
  bool get_cursorvisible() {return true;}
  point get_cursorloc() {return point(0,0);}

  // Announces that the location within the text has changed (eg, so the user
  // can display a progress meter)
  SigC::Signal2<void, int, int> location_changed;

  static keybindings *bindings;
  static void init_bindings();
};

class vs_file_pager:public vs_pager
// Like the above, but displays a file instead of an arbitrary string in
// memory.  Uses mmap. (run in terror now if you think mmap is Evil[tm] :) )
// A non-mmap option should be written..maybe just allocate a Big Buffer
// in memory and load the file into it?
{
  int fd;
  std::string error;
  // If a problem is encountered whilst loading the file, this will contain its
  // description and fd will be -1.  FIXME: this is broken.

  void paint_status();
  void dispose_mapping();
public:
  vs_file_pager();
  vs_file_pager(std::string filename);
  vs_file_pager(const char *text, int len);

  virtual void set_text(const char *_text)
  {
    dispose_mapping();
    vs_pager::set_text(_text);
  }

  virtual void set_text(const char *_text, int _len)
  {
    dispose_mapping();
    vs_pager::set_text(_text, _len);
  }

  void load_file(std::string filename);

  virtual ~vs_file_pager() {dispose_mapping();}
};

#endif
