# 
# versions.sh - print a list of versions in an archive branch
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "list the versions in an archive branch\\n"
		printf "usage: versions [options] [branch]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
		printf " -r --reverse                  sort from newest to oldest\\n"
		printf "\\n"
		printf "Print a list of versions within an archive branch.\\n"
		printf "\\n"
		printf "The list is ordinarilly sorted from oldest to newest,\\n"
		printf "but the order can be changed with -r (--reverse).\\n"
		printf "\\n"
		printf "The output format is:\\n"
		printf "\\n"
		printf "	\"%%s\\n\" \"\$category\" \\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
reverse=
debug_opt=

while test $# -ne 0 ; do

  case "$1" in 

    --debug)		shift
    			debug_opt=--debug
			printf "\n" 1>&2
			printf "versions: DEBUGGING ACTIVATED\n" 1>&2
			printf "\n" 1>&2
			set -x
			;;
			
    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "make-category: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "make-category: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;


    -r|--reverse)	shift
    			reverse=-r
			;;

    -*)			printf "versions: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: versions [options] [branch]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -ne 0 ; then
  branch="$1"
  shift
else
  branch=
fi

################################################################
# Sanity Check and Process Defaults
# 

if test -z "$branch" ; then
  version="`larch tree-version .`"
  if test -z "$archive" ; then
    archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$version\"`"
  fi
  branch="`larch parse-package-name --package \"$version\"`"
fi

larch valid-package-name -e versions -- "$branch"
archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$branch\"`"
category=`larch parse-package-name --basename "$branch"`
branch=`larch parse-package-name --package "$branch"`


################################################################
# Establish an Archive Connection
# 

if test "$WITHARCHIVE" != "$archive" ; then
  exec larch with-archive -A "$archive" larch versions $debug_opt $reverse "$archive/$branch"
fi

################################################################
# Print the List
# 

basere="([a-zA-Z]([a-zA-Z0-9]*(-[a-zA-Z0-9]+)?)*)"
vsnre="([0-9]+\\.[0-9]+)"

wftp-home

if ! wftp-cd "$category/$branch" ; then
  printf "\\n" 1>&2
  printf "versions: no such branch\\n" 1>&2
  printf "  archive: %s\\n" "$archive" 1>&2
  printf "  branch: %s\\n" "$branch" 1>&2
  printf "\\n" 1>&2
  exit 1
fi

wftp-ls \
| grep -E "^($basere)(--$basere)?(--$vsnre)$" \
| sed -e "s/.*--\\([0-9].*\\)/\\1.&/" \
| sort  -t. -k 1,1${reverse#-}n -k 2,2${reverse#-}n \
| sed -e "s/[^.]*.[^.]*.//"
