# 
# make-log.sh - initialize a new log entry file
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "initialize a new log file entry\\n"
		printf "usage: make-log [options] [[archive/]version]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
                printf " -d --dir DIR                  cd to DIR first\\n"
		printf "\\n"
		printf "Create a new log entry file in the root of the project tree\\n"
		printf "containing DIR, for the indicated VERSION.\\n"
		printf "\\n"
		printf "If an older log file (\"LOG\") already exists, it is made a\\n"
		printf "backup file (\"LOG~\"), overwritting any existing backup file.\\n"
		printf "\\n"
		printf "The name of the new log file is printed to standard output.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
dir=.

while test $# -ne 0 ; do

  case "$1" in 

    -d|--dir)		shift
			if test $# -eq 0 ; then
			  printf "make-log: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir="$1"
			shift
			;;

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "make-log: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "make-log: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;


    -*)			printf "make-log: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: make-log [options] [[archive/]version]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -gt 0 ; then
  arch_version="$1"
  shift
else
  arch_version=
fi

################################################################
# Sanity Check and Process Defaults
# 

here="`pwd`"
cd "$dir"
dir="`pwd`"

wdroot="`larch tree-root --accurate \"$dir\"`"

if test "x$arch_version" = x ; then
  arch_version=`larch tree-version`
fi

larch valid-package-name -e "make-log" --vsn -- "$arch_version"

archive=`larch parse-package-name -R "$archroot" -A "$archive" --arch "$arch_version"`
version=`larch parse-package-name -R "$archroot" -A "$archive" --package-version "$arch_version"`

logname="++log.$version--$archive"


################################################################
# Preserve an Old Log File
# 

cd "$wdroot"

if test -e "$logname" ; then
  rm -f "$logname~"
  mv "$logname" "$logname~"
fi

################################################################
# Create an "Empty" Log File
# 

cd "$wdroot"

cat > "$logname" <<EOF
Summary: 
Keywords: 

EOF

if test "x$wdroot" = "x$here" ; then
  printf "%s\\n" "$logname"
else
  printf "%s/%s\\n" "$wdroot" "$logname"
fi

