# 
# tree-repair.sh: repair a wedged project tree
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "repair a project tree after interrupted commit\\n"
		printf "usage: tree-repair [options] [dir]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf "DIR (or the current directory) should be part of a project tree\\n"
		printf "that was left in a wedged state by an interrupted commit.\\n"
		printf "The command \"larch tree-root --accurate\" will check for that condition\\n"
		printf "and those larch commands sensative to such a state will exit with an\\n"
		printf "error message while the wedged state persists.\\n"
		printf "\\n"
		printf "While in a wedged state, the project tree's patch log may or may not\\n"
		printf "be fully up-to-date (depending on whether or not the commit completed\\n"
		printf "in the archive before being interrupted.)\\n"
		printf "\\n"
		printf "This command brings the patch log up-to-date and un-wedges the project\\n"
		printf "tree.  It prints a message saying whether or not the commit actually\\n"
		printf "took place\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 


while test $# -ne 0 ; do

  case "$1" in 

    --)			shift
    			break
			;;

    -*)			printf "tree-repair: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: tree-repair [options] [dir]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -eq 1 ; then
  dir="$1"
else
  dir="."
fi

cd "$dir"
dir="`pwd`"

################################################################
# Sanity Check and Process Defaults
# 

wdroot="`larch tree-root \"$dir\"`"

cd "$wdroot/{arch}"

if test ! -e "++mid-commit" ; then

  printf "tree-repair: directory not wedged\\n" 1>&2
  printf "  root: %s\\n" "$wdroot" 1>&2
  printf "\\n"
  printf "This directory is not in a wedged state.\\n" 1>&2
  printf "No repair is necessary.\\n" 1>&2
  printf "\\n"
  exit 1

fi

archive="`grep -i -E -e ^Archive: ++mid-commit | head -1 | sed -e 's/[Aa][Rr][Cc][Hh][Ii][Vv][Ee]:[[:space:]]*//'`"
revision="`grep -i -E -e ^Revision: ++mid-commit | head -1 | sed -e 's/[Rr][Ee][Vv][Ii][Ss][Ii][Oo][Nn]:[[:space:]]*//'`"
version="`larch parse-package-name --package-version \"$revision\"`"
lvl="`larch parse-package-name --lvl \"$revision\"`"

################################################################
# We Need an Archive Connection for This
# 

if test "$WITHARCHIVE" != "$archive" ; then
  exec larch with-archive -A "$archive" \
	larch tree-repair "$wdroot"
fi


################################################################
# Did the Commit Take Place?
# 

tmpfile="$wdroot/,,tree-repair.$$"

rm -f "$tmpfile"

bail()
{
  rm -f "$tmpfile"
  exit 1
}

trap "printf \"tree-repair: interrupted -- cleaning up\\n\" 1>&2 ; bail" INT

completed=

if ! larch cat-archive-log "$archive/$revision" > "$tmpfile" 2> /dev/null ; then

  if ! larch revisions "$archive/$version" 1>&2 2> /dev/null ; then
    printf "tree-repair: error contacting archive\\n" 1>&2
    printf "\\n" 1>&2
    bail
  else
    completed=
  fi

elif ! cmp "$tmpfile" ++mid-commit ; then

  completed=

else

  completed=yes

fi


################################################################
# Update the Patch Log
# 

if test -z "$completed" ; then

  printf "tree-repair: commit did not complete (unwedging project tree)\\n"
  rm ++mid-commit

else

  printf "tree-repair: commit completed (updating project tree patch log)\\n"
  mv ++mid-commit ++commit-definite
  larch copy-to-patch-log ++commit-definite "$archive/$version" "$lvl" "$answer"
  rm ++commit-definite

fi

rm "$tmpfile"
exit 0

# tag: Tom Lord Tue Dec 11 14:47:21 2001 (project-tree/tree-repair.sh)
#
