/* with-file-lock.c: run a command with a file lock
 *
 ****************************************************************
 * Copyright (C) 2002  Tom Lord
 * 
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/os/fcntl.h"
#include "hackerlab/os/sys/types.h"
#include "hackerlab/os/sys/wait.h"



static t_uchar * program_name = "with-file-lock";
static t_uchar * usage = "with-file-lock [options] file command ...";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
				   "\n"
				   "Copyright 2002 Tom Lord\n"
				   "\n"
				   "This is free software; see the source for copying conditions.\n"
				   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
				   "PARTICULAR PURPOSE.\n"
				   "\n"
				   "Report bugs to <lord@regexps.com>.\n"
				   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", "help", 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.") \
  OP (opt_shared, 0, "shared", 0, \
      "Obtain a shared lock (exclusive is the default).")

static t_uchar long_help[] = ("execute a command while holding a file lock\n"
			      "Invoke COMMAND while holding a lock on FILE.\n"
			      "\n"
			      "With \"--shared\", hold a shared (read) lock;  otherwise hold\n"
			      "an exclusive (write) lock.\n");

enum options
{
  OPTS (OPT_ENUM, OPT_IGN)  
};

struct opt_desc opts[] = 
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
main (int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  int shared;
  char * file;
  char ** command_argv;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  shared = 0;

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, long_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
	break;
      switch (o)
	{
	default:
	  safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
	  panic ("internal error parsing arguments");

	usage_error:
	  opt_usage (2, argv[0], program_name, usage, 1);
	  exit (1);

	/* bogus_arg: */
	  safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
	  goto usage_error;

	case opt_shared:
	  shared = 1;
	  break;
	}
    }


  if (argc < 3)
    goto usage_error;

  file = argv[1];
  command_argv = argv + 2;

  {
    int file_fd;
    int subproc;

    /* obtain the lock
     */
    {
      struct flock lock_data;

      file_fd = safe_open (file, (shared ? O_RDONLY : O_WRONLY), 0);
      lock_data.l_type = (shared ? F_RDLCK : F_WRLCK);
      lock_data.l_whence = SEEK_SET;
      lock_data.l_start = 0;
      lock_data.l_len = 0;
      safe_fcntl (file_fd, F_SETLKW, (long)&lock_data);
    }

    /* execute the command
     */
    {
      subproc = fork ();

      if (subproc < 0)
	panic ("unable to fork");

      if (subproc == 0)
	{
	  safe_close (file_fd);
	  execvp (command_argv[0], command_argv);
	  safe_printfmt (2, "with-file-lock: unable to exec %s\n", command_argv[0]);
	  exit (1);
	}
    }

    /* wait for the command
     */
    {
      int status;
      int wait_pid;
      
      wait_pid = waitpid (subproc, &status, 0);

      if (wait_pid < 0)
	panic ("with-file-lock: error waiting for subprocess");

      if (wait_pid == 0)
	{
	  panic ("with-file-lock: received wait status for unrecognized process");
	}
      else if (WIFEXITED (status))
	{
	  /* propogate exit status upwards.
	   */
	  exit (WEXITSTATUS (status));
	}
      else
	{
	  panic ("with-file-lock: waitpid returned for a non-exited process");
	}
    }
  }

  return 0;
}

/* tag: Tom Lord Sun Jan 20 14:56:30 2002 (with-file-lock.c)
 */
