# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman, Enno Cramer
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.006;
use strict;
use warnings;

package ArchWay::MainWindow::RevisionBase;

use Glib qw(TRUE FALSE);
use Gtk2;

use ArchWay::Widget::Revisions;
use ArchWay::Widget::Log;

use base 'ArchWay::Util::Dialogs';

sub add_revision_actions ($) {
	my $self = shift;

	# items (name, stock id, label, accelerator, tooltip, callback)
	my @items = (
		[
			"GetTree", "gtk-open", "Get _Tree",
			undef, "Get project tree",
			sub { $self->get_tree }
		],
		[
			"GetCSet", "gtk-save", "Get C_Set",
			undef, "Download changeset",
			sub { $self->get_cset }
		],
		[
			"ViewCSet", "gtk-find", "View _CSet",
			undef, "View changeset",
			sub { $self->view_cset }
		],
		[
			"ViewLog", "gtk-find", "View _Log",
			undef, "View log message",
			sub { $self->view_log }
		],
	);

	# add menu/toolbar to ui
	my $actions = Gtk2::ActionGroup->new("RevisionActions");
	$actions->add_actions(\@items, undef);
	$self->ui->insert_action_group($actions, 0);
}

sub sync_revision_actions ($) {
	my $self = shift;

	$self->set_widget_sensitivity(
		defined $self->revisions_view->get_selection->get_selected,
		qw(
			/MenuBar/ApplicationMenus/RevisionMenu/GetTree
			/ToolBar/GetTree
		)
	);

	$self->set_widget_sensitivity(
		$self->revisions_view->get_selected_is_revision || FALSE,
		qw(
			/MenuBar/ApplicationMenus/RevisionMenu/GetCSet
			/MenuBar/ApplicationMenus/RevisionMenu/ViewCSet
			/MenuBar/ApplicationMenus/RevisionMenu/ViewLog
			/ToolBar/GetCSet
			/ToolBar/ViewCSet
			/ToolBar/ViewLog
		)
	);
}

sub revisions_view {
	my $self = shift;

	if (! exists $self->{revisions_view}) {
		my $rv = ArchWay::Widget::Revisions->new;

		$rv->get_selection->signal_connect(
			changed => sub {
				$self->sync_revision_actions;

				my $status = "";
				if (defined $rv->get_selection->get_selected) {
					$status = $rv->get_selected_is_revision
						? $rv->get_selected_revision
						: $rv->get_selected_version;
				}		

				$self->set_status($status)
					if $self->can("set_status");
			}
		);

		$self->{revisions_view} = $rv;
		$self->sync_revision_actions;
	}

	return $self->{revisions_view};
}

sub get_tree ($) {
	my $self = shift;

	my $path = $self->prompt_create_dir;
	if (defined $path) {
		my $version = $self->revisions_view->get_selected_is_revision
			? $self->revisions_view->get_selected_revision
			: $self->revisions_view->get_selected_version;

		$self->arch->get_tree(
			$version,
			$path
		);
		$self->session->activate_gui('tree', $path);
	}
}

sub get_cset ($) {
	my $self = shift;

	my $path = $self->prompt_create_dir;
	if (defined $path) {
		$self->arch->get_revision_changeset(
			$self->revisions_view->get_selected_revision,
			$path
		);
	}
}

sub view_cset ($) {
	my $self = shift;

	$self->session->activate_gui(
		'cset',
		$self->revisions_view->get_selected_revision
	);
}

sub view_log ($) {
	my $self = shift;
	my $rev = $self->revisions_view->get_selected_revision;

	my $logview = ArchWay::Widget::Log->new;
	$logview->set_editable(FALSE);
	$logview->set_cursor_visible(FALSE);
	$logview->show($self->arch->get_revision_log($rev)->get_message);

	my $scwin = Gtk2::ScrolledWindow->new;
	$scwin->set_policy('automatic', 'automatic');
	$scwin->add($logview);

	my $dialog = Gtk2::Dialog->new($rev,
		$self, 'destroy-with-parent',
		'gtk-close' => 'none',
	);
	$dialog->signal_connect(response => sub { $_[0]->destroy });
	
	$dialog->vbox->add($scwin);
	$dialog->set_default_size(400, 450);
	$dialog->show_all;
}

1;

__END__
