/*
 * Copyright (c) 2000-2001 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * ragator - argus record aggregator  with flow model manipulation.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#ifndef ArgusClient
#define ArgusClient
#endif


#include <argus_client.h>
#include <ramon.h>

#include <math.h>


#define RA_HASHTABLESIZE	65536

int RaFlowModelFile = 0;
int RaAllocHashTableHeaders = 0;
int RaAllocArgusRecordStore = 0;
int RaAllocArgusRecord      = 0;

#define RA_SVCPASSED	0x0100
#define RA_SVCFAILED	0x0200
#define RA_SVCTEST	(RA_SVCFAILED|RA_SVCPASSED)


struct ArgusServiceRecord {
   u_int status;
   struct ArgusRecord *argus;
};
 
void RaProcessThisSrvRecord (struct ArgusServiceRecord *);
void RaProcessSrvRecord (struct ArgusServiceRecord *);
 
struct ArgusRecordStore *RaThisArgusStore = NULL;

char *RaMonMatrixFlowModelFile [] = {
"Flow     100       *                  *                 *       *        *         200      3153600",
"Model    200   255.255.255.255     255.255.255.255      no      no       no",
NULL,
};

char *RaMonTopNFlowModelFile [] = {
"Flow     100       *                  *                 *       *        *         200      3153600",
"Model    200   255.255.255.255     0.0.0.0              no      no       no",
NULL,
};

int RaInitialized = 0;
int RaTopNModel = 0;

void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      RaWriteOut = 0;
   
      bzero ((char *) RaFlowArray,  sizeof(RaFlowArray));
      bzero ((char *) RaModelArray, sizeof(RaModelArray));
   
      if (Mflag == NULL) 
         usage();

      if (!(strcasecmp (Mflag, "TopN"))) {
         RaFlowModelFile = RaReadFlowModelFile (RaMonTopNFlowModelFile);
         RaTopNModel++;

      } else {
         if (!(strcasecmp (Mflag, "Matrix"))) {
            RaFlowModelFile = RaReadFlowModelFile (RaMonMatrixFlowModelFile);
         }
      }

      if (!(RaFlowModelFile))
         usage();

      if ((RaModelerQueue = RaNewQueue(256)) == NULL)
         exit(0);
   
      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RA_HASHTABLESIZE;
      }

      if (nflag) {
         hfield = 15;
         pfield =  5;
      }
      RaInitialized++;
   }
#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusClientInit: returning\n");
#endif
}

void RaPackQueue (struct RaQueueStruct *);
void RaSortQueue (struct RaQueueStruct *);
void RaPrintOutQueue (struct RaQueueStruct *);

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if (!RaParseCompleting) {
      RaParseCompleting++;
      RaPackQueue (RaModelerQueue);
      RaSortQueue (RaModelerQueue);
      RaPrintOutQueue (RaModelerQueue);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete: returning\n");
#endif
}

int ArgusHourlyUpdate = 1;
int ArgusMinuteUpdate = 1;

void
ArgusClientTimeout ()
{
   RaProcessQueue (RaModelerQueue, ARGUS_STATUS);

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusMinuteUpdate++ == 60) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s HashHdrs %d Hash %d Queue %d\n",
                     print_time(&ArgusGlobalTime), RaAllocHashTableHeaders,
                     RaHashTable.count, RaModelerQueue->count);
      ArgusMinuteUpdate = 1;
   }

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}

void
parse_arg (int argc, char**argv)
{ 
}



void
usage ()
{
   extern char version[];

   fprintf (stderr, "Ramon Version %s\n", version);
   fprintf (stderr, "usage: %s -M mode [-N num]\n", ArgusProgramName);
   fprintf (stderr, "usage: %s -M mode [-N num] [ra-options] [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "options:    -M <mode>      specify the rmon function. possible <modes> are:\n");
   fprintf (stderr, "                              TopN, Matrix\n");
   fprintf (stderr, "            -N <number>    specify the top <number> of entries to print (all).\n");
   fprintf (stderr, "ra-options: -a             print record summaries on termination.\n");
   fprintf (stderr, "            -A             print application bytes.\n");
   fprintf (stderr, "            -b             dump packet-matching code.\n");
   fprintf (stderr, "            -c             print packet and byte counts.\n");
   fprintf (stderr, "            -C             treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>     specify debug level\n");
   fprintf (stderr, "            -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "            -g             print record time duration.\n");
   fprintf (stderr, "            -G             print both start and last time values.\n");
   fprintf (stderr, "            -h             print help.\n");
   fprintf (stderr, "            -I             print transaction state and option indicators.\n");
   fprintf (stderr, "            -l             print last time values [default is start time].\n");
   fprintf (stderr, "            -n             don't convert numbers to names.\n");
   fprintf (stderr, "            -p <digits>    print fractional time with <digits> precision.\n");
   fprintf (stderr, "            -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "            -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "            -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "                  format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                           timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                               mm/dd[/yy]\n");
   fprintf (stderr, "            -T <secs>      attach to remote server for T seconds.\n");
   fprintf (stderr, "            -u             print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>      write output to <file>. '-' denotes stdout.\n");
   exit(1);
}


struct ArgusRecord * RaCopyArgusRecord (struct ArgusRecord *);
struct ArgusRecord * RaSplitArgusRecord (struct ArgusRecord *);
struct ArgusRecord * RaRmonArgusRecord (struct ArgusRecord *);

struct ArgusRecord *
RaCopyArgusRecord (struct ArgusRecord *argus)
{
   int length = 0;
   struct ArgusRecord *retn = NULL;

   if (argus && ((length = argus->ahdr.length) > 0))
      if ((retn = (struct ArgusRecord *) ArgusCalloc(1, length)) != NULL)
         bcopy ((char *) argus, (char *) retn, length);

   return (retn);
}


struct ArgusRecord *
RaRmonArgusRecord (struct ArgusRecord *argus)
{
   struct ArgusRecord *retn = NULL;
   struct ArgusMeter meter;
   struct ArgusFlow *flow1, *flow2;

   if ((retn = RaCopyArgusRecord (argus)) != NULL) {
      flow1 = &argus->argus_far.flow;
      flow2 = &retn->argus_far.flow;

      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            flow2->ip_flow.ip_src = flow1->ip_flow.ip_dst;
            flow2->ip_flow.ip_dst = flow1->ip_flow.ip_src;

            switch (flow1->ip_flow.ip_p) {
               case IPPROTO_UDP:
               case IPPROTO_TCP:
                  flow2->ip_flow.sport = flow1->ip_flow.dport;
                  flow2->ip_flow.dport = flow1->ip_flow.sport;
                  break;
            }

            meter = retn->argus_far.src;
            retn->argus_far.src = retn->argus_far.dst;
            retn->argus_far.dst = meter;

            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            break;

         default:
            break;
      }

       retn->ahdr.type |= ARGUS_RMON;
      argus->ahdr.type |= ARGUS_RMON;
   }

   return (retn);
}


void
RaProcessThisSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *arg2 = NULL;

   if (RaTopNModel) {
      arg2 = RaRmonArgusRecord(srv->argus);
      RaProcessSrvRecord(srv);
      srv->argus = arg2;
      RaProcessSrvRecord(srv);
      ArgusFree(arg2);

   } else
      RaProcessSrvRecord(srv);
}

struct ArgusRecordStore *RaFindRevArgusRecord(struct ArgusRecord *);

void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;

   struct ArgusRecordStore *store;
   struct RaPolicyStruct *rap;
   int RaTimeout = -1;

   if ((rap = RaFlowModelOverRides(argus)) != NULL) {
      RaModifyFlow(rap, argus);
      RaTimeout = rap->ArgusTimeout;
   } else
      RaModifyDefaultFlow(argus);


   if ((store = RaFindArgusRecord(argus)) == NULL) {
/*
      store = RaFindRevArgusRecord(argus);
*/
   }

   if (store) {
      RaThisArgusStore = store;

      if (RaCheckTimeout (store))
         RaSendArgusRecord(store);

      if ((srv->status & RA_SVCTEST) != (store->status & RA_SVCTEST)) {
         RaSendArgusRecord(store);
         store->status &= ~RA_SVCTEST;
         store->status |= (srv->status & RA_SVCTEST);
      }

      RaMergeArgusRecord(argus, store);
      RaUpdateArgusStore(argus, store);

   } else {
      if ((store = RaNewArgusStore(argus)) != NULL) {
         RaThisArgusStore = store;
         store->farhdrstatus = ArgusIndexRecord (store->argus, store->farhdrs);
         store->ArgusTimeout = RaTimeout;
         store->status |= RA_MODIFIED;
         store->status |= srv->status & RA_SVCTEST;

         if (store->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
            bcopy((char *)store->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&store->agr, sizeof(store->agr));

         } else {
            struct ArgusFarStruct *far = &argus->argus_far;
            if (far->type == ARGUS_FAR) {
               store->agr.type = ARGUS_AGR_DSR;
               store->agr.length = (u_char) sizeof(store->agr);
               store->agr.count = 1;
            }
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessSrvRecord: done\n");
#endif
}


struct ArgusServiceRecord ArgusThisSrv;

void
process_man (struct ArgusRecord *argus)
{
   unsigned int status;
   struct ArgusRecord *initCon = &ArgusInput->ArgusInitCon;
   
   initCon->ahdr.type |= ARGUS_RMON;
   status = ntohl(initCon->ahdr.status);

   if (RaTopNModel)
      status |= ARGUS_TOPN;
   else
      status |= ARGUS_MATRIX;

   initCon->ahdr.status = htonl(status);

/*
   if (wfile) {
#ifdef _LITTLE_ENDIAN
      ArgusHtoN(argus);
#endif

      ArgusWriteNewLogfile (wfile, argus);

#ifdef _LITTLE_ENDIAN
      ArgusNtoH(argus);
#endif
   }
*/
}


void
process_tcp (struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
         if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
            if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT)) {
               if (tcp->state & ARGUS_RESET) {
                  if (tcp->state & ARGUS_DST_RESET)
                     if (argus->argus_far.src.count && argus->argus_far.dst.count)
                        if (!(argus->argus_far.src.bytes && argus->argus_far.dst.bytes))
                           ArgusThisSrv.status = RA_SVCFAILED;
       
                  if (tcp->state & ARGUS_SRC_RESET)
                     if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                        ArgusThisSrv.status = RA_SVCFAILED;
               }
            }
    
            if (tcp->state & ARGUS_TIMEOUT)
               if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                  ArgusThisSrv.status = RA_SVCFAILED;
         }
      }
   }

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
process_icmp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
process_udp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }


   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
process_ip (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;
   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
process_arp (struct ArgusRecord *argus)
{
/*
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;
   RaProcessThisSrvRecord (&ArgusThisSrv);
*/
}


void
process_non_ip (struct ArgusRecord *argus)
{
/*
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;
   RaProcessThisSrvRecord (&ArgusThisSrv);
*/
}


void
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   char buf[MAXSTRLEN], *ptr = buf;
   struct ArgusRecord *argus = store->argus;
   struct ArgusFarHeaderStruct *farhdr;
   int length = argus->ahdr.length - sizeof(argus->ahdr);

   if (argus && (store->status & RA_MODIFIED)) {
      argus->argus_far.time.last = store->agr.lasttime;
      if (store->act.n > 0) {
         store->agr.act.mean = store->act.sumtime/store->act.n;
         store->agr.act.stdev = sqrt (store->act.sumsqrd/store->act.n - pow (store->act.sumtime/store->act.n, 2.0));
      }
      if (store->idle.n > 0) {
         store->agr.idle.mean = store->idle.sumtime/store->idle.n;
         store->agr.idle.stdev = sqrt (store->idle.sumsqrd/store->idle.n - pow (store->idle.sumtime/store->idle.n, 2.0));
      }

      bcopy ((char *) argus, buf, argus->ahdr.length);
      bcopy ((char *)&store->agr,&buf[argus->ahdr.length], store->agr.length);

      argus = (struct ArgusRecord *) ptr;
      argus->ahdr.length += store->agr.length;

      if (wfile) {

#ifdef _LITTLE_ENDIAN
         ArgusHtoN(argus);
#endif

         ArgusWriteNewLogfile (wfile, argus);

#ifdef _LITTLE_ENDIAN
         ArgusNtoH(argus);
#endif

      } else {

         if (argus->ahdr.type & ARGUS_MAR)
            printf ("%s\n", get_man_string (argus));
            
         else {
            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            switch (argus->ahdr.status & 0xFFFF) {
               case ETHERTYPE_IP:
                  switch (argus->argus_far.flow.ip_flow.ip_p) {
                     case IPPROTO_TCP:              
                        printf ("%s\n", get_tcp_string (argus));
                        break;
      
                     case IPPROTO_ICMP:              
                        printf ("%s\n", get_icmp_string (argus));
                        break;
      
                     default:
                        printf ("%s\n", get_ip_string (argus));
                        break;
                  }
                  break;
      
               case ETHERTYPE_ARP:
               case ETHERTYPE_REVARP:
                  printf ("%s\n", get_arp_string (argus));
                  break;
      
               default:
                  printf ("%s\n", get_nonip_string (argus));
                  break;
            }
         }
         fflush (stdout);
      }

      argus = store->argus;
      if (argus->ahdr.type & ARGUS_FAR) {
         int farlen;
         farhdr = (struct ArgusFarHeaderStruct *)((char *)argus + sizeof(argus->ahdr));

         while (length > 0) {
            switch (farhdr->type) {
               case ARGUS_FAR: {
                  struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;
                  far->time.start.tv_sec = 0x7FFFFFFF; far->time.start.tv_usec = 0;
                  far->time.last.tv_sec = 0x7FFFFFFF;  far->time.last.tv_usec = 0;
                  far->src.count = 0; far->src.bytes = 0;
                  far->dst.count = 0; far->dst.bytes = 0;
                  break;
               }

               case ARGUS_TCP_DSR: {
                  struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr;

                  tcp->state = 0;
                  tcp->src.seqbase = 0; tcp->src.ackbytes = 0;
                  tcp->src.rpkts = 0; tcp->src.win = 0; tcp->src.flags = 0;
                  tcp->dst.seqbase = 0; tcp->dst.ackbytes = 0;
                  tcp->dst.rpkts = 0; tcp->dst.win = 0; tcp->dst.flags = 0;

                  break;
               }

               case ARGUS_AGR_DSR: {
                  struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;

                  agr->count = 0;
                  agr->act.min = 0x7FFFFFFF;   agr->act.mean = 0;
                  agr->act.stdev = 0;  agr->act.max = 0;
                  agr->idle.min = 0x7FFFFFFF; agr->idle.mean = 0;
                  agr->idle.stdev = 0; agr->idle.max = 0;
                  break;
               }
            }
            farlen = farhdr->length;
            if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
                (farhdr->type == ARGUS_DSTUSRDATA_DSR))
               farlen = farlen * 4;
            length -= farlen;
            farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
         }
      }

      store->agr.count = 0;
      store->agr.act.n = 0;
      store->agr.act.min = 0x7FFFFFFF;
      store->agr.act.mean = 0;
      store->agr.act.stdev = 0;
      store->agr.act.max = 0;

      store->agr.idle.n = 0;
      store->agr.idle.min = 0x7FFFFFFF;
      store->agr.idle.mean = 0;
      store->agr.idle.stdev = 0;
      store->agr.idle.max = 0;

      store->act.n = 0;
      store->act.sumtime = 0;
      store->act.sumsqrd = 0;

      store->idle.n = 0;
      store->idle.sumtime = 0;
      store->idle.sumsqrd = 0;

      store->argus->ahdr.status &= ~ARGUS_MERGED;
      store->status &= ~RA_MODIFIED;
   }

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
}

int
RaCIDRAddrMatches (u_int addr, struct RaCIDRAddr *cidr)
{
   int retn = 1;

   if (cidr->addr)
      if ((addr & cidr->mask) != cidr->addr)
         retn = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaCIDRAddrMatches(0x%x, 0x%x) returned %d\n", addr, cidr, retn);
#endif

   return (retn);
}

int
RaProtoMatches (u_short p, int proto)
{
   int retn = 1;

   if (proto != 0xFF)
      retn = (p == proto) ? 1 : 0;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProtoMatches (%d, %d) returned %d\n", p, proto, retn);
#endif

   return (retn);
}


int
RaPortMatches (u_short p1, u_short p2)
{
   int retn = 1;
   
   if (p2 != 0xFFFF)
      retn = (p1 == p2) ? 1 : 0;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaPortMatches (%d, %d) returned %d\n", p1, p2, retn);
#endif

   return (retn);
}

int
RaPolicyMatch (struct ArgusRecord *argus, struct RaPolicyStruct *rap)
{
   int retn = 0;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   u_char proto = flow->ip_flow.ip_p;

   if (RaCIDRAddrMatches (flow->ip_flow.ip_src, &rap->src))

   if (RaCIDRAddrMatches (flow->ip_flow.ip_dst, &rap->dst))

   if (RaProtoMatches (flow->ip_flow.ip_p, rap->proto))

   switch (proto) {
      case IPPROTO_TCP:
      case IPPROTO_UDP:
         if (RaPortMatches (flow->ip_flow.sport, rap->sport))
         if (RaPortMatches (flow->ip_flow.dport, rap->dport))
            retn = 1;
         break;

      case IPPROTO_ICMP:
         if (RaPortMatches (flow->icmp_flow.type, rap->sport))
         if (RaPortMatches (flow->icmp_flow.code, rap->dport))
         retn = 1;
         break;

      default:
         retn = 1;
         break;
   }


#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaPolicyMatch(0x%x, 0x%x) returned %d\n", argus, rap, retn);
#endif

   return (retn);
}


struct RaPolicyStruct *
RaFlowModelOverRides (struct ArgusRecord *argus)
{
   struct RaPolicyStruct *retn = NULL;
   int i;

   if (RaThisFlowNum) {
      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            for (i = 0; i < RaThisFlowNum; i++) {
               if (RaPolicyMatch (argus, RaFlowArray[i])) {
                  retn = RaFlowArray[i];
                  break;
               }
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            break;

         default:
            break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaFlowModelOverRides(0x%x) returned 0x%x\n", argus, retn);
#endif

   return (retn);
}

#if !defined(__OpenBSD__) || !defined(_NETINET_IP_ICMP_H_)
#include <netinet/ip_icmp.h>
#define _NETINET_IP_ICMP_H_
#endif

void
RaModifyDefaultFlow (struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;

   switch (argus->ahdr.status & 0xFFFF) {
      case ETHERTYPE_IP:
         if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_FRAG_FLOWTAG) {
            argus->argus_far.flow.ip_flow.tp_p = 0;
            argus->argus_far.flow.ip_flow.ip_id = 0;
         }

         switch (argus->argus_far.flow.ip_flow.ip_p) {
            case IPPROTO_UDP:
            case IPPROTO_TCP:
               argus->argus_far.flow.ip_flow.ip_id = 0;
               break;

            case IPPROTO_ICMP:
               if (flow->icmp_flow.type != ICMP_UNREACH)
                  flow->icmp_flow.id = 0xFFFF;

               flow->icmp_flow.ip_id = 0xFFFF;
               break;

            default:
               argus->argus_far.flow.ip_flow.ip_id = 0;
               break;
         }
         break;

      case ETHERTYPE_ARP:
      case ETHERTYPE_REVARP:
         break;

      default:
         break;
   }
}

int RaFlowMajorModified = 0;

void
RaModifyFlow (struct RaPolicyStruct *rap, struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;
   struct RaPolicyStruct *model = RaModelArray[rap->RaModelId];
   u_short proto;

   RaFlowMajorModified = 0;

   if (model) {
      flow->ip_flow.ip_src = flow->ip_flow.ip_src & model->src.addr;
      flow->ip_flow.ip_dst = flow->ip_flow.ip_dst & model->dst.addr;

      proto = flow->ip_flow.ip_p;

      flow->ip_flow.ip_p  &= model->proto;

      switch (proto) {
         case IPPROTO_TCP:
         case IPPROTO_UDP:
            flow->ip_flow.ip_id = 0;
            switch (flow->ip_flow.ip_p) {
               case IPPROTO_TCP:
               case IPPROTO_UDP:
                  if (!(model->sport))
                     flow->ip_flow.sport = 0xFFFF;
                  if (!(model->dport)) {
                     flow->ip_flow.dport = 0xFFFF;
                     RaFlowMajorModified = 1;
                  }
                  break;

               default:
                  flow->ip_flow.sport = 0xFFFF;
                  flow->ip_flow.dport = 0xFFFF;
                  break;
            }
            break;
    
         case IPPROTO_ICMP:
            if (!(model->sport))
               flow->icmp_flow.type = 0xFF;

            if (!(model->dport))
               flow->icmp_flow.code = 0xFF;

            flow->icmp_flow.id = 0xFFFF;
            flow->icmp_flow.ip_id = 0xFFFF;
            break;

         default:
            flow->ip_flow.ip_id = 0;
            break;
      }

      if (flow->ip_flow.ip_p == 0) {
         flow->ip_flow.tp_p  = 0;
         flow->ip_flow.ip_id = 0;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaModifyFlow(0x%x, 0x%x) done.\n", rap, argus);
#endif

}


int RaThisHash = 0;
struct ArgusFlow *RaThisFlow;
int RaThisFlowRev = 0;

struct ArgusRecordStore *
RaFindRevArgusRecord(struct ArgusRecord *argus)
{
   struct ArgusRecordStore *retn = NULL;
   struct RaHashTableHeader *hashEntry;
   struct ArgusFlow flow = *RaThisFlow;
   struct ArgusTCPObject *tcp = NULL;

   RaThisFlowRev = 0;

   switch (flow.ip_flow.ip_p) {
      case IPPROTO_UDP:              
         RaThisFlow->ip_flow.ip_src = flow.ip_flow.ip_dst;
         RaThisFlow->ip_flow.ip_dst = flow.ip_flow.ip_src;
         RaThisFlow->ip_flow.sport  = flow.ip_flow.dport;
         RaThisFlow->ip_flow.dport  = flow.ip_flow.sport;
         RaThisFlowRev++;

         if ((hashEntry = RaFindHashObject ()) != NULL)
            retn = hashEntry->storeobj;

         *RaThisFlow = flow;
         break;

      case IPPROTO_TCP:              
         if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
            if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
               if (!(tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT))) {
                  RaThisFlow->ip_flow.ip_src = flow.ip_flow.ip_dst;
                  RaThisFlow->ip_flow.ip_dst = flow.ip_flow.ip_src;
                  RaThisFlow->ip_flow.sport  = flow.ip_flow.dport;
                  RaThisFlow->ip_flow.dport  = flow.ip_flow.sport;
                  RaThisFlowRev++;

                  if ((hashEntry = RaFindHashObject ()) != NULL)
                     retn = hashEntry->storeobj;

                  *RaThisFlow = flow;
               }
            }
         }
         break;

      default:
         break;
   }


   return (retn);
}

struct ArgusRecordStore *
RaFindArgusRecord(struct ArgusRecord *argus)
{
   struct ArgusRecordStore *retn = NULL;
   struct RaHashTableHeader *hashEntry;
   unsigned short hash = 0, *ptr;
   int i, len;

   RaThisFlow = &argus->argus_far.flow;
   ptr = (unsigned short *) RaThisFlow;

   for (i = 0, len = (sizeof(*RaThisFlow)) / sizeof(unsigned short); i < len; i++)
      hash += *ptr++;

   RaThisHash = hash;

   if ((hashEntry = RaFindHashObject ()) != NULL)
      retn = hashEntry->storeobj;

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaFindArgusRecord: returning 0x%x\n", retn);
#endif

   return (retn);
}


int RaTotalRecords = 0;


struct ArgusRecordStore *
RaNewArgusStore(struct ArgusRecord *argus)
{
   struct ArgusRecordStore *retn;
   struct ArgusRecord *argusbuf;
   int length = argus->ahdr.length;
   struct timeval lasttime;

   if ((retn = (struct ArgusRecordStore *) ArgusCalloc (1, sizeof (*retn))) != NULL) {
      RaAllocArgusRecordStore++;
      if ((argusbuf = (struct ArgusRecord *) ArgusCalloc (1, length)) != NULL) {
         bcopy((char *) argus, (char *) argusbuf, length);
         retn->argus = argusbuf;
         RaTotalRecords++;
         if ((retn->rahtblhdr = RaAddHashEntry (retn)) != NULL) {
            RaAddToQueue(RaModelerQueue, &retn->qhdr);

            retn->agr.laststartime = argus->argus_far.time.start;
            lasttime = argus->argus_far.time.last;

            retn->qhdr.lasttime = lasttime;
            retn->qhdr.logtime  = lasttime;
            retn->agr.lasttime  = lasttime;

            retn->agr.act.min  = 0x7FFFFFFF;
            retn->agr.idle.min = 0x7FFFFFFF;
            retn->agr.status  |= ARGUS_AGR_USECACTTIME;
            retn->agr.status  |= ARGUS_AGR_USECIDLETIME;

         } else {
            ArgusFree (retn);
            RaAllocArgusRecordStore--;
            retn = NULL;
            ArgusFree (argusbuf);
         }
      } else {
         ArgusFree (retn);
         RaAllocArgusRecordStore--;
         retn = NULL;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaNewArgusStore: added 0x%x\n", retn);
#endif

   return (retn);
}

void
RaRemoveArgusRecord(struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveArgusRecord: done\n");
#endif
}

long long RaGetuSecDuration (struct ArgusRecord *argus);


long long
RaGetuSecDuration (struct ArgusRecord *argus)
{
   long long retn = 0;
   int sec, usec;

   sec  = argus->argus_far.time.last.tv_sec  - argus->argus_far.time.start.tv_sec;
   usec = argus->argus_far.time.last.tv_usec - argus->argus_far.time.start.tv_usec;

   if (usec < 0) {
      sec--; usec += 1000000;
   }

   retn = (sec * 1000000) + usec;

   return (retn);
}


void
RaMergeArgusRecord(struct ArgusRecord *a1, struct ArgusRecordStore *store)
{
   u_char buf[MAXSTRLEN];
   struct ArgusRecord *a2 = NULL, *tmp = (struct ArgusRecord *)buf;
   struct ArgusFarHeaderStruct *a1farhdr[32], *a2farhdr[32];
   struct ArgusFarStruct *far1 = NULL;
   struct ArgusFarStruct *far2 = NULL;
   unsigned int a1DSRStatus = 0, a2DSRStatus = 0;
   unsigned int i, status = 0, TimeMultiplier;
   long long duration;

   if ((a2 = store->argus) != NULL) {
      bzero (buf, sizeof(buf));
      bcopy ((char *) a2, buf, a2->ahdr.length);

      a1DSRStatus = ArgusIndexRecord (a1,  a1farhdr);
      a2DSRStatus = ArgusIndexRecord (tmp, a2farhdr);

      if (RaFlowMajorModified) {
         a1DSRStatus &= (ARGUS_FAR_DSR_INDEX || ARGUS_AGR_DSR_INDEX);
         store->farhdrstatus &= (ARGUS_FAR_DSR_INDEX || ARGUS_AGR_DSR_INDEX);
      }

      status = a1DSRStatus & a2DSRStatus;

      far1 = (struct ArgusFarStruct *) a1farhdr[ARGUS_FAR_DSR_INDEX];
      far2 = (struct ArgusFarStruct *) a2farhdr[ARGUS_FAR_DSR_INDEX];

      if (!(RaThisFlowRev)) {
         if (((far2->src.count + far1->src.count) < far2->src.count) ||
             ((far2->dst.count + far1->dst.count) < far2->dst.count) ||
             ((far2->src.bytes + far1->src.bytes) < far2->src.bytes) ||
             ((far2->dst.bytes + far1->dst.bytes) < far2->dst.bytes)) {
            goto doneRaMergeArgus;
         }
 
      } else {
         if (((far2->src.count + far1->dst.count) < far2->src.count) ||
             ((far2->dst.count + far1->src.count) < far2->dst.count) ||
             ((far2->src.bytes + far1->dst.bytes) < far2->src.bytes) ||
             ((far2->dst.bytes + far1->src.bytes) < far2->dst.bytes)) {
            goto doneRaMergeArgus;
         }
      }

      for (i = 0; i < 32; i++) {
         if (status & 0x01) {
            switch (i) {
               case ARGUS_FAR_DSR_INDEX: {
                  u_int diffstartime = 0;
                  int diffusec = 0;

                  far1 = (struct ArgusFarStruct *) a1farhdr[i];
                  far2 = (struct ArgusFarStruct *) a2farhdr[i];

                  if (!(RaThisFlowRev)) {
                     if (((far2->src.count + far1->src.count) < far2->src.count) ||
                         ((far2->src.bytes + far1->src.bytes) < far2->src.bytes) ||
                         ((far2->dst.count + far1->dst.count) < far2->dst.count) ||
                         ((far2->dst.bytes + far1->dst.bytes) < far2->dst.bytes))
                        RaSendArgusRecord(store);

                     far2->src.count    += far1->src.count;
                     far2->src.bytes    += far1->src.bytes;
                     far2->dst.count    += far1->dst.count;
                     far2->dst.bytes    += far1->dst.bytes;
                     far2->src.appbytes += far1->src.appbytes;
                     far2->dst.appbytes += far1->dst.appbytes;

                     if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
                        if (far1->attr_ip.sttl != far2->attr_ip.sttl) far2->attr_ip.sttl = 0xFF;
                        if (far1->attr_ip.dttl != far2->attr_ip.dttl) far2->attr_ip.dttl = 0xFF;
                        if (far1->attr_ip.stos != far2->attr_ip.stos) far2->attr_ip.stos = 0xFF;
                        if (far1->attr_ip.dtos != far2->attr_ip.dtos) far2->attr_ip.dtos = 0xFF;
                     }

                  } else {
                     if (((far2->src.count + far1->dst.count) < far2->src.count) ||
                         ((far2->src.bytes + far1->dst.bytes) < far2->src.bytes) ||
                         ((far2->dst.count + far1->src.count) < far2->dst.count) ||
                         ((far2->dst.bytes + far1->src.bytes) < far2->dst.bytes))
                        RaSendArgusRecord(store);

                     if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
                        if (far1->attr_ip.sttl != far2->attr_ip.dttl) far2->attr_ip.dttl = 0xFF;
                        if (far1->attr_ip.dttl != far2->attr_ip.sttl) far2->attr_ip.sttl = 0xFF;
                        if (far1->attr_ip.stos != far2->attr_ip.dtos) far2->attr_ip.dtos = 0xFF;
                        if (far1->attr_ip.dtos != far2->attr_ip.stos) far2->attr_ip.stos = 0xFF;
                     }

                     far2->src.count    += far1->dst.count;
                     far2->src.bytes    += far1->dst.bytes;
                     far2->dst.count    += far1->src.count;
                     far2->dst.bytes    += far1->src.bytes;
                     far2->src.appbytes += far1->dst.appbytes;
                     far2->dst.appbytes += far1->src.appbytes;
                  }


                  diffstartime  = (far1->time.start.tv_sec  - store->agr.laststartime.tv_sec);
                  if ((diffusec = (far1->time.start.tv_usec - store->agr.laststartime.tv_usec)) < 0) {
                     diffstartime--;
                     diffusec += 1000000;
                  }

                  store->agr.laststartime = far1->time.start;

                  if (diffstartime > 0) {
                     if (diffstartime > 2147) {
                        if (store->agr.status & ARGUS_AGR_USECIDLETIME) {
                           if (store->agr.idle.min != 0x7FFFFFFF)
                              store->agr.idle.min /= 1000;
                           store->agr.idle.max /= 1000;
                           store->idle.sumtime /= 1000;
                           store->idle.sumsqrd /= 1000000.0;
                           store->agr.status &= ~ARGUS_AGR_USECIDLETIME;
                           store->agr.status |=  ARGUS_AGR_MSECIDLETIME;
                        }
                     }

                     if (store->agr.status & ARGUS_AGR_USECIDLETIME)
                        TimeMultiplier = 1000000;
                     else
                        TimeMultiplier = 1000;

                     diffstartime *= TimeMultiplier;
                     diffstartime += (diffusec * 1000) / (TimeMultiplier / 1000);

                     if ((store->agr.idle.min > diffstartime) && (diffstartime > 0))
                        store->agr.idle.min = diffstartime;

                     if ((store->agr.idle.max < diffstartime) && (diffstartime > 0))
                        store->agr.idle.max = diffstartime;

                     store->idle.sumtime += diffstartime;
                     store->idle.sumsqrd += pow(diffstartime, 2.0);
                     store->idle.n++;
                  }

                  duration = RaGetuSecDuration(a1);
                  if (duration > 0x6FFFFFFF) {
                     if (store->agr.status & ARGUS_AGR_USECIDLETIME) {
                         if (store->agr.idle.min != 0x7FFFFFFF)
                            store->agr.act.min /= 1000;
                         store->agr.act.max /= 1000;
                         store->act.sumtime /= 1000;
                         store->act.sumsqrd /= 1000000.0;
                         store->agr.status &= ~ARGUS_AGR_USECACTTIME;
                         store->agr.status |=  ARGUS_AGR_MSECACTTIME;
                     }
                  }

                  if (store->agr.status & ARGUS_AGR_USECACTTIME)
                     TimeMultiplier = 1;
                  else
                     TimeMultiplier = 1000;

                  duration /= TimeMultiplier;

                  if ((store->agr.act.min > duration) && (duration > 0))
                     store->agr.act.min = duration;

                  if ((store->agr.act.max < duration) && (duration > 0))
                     store->agr.act.max = duration;

                  store->act.sumtime += duration;
                  store->act.sumsqrd += pow(duration, 2.0);
                  store->act.n++;

                  store->agr.count++;

                  if (far1->ArgusTransRefNum != far2->ArgusTransRefNum)
                     far2->ArgusTransRefNum = 0;

                  if ((far2->time.start.tv_sec  > far1->time.start.tv_sec) ||
                     ((far2->time.start.tv_sec == far1->time.start.tv_sec) &&
                     (far2->time.start.tv_usec  > far1->time.start.tv_usec)))

                     far2->time.start = far1->time.start;

                  if ((far2->time.last.tv_sec  < far1->time.last.tv_sec) ||
                     ((far2->time.last.tv_sec == far1->time.last.tv_sec) &&
                     (far2->time.last.tv_usec  < far1->time.last.tv_usec)))

                     far2->time.last = far1->time.last;

                  if (RaThisArgusStore->qhdr.lasttime.tv_sec != far2->time.last.tv_sec) {
                     RaThisArgusStore->qhdr.lasttime = far2->time.last;
                     store->agr.lasttime = far2->time.last;
                  } else
                     if (RaThisArgusStore->qhdr.lasttime.tv_usec < far2->time.last.tv_usec) {
                        RaThisArgusStore->qhdr.lasttime = far2->time.last;
                     }

                  break;
               }

               case ARGUS_TCP_DSR_INDEX: {
                  struct ArgusTCPObject *tcp1 = (struct ArgusTCPObject *) a1farhdr[i];
                  struct ArgusTCPObject *tcp2 = (struct ArgusTCPObject *) a2farhdr[i];

                  tcp2->state |= tcp1->state;

                  if (!(tcp2->synAckuSecs))
                     tcp2->synAckuSecs = tcp1->synAckuSecs;
                  else
                     if (tcp2->synAckuSecs > tcp1->synAckuSecs)
                        if (tcp1->synAckuSecs)
                           tcp2->synAckuSecs = tcp1->synAckuSecs;

                  if (!(tcp2->ackDatauSecs))
                     tcp2->ackDatauSecs = tcp1->ackDatauSecs;
                  else
                     if (tcp2->ackDatauSecs > tcp1->ackDatauSecs)
                        if (tcp1->ackDatauSecs)
                           tcp2->ackDatauSecs = tcp1->ackDatauSecs;

                  tcp2->src.seqbase   = 0;
                  tcp2->dst.seqbase   = 0;

                  tcp2->src.ackbytes += tcp1->src.ackbytes;
                  tcp2->dst.ackbytes += tcp1->dst.ackbytes;
                  tcp2->src.bytes    += tcp1->src.bytes;
                  tcp2->dst.bytes    += tcp1->dst.bytes;
                  tcp2->src.rpkts    += tcp1->src.rpkts;
                  tcp2->dst.rpkts    += tcp1->dst.rpkts;        

                  if (tcp2->src.win > tcp1->src.win);
                     tcp2->src.win = tcp1->src.win;
                  if (tcp2->dst.win > tcp1->dst.win);
                     tcp2->dst.win = tcp1->dst.win;        

                  tcp2->src.flags    |= tcp1->src.flags;
                  tcp2->dst.flags    |= tcp1->dst.flags;        

                  break;
               }

               case ARGUS_RTP_DSR_INDEX: {
                  struct ArgusRTPObject *rtp1 = (struct ArgusRTPObject *) a1farhdr[i];
                  struct ArgusRTPObject *rtp2 = (struct ArgusRTPObject *) a2farhdr[i];

                  if ((((rtp2->sdrop + rtp1->sdrop) < rtp2->sdrop) || ((rtp2->ddrop + rtp1->ddrop) < rtp2->ddrop)) ||
                      (((rtp2->ssdev + rtp1->ssdev) < rtp2->ssdev) || ((rtp2->dsdev + rtp1->dsdev) < rtp2->dsdev))) {
                     RaSendArgusRecord(store);
                     rtp2->sdrop = rtp1->sdrop;
                     rtp2->ddrop = rtp1->ddrop;
                     rtp2->ssdev = rtp1->ssdev;
                     rtp2->dsdev = rtp1->dsdev;

                  } else {
                     rtp2->sdrop += rtp1->sdrop;
                     rtp2->ddrop += rtp1->ddrop;
                     rtp2->ssdev += rtp1->ssdev;
                     rtp2->dsdev += rtp1->dsdev;
                  }
                  break;
               }

               case ARGUS_ICMP_DSR_INDEX:
               case ARGUS_IGMP_DSR_INDEX:
               case ARGUS_ARP_DSR_INDEX:
               case ARGUS_FRG_DSR_INDEX:
                  break;

               case ARGUS_VLAN_DSR_INDEX: {
                  struct ArgusVlanStruct *vlan1 = (struct ArgusVlanStruct *) a1farhdr[i];
                  struct ArgusVlanStruct *vlan2 = (struct ArgusVlanStruct *) a2farhdr[i];

                  if ((vlan2->status & ARGUS_SRC_VLAN) && (vlan1->status & ARGUS_SRC_VLAN))
                     if (vlan2->sid != vlan1->sid)
                        vlan2->status |= ARGUS_SRC_CHANGED;

                  if ((vlan2->status & ARGUS_DST_VLAN) && (vlan1->status & ARGUS_DST_VLAN))
                     if (vlan2->did != vlan1->did)
                        vlan2->status |= ARGUS_DST_CHANGED;

                  vlan2->status |= vlan1->status;
                  vlan2->sid = vlan1->sid;
                  vlan2->did = vlan1->did;
                  break;
               }

               case ARGUS_MPLS_DSR_INDEX: {
                  struct ArgusMplsStruct *mpls1 = (struct ArgusMplsStruct *) a1farhdr[i];
                  struct ArgusMplsStruct *mpls2 = (struct ArgusMplsStruct *) a2farhdr[i];

                  if ((mpls2->status & ARGUS_SRC_MPLS) && (mpls1->status & ARGUS_SRC_MPLS))
                     if (mpls2->slabel != mpls1->slabel)
                        mpls2->status |= ARGUS_SRC_CHANGED;

                  if ((mpls2->status & ARGUS_DST_MPLS) && (mpls1->status & ARGUS_DST_MPLS))
                     if (mpls2->dlabel != mpls1->dlabel)
                        mpls2->status |= ARGUS_DST_CHANGED;

                  mpls2->status |= mpls1->status;
                  mpls2->slabel = mpls1->slabel;
                  mpls2->dlabel = mpls1->dlabel;
                  break;
               }

               case ARGUS_ESP_DSR_INDEX: {
                  struct ArgusESPStruct *esp1 = (struct ArgusESPStruct *) a1farhdr[i];
                  struct ArgusESPStruct *esp2 = (struct ArgusESPStruct *) a2farhdr[i];

                  if (esp1 && esp2) {
                     if (esp1->src.spi != esp2->src.spi)
                        esp2->src.spi = 0;

                     if (esp1->dst.spi != esp2->dst.spi)
                        esp2->dst.spi = 0;

                     esp2->src.lastseq = esp1->src.lastseq;
                     esp2->dst.lastseq = esp1->dst.lastseq;
                     esp2->src.lostseq = esp1->src.lostseq;
                     esp2->dst.lostseq = esp1->dst.lostseq;
                  }
                  break;
               }
    
               case ARGUS_TIME_DSR_INDEX: {
                  struct ArgusTimeStruct *time1 = (struct ArgusTimeStruct *) a1farhdr[i];
                  struct ArgusTimeStruct *time2 = (struct ArgusTimeStruct *) a2farhdr[i];
                  long long sumtime, sumtime1, sumtime2;
                  double sumsqrd, sumsqrd1, sumsqrd2;

                  if (time1 && time2) {
                     if ((time2->src.act.n += time1->src.act.n) > 0) {
                        sumtime1 = (time1->src.act.mean * time1->src.act.n);
                        sumtime2 = (time2->src.act.mean * time2->src.act.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->src.act.mean  = sumtime/time2->src.act.n;
                        if (time1->src.act.n)
                           sumsqrd1 = (time1->src.act.n * pow(time1->src.act.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->src.act.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->src.act.n * pow(time2->src.act.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->src.act.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->src.act.stdev = sqrt (sumsqrd/time2->src.act.n - pow (sumtime/time2->src.act.n, 2.0));
                        if (time2->src.act.max < time1->src.act.max)
                           time2->src.act.max = time1->src.act.max; 
                        if (time2->src.act.min > time1->src.act.min)
                           time2->src.act.min = time1->src.act.min; 
                     }

                     if ((time2->dst.act.n += time1->dst.act.n) > 0) {
                        sumtime1 = (time1->dst.act.mean * time1->dst.act.n);
                        sumtime2 = (time2->dst.act.mean * time2->dst.act.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->dst.act.mean  = sumtime/time2->dst.act.n;
                        if (time1->dst.act.n)
                           sumsqrd1 = (time1->dst.act.n * pow(time1->dst.act.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->dst.act.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->dst.act.n * pow(time2->dst.act.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->dst.act.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->dst.act.stdev = sqrt (sumsqrd/time2->dst.act.n - pow (sumtime/time2->dst.act.n, 2.0));
                        if (time2->dst.act.max < time1->dst.act.max)
                           time2->dst.act.max = time1->dst.act.max; 
                        if (time2->dst.act.min > time1->dst.act.min)
                           time2->dst.act.min = time1->dst.act.min; 
                     }

                     if ((time2->src.idle.n += time1->src.idle.n) > 0) {
                        sumtime1 = (time1->src.idle.mean * time1->src.idle.n);
                        sumtime2 = (time2->src.idle.mean * time2->src.idle.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->src.idle.mean  = sumtime/time2->src.idle.n;
                        if (time1->src.idle.n)
                           sumsqrd1 = (time1->src.idle.n * pow(time1->src.idle.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->src.idle.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->src.idle.n * pow(time2->src.idle.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->src.idle.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->src.idle.stdev = sqrt (sumsqrd/time2->src.idle.n - pow (sumtime/time2->src.idle.n, 2.0));
                        if (time2->src.idle.max < time1->src.idle.max)
                           time2->src.idle.max = time1->src.idle.max; 
                        if (time2->src.idle.min > time1->src.idle.min)
                           time2->src.idle.min = time1->src.idle.min; 
                     }

                     if ((time2->dst.idle.n += time1->dst.idle.n) > 0) {
                        sumtime1 = (time1->dst.idle.mean * time1->dst.idle.n);
                        sumtime2 = (time2->dst.idle.mean * time2->dst.idle.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->dst.idle.mean  = sumtime/time2->dst.idle.n;
                        if (time1->dst.idle.n)
                           sumsqrd1 = (time1->dst.idle.n * pow(time1->dst.idle.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->dst.idle.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->dst.idle.n * pow(time2->dst.idle.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->dst.idle.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->dst.idle.stdev = sqrt (sumsqrd/time2->dst.idle.n - pow (sumtime/time2->dst.idle.n, 2.0));
                        if (time2->dst.idle.max < time1->dst.idle.max)
                           time2->dst.idle.max = time1->dst.idle.max; 
                        if (time2->dst.idle.min > time1->dst.idle.min)
                           time2->dst.idle.min = time1->dst.idle.min; 
                     }
                  }

                  break;
               }

               case ARGUS_SRCUSRDATA_DSR_INDEX:
               case ARGUS_DSTUSRDATA_DSR_INDEX:
                  break;

               case ARGUS_AGR_DSR_INDEX: {
                  struct ArgusAGRStruct *agr1 = (struct ArgusAGRStruct *) a1farhdr[i];
                  unsigned int sumtime;
                  double sumsqrd;

                  if ((store->agr.lasttime.tv_sec < agr1->lasttime.tv_sec) ||
                      ((store->agr.lasttime.tv_sec == agr1->lasttime.tv_sec) &&
                       (store->agr.lasttime.tv_usec < agr1->lasttime.tv_usec))) {

                     RaThisArgusStore->qhdr.lasttime = agr1->lasttime;
                     store->agr.lasttime = agr1->lasttime;
                  }

                  if (store->agr.act.min > agr1->act.min) store->agr.act.min = agr1->act.min;
                  if (store->agr.act.max < agr1->act.max) store->agr.act.max = agr1->act.max;

                  sumtime = agr1->act.mean * agr1->act.n;
                  sumsqrd = (agr1->act.n * pow(agr1->act.stdev, 2.0)) + pow(sumtime, 2.0)/agr1->act.n;

                  store->act.sumtime += agr1->act.mean * agr1->act.n;
                  store->act.sumsqrd += sumsqrd;
                  store->act.n += agr1->act.n;

                  sumtime = agr1->idle.mean * agr1->idle.n;
                  sumsqrd = (agr1->idle.n * pow(agr1->idle.stdev, 2.0)) + pow(sumtime, 2.0)/agr1->idle.n;

                  store->idle.sumtime += agr1->idle.mean * agr1->idle.n;
                  store->idle.sumsqrd += sumsqrd;
                  store->idle.n += agr1->idle.n;

                  store->agr.count += agr1->count;
                  break;
               }
            }
         }
         status >>= 1;
      }

      store->status |= RA_MODIFIED;
      store->argus->ahdr.status |= ARGUS_MERGED;
      bcopy (buf, (char *) a2, ((struct ArgusRecord *)buf)->ahdr.length);
#ifdef ARGUSDEBUG
      ArgusDebug (5, "RaMergeArgusRecord: merged\n");
#endif
      return;

   } else {
      store->argus = RaCopyArgusRecord(a1);
#ifdef ARGUSDEBUG
      ArgusDebug (5, "RaMergeArgusRecord: swapped");
#endif
      return;
   }

doneRaMergeArgus:
   RaSendArgusRecord(store);
   ArgusFree(store->argus);
   store->argus = RaCopyArgusRecord(a1);

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaMergeArgusRecord: dumped and swapped");
#endif
}



void
RaUpdateArgusStore(struct ArgusRecord *argus, struct ArgusRecordStore *store)
{
   store->status |= RA_MODIFIED;
   store->qhdr.lasttime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaUpdateArgusStore: \n");
#endif
}

void
RaTimeoutArgusStore (struct ArgusRecordStore *store)
{
   struct ArgusRecord *argus;

   if (store && ((argus = store->argus) != NULL)) {
      if (store->status & RA_MODIFIED)
         RaSendArgusRecord(store);

      if (store->rahtblhdr)
         RaRemoveHashEntry (store->rahtblhdr);

      RaRemoveArgusRecord(argus);
      ArgusFree (argus);
      ArgusFree (store);
      RaAllocArgusRecordStore--;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaTimeoutArgusStore: 0x%x\n", store);
#endif
}


int
RaCheckTimeout (struct ArgusRecordStore *store)
{
   int retn = 0;
   struct ArgusRecord *argus = store->argus;
   unsigned int lapseTime;
   struct timeval *tvp = &ArgusGlobalTime;

   if (argus) {
      if (store->ArgusTimeout > 0) {
         lapseTime = store->qhdr.lasttime.tv_sec + store->ArgusTimeout;
         if ((tvp->tv_sec > lapseTime) || ((tvp->tv_sec == lapseTime) &&
             (tvp->tv_usec > store->qhdr.lasttime.tv_usec)))
            retn++;
   
         lapseTime = store->qhdr.logtime.tv_sec + store->ArgusTimeout;
         if ((tvp->tv_sec > lapseTime) || ((tvp->tv_sec == lapseTime) &&
             (tvp->tv_usec > store->qhdr.logtime.tv_usec)))
      
            RaSendArgusRecord(store);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaCheckTimeout: returning %d \n", retn);
#endif

   return (retn);
}


struct RaCIDRAddr *
RaParseCIDRAddr (char *str)
{
   int error = 0;
   struct RaCIDRAddr *retn = NULL;
   char *ptr = NULL, chr = '\0';
   unsigned int **alist;

   if ((ptr = strchr(str, '*')) != NULL) {
      RaCIDRPtr->addr = 0;
      RaCIDRPtr->mask = 0xFFFFFFFF;
   } else {
      if ((ptr = strchr (str, ':')) != NULL) {
         chr = *ptr;
         *ptr++ = 0;
      } else {
         if ((ptr = strchr (str, '/')) != NULL) {
            chr = *ptr;
            *ptr++ = 0;
         }
      }
   
      if ((alist = argus_nametoaddr(str)) != NULL)  {
         RaCIDRPtr->addr = **alist;
   
         if (ptr && ((chr == ':') && isdigit((int)*ptr))) {
            int maskIndex = atoi(ptr);
            u_int maskValue = 0xFFFFFFFF;
   
            if ((maskIndex) > 0) {
               maskValue <<= maskIndex;
               RaCIDRPtr->mask = maskValue;
            }
         }
      } else
         error++;
   
   }

   if (!error)
      retn = RaCIDRPtr;

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaParseCIDRAddr: returning 0x%x \n", retn);
#endif
   
   return (retn);
}


struct RaPolicyStruct *
RaParsePolicyEntry (char *str)
{
   struct RaPolicyStruct *retn = NULL;
   struct RaCIDRAddr *addr = NULL;
   int i, type, policyelements, error = 0;
   char buf[MAXSTRLEN], *ptr = NULL, *tmp = buf;

   if (strncmp (str, RA_FLOWTAGSTR, strlen(RA_FLOWTAGSTR)) == 0) {
      type = RA_FLOWLIST;
      policyelements = RA_FLOWPOLICYFIELDNUM;

   } else {
      if ((strncmp (str, RA_MODELTAGSTR, strlen(RA_MODELTAGSTR))) == 0) {
         type = RA_MODELIST;
         policyelements = RA_MODELPOLICYFIELDNUM;

      } else 
         return (NULL);
   }

   if ((retn = (struct RaPolicyStruct *) ArgusCalloc (1, sizeof (*retn))) == NULL)
      return retn;

   bzero (buf, MAXSTRLEN);
   bcopy ((char *) str, buf, strlen(str));

   for (i = 0; ((i < policyelements) && !(error)); i++) {
      if ((ptr = strtok (tmp, " \t")) && (*ptr != '\n')) {
         switch (i) {
            case RA_LABELSTRING: 
               switch (type) {
                  case RA_FLOWLIST:
                     if (strcmp (ptr,  RA_FLOWTAGSTR))
                        error++;
                     else 
                        retn->RaEntryType = RA_FLOWLIST;
                     break;

                  case RA_MODELIST:
                     if (strcmp (ptr, RA_MODELTAGSTR))
                        error++;
                     else 
                        retn->RaEntryType = RA_MODELIST;
                     break;
               }
               break;
   
            case RA_POLICYID: 
               if (!(retn->RaPolicyId = atoi (ptr))) error++;
               break;

            case RA_POLICYSRCADDR:
               if ((addr = RaParseCIDRAddr (ptr)) != NULL)
                  retn->src = *addr;
               else
                  error++;
               break;
   
            case RA_POLICYDSTADDR:
               if ((addr = RaParseCIDRAddr (ptr)) != NULL)
                  retn->dst = *addr;
               else
                  error++;
               break;
   
            case RA_POLICYPROTO: 
               switch (type) {
                  case RA_FLOWLIST:
                     if (isdigit((int)*ptr)) {
                        retn->proto = atoi(ptr);
                     } else {
                        if (*ptr == '*') {
                           retn->proto = 0xFF;
                        } else {
                           struct protoent *proto;
                           if ((proto = getprotobyname(ptr)) != NULL)
                              retn->proto = proto->p_proto;
                           else
                              error++;
                        }
                     }
                     break;

                  case RA_MODELIST:
                     if ((strstr(ptr, "yes")) != NULL)
                        retn->proto = 0xFF;
                     else
                        if ((strstr(ptr, "no")) != NULL)
                           retn->proto = 0x00;
                        else
                           error++;
                     break;
               }
               break;

            case RA_POLICYSRCPORT:
               switch (type) {
                  case RA_FLOWLIST:
                     if (isdigit((int)*ptr)) {
                        retn->sport = (arg_uint16) atoi (ptr);
                     } else {
                        if (*ptr == '*')
                           retn->sport = 0xFFFF;
                        else {
                           int proto, port;
                           if (retn->proto == IPPROTO_ICMP) {
                              if (!(strcmp (ptr, "echo"))) retn->sport = ICMP_ECHO; else
                              if (!(strcmp (ptr, "unreach"))) retn->sport = ICMP_UNREACH; else
                              if (!(strcmp (ptr, "squench"))) retn->sport = ICMP_SOURCEQUENCH; else
                              if (!(strcmp (ptr, "redirect"))) retn->sport = ICMP_REDIRECT; else
                              if (!(strcmp (ptr, "timexed"))) retn->sport = ICMP_TIMXCEED; else
                              if (!(strcmp (ptr, "param"))) retn->sport = ICMP_PARAMPROB; else
                              if (!(strcmp (ptr, "timestmp"))) retn->sport = ICMP_TSTAMP; else
                              if (!(strcmp (ptr, "info"))) retn->sport = ICMP_IREQ; else
                              if (!(strcmp (ptr, "advert"))) retn->sport = ICMP_ROUTERADVERT; else
                              if (!(strcmp (ptr, "solicit"))) retn->sport = ICMP_ROUTERSOLICIT; else
                                 error++;
                           } else {
                              if (argus_nametoport(ptr, &port, &proto)) {
                                 retn->sport = port;
                              } else
                                 error++;
                           }
                        }
                     }
                     break;

                  case RA_MODELIST:
                     if ((strstr(ptr, "yes")) != NULL)
                        retn->sport = 0xFFFF;
                     else
                        if ((strstr(ptr, "no")) != NULL)
                           retn->sport = 0x0000;
                        else
                           error++;
                     break;
               }
               break;
   
            case RA_POLICYDSTPORT:
               switch (type) {
                  case RA_FLOWLIST:
                     if (isdigit((int)*ptr)) {
                        retn->dport = (arg_uint16) atoi (ptr);
                     } else {
                        if (*ptr == '*')
                           retn->dport = 0xFFFF;
                        else {
                           if (retn->proto == IPPROTO_ICMP) {
                           } else {
                              int proto, port;
                              if (argus_nametoport(ptr, &port, &proto)) {
                                 retn->dport = port;
                              } else
                                 error++;
                           }
                        }
                     }
                     break;

                  case RA_MODELIST:
                     if ((strstr(ptr, "yes")) != NULL)
                        retn->dport = 0xFFFF;
                     else
                        if ((strstr(ptr, "no")) != NULL)
                           retn->dport = 0x0000;
                        else
                           error++;
                     break;
               }
               break;
      
            case RA_POLICYMODELST:
               if (type == RA_FLOWLIST) {
                  if (isdigit((int)*ptr))
                     retn->RaModelId = (arg_uint32) atoi (ptr);
                  else
                     error++;
               }
               break;

            case RA_POLICYTIMEOUT:
               if (type == RA_FLOWLIST) {
                  if (isdigit((int)*ptr))
                     retn->ArgusTimeout = (arg_uint32) atoi (ptr);
                  else
                     error++;
               }
               break;
         }
         tmp = NULL;

      } else
         break;
   }

   if (error) {
      ArgusFree(retn);
      retn = NULL;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaParsePolicyEntry: returning 0x%x \n", retn);
#endif

   return (retn);
}


int
RaCreatePolicyEntry (char *str)
{
   int retn = 0;
   struct RaPolicyStruct *rap;

   if ((rap = RaParsePolicyEntry (str)) != NULL) {
      rap->str = str;

      switch (rap->RaEntryType) {
         case RA_FLOWLIST:
            RaFlowArray[RaThisFlowNum++] = rap;
            break;

         case RA_MODELIST:
            RaModelArray[RaThisModelNum++] = rap;
            break;
      }
   } else {
      fprintf (stderr, "%s: RaCreatePolicyEntry: format error\n", ArgusProgramName);
      exit (-1);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaCreatePolicyEntry: returning %d\n", retn);
#endif

   return (retn);
}


int
RaReadFlowModelFile (char **model)
{
   int retn = 0, i, x;
   char *str = NULL;
   int RaPolicyLines = 0, RaPolicyLineNumber = 0;

   while ((str = *model++) != NULL) {
      RaPolicyLineNumber++;

      while (isspace((int)*str))
         str++;

      if (strlen(str)) {
         switch (str[0]) {
            case '#':
            case '\n':
            case '!':
               break;

            default:
               if ((str = strdup(str)) != NULL) {
                  if (RaCreatePolicyEntry(str) < 0) {
                     fprintf (stderr, "%s: RaReadFlowModeleFile: parse error line %d\n", ArgusProgramName,
                                RaPolicyLineNumber);
                     fprintf (stderr, "%s: RaReadFlowModeleFile: %s\n", ArgusProgramName,
                                RaPolicyError[RaParseError]);
                     exit (-1);
                  } else
                     RaPolicyLines++;
                 
               } else {
                  fprintf (stderr, "%s: RaReadFlowModeleFile: error %s\n", ArgusProgramName,
                                strerror(errno));
                  exit (-1);
               }
         }
      }
   }

   for (i = 0; i < RaThisFlowNum; i++) {
      if (RaFlowArray[i]) {
         int found = 0;
         for (x = 0; x < RaThisModelNum; x++) {
            if (RaModelArray[x])
               if (RaFlowArray[i]->RaModelId == RaModelArray[x]->RaPolicyId) {
                  RaFlowArray[i]->RaModelId = x;
                  found++;
                  break;
               }
         }
         if (!found) {
            fprintf (stderr, "%s: RaReadFlowModeleFile: Model Index %d not found\n",
                             ArgusProgramName, RaFlowArray[i]->RaModelId);
            exit (-1);
         }
      }
   }

   retn = RaThisFlowNum;

#ifdef ARGUSDEBUG
   ArgusDebug (2, "RaReadFlowModelFile: returning RaFlowArray[%d]  RaModelArray[%d]\n", RaThisFlowNum, RaThisModelNum);
#endif

   return (retn);
}


#include <stdio.h>
#include <errno.h>


#define RA_MAXQSCAN  256

int RaCompareArgusStore (const void *, const void *);

int
RaCompareArgusStore (const void *void1, const void *void2)
{
   int retn = 0;
   struct ArgusRecordStore *store1 = *(struct ArgusRecordStore **)void1;
   struct ArgusRecordStore *store2 = *(struct ArgusRecordStore **)void2;

   if ((store1 && store1->argus) && ((store2 && store2->argus))) {
      struct ArgusRecord *argus1 = store1->argus;
      struct ArgusRecord *argus2 = store2->argus;

      if (Aflag)
         retn = ((argus2->argus_far.src.bytes + argus2->argus_far.dst.bytes) -
                 (argus1->argus_far.src.bytes + argus1->argus_far.dst.bytes));

      else
         retn = ((argus2->argus_far.src.bytes + argus2->argus_far.dst.bytes) -
                 (argus1->argus_far.src.bytes + argus1->argus_far.dst.bytes));
   }

   return (retn);
}

void
RaSortQueue (struct RaQueueStruct *queue)
{
   qsort ((char *) queue->array, queue->size, sizeof (void *), RaCompareArgusStore);
}



int RaLabelCounter = 0;

char RaLabelStr[256], *RaLabel = NULL;
char *RaMatrixFlowLabel = "%*sSrcAddr%*s%*s      %*sDir  %*s DstAddr%*s%*s     %*s";
char *RaTopNFlowLabel   = "  %*sSrcAddr%*s%*s      %*s";
char *RacOptionLabel = " SrcPkt   Dstpkt    SrcBytes     DstBytes   ";
char *RamOptionLabel = "    SrcMacAddr       DstMacAddr       ";
char *RaROptionLabel = " SrcPkt   Dstpkt    Response Information    ";
char *RaTOptionLabel  = " Stos Dtos Sttl Dttl   SMinIdle    SMeanIdl    SMaxIdle     DMinIdle    DMeanIdl    DMaxIdle ";



void RaPrintLabel (struct ArgusRecord *);

void
RaPrintLabel (struct ArgusRecord *argus)
{
   int i, gflagvar = gflag, Gflagvar = Gflag;
   char RaPrecisionPad[32], RaTimePad[32], date[128];

   bzero (date, sizeof (date));
   bzero (RaTimePad, sizeof (RaTimePad));
   bzero (RaPrecisionPad, sizeof (RaPrecisionPad));

   if (Lflag) {
      printf ("%s %s Report\n", ArgusProgramName, RaTopNModel ? "TopN" : "Matrix");
      
      if (RaLabel == NULL) {
         gflag = 0; Gflag = 0;
         print_date (argus, date);
         gflag = gflagvar; Gflag = Gflagvar;
         if (strlen(date) > 10)
            for (i = 0; i < (strlen(date) - 10)/2; i++)
               strcat (RaTimePad, " ");
          
         for (i = 0; i < Uflag/2; i++)
            strcat (RaPrecisionPad, " ");

         bzero (RaLabelStr, sizeof(RaLabelStr));
         RaLabel = RaLabelStr;

         strcat (RaLabel, RaTimePad);
         if (lflag)
            strcat (RaLabel, " Last_Time ");
         else
            strcat (RaLabel, "Start_Time ");

         strcat (RaLabel, RaTimePad);
         if (strlen(date) & 0x01) strcat (RaLabel, " ");

         if (gflag) {
            strcat (RaLabel, RaPrecisionPad);
            strcat (RaLabel, " Duration ");
            strcat (RaLabel, RaPrecisionPad);
            if (Uflag &0x01) strcat (RaLabel, " ");
         } else
         if (Gflag) {
            strcat (RaLabel, RaTimePad);
            strcat (RaLabel, " Last_Time ");
            strcat (RaLabel, RaTimePad);
            if (strlen(date) & 0x01) strcat (RaLabel, " ");
         }

         if (Iflag) strcat (RaLabel, " Flgs  ");

         strcat (RaLabel, "Type ");
         if (mflag) strcat (RaLabel, RamOptionLabel);
         if (RaTopNModel)
            sprintf (&RaLabel[strlen(RaLabel)], RaTopNFlowLabel, (hfield - 8)/2, " ", (hfield - 8)/2, " ",
                                                                 (pfield - 4)/2, " ", (pfield - 4)/2, " ");
         else
            sprintf (&RaLabel[strlen(RaLabel)], RaMatrixFlowLabel, (hfield - 8)/2, " ", (hfield - 8)/2, " ",
                                                                   (pfield - 4)/2, " ", (pfield - 4)/2, " ",
                                                                   (hfield - 8)/2, " ", (hfield - 8)/2, " ",
                                                                   (pfield - 4)/2, " ", (pfield - 4)/2, " ");

         if (cflag && Rflag) strcat (RaLabel, RaROptionLabel);
         else     if (cflag) strcat (RaLabel, RacOptionLabel);

         if (zflag || Zflag) {
            strcat (RaLabel, " Status");
         } else {
            strcat (RaLabel, "State");
         }
      }

      if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
         char tmpbuf[256], *ptr = tmpbuf, *str = RaLabel;
         bzero (tmpbuf, sizeof(tmpbuf));

         while (isspace((int)*str)) str++;

         while (*str) {
            if (*str == ' ') {
               *ptr++ = RaFieldDelimiter;
               while (isspace((int)*str)) str++;
            }
            *ptr++ = *str++;
         }
         bzero (RaLabelStr, sizeof(RaLabelStr));
         bcopy (tmpbuf, RaLabelStr, strlen(tmpbuf));
      }

      if (!(RaLabelCounter++ % Lflag))
         printf ("%s\n", RaLabel);

      if (Lflag < 0)
         Lflag = 0;
   }
}


void
RaPrintOutQueue (struct RaQueueStruct *queue)
{
   struct ArgusRecordStore *obj = NULL;
   int i;

   if (Nflag == 0)
      Nflag = queue->count;

   for (i = 0; i < Nflag; i++) {
      if ((obj = (struct ArgusRecordStore *) queue->array[i]) != NULL) {
         if ((i == 0) && (wfile == NULL))
            RaPrintLabel (obj->argus);
         RaSendArgusRecord(obj);
      } else
         break;
   }
}


void
RaPackQueue (struct RaQueueStruct *queue)
{
   struct RaQueueStruct *newq;
   struct ArgusQueueHeader *obj;

   if (!((queue->head == 0) && (queue->count == queue->tail))) {
      if ((newq = RaNewQueue (queue->count)) != NULL) {
         while (queue->count)
            if ((obj = RaPopQueue(queue)) != NULL)
               RaAddToQueue(newq, obj);
   
         ArgusFree(queue->array);   
   
         queue->head  = newq->head;
         queue->tail  = newq->tail;
         queue->count = newq->count;
         queue->size  = newq->count;
         queue->array = newq->array;
      }
   } else
      queue->size = queue->count;
}
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordStore *obj = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP:
         while (queue->count) {
            if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL)
               RaTimeoutArgusStore(obj);
         }
         break;

      default:
         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  if (RaCheckTimeout(obj))
                     RaTimeoutArgusStore(obj);
                  else
                     RaAddToQueue(queue, &obj->qhdr);

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}


struct RaHashTableHeader *
RaFindHashObject ()
{
   struct RaHashTableHeader *retn = NULL;
   struct RaHashTableHeader *head = NULL, *target;

   if ((target = RaHashTable.array[RaThisHash % RaHashTable.size]) != NULL) {
      head = target;
      do {
         if (!(bcmp ((char *) RaThisFlow, (char *) &target->flow, sizeof(*RaThisFlow)))) {
               retn = target;
               break;
            } else
               target = target->nxt;
      } while (target != head);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaFindHashObject: returning 0x%x\n", retn);
#endif

   return (retn);
}



struct RaHashTableHeader *
RaAddHashEntry (struct ArgusRecordStore *store)
{
   struct RaHashTableHeader *retn = NULL;
   struct RaHashTableHeader *start = NULL;

   if ((retn = (struct RaHashTableHeader *) ArgusCalloc (1, sizeof (struct RaHashTableHeader))) != NULL) {
      RaAllocHashTableHeaders++;
      bcopy((char *) RaThisFlow, (char *)&retn->flow, sizeof (*RaThisFlow));
      retn->storeobj = store;
      retn->hash = RaThisHash;
      
      if ((start = RaHashTable.array[RaThisHash % RaHashTable.size]) != NULL) {
         retn->nxt = start;
         retn->prv = start->prv;
         retn->prv->nxt = retn;
         retn->nxt->prv = retn;
      } else
         retn->prv = retn->nxt = retn;

      RaHashTable.array[RaThisHash % RaHashTable.size] = retn;
      RaHashTable.count++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaAddHashEntry (0x%x) returning 0x%x\n", store, retn);
#endif

   return (retn);
}

void
RaRemoveHashEntry (struct RaHashTableHeader *rahtblhdr)
{
   unsigned short hash = rahtblhdr->hash;

   rahtblhdr->prv->nxt = rahtblhdr->nxt;
   rahtblhdr->nxt->prv = rahtblhdr->prv;

   if (rahtblhdr == RaHashTable.array[hash % RaHashTable.size]) {
      if (rahtblhdr == rahtblhdr->nxt)
         RaHashTable.array[hash % RaHashTable.size] = NULL;
      else
         RaHashTable.array[hash % RaHashTable.size] = rahtblhdr->nxt;

      RaHashTable.count--;
   }

   ArgusFree (rahtblhdr);
   RaAllocHashTableHeaders--;
 
#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveHashObject (0x%x) returning\n", rahtblhdr);
#endif
}


struct RaQueueStruct *
RaNewQueue (int size)
{
   struct RaQueueStruct *retn =  NULL;

   if ((retn = (struct RaQueueStruct *) ArgusCalloc (1, sizeof (struct RaQueueStruct))) != NULL) {
      retn->head = 0;
      retn->tail = 0;
      retn->count = 0;
      if ((retn->array = (struct ArgusQueueHeader **) ArgusCalloc (size, sizeof(void *))) != NULL)
         retn->size = size;
      else {
         ArgusFree(retn);   
         retn = NULL;   
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "RaNewQueue () returning 0x%x\n", retn);
#endif

   return (retn);
}

void
RaDeleteQueue (struct RaQueueStruct *queue)
{
   struct ArgusQueueHeader *obj = NULL;

   while ((obj = RaPopQueue(queue)))
      ArgusFree(obj);

#ifdef ARGUSDEBUG
   ArgusDebug (2, "RaDeleteQueue (0x%x) returning\n", queue);
#endif
}


int
RaGetQueueCount(struct RaQueueStruct *queue)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaGetQueueCount (0x%x) returning %d\n", queue, queue->count);
#endif

   return (queue->count);
}

void
RaPushQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   int start = queue->head;

   if (obj && (queue->count < queue->size)) {
      if (--start < 0)
         start = queue->size;
   
      if (queue->array[start] == NULL) {
         queue->count++;
         obj->index = start;
         queue->array[start] = obj;
         queue->head = start;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaPushQueue (0x%x, 0x%x) returning\n", queue, obj);
#endif
}

#define RA_MAXQUEUESIZE         1048576

void
RaAddToQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   struct RaQueueStruct *newq;
   struct ArgusQueueHeader *qhdr, **oarray;

   if (obj) {
      if (!((queue->count + 1) < (queue->size))) {
         if ((queue->size << 1) <= RA_MAXQUEUESIZE) {
            if ((newq = RaNewQueue (queue->size << 1)) != NULL) {
               while (queue->count)
                  if ((qhdr = RaPopQueue(queue)) != NULL)
                     RaAddToQueue(newq, qhdr);

               oarray = queue->array;

               queue->head  = newq->head;
               queue->tail  = newq->tail;
               queue->count = newq->count;
               queue->size  = newq->size;
               queue->array = newq->array;

               newq->array = oarray;
               newq->count = 0;
               RaDeleteQueue (newq);
            }
         }
   #ifdef ARGUSDEBUG
         ArgusDebug (1, "ArgusAddToQueue: expanded queue size to %d\n", queue->size);
   #endif
      }

      queue->count++;

      obj->index = queue->tail;
      queue->array[queue->tail] = obj;

      while (queue->array[queue->tail] != NULL) {
         queue->tail++;
         if (queue->tail == queue->size)
            queue->tail = 0;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaAddToQueue (0x%x, 0x%x) returning\n", queue, obj);
#endif
}



struct ArgusQueueHeader *
RaPopQueue (struct RaQueueStruct *queue)
{
   struct ArgusQueueHeader *retn = NULL;
   struct ArgusQueueHeader *obj = NULL;

   if (queue && queue->count) {
      if ((obj = queue->array[queue->head]) != NULL)
         retn = RaRemoveFromQueue(queue, obj);
      else
         ArgusLog (LOG_ERR, "RaPopQueue(0x%x) internal queue error count %d head NULL\n", queue, queue->count);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveFromQueue (0x%x) returning 0x%x\n", queue, retn);
#endif
   
   return(retn);
}


struct ArgusQueueHeader *
RaRemoveFromQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   struct ArgusQueueHeader *retn = NULL;

   if ((queue != NULL) && (obj != NULL)) {
      int index = obj->index;

      if (queue->count && (queue->array[index] == obj)) {
         queue->count--;
         queue->array[index] = NULL;
         obj->index = 0;
         retn = obj;

         if (queue->count) {
            if (queue->head == index) {
               while (queue->array[queue->head] == NULL) {
                  queue->head++;
                  if (queue->head == queue->size)
                     queue->head = 0;
               }
            }

         } else {
            queue->head = 0;
            queue->tail = 0;
         }

      } else
         ArgusLog (LOG_ERR, "RaRemoveFromQueue(0x%x, 0x%x) obj not in queue\n", queue, obj);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveFromQueue (0x%x, 0x%x) returning 0x%x\n", queue, obj, obj);
#endif

   return (retn);
}
