// IDSLog.cpp: implementation of the CIDSLog class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "IDSLog.h"
#include "misc.h"

#ifdef _WIN32

#include <winsock2.h>

#define SNPRINTF _snprintf
#define TZSET _tzset
#define TZNAME _tzname
#define TIMEZONE _timezone

extern CString sUploadType;

#else 

#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "config.h"

#define SNPRINTF snprintf
#define TZSET tzset
#define TIMEZONE timezone
#define TZNAME tzname

extern char sUploadType[];

#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


CIDSAddlInfo::CIDSAddlInfo (const char *sAddlType, const char *sAddlString)
{
	m_pNext = NULL;
	strncpy (m_sAddlType, sAddlType, LOG_STRLEN);
	strncpy (m_sAddlString, sAddlString, LOG_STRLEN);
}

CIDSAddlInfo::~CIDSAddlInfo ()
{
}

char *CIDSAddlInfo::AddlString () const
{
	return (char *) m_sAddlString;
}

char *CIDSAddlInfo::AddlType () const
{
	return (char *) m_sAddlType;
}


const char *CIDSLog::m_sStripperVersion = "1.6.2";
const char *CIDSLog::m_sXMLHeader =
	"<?xml version=\"1.0\"?>\n"
	"<!DOCTYPE idslog [\n"
	"	<!ELEMENT idslog (logheader, logentries)>\n"
	"\n"
	"	<!ELEMENT logheader (sfclientver, seriallabel, logcleandatetime, localdatetime, timezone, offsetfromgmt, productid, productname, productmajorver, productminorver, productreleasever, uploadtype, netmask)>\n"
	"	<!ELEMENT sfclientver (#PCDATA)>\n"
	"	<!ELEMENT seriallabel (#PCDATA)>\n"
	"	<!ELEMENT logcleandatetime (#PCDATA)>\n"
	"	<!ELEMENT localdatetime (#PCDATA)>\n"
	"	<!ELEMENT timezone (#PCDATA)>\n"
	"	<!ELEMENT offsetfromgmt (#PCDATA)>\n"
	"	<!ELEMENT productid (#PCDATA)>\n"
	"	<!ELEMENT productname (#PCDATA)>\n"
	"	<!ELEMENT productmajorver (#PCDATA)>\n"
	"	<!ELEMENT productminorver (#PCDATA)>\n"
	"	<!ELEMENT productreleasever (#PCDATA)>\n"
	"	<!ELEMENT uploadtype (#PCDATA)>\n"
	"	<!ELEMENT netmask (#PCDATA)>\n"
	"\n"
	"	<!ELEMENT logentries (logentry*)>\n"
	"	<!ELEMENT logentry (entry, addlinfo*)>\n"
	"		<!ELEMENT entry EMPTY>\n"
	"			<!ATTLIST entry num CDATA #REQUIRED\n"
	"				datetime CDATA #REQUIRED\n"
	"				id CDATA #REQUIRED\n"
	"				fromip CDATA #REQUIRED\n"
	"				toip CDATA #REQUIRED\n"
	"				count CDATA #REQUIRED\n"
	"				protocol CDATA #IMPLIED\n"
	"				fromport CDATA #IMPLIED\n"
	"				toport CDATA #IMPLIED\n"
	"				portscan CDATA #IMPLIED>\n"
	"		<!ELEMENT addlinfo (#PCDATA)>\n"
	"			<!ATTLIST addlinfo type CDATA #REQUIRED>\n"
	"]>\n\n";


CIDSLog::CIDSLog()
{
	m_pAddlInfoHead = NULL;
}

CIDSLog::~CIDSLog()
{
	ClearAddl ();
	FreeIPMasks ();
}

bool CIDSLog::Output (const char *sSerial, double nDate, const char *infile, const char *outfile, double& nLastDate)
{
	time_t timer;
	time (&timer);
	struct tm *ptm = gmtime (&timer);
	memcpy (&m_now, ptm, sizeof(struct tm));

	ptm = localtime (&timer);
	memcpy (&m_localnow, ptm, sizeof(struct tm));

	if (sSerial)
		strncpy (m_sSerial, sSerial, LOG_STRLEN);
	else
		m_sSerial[0] = '\0';
	
	if (!OpenLog (infile))
	{
		return false;
	}
	if (!OpenOutput (outfile))
	{
		CloseLog ();
		return false;
	}

	SeekLog (nDate);
       	m_nLastTimestamp = nDate;
	
	if (!WriteHeader ())
		return false;
	while (!LastIncident ())
	{
		ClearAddl ();
		if (ReadIncident ())
		{
			if (!WriteIncident ())
				return false;
		}
	}
	if (!WriteFooter ())
		return false;
	nLastDate = GetLastDate ();
	CloseOutput ();
	CloseLog ();
	
	return true;
}

bool CIDSLog::WriteHeader ()
{
	char sIDSProduct[LOG_STRLEN + 1], sIDSRev[LOG_STRLEN + 1];
	int nIDSID = 0, nIDSMajor = 0, nIDSMinor = 0;
	struct in_addr netmask;

	IDSVersion (sIDSProduct, nIDSID, nIDSMajor, nIDSMinor, sIDSRev);

	// the time in gmt on the machine
	double nDateTime = GetDate (m_now.tm_year + 1900, m_now.tm_mon + 1, 
		m_now.tm_mday, m_now.tm_hour, m_now.tm_min, 
		m_now.tm_sec, 0, false); // false = already converted to GMT

	// the local time on the machine, unconverted to gmt
	double nLocalDateTime = GetDate (m_localnow.tm_year + 1900, m_localnow.tm_mon + 1, 
		m_localnow.tm_mday, m_localnow.tm_hour, m_localnow.tm_min, 
		m_localnow.tm_sec, 0, false); // false = no conversion to GMT please
	
	// record the current time for checking the incident times against
	m_nCurrentTime = nDateTime;
        
	// get the time zone string
	TZSET(); // sets global time variables
	char sTimeZone[100];
	char sOffsetFromGMT[100];

	SNPRINTF( sTimeZone, 99, "%s", TZNAME[0] );

	// openbsd doesn't use the _timezone global variable so
	// bend over backwards to get it
#ifdef OPENBSD
        time_t temp;
        time( &temp );
        struct tm *ltm = localtime( &temp );
	strftime( sOffsetFromGMT, 99, "%z", ltm );

	// the offset from GMT in is returned in the format HHMM 
	// the sign is positive if east of GMT. 
	// convert it to a float (hour.fractionofhour)
	// divide the number by 100 to get the hours, 
	// divide the difference by 60 to get the fraction of the hour
	int hourmin, hour, fractionofhour;
	sscanf( sOffsetFromGMT, "%d", &hourmin );
	hour = hourmin/100;
	fractionofhour = abs(hourmin - (hour*100))/6; // divide by six to move decimal over by 1
	sprintf( sOffsetFromGMT, "%d.%d", hour, fractionofhour );
#else
	float temp = TIMEZONE; // minutes diff from GMT,  use float for half hour timezones
	float hoursdif = temp/3600;
	SNPRINTF( sOffsetFromGMT, 99, "%2.1f", hoursdif );
#endif

	if (fputs (m_sXMLHeader, m_pOutputFile) == EOF)
		return false;
	if (fputs ("<idslog>\n\n", m_pOutputFile) == EOF)
		return false;

	// clean seriallabel
	char sTmp[LOG_STRLEN + 1];
	int indx = 0, n = strlen (m_sSerial);
	for (int i = 0; (i < n) && (indx < LOG_STRLEN); i++)
	{
		if (m_sSerial[i] == '&')
		{
			int len = LOG_STRLEN - indx;
			if (len > 5)			// strlen ("&amp;")
				len = 5;
			strncpy (sTmp + indx, "&amp;", len);
			indx += len;
		}
		else if (m_sSerial[i] == '<')
		{
			int len = LOG_STRLEN - indx;
			if (len > 4)			// strlen ("&lt;")
				len = 4;
			strncpy (sTmp + indx, "&lt;", len);
			indx += len;
		}
		else if (m_sSerial[i] == '>')
		{
			int len = LOG_STRLEN - indx;
			if (len > 4)			// strlen ("&gt;")
				len = 4;
			strncpy (sTmp + indx, "&gt;", len);
			indx += len;
		}
		else if (m_sSerial[i] & 0x80)		// if extended ascii
		{
			int len = LOG_STRLEN - indx;
			if (len > 6)			// strlen ("&#xhh;")
				len = 6;
			int nChar = m_sSerial[i];
			SNPRINTF (sTmp + indx, len, "&#x%2x;", nChar & 0xff);
			indx += len;
		}
		else
			sTmp[indx++] = m_sSerial[i];
	}
	sTmp[indx] = '\0';

	memset(&netmask, 0, sizeof(struct in_addr));
	netmask.s_addr = m_nNetMask;

	if (fprintf (m_pOutputFile, "<logheader>\n"
			"<sfclientver>%s</sfclientver>\n"
			"<seriallabel>%s</seriallabel>\n"
			"<logcleandatetime>%0.8f</logcleandatetime>\n"
			"<localdatetime>%0.8f</localdatetime>\n"
			"<timezone>%s</timezone>\n"
			"<offsetfromgmt>%s</offsetfromgmt>\n"
			"<productid>%d</productid>\n"
			"<productname>%s</productname>\n"
			"<productmajorver>%d</productmajorver>\n"
			"<productminorver>%d</productminorver>\n"
			"<productreleasever>%s</productreleasever>\n"
			"<uploadtype>%s</uploadtype>\n" // no upload type in *nix version of Extractor
			"<netmask>%s</netmask>\n"
			"</logheader>\n\n",
			 m_sStripperVersion, sTmp, nDateTime, nLocalDateTime, 
			 sTimeZone, sOffsetFromGMT, nIDSID, sIDSProduct, 
			 nIDSMajor, nIDSMinor, sIDSRev , sUploadType,
			inet_ntoa(netmask)) < 0)
		return false;
	if (fputs ("<logentries>\n", m_pOutputFile) == EOF)
		return false;

	m_nEntry = 0;
	
	return true;
}

bool CIDSLog::WriteFooter ()
{
	if (fputs ("\n</logentries>\n\n", m_pOutputFile) == EOF)
		return false;
	if (fputs ("</idslog>", m_pOutputFile) == EOF)
		return false;

	return true;
}


bool CIDSLog::WriteIncident ()
{
	char sFromIP[16], sToIP[16];

	if (!StripIP (sFromIP, m_nFromIP))
		return true;
	StripIP (sToIP, m_nToIP);

	if (fputs ("\n<logentry>\n", m_pOutputFile) == EOF)
		return false;
	
	char sAtt[LOG_STRLEN + 1], sPortscan[PORTSCAN_STRLEN + 1], sEntry[LOG_STRLEN + PORTSCAN_STRLEN + 1];

	// clean m_sID
	int indx = 0, n = strlen (m_sID);
	for (int i = 0; (i < n) && (indx < LOG_STRLEN); i++)
	{
		if (m_sID[i] == '&')
		{
			int len = LOG_STRLEN - indx;
			if (len > 5)			// strlen ("&amp;")
				len = 5;
			strncpy (sAtt + indx, "&amp;", len);
			indx += len;
		}
		else if (m_sID[i] == '<')
		{
			int len = LOG_STRLEN - indx;
			if (len > 4)			// strlen ("&lt;")
				len = 4;
			strncpy (sAtt + indx, "&lt;", len);
			indx += len;
		}
		else if (m_sID[i] == '>')
		{
			int len = LOG_STRLEN - indx;
			if (len > 4)			// strlen ("&gt;")
				len = 4;
			strncpy (sAtt + indx, "&gt;", len);
			indx += len;
		}
		else
			sAtt[indx++] = m_sID[i];
	}
	sAtt[indx] = '\0';
	//
	
	sprintf (sEntry, "<entry num=\"%d\" datetime=\"%0.8f\" id=\"%s\" fromip=\"%s\" toip=\"%s\" count=\"%d\" ",
		m_nEntry++, m_nDateTime, sAtt, sFromIP, sToIP, m_nCount);

	if (m_nProtocol != 0)
	{
		// check the protocol number for a validity
		if(	( m_nProtocol == PROTO_ICMP )||
			( m_nProtocol == PROTO_IGMP )||
			( m_nProtocol == PROTO_TCP )||
			( m_nProtocol == PROTO_UDP ))
		{
			sprintf (sAtt, "protocol=\"%d\" ", m_nProtocol);
			strcat (sEntry, sAtt);
		}
	}
	if (m_nFromPort != 0)
	{
		sprintf (sAtt, "fromport=\"%d\" ", m_nFromPort);
		strcat (sEntry, sAtt);
	}
	if (m_nToPort != 0)
	{
		sprintf (sAtt, "toport=\"%d\" ", m_nToPort);
		strcat (sEntry, sAtt);
	}
	if (m_sPortScan[0] != '\0')
	{
		sprintf (sPortscan, "portscan=\"%s\" ", m_sPortScan);
		strcat (sEntry, sPortscan);
	}
	strcat (sEntry, "/>\n");
	if (fputs (sEntry, m_pOutputFile) == EOF)
		return false;

	// add warning to addlinfo if the incident occurs in the future
	// pad the current time by 20 mins = 0.014 so that we don't cry wolf
	if( m_nDateTime > ( m_nCurrentTime + 0.014 ) )
	{
		AddAddl("Warning", "Bad time");
	}

	CIDSAddlInfo *pCurr = m_pAddlInfoHead;
	while (pCurr)
	{
		if (fprintf (m_pOutputFile, "<addlinfo type=\"%s\">%s</addlinfo>\n", 
			pCurr->AddlType (),
			pCurr->AddlString ()) < 0)
			return false;
		pCurr = pCurr->m_pNext;
	}

	if (fputs ("</logentry>\n", m_pOutputFile) == EOF)
		return false;

	return true;
}

void CIDSLog::CloseOutput ()
{
	fclose (m_pOutputFile);

}

bool CIDSLog::OpenOutput (const char *filename)
{
	m_pOutputFile = fopen (filename, "w");
	
	return m_pOutputFile != NULL;
}

// used only in StripIP f'n below
unsigned long convertIPtoLong( unsigned long inIP)
{
	// the ip is stores in Big Endian format
	// -> the last ip octet is at the back of the long 
	unsigned char * octet = (unsigned char *)&inIP;
	
	unsigned long convertedIP = 0;
	convertedIP += (octet[3] & 0xff);
	convertedIP += (octet[2] & 0xff) * 0xff;
	convertedIP += (octet[1] & 0xff) * 0xffff;
	convertedIP += (octet[0] & 0xff) * 0xffffff;

	return convertedIP;		
}

// returns true if not stripped, false if stripped
// assume ips to strip are in arrays 2 * 4bytes wide and (n addresses) long
bool CIDSLog::StripIP (char *stripped, int ip) const
{
	bool strippedflag = 0;
	unsigned long base = 0;
	unsigned long range = 0;

	if (m_pIPMasks)
	{
		int i = 0;
		while (m_pIPMasks[i])
		{
			range = ~m_pIPMasks[i][1];

			if ((ip & m_pIPMasks[i][1]) == m_pIPMasks[i][0])
			{
				strippedflag = 1;

				unsigned long obfuscatedIP = ip & range;
				
				// we are interested in the last part of the ip
				// which we will become the "obfuscated ip"
				// SO convert the ip to number
				ip = convertIPtoLong(obfuscatedIP) + base;
				break;
			}

			base += convertIPtoLong(range);
			i++;
		}
	}

	if( !strippedflag)
	{
		unsigned char *octets = (unsigned char *)&ip;
		sprintf (stripped, "%d.%d.%d.%d", octets[0], octets[1], octets[2], octets[3]);
	}
	else 
		sprintf (stripped, "%d", ip);
	
	return (!strippedflag);
}

bool CIDSLog::SetIPMasks (char *masklist)
{
	// error condition: true if OK, false if input is malformed
	bool ret = true;
	
	// arbitrary maximum amt of ipmasks
#define NUM_IPMASKS 100
	int pIPStrings[NUM_IPMASKS][2];

	if (!masklist) {
		m_pIPMasks = NULL;
		return ret;
	}

	// remove , : ;
	int i = 0;
	while (masklist[i] != '\0')
	{
		if (masklist[i] == ',' || masklist[i] == ':' || masklist[i] == ';')
			masklist[i] = ' ';
		i++;
	}

	// count spaces and pre-allocate
	i = 0;
	int ipcount = 0;

	// skip any leading spaces
	while ((masklist[i] != '\0') && (masklist[i] == ' '))
		i++;
	while (masklist[i] != '\0')
	{
		pIPStrings[ipcount][0] = i;						// ipmask position

		// skip any non-spaces
		while ((masklist[i] != '\0') && (masklist[i] != ' '))
			i++;

		pIPStrings[ipcount][1] = i - pIPStrings[ipcount][0];	// ipmask length
		
		
		ipcount++;
		if (ipcount == NUM_IPMASKS)
			break;

		// skip any spaces
		while ((masklist[i] != '\0') && (masklist[i] == ' '))
			i++;
	}
	if (ipcount == 0)
	{
		m_pIPMasks = NULL;
		return ret;
	}

	m_pIPMasks = new int * [ipcount + 1];

	
	// parse addresses
	int nummasks = 0;
	for (i = 0; i < ipcount; i++)
	{
		char *sIPstring = new char[pIPStrings[i][1] + 1];
		strncpy (sIPstring, masklist + pIPStrings[i][0], pIPStrings[i][1]);
		sIPstring[pIPStrings[i][1]] = '\0';
		
		int ip1, ip2, ip3, ip4, ipm;
		int n = sscanf (sIPstring, "%d.%d.%d.%d/%d", &ip1, &ip2, &ip3, &ip4, &ipm);
		if (n >= 4)
		{
			unsigned int ipaddr = 0;
			unsigned char *octets = (unsigned char *)&ipaddr;
			octets[0] = (unsigned char) (ip1 & 0xff);
			octets[1] = (unsigned char) (ip2 & 0xff);
			octets[2] = (unsigned char) (ip3 & 0xff);
			octets[3] = (unsigned char) (ip4 & 0xff);

			m_pIPMasks[nummasks] = new int[2];
			m_pIPMasks[nummasks][0] = ipaddr;
			if (n == 4)
				ipm = 32;

			unsigned int tmp, ipmask;
			octets = (unsigned char *)&ipmask;
			if (ipm == 0)
				tmp = 0;
			else
				tmp = 0xffffffff << (32 - ipm);
			octets[0] = (unsigned char) ((tmp >> 24) & 0xff);
			octets[1] = (unsigned char) ((tmp >> 16) & 0xff);
			octets[2] = (unsigned char) ((tmp >> 8) & 0xff);
			octets[3] = (unsigned char) (tmp & 0xff);
			m_pIPMasks[nummasks][1] = ipmask;

			nummasks++;
		}
		else
			ret = false;
		
		delete[] sIPstring;
	}

	m_pIPMasks[nummasks] = NULL;
	return ret;
}

void CIDSLog::FreeIPMasks ()
{
	if (!m_pIPMasks)
		return;

	int i = 0;
	while (m_pIPMasks[i] != NULL)
	{
		delete[] m_pIPMasks[i];
		i++;
	}

	delete[] m_pIPMasks;
	m_pIPMasks = NULL;
}

double CIDSLog::GetDate (int yr, int mt, int dy, int hr, int mn, int sc, int ms)
{
	if( m_nLoggingInGMT == 0)
	{
		struct tm localtimestruct;
		struct tm GMTTime;

		// init a timestruct for timezone etc.
		time_t now;
		time(&now);
		localtimestruct = *localtime(&now);


		localtimestruct.tm_year = yr - EPOCH_YEAR;
		localtimestruct.tm_mon = mt-1;
		localtimestruct.tm_mday = dy; // mday = day of the month
		localtimestruct.tm_hour = hr;
		localtimestruct.tm_min = mn;
		localtimestruct.tm_sec = sc;

		time_t localtimelong = mktime( &localtimestruct );

		// convert to GMT
		GMTTime = *gmtime(&localtimelong);

		yr = GMTTime.tm_year + EPOCH_YEAR;
		mt = GMTTime.tm_mon+1;
		dy = GMTTime.tm_mday;
		hr = GMTTime.tm_hour;
		mn = GMTTime.tm_min;
		sc = GMTTime.tm_sec;
		
	}
	
	// date
	int nDays = sub_year (yr, EPOCH_YEAR) - EPOCH_DAYS;
	nDays += calc_dayofyear (yr, mt, dy);

	// time
	double nDate;
	nDate = (double) (1000 * (60 * (60 * hr + mn) + sc) + ms);
	nDate /= (24 * 60 * 60 * 1000);
	nDate += (double) nDays;

	return nDate;
}

// this function is only called from main.cpp BUT its a mirror of CIDSLog::GetDate
// so it stays heuh
double CIDSLog::GetDate (int yr, int mt, int dy, int hr, int mn, int sc, int ms, int convertToGMT)
{
	if( convertToGMT)
	{
		struct tm when;
		time_t now;

		time( &now );
		when = *localtime( &now );

		struct tm localtimestruct;
		struct tm GMTTime;

		localtimestruct.tm_year = yr - EPOCH_YEAR;
		localtimestruct.tm_mon = mt-1;
		localtimestruct.tm_mday = dy; // mday = day of the month
		localtimestruct.tm_hour = hr;
		localtimestruct.tm_min = mn;
		localtimestruct.tm_sec = sc;

		time_t localtimelong = mktime( &localtimestruct );

		// convert to GMT
		GMTTime = *gmtime(&localtimelong);

		yr = GMTTime.tm_year + EPOCH_YEAR;
		mt = GMTTime.tm_mon+1;
		dy = GMTTime.tm_mday;
		hr = GMTTime.tm_hour;
		mn = GMTTime.tm_min;
		sc = GMTTime.tm_sec;
	}
	
	// date
	int nDays = sub_year (yr, EPOCH_YEAR) - EPOCH_DAYS;
	nDays += calc_dayofyear (yr, mt, dy);

	// time
	double nDate;
	nDate = (double) (1000 * (60 * (60 * hr + mn) + sc) + ms);
	nDate /= (24 * 60 * 60 * 1000);
	nDate += (double) nDays;

	return nDate;
}

void CIDSLog::GetDate (double nDate, int& yr, int& mt, int& dy, int& hr, int& mn, int& sc, int& ms)
{
	// get date from nDate
	int nDays = (int) nDate + EPOCH_DAYS;
	int nDayOfYear = 0;
	add_days (EPOCH_YEAR, nDays, yr, nDayOfYear);
	calc_date (yr, nDayOfYear, mt, dy);

	// only need time from nDate now
	nDate -= floor (nDate);
	int msec = (int) (nDate * (1000.0 * 60.0 * 60.0 * 24.0));
	hr = msec / 3600000;
	mn = (msec / 60000) % 60;
	sc = (msec / 1000) % 60;
	ms = msec % 1000;
}

void CIDSLog::AddAddl (const char *sAddlType, const char *sAddlString)
{
	// clean sAddlString
	char sTmp[LOG_STRLEN + 1];
	int indx = 0, n = strlen (sAddlString);
	for (int i = 0; (i < n) && (indx < LOG_STRLEN); i++)
	{
		if (sAddlString[i] == '&')
		{
			int len = LOG_STRLEN - indx;
			if (len > 5)			// strlen ("&amp;")
				len = 5;
			strncpy (sTmp + indx, "&amp;", len);
			indx += len;
		}
		else if (sAddlString[i] == '<')
		{
			int len = LOG_STRLEN - indx;
			if (len > 4)			// strlen ("&lt;")
				len = 4;
			strncpy (sTmp + indx, "&lt;", len);
			indx += len;
		}
		else if (sAddlString[i] == '>')
		{
			int len = LOG_STRLEN - indx;
			if (len > 4)			// strlen ("&gt;")
				len = 4;
			strncpy (sTmp + indx, "&gt;", len);
			indx += len;
		}
		else if (sAddlString[i] & 0x80)		// if extended ascii
		{
			int len = LOG_STRLEN - indx;
			if (len > 6)			// strlen ("&#xhh;")
				len = 6;
			int nChar = sAddlString[i];
			SNPRINTF (sTmp + indx, len, "&#x%2x;", nChar & 0xff);
			indx += len;
		}
		else
			sTmp[indx++] = sAddlString[i];
	}
	sTmp[indx] = '\0';

	CIDSAddlInfo *pNew = new CIDSAddlInfo (sAddlType, sTmp);
	CIDSAddlInfo *pNext = m_pAddlInfoHead;
	while (pNext)
	{
		if (pNext->m_pNext)
			pNext = pNext->m_pNext;
		else
			break;
	}
	if (pNext)
		pNext->m_pNext = pNew;
	else
		m_pAddlInfoHead = pNew;
}

void CIDSLog::ClearAddl ()
{
	CIDSAddlInfo *pCurr = m_pAddlInfoHead;
	while (pCurr)
	{
		CIDSAddlInfo *pNext = pCurr->m_pNext;
		delete pCurr;
		pCurr = pNext;
	}
	m_pAddlInfoHead = NULL;
}

int CIDSLog::sub_year (int y1, int y2)
{
	int days = 0;
	for (int i=y2; i < y1; i++)
	{
		if (i % 4 != 0)
			days += 365;
		else if (i % 400 == 0)
			days += 366;
		else if (i % 100 == 0)
			days += 365;
		else
			days += 366;
	}

	return days;
}

void CIDSLog::add_days (int y, int d, int& y2, int& d2)
{
	for (; ; y++)
	{
		if (y % 4 != 0)
		{
			if (d >= 365)
				d -= 365;
			else
				break;
		}
		else if (y % 400 == 0)
		{
			if (d >= 366)
				d -= 366;
			else
				break;
		}
		else if (y % 100 == 0)
		{
			if (d >= 365)
				d -= 365;
			else
				break;
		}
		else
		{
			if (d >= 366)
				d -= 366;
			else
				break;
		}
	}
	
	y2 = y;
	d2 = d;
}

void CIDSLog::calc_date (int y, int d, int& mt, int& dy)
{
	bool leap;

	if (y % 4 != 0)
		leap = false;
	else if (y % 400 == 0)
		leap = true;
	else if (y % 100 == 0)
		leap = false;
	else
		leap = true;

	mt = 0;
	for (; ; mt++)
	{
		if (mt == 0 || mt == 2 || mt == 4 || mt == 6 || mt == 7 || mt == 9 || mt == 11)
		{
			if (d >= 31)
				d -= 31;
			else
				break;
		}
		else if (mt == 3 || mt == 5 || mt == 8 || mt == 10)
		{
			if (d >= 30)
				d -= 30;
			else
				break;
		}
		else if (mt == 1 && leap)
		{
			if (d >= 29)
				d -= 29;
			else
				break;
		}
		else if (mt == 1 && !leap)
		{
			if (d >= 28)
				d -= 28;
			else
				break;
		}
		else
			break;
	}
	mt++;				// months are 1-based
	dy = d + 1;			// as are days
}


int CIDSLog::calc_dayofyear (int y, int m, int d)
{
	bool leap;

	if (y % 4 != 0)
		leap = false;
	else if (y % 400 == 0)
		leap = true;
	else if (y % 100 == 0)
		leap = false;
	else
		leap = true;

	m--;				// months are 1-based
	d--;				// as are days
	
	int days = 0;
	for (int i=0; i < m; i++)
	{
		if (i == 0 || i == 2 || i == 4 || i == 6 || i == 7 || i == 9 || i == 11)
			days += 31;
		else if (i == 3 || i == 5 || i == 8 || i == 10)
			days += 30;
		else if (i == 1 && leap)
			days += 29;
		else if (i == 1 && !leap)
			days += 28;
	}
	
	days += d;
	return days;
}

int CIDSLog::GetNumRecords() const
{
	return m_nEntry;
}

bool CIDSLog::SkippedRecords() const
{
	return m_bSkippedRecords;
}

