#ifdef _WIN32
#include <winsock.h>
#else
#include <config.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#endif
#include <string.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>


#include "SnortLog.h"
#include "misc.h"

#if !defined(INADDR_NONE)
#define INADDR_NONE (in_addr_t)-1
#endif

#define FLOAT_GT_OR_EQ(x, y) ( ((x) - (y)) >= (-0.00000001))

CSnortLog::CSnortLog()
{
  m_pLogfile = NULL;
  m_pPortScanFile = NULL;
  m_sPortScan[0] = 0;
  m_savedTime = 0.0;
  m_lastTime = 0.0;

  m_nPortScanLines = 0;
  m_nAlertLines = 0;
  m_nPortScanEvents = 0;
  m_nAlertEvents = 0;
  m_nPortScanSkipped = 0;
  m_nAlertSkipped = 0;
}

CSnortLog::~CSnortLog()
{
}


double CSnortLog::GetLastDate()
{
  return m_lastTime;
}
//
// CSnortLog::IDSVersion()
//
// Args:
//   char *idsver;  // pointer to a buffer
//
// Purpose:
//   Copies name and version of IDS for which we
//   are parsing logs into buffer
//

void CSnortLog::IDSVersion(char *sIDSProduct, int &nIDSID, int &nIDSMajor, int &nIDSMinor, char *sIDSRev) const
{

  memset(sIDSProduct, 0, LOG_STRLEN);
  memset(sIDSRev, 0, LOG_STRLEN);
#if defined(_WIN32)
  strncpy(sIDSProduct, "Win32 Snort", LOG_STRLEN - 1);
#else
  strncpy(sIDSProduct, "Unix Snort", LOG_STRLEN - 1);
#endif
  nIDSID = 2; // ID Code for snort
  nIDSMajor = 1;
  nIDSMinor = 6;
  strncpy(sIDSRev, "or greater", LOG_STRLEN - 1);	/* 1.01beta3 */
  return;
}

//
// CSnortLog::SeekLog()
//
// Args:
//   double nDate;  // Date and time represented as a double
//
// Purpose:
//   Seek into both alert and portscan logs until a record is reached 
//   which is newer than the date specified.  This is implemented by
//   setting a member variable which is used by the log reading routines to 
//   skip over entries
//


void CSnortLog::SeekLog(double nDate)
{
  m_savedTime = nDate;

  // m_lastTime needs to be >= m_savedTime
  m_lastTime = m_savedTime;
  return;
}


//
// CSnortLog::OpenLog()
//
// Args:
//   char *filename;  // Contains name of both alert and portscan logfiles 
//                    // separated by a comma ',' or alert filename only
// Purpose:
//   Separate filename argument into two components if it contains a comma
//   and opens both files.  Otherwise, opens alert file only and sets 
//   m_pPortscanFile to NULL
//
// Return:
//  true if logfiles are successfully opened.  false otherwise
//

bool CSnortLog::OpenLog(const char *filename)
{
  char *tmp;
  char *logfile, *portscanfile;
  
  
  
  assert(m_pLogfile == NULL);

  tmp = strdup(filename);

  //
  // 'filename' contains both the alert logfile and the portscan
  //  logfile separated by a comma
  //
  logfile = tmp;
  if((portscanfile = strchr(tmp, ','))) {
    *portscanfile = '\0';
    portscanfile++;
  }

  if((m_pLogfile = fopen(logfile, "r")) == NULL) {
    debug("Could not open: %s\n", filename);
    free(tmp);
    return false;
  }

  m_pPortScanFile = NULL;
  if(portscanfile) {
    if((m_pPortScanFile = fopen(portscanfile, "r")) == NULL) {
      debug("Could not open: %s\n", filename);
      free(tmp);
      return false;
    }
  }

  free(tmp);

  /// spp
  m_nSpp = 0;
  m_nSppDateTime = 0.0;
  //

  return true;
}

//
// CSnortLog::LastIncident()
//
// Purpose:
//   Determine if the end of both logfiles has been reached.
//
// Return:
//   true if at EOF for both logfiles.  false otherwise.
//
bool CSnortLog::LastIncident()
{
  assert(m_pLogfile);

  // Alert file only
  if(!m_pPortScanFile && feof(m_pLogfile))
    return true;

  // Alert and Portscan file
  if(feof(m_pLogfile) && feof(m_pPortScanFile))
    return true;

  return false;
}

int CSnortLog::guessYear(int month, int day)
{
  time_t t;
  struct tm *tm;
  int year;
  //
  // Since snort (<= 1.7) doesn't save the year in timestamps we have to guess.  If the date is before
  // the present day we'll assume the present year.  If it's after, then we'll guess last year.
  // If your logs have entries more than a year old, all bets are off.
  t = time(NULL);
  tm = localtime(&t);
  year = tm->tm_year + 1900;

  if(month == (tm->tm_mon + 1)) {
    if(day > tm->tm_mday)
      year--;
  }
  else
    if(month > (tm->tm_mon + 1))
      year --;

  return year;
}

//
//  CSnortLog::do_date()
//    
//  Purpose:
//    Convert components of date and time into double representation.  Guess 
//    what year it might be.
//
//  Returns:
//    Date as a double 
//

double CSnortLog::do_date()
{

  if (m_nYear < 0)
    m_nYear = guessYear(m_nMonth, m_nDay); 
  return (GetDate(m_nYear, 
		  m_nMonth, m_nDay, m_nHour, m_nMinute, m_nSecond, m_nFract / 1000));
}

//
// CSnortLog::ReadIncident()
//
// Purpose:
//   This is the function called by the main log parsing loop to read a single
//   event from a log file.  Since Snort requires that we read information from
//   both the alert log file and the portscan log file this has been split into
//   two other functions.  Each call to ReadIncident() calls DoAlertLog() until 
//   the end of the alert log is reached.  Then DoPortScanLog() is called each 
//   time until the port scan log is finished.  The caller of this function 
//   expects certain member variables to be set if this function returns true.
//   This work is done in both DoAlertLog() and DoPortScanLog() and the variables
//   contain all of the relevant information about the event that was just processed.
//
// Return:
//   True if an event was successfully parsed.  False otherwise.
//
bool CSnortLog::ReadIncident()
{
  if(!feof(m_pLogfile))
    return (DoAlertLog());
  if(m_pPortScanFile && !feof(m_pPortScanFile))
    return (DoPortScanLog());
  return false;
}
//
// CSnortLog::DoPortScanLog()
//
// Purpose:
//   This function is called to parse an "event" from the portscan logfile.  Snort stores portscan
//   data with one line for every port probed, but ideally we'd like to collect all of these lines
//   into a single portscan event.  This function first reads one line and makes note of the source 
//   and destination IP addresses as well as the scan type.  Then lines are read until a line 
//   containing either a different pair of IP addresses or a different scan type is reached.  This
//   function adds additional data to the event containing the ports scanned represented as 
//   comma delimited ports and ranges.
//
//   example:  "23,25,139,6000-6010"
//
// Returns:
//   True if event successfully created from portscan logfile.  False otherwise.
//

bool CSnortLog::DoPortScanLog()
{
  char buffer[LOG_STRLEN];
  char scanType[41];
  char curType[41];
  char *months[] = {"NONE", "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
  char strMon[4];
  int month, day, hour, minute, second, i;
  char srcIP[16], dstIP[16];
  char type[21], flags[21];
  int srcPort, dstPort;
  long filepos;
  struct in_addr in1, in2;
  

  m_nToIP = 0;
  clearports();
  m_nFromPort = m_nToPort = 0;
  m_nCount = 1;

  // problem: if the portscan file has an extra empty line in it
  // the last lines of the portscan file get dropped from the xml file
  // 
  // for the last empty line, the sscanf statement fails, restarting 
  // at the top of the for loop.
  // the fgets then fails (being at the end of file)
  // returning false.
  // sooo, only return false from the fgets if the for loop 
  // hasn't been successful for at least one loop

  bool portsToDump = false;

  for(;;) {
    filepos = ftell(m_pPortScanFile);

    if(!fgets(buffer, sizeof(buffer) - 1, m_pPortScanFile)) 
	{
	  if( portsToDump )
	  {
		// copy of code below... 
		(void)dumpports();     // This fills m_sPortScan[] in IDSLog 
		fseek(m_pPortScanFile, filepos, SEEK_SET);
		m_nDateTime = do_date();

		if(m_nDateTime > m_lastTime)
		  m_lastTime = m_nDateTime;

		if(strncmp(scanType, "UDP", 3) == 0)
		  m_nProtocol = PROTO_UDP;
		else
		  m_nProtocol = PROTO_TCP;
		strcpy(m_sID, "Portscan: ");
		strcat(m_sID, scanType);
		m_nPortScanEvents++;
		return true;
	  }
	  else
	    return false;
	}

    m_nPortScanLines++;

    flags[0] = 0;
    if(sscanf(buffer, "%3c %d %d:%d:%d %15[.0-9]:%d -> %15[.0-9]:%d %20s %20s ", strMon, &day, &hour, &minute, &second,
	      srcIP, &srcPort, dstIP, &dstPort, type, flags) < 10) {
      debug("Could not parse line: %s\n", buffer);
      continue;
    }
   
    strMon[3] = 0;
    for(i = 1; i <= 12; i++)
      if(!strcmp(strMon, months[i])) {
	month = i;
	break;
      }
    if(i > 12) {
      debug("Could not parse month: %s\n", strMon);
      continue;
    }

    //
    // Keep skipping lines until we pass the date in m_savedTime 
    //
    if(FLOAT_GT_OR_EQ(m_savedTime,GetDate(guessYear(month, day), month, day, hour, minute, second, 0))) {
      m_nPortScanSkipped++;
      continue;
    }
    
    strcpy(curType, type);
    //    if(strlen(flags)) {
    //  strcat(curType, " ");
    //  strcat(curType, flags);
    // }
    
	if(!m_nToIP) 
	{
      strcpy(scanType, curType);
      // Since we're condensing all of the ports probed into
      // a single event, we'll use timestamp of the first
      // probe for the event.
      m_nMonth = month;
      m_nDay = day; 
      m_nHour = hour;
      m_nMinute = minute;
      m_nSecond = second;
      m_nFract = 0;
      m_nYear = -1;
      addport(dstPort);

#if  defined(_WIN32) || !defined(HAVE_INET_ATON) 
	  if ((in1.s_addr = inet_addr(srcIP)) == INADDR_NONE)
#else
	  if(inet_aton(srcIP, &in1) == 0)
#endif
	  {
	debug("Could not parse source IP address: %s\n", buffer);
	continue;
      }
      m_nFromIP = in1.s_addr;
#if defined(_WIN32) || !defined(HAVE_INET_ATON)
	  if ((in1.s_addr = inet_addr(dstIP)) == INADDR_NONE)
#else
	  if(inet_aton(dstIP, &in1) == 0)
#endif
	  {
	debug("Could not parse destination IP address: %s\n", buffer);
	continue;
      }
      m_nToIP = in1.s_addr;
    }
    else 
	{
#if defined(_WIN32) || !defined(HAVE_INET_ATON)
	  if ((in1.s_addr = inet_addr(srcIP)) == INADDR_NONE)
#else
	  if(inet_aton(srcIP, &in1) == 0)
#endif
	  {
		debug("Could not parse source IP address: %s\n", buffer);
		continue;
      }

#if defined(_WIN32) || !defined(HAVE_INET_ATON)
	  if ((in2.s_addr = inet_addr(dstIP)) == INADDR_NONE)
#else
      if(inet_aton(dstIP, &in2) == 0)
#endif
	  {
		debug("Could not parse destination IP address: %s\n", buffer);
		continue;
      }
      
      if(in1.s_addr != m_nFromIP ||
		in2.s_addr != m_nToIP ||
		strcmp(scanType, curType)) 
	  {
		(void)dumpports();     // This fills m_sPortScan[] in IDSLog 
		fseek(m_pPortScanFile, filepos, SEEK_SET);
		m_nDateTime = do_date();

		if(m_nDateTime > m_lastTime)
		  m_lastTime = m_nDateTime;

		if(strncmp(scanType, "UDP", 3) == 0)
		  m_nProtocol = PROTO_UDP;
		else
		  m_nProtocol = PROTO_TCP;
		strcpy(m_sID, "Portscan: ");
		strcat(m_sID, scanType);
		m_nPortScanEvents++;
		return true;
      }
      addport(dstPort);
	  portsToDump = true;
    }
  }
}

//
// CSnortLog::clearports()
//
// Purpose:
//   Zero out bit array for storing port information.
//
void CSnortLog::clearports()
{
  for(int i = 0; i < 8192; i++)
    m_pPortArray[i] = 0;
}

//
// CSnortLog::addport()
//
// Args:
//   int port;  // port to add
//
// Purpose:
//   Set bit in port array cooresponding to port argument.
//

void CSnortLog::addport(int port)
{
  m_pPortArray[port / 8] |= 1 << (port % 8);
}

//
// CSnortLog::isportset()
//
// Args:
//   int port;  // port to test
//
// Return:
//   True if bit for port is set in port array.  False if it is not set.
//

bool CSnortLog::isportset(int port)
{
  return ( (m_pPortArray[port / 8] & (1 << (port % 8))) != 0 );
}

//
// CSnortLog::dumpports()
//
// Purpose:
//   Convert array of bits containing port information into a string
//   listing the ports which were set in the array as single ports
//   and ranges of ports separated by commas.
//
//   example:   "23,25,139,6000-6010"
//
// Returns:
//   A pointer the the string. (Which is actually a member variable)
//

char *CSnortLog::dumpports()
{
  int lowPort = 0;
  char range[20];
  int i;

  m_sPortScan[0] = 0;
  for(i = 0; i < 65536; i++) {
    if(!lowPort) {
      if(isportset(i))
	lowPort = i;
      continue;
    }
    if(!isportset(i)) {
      if(i - lowPort == 1)
	sprintf(range, "%d", lowPort);
      else
	sprintf(range, "%d-%d", lowPort, i - 1);

      if(strlen(m_sPortScan) > (PORTSCAN_STRLEN - 25)) {
	strcat(m_sPortScan, "...");
	return m_sPortScan;
      }
	
      if(strlen(m_sPortScan))
	strcat(m_sPortScan, ",");
      strcat(m_sPortScan, range);
      lowPort = 0;
    }
  }
  return m_sPortScan;
}

//
// CSnort::parseAddresses()
//
// Args:
//   char *addresses
//   char *srcIP, *dstIP
//   int  &srcPort, &dstPort
//
// Purpose:
//   Returns true if 'addresses' is in one of two different forms:
//
//   x.x.x.x:srcport -> y.y.y.y : dstport
//   x.x.x.x -> y.y.y.y
//
//   If 'addresses' can be parsed, fills in appropriate variables
//
bool CSnortLog::parseAddresses(const char *addresses, char *srcIP, int &srcPort, char *dstIP, int &dstPort, char *strProto)
{

  srcPort = 0;
  dstPort = 0;


  if(sscanf(addresses, "%15[.0-9] -> %15[.0-9] %4s ", srcIP, dstIP, strProto) >= 2) 
    return true;


    if(sscanf(addresses, "%15[.0-9]:%d -> %15[.0-9]:%d %4s ", srcIP, &srcPort, dstIP, &dstPort, strProto) >= 4) 
      return true;

  return false;
}

void
CSnortLog::cleanID(char *id)
{
  char *tmpStr = strdup(id);
  char *p = tmpStr;

  // Skip over any leading whitespace
  while(isspace(*p))
    p++;
  
  // Skip over interface name if it is present
  if(*p == '<') {
    p = strchr(p, '>');
    if(!p) 
      goto bye;
    p += 2;
  }

  strcpy(id, p);
  
bye:

  if(tmpStr)
    free(tmpStr);
  return;
}

//
// CSnortLog::identifyLog()
//
// Args:
//   char *logLine
//
// Purpose:
//   Tries to guess what format of Snort logfile is currently being parsed by examining
//   logLine.
//
// Return:
//   An enumerated constant indicating type of logfile or SNORT_LOG_DONE if not
//   identified.
//
CSnortLog::log_type CSnortLog::identifyLog(const char *logLine)
{
  /*char srcIP[16], dstIP[16];*/
  char *delim = "[**]";
  int n1, n2, n3, n4, n5, n6, n7;
  char buf[64];
  char month[4];
  /*char proto[6];*/
  char *p, *q;


  if(strlen(logLine) > strlen(delim) &&
     strncmp(logLine, delim, strlen(delim)) == 0 &&
     strstr(logLine + strlen(delim), delim) != NULL) 
    return SNORT_LOG_FULL;
  
  if((sscanf(logLine, "%d/%d-%d:%d:%d.%d", &n1, &n2, &n3, &n4, &n5, &n6) == 6 ||		// snort <= 1.7
      sscanf(logLine, "%d/%d/%d-%d:%d:%d.%d", &n1, &n2, &n3, &n4, &n5, &n6, &n7) == 7) &&	// snort >= 1.8 
      (p = strstr(logLine, delim)) &&
      (q = strstr(p + strlen(delim), delim)) /* &&
      parseAddresses(q + strlen(delim) + 1, srcIP, n1, dstIP, n2, proto)*/)
    return SNORT_LOG_FAST;

  if( sscanf(logLine, "%3s %d %d:%d:%d %64s snort[%d]:", month, &n1, &n2, &n3, &n4, buf, &n5) == 7 &&
      (p = strstr(logLine, "]:")) &&
      (q = strstr(p + 2, ":")) /* &&
      parseAddresses(q + 2, srcIP, n1, dstIP, n2, proto) == true*/)
    return SNORT_LOG_SYSLOG;

  if( sscanf(logLine, "%3s %d %d:%d:%d %64s snort:", month, &n1, &n2, &n3, &n4, buf) == 6 &&
      (p = strstr(logLine, "snort:")) &&
      (q = strstr(p + 6, ":")) /* &&
      parseAddresses(q + 2, srcIP, n1, dstIP, n2, proto) == true*/) 
    return SNORT_LOG_SYSLOG;

  return SNORT_LOG_NONE;
}
//
// CSnortLog::DoAlertLog()
//
// Purpose:
//   Process a single event from a Snort alert file.  This function sets the following member 
//   variables from CIDSLog which are later used to write the event in XML to the output file:
//
//   m_sID       -- String containing ID of event
//   m_nDateTime -- Date and Time stored as a double
//   m_nProtocol -- The protocol number of the event
//   m_nFromIP   -- Source IP address of event
//   m_nToIP     -- Destination IP address of event
//   m_nFromPort -- Source port of event if relevant (0 otherwise)
//   m_nToPort   -- Destination port of event if relevant (0 otherwise)
//   m_nCount    -- Always set to 1
//
// Return:
//   True if an event was successfully processed.  False otherwise.
//

bool CSnortLog::DoAlertLog()
{
  char buffer[LOG_STRLEN];
  log_type logtype;


  for(;;) {
    // We bailed from the previous event as soon as we had all the information that we needed
    // but this will skip over any remaining lines
    for(;;) {
      if(!fgets(buffer, sizeof(buffer) - 1, m_pLogfile))
	return false;
      m_nAlertLines++;

      if( (logtype = identifyLog(buffer)) != SNORT_LOG_NONE)
	break;
    }

    switch(logtype) {
      case SNORT_LOG_FULL:
        if(ProcessLogFull(buffer)) {
	  m_nAlertEvents++;
	  return true;
	}
        break;
      case SNORT_LOG_FAST:
        if(ProcessLogFast(buffer)) {
	  m_nAlertEvents++;
	  return true;
	}
        break;
      case SNORT_LOG_SYSLOG:
        if(ProcessLogSyslog(buffer)) {
	  m_nAlertEvents++;
	  return true;
	}
        break;
      case SNORT_LOG_NONE:
      default:
	// not possible
        break;

    }

  }
  return false;
}

bool CSnortLog::ProcessLogSyslog(char *line)
{

  char strMon[4];
  char proto[6];
  char srcIP[16], dstIP[16];
  struct in_addr in;
  char *months[] = {"NONE", "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
  char *p, *q;
  int i;

  // Skip all of the plugin events except http_decode
  if(!strstr(line, "spp_http_decode"))
  {
    m_nSpp--;           // change state   
    if (m_nSpp < 0)
      m_nSpp = 0;       // "nevermind" state
    if(strstr(line, "spp_"))
      return false;
  }
  else
    m_nSpp = 2;		// "received an spp event" state

  if(sscanf(line, "%3c %d %d:%d:%d ", strMon, &m_nDay, &m_nHour, &m_nMinute, &m_nSecond) != 5) {
    debug("Could not parse date: %s\n", line);
    return false;
  }
  strMon[3] = 0;
  for(i = 1; i <= 12; i++) 
    if(!strcmp(strMon, months[i])) {
      m_nMonth = i;
      break;
    }
  if(i > 12) {
    debug("Could not parse month: %s\n", strMon);
    return false;
  }

  m_nFract = 0;
  m_nYear = -1;

  // 
  // If this entry is older than saved timestamp return false. calling function should loop and read another line
  //
  if(FLOAT_GT_OR_EQ(m_savedTime, do_date ())) {
    m_nAlertSkipped++;
    return false;
  }

  m_nDateTime = do_date();


  // set or compare against the date of the most recent spp_http_decode event
  if (m_nSpp == 2)
    m_nSppDateTime = m_nDateTime;
  else if (m_nSpp == 1)
  {
    m_nSpp = 0;
    if (m_nSppDateTime == m_nDateTime)  
      return false;                             // skip next event
  }
  /////////////


  if (strstr (line, "snort["))
  {
    if((p = strstr(line, "]:")))
      p+=3;
    else
      return false;
  } 
  else if((p = strstr(line, "snort:")))
    p+=7;
  else
    return false;

  /* Second Delimiter */
  if ((q = strstr(p, " [Classification:")))
  {
    /* Third Delimiter */
    char *pSkip = strstr(q, "]:");
    *q = 0;
    if (pSkip)
      q = pSkip + 3;
    else
      return false;
  }
  else if ((q = strstr (p, " {")))
  {
    *q = 0;
    q++;
  }
  else if((q = strstr(p, ":")) == NULL)
    return false;
  else
  {
    *q = 0;
    q += 2;
  }

  // cph
  // remove etherport string <eth1>
  // depending on the format, and how the strings have been
  // parsed to this point, it may be at the beginning of string q
  // or at the end of string p
  //
  // check p
  char *pSkip1 = strstr(p, " <");
  if(pSkip1)
  {
    char *pSkip2 = strstr(p, ">");
    if (pSkip2)
	{
	  // terminate string p before the first '<'
	  *pSkip1 = 0;
	}
    else
      return false;
  }
  else // check q
  {
    pSkip1 = strstr(q, "<");
    if(pSkip1)
	{
  	  char *pSkip2 = strstr(q, "> ");
  	  if (pSkip2)
	  {
	    q = pSkip2 + 2;
	  }
      else
	    return false;
	}
  }


  // now look for the [n:n:n] pattern from Snort 1.8+
  char *sFinalSigID = p;
  unsigned int tmp_n0 = 0, tmp_n1 = 0, tmp_n2 = 0;
  /* as a prefix to the signature id */
  if (sscanf (sFinalSigID, "[%u:%u:%u] ", &tmp_n0, &tmp_n1, &tmp_n2) == 3)
    sFinalSigID = strchr (sFinalSigID, ']') + 2;
  /* as a suffix to the signature id */
  char *sFinalSigIDEnd = strstr (sFinalSigID, " [");
  if (sFinalSigIDEnd)
  {
    if (sscanf (sFinalSigIDEnd, " [%u:%u:%u]", &tmp_n0, &tmp_n1, &tmp_n2) == 3) 
      *sFinalSigIDEnd = 0;
  }

  memset(m_sID, 0, LOG_STRLEN + 1);
  strncpy(m_sID, sFinalSigID, LOG_STRLEN);
  cleanID(m_sID);

  // this is actually defined in 1.8+
  m_nProtocol = 0;
  if (*q == '{')                        // Snort 1.8+
  {
    char *sProtoStart = q + 1;
    char *sProtoEnd = strstr (q, "} ");
    if (sProtoEnd)
    {
     *sProtoEnd = '\0';
      q = sProtoEnd + 2;
    }
    // now sProtoStart contains the protocol information
    if (!strcmp (sProtoStart, "TCP"))  
      m_nProtocol = PROTO_TCP;
    else if (!strcmp (sProtoStart, "UDP"))
      m_nProtocol = PROTO_UDP;
    else if (!strcmp (sProtoStart, "ICMP"))
      m_nProtocol = PROTO_ICMP;
  }

  if(!parseAddresses(q, srcIP, m_nFromPort, dstIP, m_nToPort, proto)) {
    debug("Failed to parse IP and port information: %s\n", q);
    return false;
  }

#if defined(_WIN32) || !defined(HAVE_INET_ATON)
  if ((in.s_addr = inet_addr(srcIP)) == INADDR_NONE)
#else
  if(inet_aton(srcIP, &in) == 0)
#endif
  {
    debug("Could not parse source IP address: %s\n", srcIP);
    return false;
  }
  m_nFromIP = in.s_addr;

#if defined(_WIN32) || !defined(HAVE_INET_ATON)
  if ((in.s_addr = inet_addr(dstIP)) == INADDR_NONE)
#else
  if(inet_aton(dstIP, &in) == 0) 
#endif
  {
    debug("Could not parse destination IP address: %s\n", dstIP);
    return false;
  }

  m_nToIP = in.s_addr;

  m_nCount = 1; // always 1 for Snort

  m_sPortScan[0] = 0;

  if(m_nDateTime > m_lastTime)
    m_lastTime = m_nDateTime;

  return true;
  
}

bool CSnortLog::ProcessLogFast(char *line)
{

  char *delim = "[**]";
  char *p, *q;
  char proto[6];
  char srcIP[16], dstIP[16];
  struct in_addr in;

  //
  // Skip plugin generated events except http_decode
  //
  if(!strstr(line, "spp_http_decode"))
  {
    m_nSpp--;           // change state   
    if (m_nSpp < 0)
      m_nSpp = 0;       // "nevermind" state
    if(strstr(line, "spp_"))
      return false;
  }
  else
    m_nSpp = 2;         // "received an spp event" state

  if(sscanf(line, "%d/%d/%d-%d:%d:%d.%d", &m_nMonth, &m_nDay, &m_nYear, &m_nHour,
  	    &m_nMinute, &m_nSecond, &m_nFract) != 7) {
    m_nYear = -1;
  
    if(sscanf(line, "%d/%d-%d:%d:%d.%d", &m_nMonth, &m_nDay, &m_nHour, 
  	      &m_nMinute, &m_nSecond, &m_nFract) != 6) {
      debug("Date parse failed: %s\n", line);
      return false;
    }
  }
  else
  {
    m_nYear += 2000;			/* snort 1.8+ years are two-digit */
  }

  //
  // Check to see if this record is newer than saved timestamp.
  //  If not, return false and calling function should loop and 
  //  read another line.
  //
  if(FLOAT_GT_OR_EQ(m_savedTime, do_date ())) {
    m_nAlertSkipped++;
    return false;
  }

  m_nDateTime = do_date();


  // set or compare against the date of the most recent spp_http_decode event
  if (m_nSpp == 2)
    m_nSppDateTime = m_nDateTime;
  else if (m_nSpp == 1)
  {
    m_nSpp = 0;
    if (m_nSppDateTime == m_nDateTime)  
      return false;                             // skip next event
  }
  /////////////


  if((p = strstr(line, delim)) == NULL) // First delimiter
    return false;
  p += strlen(delim) + 1;
  
  if((q = strstr(p, delim)) == NULL) // Second delimiter
    return false;
  *(q - 1)= 0; // back up to the space
  q += strlen(delim) + 1; // skip over delimiter and space

  // now look for the [n:n:n] pattern from Snort 1.8+
  char *sFinalSigID = p;
  unsigned int tmp_n0 = 0, tmp_n1 = 0, tmp_n2 = 0;
  if (sscanf (sFinalSigID, "[%u:%u:%u] ", &tmp_n0, &tmp_n1, &tmp_n2) == 3)
    sFinalSigID = strchr (sFinalSigID, ']') + 2;
  
  memset(m_sID, 0, LOG_STRLEN + 1);
  strncpy(m_sID, sFinalSigID, LOG_STRLEN);
  cleanID(m_sID);


  /* skip Classification and Priority (Snort 1.8+) */
  char *pClas = strstr (q, "[Classification: ");
  if (pClas)
  {
    char *pClas2 = strstr (pClas, "] ");
    if (pClas2)
    {
      q = pClas2 + 2;					// text after Classification
      char *pPri = strstr (q, "[Priority: ");
      if (pPri)
      {
        char *pPri2 = strstr (pPri, "] ");
        if (pPri2)
          q = pPri2 + 2;				// text after priority
      }
    }
  }

  // this is actually defined in 1.8+
  m_nProtocol = 0;
  if (*q == '{')			// Snort 1.8+
  {
    char *sProtoStart = q + 1;
    char *sProtoEnd = strstr (q, "} ");
    if (sProtoEnd)
    {
     *sProtoEnd = '\0';
      q = sProtoEnd + 2;
    }
    // now sProtoStart contains the protocol information
    if (!strcmp (sProtoStart, "TCP"))
      m_nProtocol = PROTO_TCP;
    else if (!strcmp (sProtoStart, "UDP"))
      m_nProtocol = PROTO_UDP;
    else if (!strcmp (sProtoStart, "ICMP"))
      m_nProtocol = PROTO_ICMP;
  }

  if(!parseAddresses(q, srcIP, m_nFromPort, dstIP, m_nToPort, proto)) {
    debug("Failed to parse IP, and port information: %s\n", q);
    return false;
  }

#if defined(_WIN32) || !defined(HAVE_INET_ATON)
  if ((in.s_addr = inet_addr(srcIP)) == INADDR_NONE)
#else
  if(inet_aton(srcIP, &in) == 0)
#endif
  {
    debug("Could not parse source IP address: %s\n", srcIP);
    return false;
  }
  m_nFromIP = in.s_addr;

#if defined(_WIN32) || !defined(HAVE_INET_ATON)
  if ((in.s_addr = inet_addr(dstIP)) == INADDR_NONE)
#else
  if(inet_aton(dstIP, &in) == 0) 
#endif
  {
    debug("Could not parse destination IP address: %s\n", dstIP);
    return false;
  }

  m_nToIP = in.s_addr;
  
  m_nCount = 1; // always 1 for Snort

  m_sPortScan[0] = '\0';

  if(m_nDateTime > m_lastTime)
    m_lastTime = m_nDateTime;

  return true;
  
}
bool CSnortLog::ProcessLogFull(char *line)
{
  char *p;
  char *delim = "[**]";
  char buffer[LOG_STRLEN];
  char proto[6];
  char srcIP[16], dstIP[16];
  struct in_addr in;
  

  //
  // Skip all of the plugin generated events except http_decode
  //
  if(!strstr(line, "spp_http_decode"))
  {
    m_nSpp--;           // change state
    if (m_nSpp < 0)
      m_nSpp = 0;       // "nevermind" state
    if(strstr(line, "spp_"))
      return false;
  }
  else
    m_nSpp = 2;         // "received an spp event" state


  //
  // Snort event ID lines look like this:
  //   [**] IDS177/netbios-name-query [**]
  // Or like this: (1.8+)
  //   [**] [1:5:1] IDS177/netbios-name-query [**]
  // So we need to remove the [**] sequences and the spaces to get the event ID string
  
  if((p = strstr(line, delim)) == NULL) // First delimiter
    return false;
  p += strlen(delim);
  if((p = strstr(p, delim)) == NULL) // Second delimiter
    return false;
  p--; // Back up to the space
  *p = 0;

  // now look for the [n:n:n] pattern from Snort 1.8+ 
  char *sFinalSigID = line + strlen(delim) + 1;
  unsigned int tmp_n0 = 0, tmp_n1 = 0, tmp_n2 = 0;
  if (sscanf (sFinalSigID, "[%u:%u:%u] ", &tmp_n0, &tmp_n1, &tmp_n2) == 3)
    sFinalSigID = strchr (sFinalSigID, ']') + 2;

  memset(m_sID, 0, LOG_STRLEN + 1);
  strncpy(m_sID, sFinalSigID, LOG_STRLEN);
  cleanID(m_sID);

  if(!fgets(buffer, sizeof(buffer) - 1, m_pLogfile))
    return false;
  m_nAlertLines++;

  if (strstr (buffer, "[Classification: "))			/* (Snort 1.8+) skip Classification and Priority */
  {
    if(!fgets(buffer, sizeof(buffer) - 1, m_pLogfile))
      return false;
    m_nAlertLines++;
  }

  if(sscanf(buffer, "%d/%d/%d-%d:%d:%d.%d", &m_nMonth, &m_nDay, &m_nYear, &m_nHour,
            &m_nMinute, &m_nSecond, &m_nFract) != 7) {
    m_nYear = -1;

    if(sscanf(buffer, "%d/%d-%d:%d:%d.%d", &m_nMonth, &m_nDay, &m_nHour,
              &m_nMinute, &m_nSecond, &m_nFract) != 6) {
      debug("date parse failed: %s\n", buffer);
      return false;
    }
  }
  else
  {
    m_nYear += 2000;					/* Snort 1.8+ years are two-digit */
  }

  //
  //  Here's where we check to see if we're processing a record that is newer than the saved timestamp.
  //  If not, we return false and the calling function go back into a loop and read another record.
  //
  if(FLOAT_GT_OR_EQ(m_savedTime,
			   do_date ())) {
    m_nAlertSkipped++;
    return false;
  }

  m_nDateTime = do_date();


  // set or compare against the date of the most recent spp_http_decode event
  if (m_nSpp == 2)
    m_nSppDateTime = m_nDateTime;
  else if (m_nSpp == 1)
  {
    m_nSpp = 0;
    if (m_nSppDateTime == m_nDateTime)
      return false;				// skip next event
  }
  /////////////


  //
  // Ok, now we need to determine if the log was recorded with the ethernet flag (-e)
  //
  for(p = buffer; *p != ' '; p++) // find the first space
    if(*p == 0)
      return false;

  memset(proto, 0, sizeof(proto));
  if(!parseAddresses(p + 1, srcIP, m_nFromPort, dstIP, m_nToPort, proto)) {
    if(!fgets(buffer, sizeof(buffer) - 1, m_pLogfile))
      return false;
    m_nAlertLines++;
    if(!parseAddresses(buffer, srcIP, m_nFromPort, dstIP, m_nToPort, proto)) {
      debug("Failed to parse IP, port, and protocol information: %s\n", buffer);
      return false;
    }
  }
  else { // If no ethernet line then protocol at the beginning of next line
    if(!fgets(buffer, sizeof(buffer) - 1, m_pLogfile))
      return false;
    m_nAlertLines++;
    if(sscanf(buffer, "%4s ", proto) != 1) {
      debug("Could not parse protocol from: %s\n", buffer);
      return false;
    }
      
  }

  if(!strcmp(proto, "UDP"))
    m_nProtocol = PROTO_UDP;
  else if (!strcmp(proto, "TCP"))
    m_nProtocol = PROTO_TCP;
  else if (!strcmp(proto, "ICMP"))
    m_nProtocol = PROTO_ICMP;
  else {
    debug("Could not understand protocol: %s\n", proto);
    return false;
  }

#if defined(_WIN32) || !defined(HAVE_INET_ATON)
  if ((in.s_addr = inet_addr(srcIP)) == INADDR_NONE)
#else
  if(inet_aton(srcIP, &in) == 0)
#endif
  {
    debug("Could not parse source IP address: %s\n", buffer);
    return false;
  }
  m_nFromIP = in.s_addr;

#if defined(_WIN32) || !defined(HAVE_INET_ATON)
  if ((in.s_addr = inet_addr(dstIP)) == INADDR_NONE)
#else
  if(inet_aton(dstIP, &in) == 0) 
#endif
  {
    debug("Could not parse destination IP address: %s\n", buffer);
    return false;
  }

  m_nToIP = in.s_addr;
  m_nCount = 1; // Always 1 for Snort

  m_sPortScan[0] = '\0';

  if(m_nDateTime > m_lastTime)
    m_lastTime = m_nDateTime;

  return true;
}
//
// CSnortLog::CloseLog()
//
// Purpose:
//   Close both log files.
//
void CSnortLog::CloseLog()
{
  if(m_pLogfile)
    fclose(m_pLogfile);
  m_pLogfile = NULL;

  if(m_pPortScanFile)
    fclose(m_pPortScanFile);
  m_pPortScanFile = NULL;

  return;
}

void CSnortLog::printStats()
{
  if(m_nPortScanLines) {
    printf("Read %d lines from portscan logfile and\n", m_nPortScanLines);
    printf("skipped %d lines that were older than saved timestamp and\n", m_nPortScanSkipped);
    printf("created %d events from portscan data.\n\n", m_nPortScanEvents);
  }
  
  printf("Read %d lines from alert logfile and\n", m_nAlertLines);
  printf("skipped %d events that were older than saved timestamp and\n", m_nAlertSkipped);
  printf("created %d events from alert data.\n\n", m_nAlertEvents);
}

