/* $Id: ArkMaterial.cpp,v 1.23 2003/03/19 09:25:05 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sstream>

#include <Ark/ArkModel.h>
#include <Ark/ArkTexture.h>
#include <Ark/ArkLexer.h>
#include <Ark/ArkCache.h>

namespace Ark
{
   // ========================================================================
   // Shaders
   // ========================================================================
   ShaderPass::ShaderPass (const TexturePtr& texture) : 
      m_Flags(0),
      m_Texture(texture),
      m_BlendColor (1.0f, 1.0f, 1.0f, 1.0f)
   {
   }

   ShaderPass::ShaderPass() :
      m_Flags(PASS_HAS_DEPTHWRITE | PASS_HAS_DEPTHFUNC),
      m_Texture(),
      m_DepthFunc(DEPTHFUNC_LEQUAL),
      m_DepthTest(true),
      m_DepthWrite(true),
      m_BlendColor(1.0f, 1.0f, 1.0f, 1.0f)
   {
   }

   ShaderPass::~ShaderPass ()
   {
   }

   BlendFunc GetBlendFuncFromString (const String &from)
   {
      if (from == "One")
	 return BLEND_ONE;
      if (from == "Zero")
	 return BLEND_ZERO;
      if (from == "DstColor")
	 return BLEND_DST_COLOR;
      if (from == "OneMinusDstColor")
	 return BLEND_ONE_MINUS_DST_COLOR;
      if (from == "SrcColor")
	 return BLEND_SRC_COLOR;
      if (from == "OneMinusSrcColor")
	 return BLEND_ONE_MINUS_SRC_COLOR;
      if (from == "DstAlpha")
	 return BLEND_DST_ALPHA;
      if (from == "OneMinusDstAlpha")
	 return BLEND_ONE_MINUS_DST_ALPHA;
      if (from == "SrcAlpha")
	 return BLEND_SRC_ALPHA;
      if (from == "OneMinusSrcAlpha")
	 return BLEND_ONE_MINUS_SRC_ALPHA;
      if (from == "ConstantAlpha")
	 return BLEND_CONSTANT_ALPHA;
      if (from == "OneMinusConstantAlpha")
	 return BLEND_ONE_MINUS_CONSTANT_ALPHA;

      return BLEND_UNDEFINED;
   }

   TextureEnv GetTexEnvFromString (const String &from)
   {
      if (from == "Modulate")
	 return TEXTUREENV_MODULATE;
      if (from == "Decal")
	 return TEXTUREENV_DECAL;
      if (from == "Replace")
	 return TEXTUREENV_REPLACE;

      return TEXTUREENV_MODULATE;
   }

   bool GetBoolFromString (const String &from)
   {
      if (from == "True" || from == "1")
	 return true;
      else if (from == "False" || from == "0")
	 return false;

      return true;
   }

   DepthFunc GetDepthFuncFromString (const String &from)
   {
      if (from == "Equal")
	 return DEPTHFUNC_EQUAL;
      else if (from == "LEqual")
	 return DEPTHFUNC_LEQUAL;
      else if (from == "GEqual")
	 return DEPTHFUNC_GEQUAL;
      else if (from == "Always")
	 return DEPTHFUNC_ALWAYS;

      return DEPTHFUNC_UNDEFINED;
   }

   MapRepeatMode GetTextureRepeatModeFromString (const String &from)
   {
      if (from == "Repeat")
	 return TEXTURE_REPEAT_REPEAT;
      else if (from == "Clamp")
	 return TEXTURE_REPEAT_CLAMP;

      return TEXTURE_REPEAT_UNKNOWN;
   }

   bool 
   ShaderPass::Parse (Lexer &lexer, Cache &cache)
   {
      String token;
      lexer.CheckToken ("{");

      do 
      {
	 token = lexer.GetToken();

	 if (token == "}")
	    break;
	 else if (token == "TextureEnv")
	 {
	    lexer.CheckToken("=");
	    m_TextureEnv = GetTexEnvFromString (lexer.GetToken());
	    lexer.CheckToken (";");

	    if (m_TextureEnv != TEXTUREENV_UNDEFINED)
	       m_Flags |= PASS_HAS_TEXTUREENV;
	 }
	 else if (token == "TextureGen")
	 {
	    lexer.CheckToken("=");
	    String value = lexer.GetToken (Ark::Lexer::IDENTIFIER);

	    if (value == "Environment")
	    {
	       m_Flags |= PASS_HAS_TEXTUREGEN;
	       m_TextureGen = TEXTUREGEN_ENVIRONMENT;
	    }
	    else if (value == "Linear")
	    {
	       m_Flags |= PASS_HAS_TEXTUREGEN;
	       m_TextureGen =  TEXTUREGEN_LINEAR;

	       lexer.CheckToken("(");
	       lexer.ReadScalarVector
		  (m_TextureGenData.m_Linear.m_PlaneS, 4);
	       lexer.CheckToken(",");
	       lexer.ReadScalarVector
		  (m_TextureGenData.m_Linear.m_PlaneT, 4);
	       lexer.CheckToken(")");
	    }
	    lexer.CheckToken (";");
	 }
	 else if (token == "Blending")
	 {
	    lexer.CheckToken ("=");
	    lexer.CheckToken ("(");
	    m_BlendSrc = GetBlendFuncFromString (lexer.GetToken());
	    lexer.CheckToken (",");
	    m_BlendDst = GetBlendFuncFromString (lexer.GetToken());
	    lexer.CheckToken (")");
	    lexer.CheckToken (";");

	    if (m_BlendSrc != BLEND_UNDEFINED 
		&& m_BlendDst != BLEND_UNDEFINED)
	       m_Flags |= PASS_HAS_BLENDING;
	 }
	 else if (token == "AlphaTest")
	 {
	    lexer.CheckToken ("=");
	    lexer.CheckToken ("(");
	    m_AlphaFunc = GetDepthFuncFromString (lexer.GetToken());
	    lexer.CheckToken (",");
	    lexer.ReadScalar(&m_AlphaValue);
	    lexer.CheckToken (")");
	    lexer.CheckToken (";");

	    if (m_AlphaFunc != DEPTHFUNC_UNDEFINED)
		{
	       m_Flags |= PASS_HAS_ALPHATEST;
		}
	 }
	 else if (token == "BlendColor")
	 {
	    lexer.CheckToken ("=");
	    lexer.ReadScalarVector (&m_BlendColor.R, 4);
	    lexer.CheckToken(";");
	 }
	 else if (token == "DepthTest")
	 {
	    lexer.CheckToken ("=");
	    m_DepthTest = GetBoolFromString (lexer.GetToken());
	    lexer.CheckToken (";");

	    m_Flags |= PASS_HAS_DEPTHFUNC;
	 }
	 else if (token == "DepthFunc")
	 {
	    lexer.CheckToken ("=");
	    m_DepthFunc = GetDepthFuncFromString (lexer.GetToken());
	    lexer.CheckToken (";");

	    if (m_DepthFunc != DEPTHFUNC_UNDEFINED)
	       m_Flags |= PASS_HAS_DEPTHFUNC;
	 }
	 else if (token == "RepeatMode")
	 {
	    lexer.CheckToken ("=");
	    m_RepeatMode = GetTextureRepeatModeFromString(lexer.GetToken());
	    lexer.CheckToken (";");
	 }
	 else if (token == "DepthWrite")
	 {
	    lexer.CheckToken ("=");
	    m_DepthWrite = GetBoolFromString (lexer.GetToken());
	    lexer.CheckToken (";");

	    m_Flags |= PASS_HAS_DEPTHWRITE;
	 }
	 else if (token == "Map")
	 {
	    lexer.CheckToken ("=");
	    String tmap = lexer.GetToken();

	    // Animated texture...
	    if (tmap == "Anim")
	    {
	       lexer.CheckToken("(");
	       lexer.ReadInteger (&m_TexPerSec);
	       lexer.CheckToken(",");

	       String tok;
	       do
	       {
		  tok = lexer.GetToken(Lexer::STRING);

		  if (tok == "")
		     break;

		  TexturePtr texture;
		  if (cache.Get (V_TEXTURE, UnquoteString(tok), texture))
		  {
		      texture->Configure();
		      m_AnimTex.push_back(texture);
		  }

		  tok = lexer.GetToken(Lexer::SYMBOL);

		  if (tok == ")" || tok != ",")
		     break;
	       } while(1);

	       m_Flags |= PASS_HAS_TEXTUREANIM;
	    }   
	    else
	    {
	       tmap = UnquoteString(tmap);
	       cache.Get(V_TEXTURE, tmap, m_Texture);
	    }

	    lexer.CheckToken (";");

	    m_Flags |= PASS_HAS_TEXTURE;
	 }

      } while (1);

      if (token != "}")
      {
	  std::ostringstream os;
	  os << "Unexpected token '" << token << "'.";
	  lexer.Error( os.str() );
	 return false;
      }

      // Now the pass is configured, setup the texture, if any
      if (m_Texture)
      {
	  switch(m_RepeatMode)
	  {
	      case TEXTURE_REPEAT_REPEAT:
		  m_Texture->m_RepeatMode = Image::REPEAT;
		  break;
	      case TEXTURE_REPEAT_CLAMP:
		  m_Texture->m_RepeatMode = Image::CLAMP;
		  break;
	      default:
		  // nothing to do, keep default
		  break;
	  }

	  // Updload texture to renderer
	  m_Texture->Configure();
      }
	  
      return true;
   }


   // ========================================================================
   // Material
   // ========================================================================

   Material::Material (const String &name) :
      Object (name)
   {
      m_Type = V_MATERIAL;
      m_Flags = MATERIAL_IS_WALKABLE;
   }

   Material::~Material()
   {}

   bool
   Material::Parse (const String& name, Stream &stream, Cache &cache)
   {
      Lexer lexer (name, stream);
      return Parse (name, lexer, cache);
   }

   bool
   Material::Parse (const String& name, Lexer &lexer, Cache &cache)
   {
      int npass = 0;
      String token;

      m_DecorName = UnquoteString(lexer.GetToken (Lexer::STRING));
      lexer.CheckToken ("{");

      do 
      {
	 token = lexer.GetToken();

	 if (token == "}")
	    break;
	 else if (token == "Pass")
	 {
	    m_Flags |= MATERIAL_HAS_PASS1 << npass;
	    m_Passes[npass++].Parse (lexer, cache);
	 }
	 else if (token == "Ambient")
	 {
	    lexer.CheckToken ("=");
	    lexer.ReadScalarVector (&m_Ambient.R, 4);
	    lexer.CheckToken(";");

	    m_Flags |= MATERIAL_HAS_LIGHTING;
	 }
	 else if (token == "Diffuse")
	 {
	    lexer.CheckToken ("=");
	    lexer.ReadScalarVector (&m_Diffuse.R, 4);
	    lexer.CheckToken(";");

	    m_Flags |= MATERIAL_HAS_LIGHTING;
	 }
	 else if (token == "Specular")
	 {
	    lexer.CheckToken ("=");
	    lexer.ReadScalarVector (&m_Specular.R, 4);
	    lexer.CheckToken(";");

	    m_Flags |= MATERIAL_HAS_LIGHTING;
	 }
	 else if (token == "Walkable")
	 {
	    lexer.CheckToken ("=");
	    String walkable = lexer.GetToken ();
	    lexer.CheckToken(";");

	    if (GetBoolFromString (walkable))
	       m_Flags |= MATERIAL_IS_WALKABLE;
	    else
	       m_Flags &= ~MATERIAL_IS_WALKABLE;
	 }
	 else if (token == "DoubleSided")
	 {
	    lexer.CheckToken ("=");
	    String dsided = lexer.GetToken ();
	    lexer.CheckToken(";");

	    if (GetBoolFromString (dsided))
	       m_Flags |= MATERIAL_IS_DOUBLESIDED;
	    else
	       m_Flags &= ~MATERIAL_IS_DOUBLESIDED;
	 }

      } while (1);

      if (token != "}")
      {
	  std::ostringstream os;
	  os << "Unexpected token '" << token << "'.";
	  lexer.Error( os.str() );
      }
   
      return true;
      
   }

   /// Return the number of passes.
   int
   Material::NumPasses() const
   {
      if (m_Flags & MATERIAL_HAS_PASS4) return 4;
      if (m_Flags & MATERIAL_HAS_PASS3) return 3;
      if (m_Flags & MATERIAL_HAS_PASS2) return 2;
      if (m_Flags & MATERIAL_HAS_PASS1) return 1;
      return 0;
   }

   /// Returns a string describing the model
   String
   Material::Dump (bool long_version)
   {
       std::ostringstream os;
       os << "[mat] " << Name();

       if (long_version)
       {
	   os << "\n\tnum_passes : " << NumPasses();
	   os << "\n\thas_lighting : " << ((m_Flags & MATERIAL_HAS_LIGHTING) ? "true" : "false");
	   os << "\n\tis_walkable : " << ((m_Flags & MATERIAL_IS_WALKABLE) ? "true" : "false");
       }

       return os.str();
   }

}
