/* $Id: ArkModelRead.cpp,v 1.34 2003/03/20 17:23:25 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <Ark/ArkModel.h>
#include <Ark/ArkModelBuild.h>
#include <Ark/ArkLoader.h>
#include <Ark/ArkLexer.h>
#include <Ark/ArkCache.h>

namespace Ark
{
   class LoaderArkModel : public Loader
   {
      public:
	 LoaderArkModel()
	 {}

	 bool CanLoad (ObjectType type, Stream &file, const String &name,
		       const String &args)
	 {
	    if (type != V_MODEL)
	       return false;

	    String sig;
	    file >> sig;
	    
	    if (sig != "ArkModel")
	       return false;

	    return true;
	 }

	 bool Load (Object *vis, Stream &file, const String &name,
		    const String &args, Cache *cache = NULL,
		    Progress *progress = NULL, int granularity = 0)
	 {
	    if (vis == NULL || vis->Type() != V_MODEL)
	       return false;

	    Model *mdl = (Model*) vis;

	    bool r = mdl->Read (name, file, cache);

	    scalar height;	    
	    if (StringToScalar(args, height))
	    {
	       if (height == 0.0)
		  return true;

	       mdl->SetRealHeight (height);
	    }

	    return r;
	 }

	 String GetInformations()
	 {
	    return "Ark Model";
	 }
	 
   }; // class LoaderArkModel

   void ark_AddArkMDLLoader (Loaders *loaders)
   {
      loaders->Add (new LoaderArkModel());
   }

///////////////////////////////////////////////////////////

   int Skeleton::OrderChildren(int cur, int idx)
   {
      m_BoneOrder[cur++] = idx;
      
      for (size_t i = 0; i < m_Bones.size(); i++)
      {
	 if (m_Bones[i].m_Parent == idx)
	    cur = OrderChildren(cur, i);
      }
      
      return cur;
   }

   void
   Skeleton::ApplyReorder()
   {
      std::vector<Bone> bonecopies = m_Bones;
      std::vector<int> invOrder;
      size_t i;
     
      invOrder.resize(m_BoneOrder.size());
      for (i = 0; i < m_BoneOrder.size(); ++i)
	 invOrder[m_BoneOrder[i]] = i;

      for (i = 0; i < m_Bones.size(); ++i)
      {
	 m_Bones[i] = bonecopies[m_BoneOrder[i]];
	 if (m_Bones[i].m_Parent>=0)
	    m_Bones[i].m_Parent = invOrder[m_Bones[i].m_Parent];
      }
   }

   void
   Skeleton::ComputeBoneOrder()
   {
      std::map<String,int> namemap;
      size_t i,j;

      for (i = 0; i < m_Bones.size(); i++)
      {
	 if (namemap.find(m_Bones[i].m_Name) != namemap.end())
	    Sys()->Error("'%s': Two bones with the same name (%s)",
			 m_Name.c_str(), m_Bones[i].m_Name.c_str());

	 namemap[m_Bones[i].m_Name] = i;
      }

      std::vector<size_t> namereorder;

      namereorder.resize(m_Bones.size());
      j = 0;
      for (std::map<String,int>::iterator nameIterator = namemap.begin();
	   nameIterator != namemap.end(); ++nameIterator, ++j)
	   {
		namereorder[j] = nameIterator->second;
	   }

      // Reorder bones according to "name" order.
      std::vector<Bone> bonecopies = m_Bones;
      m_BoneOrder = namereorder;
      ApplyReorder();

      m_BoneOrder.resize(m_Bones.size());
      for (i = 0; i < m_Bones.size(); ++i) m_BoneOrder[i] = i;

      size_t cur = 0;
      for (i = 0; i < m_Bones.size(); ++i)
      {
	 if (m_Bones[i].m_Parent != -1) 
	    continue;
	 
	 cur = OrderChildren(cur, i);
      }
     
      // Put bones in the initial order
      m_Bones = bonecopies;

      // Final reorder = name reorder (bone reorder (i))
      std::vector<size_t> copyorder = m_BoneOrder;
      for(i = 0; i < m_Bones.size(); i++)
	 m_BoneOrder[i] = namereorder[copyorder[i]];
   }

   /** Fill the skeleton with data read from a native file. */
   void
   Skeleton::Read (Lexer &lexer)
   {
      lexer.CheckToken("Skeleton");
      int nbones, i, j;
      StringToInt (lexer.GetToken (Lexer::CONSTANT), nbones);

      lexer.CheckToken ("{");
      
      std::vector<String> links;

      links.resize(nbones);
      m_Bones.resize(nbones);

      for (i = 0; i < nbones; i++)
      {
	 Bone &b = m_Bones[i];
	 b.m_Name = UnquoteString (lexer.GetToken (Lexer::STRING));

	 StringToScalar (lexer.GetToken (Lexer::CONSTANT), b.m_DefPosition.X);
	 StringToScalar (lexer.GetToken (Lexer::CONSTANT), b.m_DefPosition.Y);
	 StringToScalar (lexer.GetToken (Lexer::CONSTANT), b.m_DefPosition.Z);

	 StringToScalar (lexer.GetToken (Lexer::CONSTANT), b.m_DefRotation.X);
	 StringToScalar (lexer.GetToken (Lexer::CONSTANT), b.m_DefRotation.Y);
	 StringToScalar (lexer.GetToken (Lexer::CONSTANT), b.m_DefRotation.Z);

	 for (j = 0; j < 6; ++j)
	    b.m_BoneControllers[j] = -1;

	 links[i] = UnquoteString (lexer.GetToken (Lexer::STRING));
      }


      lexer.CheckToken("SeqList");
      lexer.CheckToken("=");
      m_SeqList = UnquoteString(lexer.GetToken(Lexer::STRING));
      lexer.CheckToken(";");
      
      if (lexer.GetToken() == "FixXRot")
      {
	 Matrix44 m;
	 lexer.CheckToken(";");
	 m_RootMatrix.MakeRotationX(-90.0f);
	 m.MakeRotationY(90.0f);
	 m_RootMatrix.Multiply(m);
      }
      else lexer.UngetToken(); 

      lexer.CheckToken ("}");

      for (i = 0; i < nbones; i++)
      {
	 Bone &b = m_Bones[i];
	 bool found = false;

	 for (j = 0; j < nbones; j++)
	 {
	    if (m_Bones[j].m_Name == links[i])
	    {
	       found = true;
	       break;
	    }
	 }

	 if (found) b.m_Parent = j;
	 else b.m_Parent = -1;
      }

      ComputeBoneOrder();

      AutoReadStream as (m_SeqList);

      Lexer aslex (m_SeqList, as.Get());
      aslex.CheckToken("SequenceList");
      aslex.CheckToken("{");
      while(1)
      {
	 String str, val;
	 
	 str = aslex.GetToken();
	 if (str == "}" || str == "") break;

	 str = UnquoteString(str);
	 aslex.CheckToken("=");	 
	 val = UnquoteString(aslex.GetToken(Lexer::STRING));
	 aslex.CheckToken(";");
	 m_Sequences[str] = val;
      }
   }

   /** Fill this model with data read from a native file. */
   bool
   Model::Read (const String& name, Stream &file, Cache *cache)
   {
      Lexer lexer (name, file);
      String tok;

      lexer.CheckToken("ArkModel");

      //=============
      lexer.CheckToken("Materials");
      lexer.CheckToken("{");

      String directory = name;
      String::size_type slashpos = directory.rfind ('.');

      if (slashpos != String::npos)
	 directory.resize (slashpos);

      directory += '/';

      do
      {
	 String nmat = lexer.GetToken();

	 if (nmat == "}")
	    break;

	 nmat = directory + UnquoteString(nmat) + ".mat";

	 MaterialPtr mat;
	 cache->Get (V_MATERIAL, nmat, mat);

	 m_Skin.Insert (mat);
      } while (!file.eof());


      //=============
      size_t nsubmodels;

      lexer.CheckToken("SubModels");
      lexer.ReadInteger((int*) &nsubmodels);
      lexer.CheckToken("{");

      m_SubModels.resize (nsubmodels);
      bool hasskeleton = false;

      size_t subm = 0;
      while (subm < nsubmodels)
      {
	 SubModel &sm = m_SubModels[subm++];

	 bool bone = false, uv0 = false, color = false;
	 size_t nvertex = 0;
	 int vbflags = VertexBuffer::VB_HAS_COORD |
	    VertexBuffer::VB_HAS_NORMAL;

	 lexer.CheckToken("SubModel");
	 lexer.CheckToken("{");
	 lexer.CheckToken("VertexBuffer");
	 lexer.ReadInteger((int*) &nvertex);

	 if (nvertex == 0)
	    return false;
	 
	 while ((tok = lexer.GetToken()) != "{"
		&& tok != "")
	 {
	    if (tok == "bone") bone = hasskeleton = true;
	    if (tok == "uv0")
	    {
	       uv0 = true;
	       vbflags |= VertexBuffer::VB_HAS_UV0;
	    }

	    if (tok == "color")
	    {
	       color = true;
	       vbflags |= VertexBuffer::VB_HAS_COLOR;
	    }
	 }

	 sm.m_VB.SetFormat (VertexBuffer::VertexFormat(vbflags));
	 sm.m_VB.Resize (nvertex);

	 if (bone)
	    sm.m_BoneBindings.resize (nvertex);

	 for (size_t i = 0; i < nvertex; i++)
	 {
	    Vector3 &coord = sm.m_VB.Coord (i);
	    Vector3 &normal = sm.m_VB.Normal (i);

	    lexer.ReadScalar (&coord.X);
	    lexer.ReadScalar (&coord.Y);
	    lexer.ReadScalar (&coord.Z);
	    sm.m_BBox.AddPoint (coord);

	    lexer.ReadScalar (&normal.X);
	    lexer.ReadScalar (&normal.Y);
	    lexer.ReadScalar (&normal.Z);

	    if (uv0)
	    {
	       Vector2 &uv = sm.m_VB.UV0(i);
	       
	       lexer.ReadScalar (&uv.X);
	       lexer.ReadScalar (&uv.Y);
	    } 

	    if (color)
	    {
	       RGBA &cl = sm.m_VB.Color4(i);
	       int r,g,b,a;
	       
	       lexer.ReadInteger (&r); cl.R = r;
	       lexer.ReadInteger (&g); cl.G = g;
	       lexer.ReadInteger (&b); cl.B = b;
	       lexer.ReadInteger (&a); cl.A = a;
	    } 
	 }
	 lexer.CheckToken("}");

	 if (bone)
	 {
	    lexer.CheckToken("BoneBindings");
	    lexer.CheckToken("{");
	    for (size_t i = 0; i < nvertex; i++)
	    {
	       int b;
	       lexer.ReadInteger (&b);
	       sm.m_BoneBindings[i] = (uchar)b;
	       
	    }
	    lexer.CheckToken("}");
	 }
	    
	 while (lexer.GetToken() == "Mesh")
	 {
	    sm.m_Meshes.resize (sm.m_Meshes.size()+1);
	    Mesh &m = sm.m_Meshes.back();

	    lexer.ReadInteger (&m.m_Material);

	    lexer.CheckToken("{");
	    while (lexer.GetToken() == "Triangle")
	    {
	       PrimitiveBlock pb;
	       int ntri;

	       String type = lexer.GetToken(Lexer::IDENTIFIER);
	       lexer.ReadInteger (&ntri);

	       if (type == "block") pb.SetType(PRIM_TRIANGLES);
	       if (type == "fan")   pb.SetType(PRIM_TRIANGLE_FAN);
	       if (type == "strip") pb.SetType(PRIM_TRIANGLE_STRIP);

	       // Allocate space and read triangle data.
	       lexer.CheckToken("{");
	       int current = 0;
	       pb.Resize (ntri);
	       while (ntri--)
	       {
		   int toBeRead;
		   lexer.ReadInteger( &toBeRead );
		   pb[current++] = toBeRead;
	       }
	       lexer.CheckToken("}");

	       // adds block to list
	       m.m_Blocks.push_back( pb );
	    }

	    lexer.UngetToken();
	    lexer.CheckToken("}");

	 } // Mesh

	 lexer.UngetToken();
	 lexer.CheckToken("}");
      } // SubModel

      lexer.CheckToken("}");

      SetupBBox(false);

      if (hasskeleton)
      {
	 m_Skeleton = new Skeleton (Name());
	 m_Skeleton->Read (lexer);

	 // Bone reordering.
	 std::vector<int> invOrder;
	 size_t i;
     
	 invOrder.resize(m_Skeleton->m_BoneOrder.size());
	 for (i = 0; i < m_Skeleton->m_BoneOrder.size(); ++i)
	    invOrder[m_Skeleton->m_BoneOrder[i]] = i;

	 for(SubModelLI inverseSubmodel = m_SubModels.begin();
	     inverseSubmodel != m_SubModels.end(); ++inverseSubmodel)
	 {
	    for (SubModel::BoneBindingLI bb = inverseSubmodel->m_BoneBindings.begin();
		 bb != inverseSubmodel->m_BoneBindings.end(); ++bb)
	       (*bb) = (uchar) invOrder[*bb];
	 }
	 m_Skeleton->ApplyReorder();

	 // Model transformation (put coords in bone space).
	 Transform (m_Skeleton->m_RootMatrix, false);

	 ModelState mstate (this);

	 Matrix44 *mats = new Matrix44[m_Skeleton->m_Bones.size()];

	 m_Skeleton->SetupBones (mstate, mats);

	 for (i = 0; i < m_Skeleton->m_Bones.size(); i++)
	    mats[i].Invert();

	 for (SubModelLI it = m_SubModels.begin();
	      it != m_SubModels.end(); ++it)
	 {
	    if (it->m_BoneBindings.empty()) continue;

	    for (size_t i = 0; i < it->m_VB.Size(); i++)
	    {
	       it->m_VB.Coord(i) = mats[it->m_BoneBindings[i]].
		  Transform(it->m_VB.Coord(i));
	    }
	 }	 

	 // BBox might have been modified.
	 SetupBBox(false);

	 //FIXME: this shouldn't be necessary
	 ((Object*)m_Skeleton)->PostLoad(cache);
      }

      tok = lexer.GetToken ();
      if (tok == "Optimized")
	 m_Optimized = true;
      else
      {
	 Sys()->Log ("optimizing: %s\n", Name().c_str());
	 Optimize();
	 lexer.UngetToken();
      }

      tok = lexer.GetToken();
      if (tok == "SharedCDModel")
      {
	 lexer.CheckToken("=");
	 tok = lexer.GetToken(Lexer::STRING);
	 lexer.CheckToken(";");

	 m_SharedCDModelName = UnquoteString(tok);
      }

      return true;
   }
}
