/* partable.c: utilities for scanning the in-core partition table
 *
 * Copyright (C) 1995-97 Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *               1996-97 Michael Schlueter <schlue00@marvin.informatik.uni-dortmund.de>
 *
 * This program is free software.  You can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation: either version 2 or
 * (at your option) any later version.
 *
 */

/* $Id: partable.c,v 1.8 1997/08/22 15:37:17 rnhodek Exp $
 *
 * $Log: partable.c,v $
 * Revision 1.8  1997/08/22 15:37:17  rnhodek
 * Implemented basic support for ICD format. Should work, but conversion
 * AHDI<->ICD isn't very clever yet.
 *
 * Revision 1.7  1997/07/10 14:46:50  rnhodek
 * Fixed bug if no free sectors before possible aux. rs
 *
 * Revision 1.6  1997/07/08 18:59:05  rnhodek
 * Ouch, too many changes without commits in between...
 * Implemented moving of partitions
 * Implemented checking of "extended start condition" (great changes in
 *   new_partition())
 * Some more tests in verify()
 *
 * Revision 1.5  1997/06/22 10:30:56  rnhodek
 * Add __attribute__((unused)) to cvid
 *
 * Revision 1.4  1997/06/21 20:47:47  rnhodek
 * Added RCS keywords
 *
 * Revision 1.3  1997/06/13 12:51:22  rnhodek
 * Bug fixes and improvements to new_partition; little fixes in verify
 * 
 * Revision 1.2  1997/06/12 13:59:34  rnhodek
 * Fix comparing unsigned var < 0 bug
 * 
 * Revision 1.1  1997/06/11 14:36:36  rnhodek
 * Initial revision
 * 
 * Revision 1.1.1.1  1997/06/11 14:36:36  rnhodek
 * Started using CVS for atafdisk
 *
 */

#ifndef lint
static char vcid[] __attribute__ ((unused)) =
"$Id: partable.c,v 1.8 1997/08/22 15:37:17 rnhodek Exp $";
#endif /* lint */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include "fdisk.h"
#include "partable.h"
#include "util.h"


/* sector inside partition? */
int inside_part( PARTITION *p, unsigned long sec )
{
    return( p->start <= sec && sec < p->start + p->size );
}

/* return number of primary partitions */
int n_primary( void )
{
    int i, n = 0;
    
    for( i = 0; i < partitions; ++i )
	if (!IS_EXTENDED(part_table[i]))
	    ++n;
    return( n );
}

/* in which primary slot the XGM will go? */
int XGM_slot( void )
{
    int i;
    
    for( i = 0; i < partitions && !IS_EXTENDED(part_table[i]); ++i )
	;
    return( i );
}

/* determine boundaries of XGM partition */
void XGM_boundary( unsigned long *start, unsigned long *size )
{
    unsigned long sta, end;
    int i;
    PARTITION *p;

    sta = hd_size;
    end = 0;
    for( i = 0; i < partitions; ++i ) {
	p = &part_table[i];
	if (IS_EXTENDED(*p)) {
	    sta = min( sta, min( p->start, p->rootsec ));
	    end = max( end, p->start + p->size );
	}
    }

    *start = sta;
    *size  = end - sta;
}

void table_infos( int *n_prim, int *n_ext, int *first_ext, int *last_ext )
{
    int np = 0, ne = 0, fe = -1, le = -1;
    int i;
    
    for( i = 0; i < partitions; ++i ) {
	if (IS_EXTENDED(part_table[i])) {
	    ++ne;
	    if (fe < 0)
		fe = i;
	    le = i;
	}
	else
	    ++np;
    }

    if (n_prim) *n_prim = np;
    if (n_ext) *n_ext = ne;
    if (first_ext) *first_ext = fe;
    if (last_ext) *last_ext = le;
}

/* return canonical number for a partition (by disk location order) */
int canonical_number( unsigned long start )
{
    int i, new_num = 0;
    
    for( i = 0; i < partitions; ++i ) {
	if (part_table[i].start < start)
	    ++new_num;
    }
    return( new_num );
}

/* Can a new partition NEW_NUM be a primary partition given the rest of
 * layout? If not, return reason string in WHY if not NULL. */
int primary_possible( int new_num, int n_prim, int n_ext, int first_ext,
		      int last_ext, char **why )
{
    if (n_prim + (n_ext > 0) >= MAX_PRIMARY) {
	if (why)
	    *why = "no primary slots available";
	return( 0 );
    }
    if (n_ext > 0 &&
	new_num >= first_ext + (new_num <= first_ext) &&
	new_num <= last_ext  + (new_num <= last_ext )) {
	if (why)
	    *why = "would break XGM chain into two";
	return( 0 );
    }
    return( 1 );
}

/* Can a new partition NEW_NUM be an extended partition given the rest of
 * layout? If not, return reason string in WHY if not NULL. */
int extended_possible( int new_num, int n_prim, int n_ext, int first_ext,
		       int last_ext, unsigned long start, char **why )
{
    unsigned long min_ars, max_ars;

    if (xpart_fmt == xfmt_ICD) {
	if (why)
	    *why = "no extended partition with ICD format";
	return( 0 );
    }
    
    if (new_num == 0) {
	if (why)
	    *why = "first partition may not be extended";
	return( 0 );
    }
    if (n_ext == 0 && n_prim == 4) {
	if (why)
	    *why = "no primary slot for XGM available";
	return( 0 );
    }
    if (n_ext > 0 &&
	(new_num < first_ext + (new_num <= first_ext) - 1 ||
	 new_num > last_ext  + (new_num <= last_ext ) + 1)) {
	if (why)
	    *why = "not in or next to existing XGM area";
	return( 0 );
    }

    /* check if there is space for the aux. rootsector */

    if (n_ext == 0 || new_num <= first_ext) {
	/* this will then be the first extended part. => its aux. rootsector
	 * must be before the current first ars and before the part. data */
	min_ars = 1;
	max_ars = min( start, lowest_ars() - 1 );
    }
    else {
	/* this will be a non-first ext. part => its aux. rootsector must be
	 * after the first ars, and before the partition data */
	min_ars = lowest_ars() + 1;
	max_ars = start;
    }
    if (min_ars > max_ars) {
	if (why)
	    *why = "aux. rootsector can't be first XGM sector";
	return( 0 );
    }

    /* skip already-allocated sectors */
    min_ars += allocated_sectors_at( min_ars );
    max_ars -= allocated_sectors_before( max_ars+1 );
    /* max_ars is -1 or UINT_MAX if no free space */
    if (min_ars > max_ars || max_ars == UINT_MAX) {
	if (why)
	    *why = "no space for aux. rootsector";
	return( 0 );
    }
    
    return( 1 );
}

/* find lowest-numbered aux. root sector */
unsigned long lowest_ars( void )
{
    int i;
    unsigned long xbegin = hd_size;
    
    for( i = 0; i < partitions; ++i ) {
	if (IS_EXTENDED(part_table[i]))
	    xbegin = min( xbegin, part_table[i].rootsec );
    }
    return( xbegin );
}

/* Verify the condition that the aux rs of the first ext. part. must be the
 * start of the XGM area. PART/ROOTSEC describe an additional part.
 * entry:
 *   op==NONE : ignored
 *   op==ADD  : assume additional part with ars ROOTSEC inserted as PART
 *   op==DEL  : assume part. PART non-existant
 *   op=CHANGE: replace part. PART's rootsec with ROOTSEC
 */
int check_ext_start_condition( enum check_ext_cond_code code, int part,
			       unsigned long rootsec )
{
    unsigned long xbegin, pr, first_ars = 0;
    int i, first_ext = -1;

    if (xpart_fmt == xfmt_ICD)
	/* condition always ok if no extended part.s */
	return( 1 );
    
    /* determine begin of XGM area */
    xbegin = hd_size;
    for( i = 0; i < partitions; ++i ) {
	if (!IS_EXTENDED(part_table[i]))
	    continue;

	pr = part_table[i].rootsec;
	if (code == DEL && i == part)
	    continue;
	if (code == CHANGE && i == part)
	    pr = rootsec;
	xbegin = min( xbegin, pr );

	if (!first_ars) {
	    first_ars = pr;
	    first_ext = i;
	}
    }

    if (code == ADD && rootsec) {
	xbegin = min( xbegin, rootsec );
	if (!first_ars || part <= first_ext)
	    first_ars = rootsec;
    }

    return( !first_ars || first_ars == xbegin );
}

/* return number of unallocated sectors starting at 'sec' */
int free_sectors_at( unsigned long sec )
{
    int i;
    unsigned long n, start, end, rs;

    /* assume everything free to end of disk... */
    n = hd_size - sec;

    /* scan partitions and cut free area if necessary */
    for( i = 0; i < partitions; ++i ) {
	start = part_table[i].start;
	end   = start + part_table[i].size;
	if (end > start && start < sec+n && end > sec) {
	    if (start <= sec)
		/* overlapping 'sec' -> 'sec' not free at all */
		return( 0 );
	    else
		n = start - sec;
	}
	/* consider aux rs if part. is extended */
	rs = part_table[i].rootsec;
	if (rs) {
	    if (rs == sec)
		return( 0 );
	    else if (rs > sec && rs < sec+n)
		n = rs - sec;
	}
    }

    /* also compare with bad sector list */
    if (bsl_size && bsl_start < sec+n && bsl_start+bsl_size > sec) {
	if (bsl_start <= sec)
	    return( 0 );
	else
	    n = bsl_start - sec;
    }
    
    return( n );
}

/* return number of unallocated sectors before (and excluding) 'sec' */
int free_sectors_before( unsigned long sec )
{
    int i;
    unsigned long s, start, end,  rs;

    /* assume sectors [1..sec-1] free */
    s = 1;

    /* scan partitions and cut free area if necessary */
    for( i = 0; i < partitions; ++i ) {
	start = part_table[i].start;
	end   = start + part_table[i].size;
	if (end > start && start < sec && end > s) {
	    if (end >= sec)
		/* overlapping 'sec'-1 -> 'sec'-1 not free at all */
		return( 0 );
	    else
		s = end;
	}
	/* consider aux rs if part. is extended */
	rs = part_table[i].rootsec;
	if (rs) {
	    if (rs == sec-1)
		return( 0 );
	    else if (rs < sec && rs >= s)
		s = rs + 1;
	}
    }

    /* also compare with bad sector list */
    end = bsl_start + bsl_size;
    if (bsl_size && bsl_start < sec && end > s) {
	if (end >= sec)
	    return( 0 );
	else
	    s = end;
    }
    
    return( sec - s );
}

/* return number of allocated sectors starting at 'sec' */
int allocated_sectors_at( unsigned long sec )
{
    int i;
    unsigned long end;

    end = sec;
  rescan:
    for( i = 0; i < partitions; ++i ) {
	if (inside_part( &part_table[i], end )) {
	    /* current end is inside this partition -> new end of allocated
	     * region is end of this partition; need to rescan table from the
	     * beginning now, since part.s need not be in ascending order! */
	    end = part_table[i].start + part_table[i].size;
	    goto rescan;
	}
	if (end == part_table[i].rootsec) {
	    /* current end is aux rs for some extended partition */
	    ++end;
	    goto rescan;
	}
    }

    /* also bad sector list allocates sectors... */
    if (bsl_size && bsl_start <= end && end < bsl_start + bsl_size) {
	end = bsl_start + bsl_size;
	goto rescan;
    }
    
    return( end - sec );
}

/* return number of allocated sectors before 'sec' */
int allocated_sectors_before( unsigned long sec )
{
    int i;
    unsigned long beg;

    beg = sec;
  rescan:
    for( i = 0; i < partitions; ++i ) {
	if (inside_part( &part_table[i], beg-1 )) {
	    beg = part_table[i].start;
	    goto rescan;
	}
	if (beg-1 == part_table[i].rootsec) {
	    --beg;
	    goto rescan;
	}
    }

    /* also bad sector list allocates sectors... */
    if (bsl_size && bsl_start <= beg-1 && beg-1 < bsl_start + bsl_size) {
	beg = bsl_start;
	goto rescan;
    }
    
    return( sec - beg );
}

/* Local Variables: */
/* tab-width: 8     */
/* End:             */
