# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::tree::star_merge_undo;
use base 'AXP::Command::tree';

use Arch::Util qw(run_tla);
use Arch::TempFiles;

sub infoline {
	"undo star-merge command leaving the original changes"
}

sub optusage {
	"[star-merge args and options] version"
}

sub own_options {
	(
		archive   => { sh => 'A', type => "=s", arg => 'ARCHIVE', desc => "override tree archive" },
		reference => { sh => 'r', type => "=s", arg => 'VERSION', desc => "set reference version" },
		forward   => { sh => 'N', desc => "pass the --forward option to 'patch'" },
		three_way => { sh => 't', desc => "perform a 3-way (diff3-style) merge" },
	)
}

sub helptext {
	q{
		Undo changes done by the previous star-merge.  Note, it only
		makes sense to issue star-merge-undo with the same parameters
		as the actual last star-merge.  All other usages may produce
		unpredictable results.

		The --archive, --reference, --forward, --three-way and --dir
		options as well as the version are passed to star-merge.

		The potential errors in the intermediate process should be
		handled properly.
	}
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	my $version = shift @ARGV;
	die "No star-merge version given, exiting\n" unless $version;

	die "The --test option is not implemented yet\n" if $opt{test};
	$ENV{DEBUG} = 1 if $opt{verbose};

	my $tree = $self->tree;
	my $dir = $tree->root;

	my $tmp = Arch::TempFiles->new;
	my $tmp_dir = $tmp->dir_name;

	print "* undoing all changes\n"
		unless $opt{quiet};
	run_tla("undo", "-d", $dir);
	die "Can't undo in tree $dir: exit status $?\n" if $?;

	my @sm_opts = ();
	push @sm_opts, "--archive", $opt{archive} if $opt{archive};
	push @sm_opts, "--reference", $opt{reference} if $opt{reference};
	push @sm_opts, "--forward" if $opt{forward};
	push @sm_opts, "--three-way" if $opt{three_way};
	push @sm_opts, "--dir", $dir;

	print "* localizing star-merge changes\n"
		unless $opt{quiet};
	run_tla("star-merge", @sm_opts, "--changes", $tmp_dir, $version);
	my $sm_status = $?;

	print "* redoing all changes\n"
		unless $opt{quiet};
	run_tla("redo", "-d", $dir);
	die "Can't redo in tree $dir: exit status $?\n" if $?;

	die "Error while executed star-merge: exit status $sm_status\n" if $sm_status;

	print "* applying localized star-merge changes in reverse\n"
		unless $opt{quiet};
	run_tla("apply-changeset --reverse", $tmp_dir, $dir);
	die "Can't apply-changeset in tree $dir: exit status $?\n" if $?;
}

1;
