# Arch Perl library, Copyright (C) 2005 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::triggers::add;
use base 'AXP::Command::triggers';

use Arch::Util qw(run_cmd);
use Arch::Backend qw(arch_backend);

sub optusage {
	"[options] limit [slave_limit ...]"
}

sub infoline {
	"add a new trigger"
}

sub options {
	(
		email   => { sh => 'e', desc => "use EMAIL_ADDR in standard send-mail action", type => "=s", arg => 'EMAIL_ADDR' },
		plain   => { sh => 'p', desc => "use plain-log-style rather than changelog-style" },
		action  => { sh => 'a', desc => "set SHELL_CMD to be the action (experts only)", type => "=s", arg => 'SHELL_CMD' },
		edit    => { sh => 'E', desc => "invoke editor to tweak default action" },
		no_skip => { sh => 'S', desc => "do not skip all already existing revisions" },
		master  => { sh => 'm', desc => "link parameter limits to MASTER (limit or number)", type => "=s", arg => 'MASTER' },
		var     => { sh => 'v', desc => 'set user vars: -v LABEL=prj1 -v EMAIL=me@me.org', type => "=s%", arg => 'VAR=VALUE' },
		__PACKAGE__->quiet_option,
	)
}

sub helptext {
	my $prefix = $_[0]->{prefix};
	qq{
		Add a new trigger for LIMIT parameter.  LIMIT may be an arch
		archive, category, branch or version.

		If you use --edit option, this will open an editor where you can
		enter commands that will be executed whenever a new revision is
		detected in LIMIT.  Revisions that already exist by the time the
		trigger is created will not cause the commands to be executed,
		unless you add the --no-skip option.

		The default commands send changelog for every new revision to
		your local account or the one specified in --email option.

		You can change the commands and variables later using 'edit'
		command.  You may view all trigger data using 'list' command.

		With --master, do not create its own action for a new trigger,
		but share it with the action of already existing master trigger.
		So, 'add lim1 lim2' is just 'add lim1; add --master lim1 lim2'.

		Example:
			$prefix --email me\@me.org,list\@prj.org devel\@prg.org--2005
	}
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	$opt{var}->{SEND_PLAINLOG} = 1 if $opt{plain};

	my $master_trigger = undef;
	if ($opt{master}) {
		$master_trigger = $self->load_trigger($opt{master});
		goto PROCESS_SLAVES;
	}

	my $name = shift @ARGV;
	my $trigger = $self->create_trigger($name);

	$self->apply_vars($trigger, $opt{var});

	my $user = $ENV{USER} || $ENV{USERNAME} || getpwuid($>);
	my $email = $opt{email} || "$user\@localhost";

	my $sh_comment = q{
#!/bin/sh
#
# Run on every } . $name . q{ commit.
#
# The following special variables are available in this script:
#   $ARCH_ARCHIVE, $ARCH_REVISION (full non-archive revision),
#   $FULL_REVISION, $CATEGORY, $BRANCH, $VERSION, $REVISION,
#   $LOG_SUMMARY, $LOG_CREATOR, $LOG_EMAIL, $LOG_USERNAME,
#   $LOG_DATE, $LOG_KIND, $LOG_BODY, $LOG_TEXT, $LOG_FILE,
#   $AXP (} . $0 . q{), $ARCH_BACKEND (} . arch_backend() . q{).
#
# Other variables used here may be defined using a simple interface:
#   axp triggers edit --var EMAIL=my@list.org --var SEND_PATCH=1 limit
#   axp triggers list --show-vars

};

	my $sh_action = $opt{action} || q%
# send ChangeLog formatted output unless directed otherwise
if [ "$SEND_PLAINLOG $SEND_CHANGES $SEND_PATCH" = "  " ]; then
	SEND_CHANGELOG=1
fi

# use defaults for some variables if not set yet;
# one quick way to disable mail sending is to comment out the next line
: ${MAIL_S=mail -s}
: ${MAIL_S=true}
: ${EMAIL=% . $email . q%}

# actually send the mail with one or another output in the body
(
	[ "$SEND_CHANGELOG" ] && $AXP changelog --use-log $LOG_FILE
	[ "$SEND_PLAINLOG" ]  && cat $LOG_FILE
	[ "$SEND_CHANGES" ]   && $AXP cset show $FULL_REVISION
	[ "$SEND_PATCH" ]     && $AXP cset show --diffs $FULL_REVISION
	[ "$ARCHZOOM_URL" ]   && echo && echo -n $ARCHZOOM_URL
	[ "$ARCHZOOM_URL" ]   && echo "/$FULL_REVISION?log?expand"
) | $MAIL_S "[COMMIT $LOG_USERNAME $ARCH_REVISION] $LOG_SUMMARY" $EMAIL

# Ideas for the project specific actions that may be performed here:
# MAKE_TESTS, CHECK_DIST, UPDATE_RSS, UPDATE_DOC, UPDATE_BTS.

%;

	# force correct surrounding newlines
	$sh_comment =~ s/^\n?(.*)$/$1/s;
	$sh_action =~ s/^\n?(.*?)\n?$/$1\n/s;

	$trigger->{action} = $sh_comment . $sh_action;

	$self->edit_trigger($trigger)
		or die "Error or canceled by user; $name trigger is not added\n"
		if $opt{edit};

	$self->skip_revisions($trigger) unless $opt{no_skip};
	$self->save_trigger($trigger);
	run_cmd("chmod +x", $trigger->{action_file});

	print "* trigger for $trigger->{name} is added\n"
		unless $opt{quiet};

	$master_trigger = $trigger if @ARGV;

	PROCESS_SLAVES:
	if ($master_trigger) {
		foreach my $name (@ARGV) {
			my $trigger = $self->create_trigger($name);
			$trigger->{is_slave} = 1;
			$self->apply_vars($trigger, $opt{var});
			symlink($master_trigger->{action_file}, $trigger->{action_file});
			$self->skip_revisions($trigger) unless $opt{no_skip};
			$self->save_trigger($trigger);
			print "* slave trigger for $trigger->{name} is added\n"
				unless $opt{quiet};
		}
	}
}

1;
