package com.holub.asynch;


import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.LinkedList;

/**
 *	A generic server-side socket that efficiently executes client-related
 *  actions on their own threads. Use it like this:
 *	<pre>
 *	public class Action implements Socket_server.Client_action
 *	{	public void action( Socket socket )
 *		{	// perform any action that's needed to talk to
 *			// a single client throgh the "socket". This
 *			// method executes on its own thread.
 *		}
 *	};
 *
 *	Socket_server echo_server = new Socket_server (
 *                                      		port_number, 
 *							expected_connection_count,
 *							Action.class,
 *							new Socket_server.Death()
 *							{	public void action( Exception e )
 *								{ // peformed when server aborts or
 *								  // is killed
 *								}
 *							}
 *                                              	);
 *	echo_server.start();
 *	//...
 *	echo_server.kill();
 *  </pre>
 * The <code>Client_action</code> object encapsulates whatever action must
 * be performed when a client connects to the current server. The
 * <code>action()</code> method runs on its own thread, and is passed
 * the Socket that's connected to the client. <code>Client_action</code> objects
 * are manufactured by the <code>Socket_server</code>, and a minimal
 * number of objects are created and recycled as necessary. Consequently,
 * the <code>socket</code> argument should not be cached. Moreover,
 * the <code>Client_action</code> object should reinitialize itself
 * every time the <code>action()</code> method is called.
 * (This is <u>not</u> the same architecture as a Servelet.
 * One instance of the <code>Client_action</code> will exist for each
 * connection, so the object can maintain a unique state in local
 * fields if it wishes.
 * Once the action completes, however, the object might be used again
 * for another connection, so it should reinitialize itself at the
 * top of the <code>action</code> method.)
 *
 * <br><br>
 * <table border=1 cellspacing=0 cellpadding=5><tr><td><font size=-1><i>
 * <center>(c) 1999, Allen I. Holub.</center>
 * <p>
 * This code may not be distributed by yourself except in binary form,
 * incorporated into a java .class file. You may use this code freely
 * for personal purposes, but you may not incorporate it into any
 * commercial product without express permission of Allen I. Holub in
 * writing.
 * </td></tr></table>
 */

public class Socket_server extends Thread
{
	/**
	 * Implementations of this class service individual client connections.
	 * The <code>action()</code> method is called every time a client
	 * connects. There is a one-to-one mapping of client connections to
	 * <code>Client_action</code> objects, but a given
	 * <code>Client_action</code> object may be
	 * recycled to service many clients in sequence.
	 */
	public interface Client_action
	{	void	action( Socket client );
	}

	/**
	 *	The Socket_server.Death object is passed into the server to
	 *  specify a shut-down action. It's <code>action()</code> method
	 *  is called when the server thread shuts down. The argument is
	 *	null for a normal shutdown, otherwise it's the exception
	 *  object that caused the shutdown.
	 */
	public interface Death
	{	void	action( Exception e );
	}

	/**
	 * A wrapper that makes it a bit easier for the user to implement
	 * the client-connection operation. Makes all the mechanics of
	 * thread manipulation internal.
	 */
	private class Client implements Runnable
	{	private final Client_action action; // Clone of user-supplied action 
		public		  Socket 		socket;	// Changes with each use

		Client( Client_action action ){ this.action = action; }
		public void run()
		{	
			action.action(socket);
			synchronized( connections )
			{	connections.addLast( this );	// Put self back into queue to
			}									// be used again
		}
	}

	// All of the following should be final. A compiler bug (that's
	// in all compiler versions up to and including 1.2) won't permit it.

	private /*final*/ ServerSocket 	main;
	private /*final*/ Thread_pool 	pool;
	private /*final*/ Class 		action;
	private /*final*/ Death			shutdown;
	private	/*final*/ LinkedList 	connections = new LinkedList();

	/**
	 * Thrown by the <code>Socket_server</code> construtor if it can't
	 * be created for some reason.
	 */
	public class Not_created extends RuntimeException
	{	
        // Unique SerialVersionUID (should NEVER change)
        static final long serialVersionUID = 0L;
        
        public Not_created( Exception e )
		{	super("Couldn't create socket: " + e.getMessage() );
		}
	}

	/**
	 * Create a socket-sever thread that creates and maintains a server-side
	 * socket, dispatching client actions on individual threads when
	 * clients connect.
	 * @param port_number			The port number to use for the main socket
	 * @param expected_connections	The number of simulteneous connections
	 *								that you expect. More connections are
	 *								supported, but the threads that service
	 *								them may need to be created dynamically
	 *								when the client connects.
	 * @param action				The <code>Class</code> object that
	 *								represents a <code>Socket_server.Client_action</code>
	 *								derivitive. Objects are created dynamically
	 *								by the system to service clients.
	 * @param shutdown				Command object that encapsulates the
	 *								action to take when the server thread
	 *								shuts down (either by being passed
	 *								a {@link kill} message, or by some
	 *								internal error).
	 * @throws Not_created			If the object can't be created successfully
	 *								for one of various reasons (the socket
	 *								can't be opened, client actions can't
	 *								be manufactured, etc.).
	 */

	public Socket_server(int port_number, int expected_connections, 
                             Class action, Death shutdown) throws Not_created
	{
            try	{
                main		= new ServerSocket(port_number);
                pool		= new Thread_pool (expected_connections, 0); //zero as second parameter will make max threads unlimited.
                this.action 	= action;
                this.shutdown   = shutdown;

                System.out.println("set variables. about to create pool");

                for( int i = expected_connections; --i >= 0 ; ) 
                        connections.addLast( 
                                        new Client((Client_action)(action.newInstance())) );

                System.out.println("created pool.");

                setDaemon( true );		// Don't keep the process alive
	    }
	    catch( Exception e ){ throw new Not_created(e); }
	}

	/*******************************************************************
	 * Implements the "accept" loop, waits for clients to connect and
	 * when a client does connect, executes the <code>action()</code>
	 * method of a clone of the <code>Client</code>
	 * prototype passed into the constructor on its own thread.
	 * If the <code>accept()</code> fails catostrophically, the
	 * thread shuts down, and the shut_down object passed to the
	 * constructor is notified (with a null argument).
	 */

	public void run()
	{
            try
            {	
                Client client;
                while( !isInterrupted() )				//#accept_loop
                {
                    synchronized( connections )
                    {
                        //client = (connections.size() > 0)  ? (Client)(connections.removeLast()) : new Client((Client_action)(action.newInstance())) ;
                        if (connections.size() > 0) {
                            client = (Client)(connections.removeLast());
                            System.out.println("reusing client");
                        } else {
                            client = new Client((Client_action)(action.newInstance()));
                            System.out.println("making new client");
                        }
                     }
                        client.socket = main.accept();	
                        if( isInterrupted() )
                                break;
                        pool.execute( client );
                }
            }
            catch(Exception e)
            {	if( shutdown != null )
                            shutdown.action(e);
            }
            finally
            {	pool.close();
                    if( shutdown != null )
                            shutdown.action(null);
            }
	}

	/*******************************************************************
	 * Shut down the Socket_server thread gracefully. All associated
	 * threads are terminated (but those that are working on serving
	 * a client will remain active until the service is complete).
	 * The main socket is closed as well.
	 */

	public void kill()
	{	try
		{	pool.close();
			interrupt();
			main.close();
		}
		catch( IOException e ){ /*ignore*/ }
	}

	/*******************************************************************
	 * A small test class. Creates a socket server that implements an]
	 * echo server, then opens two connections to it and runs
	 * a string through each connection.
	 */

	static public class Test
	{
		public static void main( String[] args ) throws Exception
		{
			class Action implements Socket_server.Client_action
			{	public void action( Socket socket )
				{	try
					{
						BufferedReader in =
							new BufferedReader(
								new InputStreamReader(socket.getInputStream(), "UTF-8"));
						OutputStreamWriter out =
								new OutputStreamWriter(socket.getOutputStream(), "UTF-8");

						String line;
						while( (line = in.readLine()) != null )
						{	out.write( line, 0, line.length() );
							out.write( "\n", 0, 1 			  );
							out.flush();
						}
						socket.close();
					}
					catch( Exception e )
					{	System.out.println(e.toString());
					}
				}
			};

			Socket_server echo_server = new Socket_server( 9999, 10, Action.class, 
										new Socket_server.Death()
										{	public void action( Exception e )
											{ System.out.println("goodbye world (" + e + ")");
											}
										}
									);
			echo_server.start();
			connect("Hello\n");
			connect("World\n");
			echo_server.kill();
		}

		private static void connect( String s ) throws Exception
		{	
			Socket 			client  = new Socket( "localhost", 9999 );
			BufferedReader	in		= new BufferedReader(
										new InputStreamReader(
											client.getInputStream(), "UTF-8" ));
			OutputStreamWriter out	= new OutputStreamWriter(
											client.getOutputStream(), "UTF-8");

			out.write( s, 0, s.length() );
			out.flush();

			s = in.readLine();
			System.out.println( s );

			client.close();
		}
	}
}