<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    db-oracle.php                                           */
/* Author:      Paul Waite, Mark Kessell                                */
/* Description: Definitions for ORACLE 8i database access. This was     */
/*              written for the OCI8 interface for Php4.                */
/*                                                                      */
/* ******************************************************************** */
/** @package database */

/**
* ORACLE 8 database interface
* This is a database interface class. It is an impedance-matcher
* between the high-level Axyl functions for accessing data, and
* the specific functions suplpied by Php to access a particular
* flavour of databse such as Postgres, MS-SQL Server, Sybase etc.
* @package database
* @access private
*/
class db_oracle extends database {
  var $result;
  var $numrows = 0;
  // ....................................................................
  /** Constructor */
  function db_oracle($name="", $user="", $passwd="", $host="", $port=1521) {
    $this->database($name, $user, $passwd, $host, $port);
    $this->type = "oracle";
  }
  // ....................................................................
  /**
  * Connect to the database.
  * @param boolean $persistent Whether to connect persistently or not
  * @return boolean Status true if connected successfully
  */
  function connect($persistent=NOT_PERSISTENT) {
    if (!$this->connected) {
      if ($persistent)
        $this->dbid = ociplogon($this->user, $this->passwd, $this->name);
      else
        $this->dbid = ocilogon($this->user, $this->passwd, $this->name);
      if ($this->dbid) {
        $this->connected = true;
      }
    }
    return $this->connected;
  }
  // ....................................................................
  /** Disconnect from the database, if connected. */
  function disconnect() {
    if (ocilogoff($this->dbid)) {
      $this->connected = false;
    }
  }
  // ....................................................................
  /**
  * Execute a query on the connected database.
  * @param string $sql The SQL query to execute on the database
  * @return resource A database query resource ID, or false if query failed
  */
  function query($sql) {
    $sql = $this->convert_boolean_syntax($sql);
    $this->timer->restart();
    $rid = ociparse($this->dbid, $sql);
    if ($rid !== false) {
      $didexecute = ociexecute($rid);
    }
    $this->timer->stop();
    $this->executable_sql = $sql;
    $this->rid = $rid;
    $this->query_report();
    return ($didexecute ? $rid : false);
  }
  // ....................................................................
  /**
  * Return the number of rows returned by a SELECT query.
  * @param resource $rid The resource ID for the executed query
  * @return integer The number of rows returned by the query
  */
  function numrows($rid) {
    if ($rid !== false) {
      ociexecute($rid);
    }
    $this->numrows = ocifetchstatement($rid, $this->records);
    return $this->numrows;
  }
  // ....................................................................
  /**
  * Return the number of rows affected by a query.
  * @param resource $rid The resource ID for the executed query
  * @return integer The number of rows affected by the query
  */
  function affectedrows($rid) {
    return ocirowcount($rid);
  }
  // ....................................................................
  /**
  * Free a resource.
  * @param resource $rid The resource ID for the executed query
  */
  function freeresult($rid) {
    ocifreestatement($rid);
  }
  // ....................................................................
  /**
  * Return the last error message.
  * @return string The last error message which was generated
  */
  function errormessage() {
    return ocierror($this->dbid);
  }
  // ....................................................................
  /**
  * Return the specified row, as a standard (enumerated) array of
  * field values.
  * @param resource $rid The resource ID for the executed query
  * @param integer $rowno Row number (zero-based) of row to return
  * @return array Enumerated array of field values
  */
  function fetch_row($rid, $rowno) {
    if ($rid !== false) {
      ociexecute($rid);
    }
    if (!isset($this->result)) {
      $this->result = array();
      $row = array();
      while (ocifetchinto($rid, $row, OCI_NUM+OCI_RETURN_LOBS)) {
        $this->result[] = convert_oci_array($row);
      }
    }
    return $this->result[$rowno];
  }
  // ....................................................................
  /**
  * Return the specified row, as an associative array of fields
  * in a fieldname => value format.
  * @param resource $rid The resource ID for the executed query
  * @param integer $rowno Row number (zero-based) of row to return
  * @return array Associative array of field values
  */
  function fetch_array($rid, $rowno) {
    if ($rid !== false) {
      ociexecute($rid);
    }
    if (!isset($this->result)) {
      $this->result = array();
      $row = array();
      while (ocifetchinto($rid, $row, OCI_ASSOC+OCI_RETURN_LOBS)) {
        $this->result[] = convert_oci_array($row);
      }
    }
    return $this->result[$rowno];
  }
  function lock($tablelist, $mode) {
    $res = true;
    $tables = explode(",", $tablelist);
    foreach($tables as $table) {
      $rid = $this->query("LOCK $table IN $mode");
      if ($rid === false) {
        $res = false;
      }
    }
    return $res;
  }
}

// ----------------------------------------------------------------------
// UTILITY FUNCTIONS
/**
* Function to convert the Oracle returned array format into the more
* useful associative array format we usually deal with.
* @param array $array Original Oracle formatted array of results
* @return array The converted associative array of results
* @access private
*/
function convert_oci_array($array) {
  // converts the array's key values to lower case
  $temp = array();
  foreach ($array as $k => $v ) {
    $tk = strtolower($k);
    $temp["$tk"] = $v;
  }
  return $temp;
}

// ----------------------------------------------------------------------
?>